#!/usr/bin/env python3
"""
Create test match for debugging chat functionality
"""

import asyncio
import mysql.connector
import os
from dotenv import load_dotenv

load_dotenv()

async def create_test_match():
    """Create a test match between two users"""
    
    try:
        # Database connection
        host = os.getenv('DB_HOST', 'localhost')
        port = int(os.getenv('DB_PORT', 3306))
        user = os.getenv('DB_USER', 'chatmilos_db')
        password = os.getenv('DB_PASSWORD', '')
        database = os.getenv('DB_NAME', 'chatmilos_db')
        
        print(f"🔗 Connecting to: {user}@{host}:{port}/{database}")
        
        conn = mysql.connector.connect(
            host=host,
            port=port,
            user=user,
            password=password,
            database=database
        )
        cursor = conn.cursor()
        
        # Check existing users
        cursor.execute("SELECT user_id, username, first_name FROM users LIMIT 10")
        users = cursor.fetchall()
        
        print(f"👥 Found {len(users)} users:")
        for user_id, username, first_name in users:
            name = username or first_name or "کاربر"
            print(f"   - {name} (ID: {user_id})")
        
        if len(users) < 2:
            print("❌ Need at least 2 users to create a match")
            return
        
        user_a = users[0][0]  # First user ID
        user_b = users[1][0]  # Second user ID
        
        # Check if match already exists
        cursor.execute("""
            SELECT id, status FROM matches 
            WHERE (user_a = %s AND user_b = %s) OR (user_a = %s AND user_b = %s)
        """, (user_a, user_b, user_b, user_a))
        
        existing_match = cursor.fetchone()
        
        if existing_match:
            match_id, status = existing_match
            print(f"✅ Match already exists: ID {match_id}, Status {status}")
            
            # Update status to active if needed
            if status != 1:
                cursor.execute("UPDATE matches SET status = 1 WHERE id = %s", (match_id,))
                conn.commit()
                print(f"✅ Updated match status to active")
        else:
            # Create new match
            cursor.execute("""
                INSERT INTO matches (user_a, user_b, status, created_at) 
                VALUES (%s, %s, 1, NOW())
            """, (user_a, user_b))
            conn.commit()
            match_id = cursor.lastrowid
            print(f"✅ Created new match: ID {match_id}")
        
        # Check existing messages
        cursor.execute("SELECT COUNT(*) FROM chat_messages WHERE match_id = %s", (match_id,))
        msg_count = cursor.fetchone()[0]
        
        print(f"💬 Match has {msg_count} messages")
        
        if msg_count == 0:
            # Create a test message
            cursor.execute("""
                INSERT INTO chat_messages (match_id, sender_id, content, message_type, created_at) 
                VALUES (%s, %s, %s, 'text', NOW())
            """, (match_id, user_a, "سلام! این یک پیام تست است."))
            conn.commit()
            print("✅ Created test message")
        
        print(f"\n🎉 Test setup complete!")
        print(f"   Match ID: {match_id}")
        print(f"   User A: {user_a}")
        print(f"   User B: {user_b}")
        print(f"   Status: Active (1)")
        
        cursor.close()
        conn.close()
        
    except Exception as e:
        print(f"❌ Error: {e}")

if __name__ == "__main__":
    asyncio.run(create_test_match())
