#!/usr/bin/env python3
"""
MySQL Database Initialization Script for ChatDate Bot
مناسب برای استفاده در cPanel
"""

import os
import sys
from pathlib import Path
import mysql.connector
from mysql.connector import Error
from dotenv import load_dotenv
from urllib.parse import urlparse

# Load environment variables
load_dotenv()

def parse_database_url(url):
    """Parse MySQL URL to connection parameters"""
    parsed = urlparse(url)
    return {
        'host': parsed.hostname or 'localhost',
        'port': parsed.port or 3306,
        'user': parsed.username,
        'password': parsed.password,
        'database': parsed.path.lstrip('/')
    }

def create_tables(cursor):
    """Create all necessary tables for ChatDate Bot"""
    
    # Users table
    cursor.execute("""
    CREATE TABLE IF NOT EXISTS users (
        user_id BIGINT PRIMARY KEY,
        username VARCHAR(255),
        first_name VARCHAR(255),
        last_name VARCHAR(255),
        language_code VARCHAR(10) DEFAULT 'fa',
        age INT,
        gender VARCHAR(10),
        bio TEXT,
        interests TEXT,
        looking_for VARCHAR(20),
        location_lat DECIMAL(10, 8),
        location_lon DECIMAL(11, 8),
        city VARCHAR(255),
        country VARCHAR(255),
        photos JSON,
        is_premium BOOLEAN DEFAULT FALSE,
        premium_until DATETIME,
        is_verified BOOLEAN DEFAULT FALSE,
        is_active BOOLEAN DEFAULT TRUE,
        is_banned BOOLEAN DEFAULT FALSE,
        last_active TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        settings JSON DEFAULT '{}',
        INDEX idx_location (location_lat, location_lon),
        INDEX idx_active (is_active, is_banned),
        INDEX idx_premium (is_premium, premium_until)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    """)
    
    # Matches table
    cursor.execute("""
    CREATE TABLE IF NOT EXISTS matches (
        match_id INT AUTO_INCREMENT PRIMARY KEY,
        user1_id BIGINT,
        user2_id BIGINT,
        status VARCHAR(20) DEFAULT 'pending',
        matched_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        chat_started BOOLEAN DEFAULT FALSE,
        chat_started_at DATETIME,
        last_message_at DATETIME,
        user1_liked BOOLEAN DEFAULT FALSE,
        user2_liked BOOLEAN DEFAULT FALSE,
        compatibility_score FLOAT,
        FOREIGN KEY (user1_id) REFERENCES users(user_id) ON DELETE CASCADE,
        FOREIGN KEY (user2_id) REFERENCES users(user_id) ON DELETE CASCADE,
        UNIQUE KEY unique_match (user1_id, user2_id),
        INDEX idx_status (status),
        INDEX idx_users (user1_id, user2_id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    """)
    
    # Messages table
    cursor.execute("""
    CREATE TABLE IF NOT EXISTS messages (
        message_id INT AUTO_INCREMENT PRIMARY KEY,
        match_id INT,
        sender_id BIGINT,
        receiver_id BIGINT,
        message_text TEXT,
        message_type VARCHAR(20) DEFAULT 'text',
        media_url TEXT,
        is_read BOOLEAN DEFAULT FALSE,
        read_at DATETIME,
        sent_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        expires_at DATETIME,
        is_deleted BOOLEAN DEFAULT FALSE,
        FOREIGN KEY (match_id) REFERENCES matches(match_id) ON DELETE CASCADE,
        FOREIGN KEY (sender_id) REFERENCES users(user_id) ON DELETE CASCADE,
        FOREIGN KEY (receiver_id) REFERENCES users(user_id) ON DELETE CASCADE,
        INDEX idx_match (match_id),
        INDEX idx_sender_receiver (sender_id, receiver_id),
        INDEX idx_sent_at (sent_at)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    """)
    
    # Reports table
    cursor.execute("""
    CREATE TABLE IF NOT EXISTS reports (
        report_id INT AUTO_INCREMENT PRIMARY KEY,
        reporter_id BIGINT,
        reported_id BIGINT,
        reason VARCHAR(255),
        description TEXT,
        evidence JSON,
        status VARCHAR(20) DEFAULT 'pending',
        resolved_at DATETIME,
        resolved_by BIGINT,
        action_taken TEXT,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        FOREIGN KEY (reporter_id) REFERENCES users(user_id) ON DELETE CASCADE,
        FOREIGN KEY (reported_id) REFERENCES users(user_id) ON DELETE CASCADE,
        INDEX idx_status (status),
        INDEX idx_reported (reported_id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    """)
    
    # Blocks table
    cursor.execute("""
    CREATE TABLE IF NOT EXISTS blocks (
        block_id INT AUTO_INCREMENT PRIMARY KEY,
        blocker_id BIGINT,
        blocked_id BIGINT,
        reason VARCHAR(255),
        blocked_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        expires_at DATETIME,
        FOREIGN KEY (blocker_id) REFERENCES users(user_id) ON DELETE CASCADE,
        FOREIGN KEY (blocked_id) REFERENCES users(user_id) ON DELETE CASCADE,
        UNIQUE KEY unique_block (blocker_id, blocked_id),
        INDEX idx_blocker_blocked (blocker_id, blocked_id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    """)
    
    # Likes table (for premium features)
    cursor.execute("""
    CREATE TABLE IF NOT EXISTS likes (
        like_id INT AUTO_INCREMENT PRIMARY KEY,
        liker_id BIGINT,
        liked_id BIGINT,
        is_super_like BOOLEAN DEFAULT FALSE,
        liked_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        FOREIGN KEY (liker_id) REFERENCES users(user_id) ON DELETE CASCADE,
        FOREIGN KEY (liked_id) REFERENCES users(user_id) ON DELETE CASCADE,
        UNIQUE KEY unique_like (liker_id, liked_id),
        INDEX idx_liked (liked_id),
        INDEX idx_liker (liker_id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    """)
    
    # Boosts table (for premium features)
    cursor.execute("""
    CREATE TABLE IF NOT EXISTS boosts (
        boost_id INT AUTO_INCREMENT PRIMARY KEY,
        user_id BIGINT,
        started_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        expires_at DATETIME,
        boost_type VARCHAR(20) DEFAULT 'standard',
        FOREIGN KEY (user_id) REFERENCES users(user_id) ON DELETE CASCADE,
        INDEX idx_user (user_id),
        INDEX idx_expires (expires_at)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    """)
    
    # Payments table
    cursor.execute("""
    CREATE TABLE IF NOT EXISTS payments (
        payment_id INT AUTO_INCREMENT PRIMARY KEY,
        user_id BIGINT,
        amount DECIMAL(10, 2),
        currency VARCHAR(3) DEFAULT 'USD',
        payment_method VARCHAR(50),
        transaction_id VARCHAR(255),
        status VARCHAR(20) DEFAULT 'pending',
        description TEXT,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        completed_at DATETIME,
        FOREIGN KEY (user_id) REFERENCES users(user_id) ON DELETE CASCADE,
        INDEX idx_user (user_id),
        INDEX idx_status (status),
        INDEX idx_transaction (transaction_id)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    """)
    
    # Analytics table
    cursor.execute("""
    CREATE TABLE IF NOT EXISTS analytics (
        id INT AUTO_INCREMENT PRIMARY KEY,
        user_id BIGINT,
        event_type VARCHAR(50),
        event_data JSON,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        INDEX idx_user (user_id),
        INDEX idx_event (event_type),
        INDEX idx_created (created_at)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
    """)
    
    print("✅ All tables created successfully!")

def create_indexes(cursor):
    """Create additional indexes for performance"""
    
    # Spatial index alternative for location-based queries
    # Since MySQL in cPanel might not have spatial extensions
    cursor.execute("""
    CREATE INDEX IF NOT EXISTS idx_location_search 
    ON users(is_active, is_banned, location_lat, location_lon)
    """)
    
    print("✅ Indexes created successfully!")

def insert_sample_data(cursor):
    """Insert sample admin user"""
    
    admin_id = os.getenv('ADMIN_USER_IDS', '').split(',')[0].strip()
    if admin_id:
        cursor.execute("""
        INSERT IGNORE INTO users (
            user_id, username, first_name, language_code, 
            is_verified, is_active, is_premium
        ) VALUES (%s, %s, %s, %s, %s, %s, %s)
        """, (
            int(admin_id), 
            'admin', 
            'Admin', 
            'fa',
            True, 
            True, 
            True
        ))
        print(f"✅ Admin user {admin_id} added/updated!")

def main():
    """Main function to initialize MySQL database"""
    
    # Get database URL from environment
    database_url = os.getenv('DATABASE_URL')
    
    if not database_url or not database_url.startswith('mysql://'):
        print("❌ Error: DATABASE_URL not set or not MySQL URL in .env file")
        print("Example: mysql://username:password@localhost/database_name")
        sys.exit(1)
    
    # Parse connection parameters
    try:
        conn_params = parse_database_url(database_url)
    except Exception as e:
        print(f"❌ Error parsing DATABASE_URL: {e}")
        sys.exit(1)
    
    # Connect to MySQL
    try:
        print(f"🔗 Connecting to MySQL database: {conn_params['database']}...")
        
        # First, try to create database if it doesn't exist
        conn_without_db = mysql.connector.connect(
            host=conn_params['host'],
            port=conn_params['port'],
            user=conn_params['user'],
            password=conn_params['password']
        )
        cursor_db = conn_without_db.cursor()
        cursor_db.execute(f"CREATE DATABASE IF NOT EXISTS {conn_params['database']} "
                         f"CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci")
        cursor_db.close()
        conn_without_db.close()
        
        # Now connect to the specific database
        conn = mysql.connector.connect(**conn_params)
        cursor = conn.cursor()
        
        print("✅ Connected to MySQL successfully!")
        
        # Create tables
        print("📝 Creating tables...")
        create_tables(cursor)
        
        # Create indexes
        print("📊 Creating indexes...")
        create_indexes(cursor)
        
        # Insert sample data
        print("📥 Inserting sample data...")
        insert_sample_data(cursor)
        
        # Commit changes
        conn.commit()
        
        print("\n" + "="*50)
        print("🎉 Database initialization completed successfully!")
        print("="*50)
        
    except Error as e:
        print(f"❌ MySQL Error: {e}")
        sys.exit(1)
    
    finally:
        if 'cursor' in locals():
            cursor.close()
        if 'conn' in locals():
            conn.close()

if __name__ == "__main__":
    main()
