from sqlalchemy import Column, Integer, BigInteger, String, Text, Boolean, SmallInteger, TIMESTAMP, ForeignKey, UniqueConstraint
from sqlalchemy.ext.declarative import declarative_base
from sqlalchemy.orm import relationship
from sqlalchemy.sql import func

Base = declarative_base()

class User(Base):
    __tablename__ = 'users'
    
    user_id = Column(BigInteger, primary_key=True)
    username = Column(String(255))
    first_name = Column(String(255))
    last_name = Column(String(255))
    language_code = Column(String(10), default='fa')
    age = Column(Integer)
    gender = Column(String(10))
    bio = Column(Text)
    interests = Column(Text)
    looking_for = Column(String(20))
    location_lat = Column(String(20))
    location_lon = Column(String(20))
    city = Column(String(255))
    country = Column(String(255))
    photos = Column(Text)
    is_premium = Column(Boolean, default=False)
    premium_until = Column(String(50))
    is_verified = Column(Boolean, default=False)
    is_active = Column(Boolean, default=True)
    is_banned = Column(Boolean, default=False)
    last_active = Column(TIMESTAMP)
    created_at = Column(TIMESTAMP)
    settings = Column(Text)

class Preference(Base):
    __tablename__ = 'preferences'
    
    user_id = Column(BigInteger, ForeignKey('users.user_id'), primary_key=True)
    age_min = Column(SmallInteger)
    age_max = Column(SmallInteger)
    gender_pref = Column(SmallInteger)
    radius_km = Column(SmallInteger)
    country_code = Column(String(3))
    city = Column(String(100))
    mood = Column(String(20))

class Match(Base):
    __tablename__ = 'matches'
    
    id = Column(BigInteger, primary_key=True)
    user_a = Column(BigInteger, ForeignKey('users.user_id'))
    user_b = Column(BigInteger, ForeignKey('users.user_id'))
    status = Column(Integer, default=0)
    last_message_at = Column(TIMESTAMP)
    last_message_by = Column(BigInteger)
    unread_count_a = Column(Integer, default=0)
    unread_count_b = Column(Integer, default=0)
    created_at = Column(TIMESTAMP)
    updated_at = Column(TIMESTAMP)
    ended_at = Column(TIMESTAMP)
    ended_by = Column(BigInteger)

class ChatMessage(Base):
    __tablename__ = 'chat_messages'
    
    id = Column(BigInteger, primary_key=True, autoincrement=True)
    match_id = Column(BigInteger, ForeignKey('matches.id'), nullable=False)
    sender_id = Column(BigInteger, ForeignKey('users.user_id'), nullable=False)
    message_type = Column(String(20), default='text')
    content = Column(Text)
    is_read = Column(Boolean, default=False)
    created_at = Column(TIMESTAMP, default=func.now())

class Swipe(Base):
    __tablename__ = 'swipes'
    
    id = Column(BigInteger, primary_key=True)
    from_user = Column(BigInteger, ForeignKey('users.user_id'))
    to_user = Column(BigInteger, ForeignKey('users.user_id'))
    action = Column(SmallInteger)
    created_at = Column(TIMESTAMP)

class Report(Base):
    __tablename__ = 'reports'
    
    id = Column(BigInteger, primary_key=True)
    reporter = Column(BigInteger, ForeignKey('users.user_id'))
    target = Column(BigInteger, ForeignKey('users.user_id'))
    reason = Column(SmallInteger)
    notes = Column(Text)
    status = Column(SmallInteger)
    created_at = Column(TIMESTAMP)

class Payment(Base):
    __tablename__ = 'payments'
    
    id = Column(BigInteger, primary_key=True)
    user_id = Column(BigInteger, ForeignKey('users.user_id'))
    plan = Column(String(20))
    amount_cents = Column(Integer)
    currency = Column(String(3))
    provider = Column(String(20))
    provider_ref = Column(String(100))
    status = Column(SmallInteger)
    created_at = Column(TIMESTAMP)

class Event(Base):
    __tablename__ = 'events'
    
    id = Column(BigInteger, primary_key=True)
    user_id = Column(BigInteger)
    event_type = Column(String(50))
    event_data = Column(Text)
    created_at = Column(TIMESTAMP)

class Notification(Base):
    __tablename__ = 'notifications'
    
    id = Column(BigInteger, primary_key=True)
    user_id = Column(BigInteger, ForeignKey('users.user_id'))
    type = Column(String(50))
    title = Column(String(255))
    message = Column(Text)
    related_user_id = Column(BigInteger)
    related_match_id = Column(BigInteger)
    is_read = Column(Boolean, default=False)
    created_at = Column(TIMESTAMP)

class ProfileView(Base):
    __tablename__ = 'profile_views'
    
    id = Column(BigInteger, primary_key=True)
    viewer_id = Column(BigInteger, ForeignKey('users.user_id'))
    viewed_user_id = Column(BigInteger, ForeignKey('users.user_id'))
    created_at = Column(TIMESTAMP)

class UserSession(Base):
    __tablename__ = 'user_sessions'
    
    id = Column(BigInteger, primary_key=True)
    user_id = Column(BigInteger)
    session_token = Column(String(255))
    created_at = Column(TIMESTAMP)
    expires_at = Column(TIMESTAMP)
