#!/usr/bin/env python3
"""Real database models for ChatMilos Bot"""

from sqlalchemy import Column, BigInteger, String, Text, Boolean, TIMESTAMP, Enum, Integer, ForeignKey, UniqueConstraint, Index
from sqlalchemy.sql import func
from sqlalchemy.orm import relationship
from database.models import Base

class ChatMessage(Base):
    __tablename__ = 'chat_messages'
    __table_args__ = {'extend_existing': True}
    
    id = Column(BigInteger, primary_key=True, autoincrement=True)
    match_id = Column(BigInteger, ForeignKey('matches.id'), nullable=False)
    sender_id = Column(BigInteger, ForeignKey('users.user_id'), nullable=False)
    receiver_id = Column(BigInteger, ForeignKey('users.user_id'), nullable=False)
    message_type = Column(Enum('text', 'photo', 'voice', 'sticker', 'document'), default='text')
    content = Column(Text)
    file_id = Column(String(255))  # برای فایل‌های تلگرام
    is_read = Column(Boolean, default=False)
    is_deleted = Column(Boolean, default=False)
    created_at = Column(TIMESTAMP, default=func.now())
    updated_at = Column(TIMESTAMP, default=func.now(), onupdate=func.now())
    
    # Relationships (simplified - no back_populates)
    # match = relationship("Match")  # Commented out to avoid circular reference
    # sender = relationship("User", foreign_keys=[sender_id])
    # receiver = relationship("User", foreign_keys=[receiver_id])
    
    # Indexes
    __table_args__ = (
        Index('idx_match_messages', 'match_id', 'created_at'),
        Index('idx_user_messages', 'sender_id', 'created_at'),
        Index('idx_unread_messages', 'receiver_id', 'is_read', 'created_at'),
        {'extend_existing': True}
    )

class Swipe(Base):
    __tablename__ = 'swipes'
    __table_args__ = {'extend_existing': True}
    
    id = Column(BigInteger, primary_key=True, autoincrement=True)
    from_user = Column(BigInteger, ForeignKey('users.user_id'), nullable=False)
    to_user = Column(BigInteger, ForeignKey('users.user_id'), nullable=False)
    action = Column(Enum('like', 'pass', 'super_like'), nullable=False)
    created_at = Column(TIMESTAMP, default=func.now())
    
    # Relationships (simplified)
    # swiper = relationship("User", foreign_keys=[from_user])
    # swiped = relationship("User", foreign_keys=[to_user])
    
    # Constraints
    __table_args__ = (
        UniqueConstraint('from_user', 'to_user', name='unique_swipe'),
        Index('idx_user_swipes', 'from_user', 'created_at'),
        Index('idx_received_swipes', 'to_user', 'action', 'created_at'),
        {'extend_existing': True}
    )

class Notification(Base):
    __tablename__ = 'notifications'
    __table_args__ = {'extend_existing': True}
    
    id = Column(BigInteger, primary_key=True, autoincrement=True)
    user_id = Column(BigInteger, ForeignKey('users.user_id'), nullable=False)
    type = Column(Enum('match', 'like', 'message', 'profile_view', 'premium_expiry'), nullable=False)
    title = Column(String(255), nullable=False)
    message = Column(Text, nullable=False)
    related_user_id = Column(BigInteger, ForeignKey('users.user_id'), nullable=True)
    related_match_id = Column(BigInteger, ForeignKey('matches.id'), nullable=True)
    is_read = Column(Boolean, default=False)
    created_at = Column(TIMESTAMP, default=func.now())
    
    # Relationships (simplified)
    # user = relationship("User", foreign_keys=[user_id])
    # related_user = relationship("User", foreign_keys=[related_user_id])
    # related_match = relationship("Match", foreign_keys=[related_match_id])
    
    # Indexes
    __table_args__ = (
        Index('idx_user_notifications', 'user_id', 'is_read', 'created_at'),
        Index('idx_notification_type', 'user_id', 'type', 'created_at'),
        {'extend_existing': True}
    )

class ProfileView(Base):
    __tablename__ = 'profile_views'
    __table_args__ = {'extend_existing': True}
    
    id = Column(BigInteger, primary_key=True, autoincrement=True)
    viewer_id = Column(BigInteger, ForeignKey('users.user_id'), nullable=False)
    viewed_user_id = Column(BigInteger, ForeignKey('users.user_id'), nullable=False)
    created_at = Column(TIMESTAMP, default=func.now())
    
    # Relationships (simplified)
    # viewer = relationship("User", foreign_keys=[viewer_id])
    # viewed_user = relationship("User", foreign_keys=[viewed_user_id])
    
    # Constraints
    __table_args__ = (
        Index('idx_profile_views', 'viewed_user_id', 'created_at'),
        Index('idx_viewer_history', 'viewer_id', 'created_at'),
        {'extend_existing': True}
    )

# Update Match model to include message relationships
# This would be added to the existing Match class in models.py:
"""
# Add to Match class:
last_message_at = Column(TIMESTAMP)
last_message_by = Column(BigInteger)
unread_count_a = Column(Integer, default=0)
unread_count_b = Column(Integer, default=0)

# Relationship
messages = relationship("ChatMessage", back_populates="match", cascade="all, delete-orphan")
"""
