#!/usr/bin/env python3
"""
Direct fix for ChatMessage table - No confirmation needed
"""

import mysql.connector
import os
from dotenv import load_dotenv
from urllib.parse import urlparse

load_dotenv()

print("🔧 Direct ChatMessage Table Fix")
print("="*40)

try:
    # Get database connection info
    db_url = os.getenv('DATABASE_URL')
    if not db_url:
        print("❌ DATABASE_URL not found")
        exit(1)
    
    print(f"🔗 Database URL: {db_url[:50]}...")
    
    # Try to parse URL, if fails use environment variables
    try:
        parsed = urlparse(db_url)
        host = parsed.hostname or 'localhost'
        port = parsed.port or 3306
        user = parsed.username
        password = parsed.password
        database = parsed.path.lstrip('/')
    except:
        # Fallback to individual environment variables
        host = os.getenv('DB_HOST', 'localhost')
        port = int(os.getenv('DB_PORT', 3306))
        user = os.getenv('DB_USER', 'chatmilos_db')
        password = os.getenv('DB_PASSWORD', '')
        database = os.getenv('DB_NAME', 'chatmilos_db')
    
    print(f"🔗 Connecting to: {user}@{host}:{port}/{database}")
    
    # Connect to database
    conn = mysql.connector.connect(
        host=host,
        port=port,
        user=user,
        password=password,
        database=database
    )
    
    cursor = conn.cursor()
    print("✅ Connected to database")
    
    # Drop and recreate table
    print("🔧 Recreating chat_messages table...")
    
    cursor.execute("DROP TABLE IF EXISTS chat_messages")
    print("✅ Dropped old table")
    
    cursor.execute("""
        CREATE TABLE chat_messages (
            id BIGINT AUTO_INCREMENT PRIMARY KEY,
            match_id BIGINT NOT NULL,
            sender_id BIGINT NOT NULL,
            message_type VARCHAR(20) DEFAULT 'text',
            content TEXT,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            INDEX idx_match_id (match_id),
            INDEX idx_sender_id (sender_id)
        )
    """)
    print("✅ Created new table")
    
    # Verify structure
    cursor.execute("DESCRIBE chat_messages")
    columns = cursor.fetchall()
    
    print("\n📋 Table structure:")
    for col in columns:
        print(f"  ✅ {col[0]} ({col[1]})")
    
    # Test insert
    cursor.execute("""
        INSERT INTO chat_messages (match_id, sender_id, message_type, content) 
        VALUES (999, 123456789, 'text', 'Test message')
    """)
    print("✅ Test insert successful")
    
    # Clean up test
    cursor.execute("DELETE FROM chat_messages WHERE match_id = 999")
    print("✅ Test data cleaned")
    
    # Commit
    conn.commit()
    print("💾 Changes committed")
    
    cursor.close()
    conn.close()
    
    print("\n🎉 TABLE FIXED SUCCESSFULLY!")
    print("🔄 Now restart the bot:")
    print("nohup python3 main.py > bot.log 2>&1 &")
    
except Exception as e:
    print(f"❌ Error: {e}")
    import traceback
    traceback.print_exc()
