"""
Admin panel handler for ChatDate bot
"""

from aiogram import Router, F
from aiogram.types import Message, CallbackQuery, InlineKeyboardMarkup, InlineKeyboardButton
from aiogram.filters import Command
from aiogram.fsm.context import FSMContext
from aiogram.fsm.state import State, StatesGroup
from datetime import datetime, timedelta
from typing import List
import os
from sqlalchemy import select, func, and_, or_, delete

from database.database import db
from database.models import User, Match, Report, Swipe
from localization.translations import get_text

router = Router()

# Admin user IDs from environment
ADMIN_USER_IDS = [int(uid) for uid in os.getenv('ADMIN_USER_IDS', '').split(',') if uid]

class AdminStates(StatesGroup):
    broadcast_message = State()
    ban_user = State()
    grant_premium = State()
    search_user = State()

def is_admin(user_id: int) -> bool:
    """Check if user is admin"""
    return user_id in ADMIN_USER_IDS

async def admin_only(message: Message) -> bool:
    """Filter for admin-only commands"""
    if not is_admin(message.from_user.id):
        await message.answer("⛔ Access denied. Admin only.")
        return False
    return True

@router.message(Command("admin"))
@router.message(F.text == "🔐 پنل ادمین")
async def admin_panel(message: Message):
    """Show admin panel"""
    if not await admin_only(message):
        return
    
    keyboard = InlineKeyboardMarkup(inline_keyboard=[
        [InlineKeyboardButton(text="📊 Statistics", callback_data="admin_stats")],
        [InlineKeyboardButton(text="👥 User Management", callback_data="admin_users")],
        [InlineKeyboardButton(text="🚨 Reports", callback_data="admin_reports")],
        [InlineKeyboardButton(text="📢 Broadcast", callback_data="admin_broadcast")],
        [InlineKeyboardButton(text="🔧 System", callback_data="admin_system")],
        [InlineKeyboardButton(text="❌ Close", callback_data="cancel")]
    ])
    
    await message.answer(
        "🔐 **Admin Panel**\n\n"
        "Select an option:",
        reply_markup=keyboard,
        parse_mode="Markdown"
    )

@router.callback_query(F.data == "admin_stats")
async def show_statistics(callback: CallbackQuery):
    """Show bot statistics"""
    if not is_admin(callback.from_user.id):
        await callback.answer("Access denied", show_alert=True)
        return
    
    async with db.get_session() as session:
        # User statistics
        total_users = await session.scalar(select(func.count(User.id)))
        active_users = await session.scalar(
            select(func.count(User.id)).where(User.is_active == True)
        )
        premium_users = await session.scalar(
            select(func.count(User.id)).where(User.is_premium == True)
        )
        banned_users = await session.scalar(
            select(func.count(User.id)).where(User.is_banned == True)
        )
        
        # Today's statistics
        today = datetime.utcnow().date()
        new_users_today = await session.scalar(
            select(func.count(User.id)).where(
                func.date(User.created_at) == today
            )
        )
        
        # Match statistics
        total_matches = await session.scalar(select(func.count(Match.id)))
        active_matches = await session.scalar(
            select(func.count(Match.id)).where(Match.status == 'active')
        )
        
        # Language distribution
        lang_stats = await session.execute(
            select(User.language, func.count(User.id))
            .group_by(User.language)
            .order_by(func.count(User.id).desc())
        )
        lang_dist = "\n".join([f"  {lang}: {count}" for lang, count in lang_stats])
        
        stats_text = f"""📊 **Bot Statistics**

👥 **Users:**
• Total: {total_users}
• Active: {active_users}
• Premium: {premium_users}
• Banned: {banned_users}
• New Today: {new_users_today}

💬 **Matches:**
• Total: {total_matches}
• Active Chats: {active_matches}

🌍 **Languages:**
{lang_dist}

📅 Last Update: {datetime.utcnow().strftime('%Y-%m-%d %H:%M UTC')}"""
        
        keyboard = InlineKeyboardMarkup(inline_keyboard=[
            [InlineKeyboardButton(text="🔄 Refresh", callback_data="admin_stats")],
            [InlineKeyboardButton(text="📈 Detailed Stats", callback_data="admin_stats_detailed")],
            [InlineKeyboardButton(text="🔙 Back", callback_data="admin_menu")]
        ])
        
        await callback.message.edit_text(stats_text, reply_markup=keyboard, parse_mode="Markdown")
        await callback.answer()

@router.message(F.text == "📊 آمار سریع")
async def quick_stats(message: Message):
    """Show quick statistics for admin"""
    if not is_admin(message.from_user.id):
        await message.answer("⛔ Access denied. Admin only.")
        return
    
    async with db.get_session() as session:
        # User statistics
        total_users = await session.scalar(select(func.count(User.id)))
        active_users = await session.scalar(
            select(func.count(User.id)).where(User.is_active == True)
        )
        premium_users = await session.scalar(
            select(func.count(User.id)).where(User.is_premium == True)
        )
        
        # Today's statistics
        today = datetime.utcnow().date()
        new_users_today = await session.scalar(
            select(func.count(User.id)).where(
                func.date(User.created_at) == today
            )
        )
        
        # Match statistics
        total_matches = await session.scalar(select(func.count(Match.id)))
        active_matches = await session.scalar(
            select(func.count(Match.id)).where(Match.status == 'active')
        )
        
        stats_text = f"""📊 **آمار سریع**

👥 **کاربران:**
• کل: {total_users}
• فعال: {active_users}
• پریمیوم: {premium_users}
• جدید امروز: {new_users_today}

💬 **مچ‌ها:**
• کل: {total_matches}
• گفتگوهای فعال: {active_matches}

📅 به‌روزرسانی: {datetime.utcnow().strftime('%H:%M UTC')}"""
        
        await message.answer(stats_text, parse_mode="Markdown")

@router.message(F.text == "👥 مدیریت کاربران")
async def user_management_text(message: Message):
    """Show user management menu via text"""
    if not is_admin(message.from_user.id):
        await message.answer("⛔ Access denied. Admin only.")
        return
    
    keyboard = InlineKeyboardMarkup(inline_keyboard=[
        [InlineKeyboardButton(text="🔍 جستجوی کاربر", callback_data="admin_search_user")],
        [InlineKeyboardButton(text="🚫 مسدود کردن", callback_data="admin_ban_user")],
        [InlineKeyboardButton(text="✅ رفع مسدودی", callback_data="admin_unban_user")],
        [InlineKeyboardButton(text="⭐ اعطای پریمیوم", callback_data="admin_grant_premium")],
        [InlineKeyboardButton(text="📋 کاربران اخیر", callback_data="admin_recent_users")],
        [InlineKeyboardButton(text="❌ بستن", callback_data="cancel")]
    ])
    
    await message.answer(
        "👥 **مدیریت کاربران**\n\nیک گزینه را انتخاب کنید:",
        reply_markup=keyboard,
        parse_mode="Markdown"
    )

@router.message(F.text == "🚨 گزارشات")
async def reports_text(message: Message):
    """Show reports via text"""
    if not is_admin(message.from_user.id):
        await message.answer("⛔ Access denied. Admin only.")
        return
    
    async with db.get_session() as session:
        reports = await session.execute(
            select(Report)
            .order_by(Report.created_at.desc())
            .limit(5)
        )
        reports = reports.scalars().all()
        
        if not reports:
            await message.answer("📋 گزارشی یافت نشد")
            return
        
        text = "🚨 **گزارشات اخیر**\n\n"
        for report in reports:
            reporter = await session.get(User, report.reporter_id)
            reported = await session.get(User, report.reported_id)
            
            text += f"📝 گزارش #{report.id}\n"
            text += f"  از: {reporter.name if reporter else 'Unknown'} ({report.reporter_id})\n"
            text += f"  درباره: {reported.name if reported else 'Unknown'} ({report.reported_id})\n"
            text += f"  دلیل: {report.reason}\n"
            text += f"  تاریخ: {report.created_at.strftime('%Y-%m-%d %H:%M')}\n\n"
        
        await message.answer(text, parse_mode="Markdown")

@router.message(F.text == "📢 پیام همگانی")
async def broadcast_text(message: Message, state: FSMContext):
    """Start broadcast via text"""
    if not is_admin(message.from_user.id):
        await message.answer("⛔ Access denied. Admin only.")
        return
    
    await message.answer(
        "📢 **پیام همگانی**\n\n"
        "پیامی که می‌خواهید برای همه کاربران ارسال شود را بنویسید.\n"
        "برای لغو /cancel را بزنید.",
        parse_mode="Markdown"
    )
    await state.set_state(AdminStates.broadcast_message)

@router.message(F.text == "🔧 تنظیمات سیستم")
async def system_settings_text(message: Message):
    """Show system settings via text"""
    if not is_admin(message.from_user.id):
        await message.answer("⛔ Access denied. Admin only.")
        return
    
    keyboard = InlineKeyboardMarkup(inline_keyboard=[
        [InlineKeyboardButton(text="🗑️ پاکسازی دیتا", callback_data="admin_clean_data")],
        [InlineKeyboardButton(text="🔄 ریست مچ‌ها", callback_data="admin_reset_matches")],
        [InlineKeyboardButton(text="📊 اطلاعات دیتابیس", callback_data="admin_db_info")],
        [InlineKeyboardButton(text="⚙️ تنظیمات بات", callback_data="admin_settings")],
        [InlineKeyboardButton(text="❌ بستن", callback_data="cancel")]
    ])
    
    await message.answer(
        "🔧 **تنظیمات سیستم**\n\nیک گزینه را انتخاب کنید:",
        reply_markup=keyboard,
        parse_mode="Markdown"
    )

@router.message(F.text == "🧪 حالت تست")
async def test_mode(message: Message):
    """Enable test mode for admin"""
    if not is_admin(message.from_user.id):
        await message.answer("⛔ Access denied. Admin only.")
        return
    
    await message.answer(
        "🧪 **حالت تست**\n\n"
        "در این حالت می‌توانید:\n"
        "• با /start به عنوان کاربر عادی ثبت‌نام کنید\n"
        "• تمام قابلیت‌های کاربری را تست کنید\n"
        "• پروفایل تستی ایجاد کنید\n\n"
        "برای بازگشت به حالت ادمین /admin را بزنید.",
        parse_mode="Markdown"
    )

@router.callback_query(F.data == "admin_users")
async def user_management_menu(callback: CallbackQuery):
    """Show user management menu"""
    if not is_admin(callback.from_user.id):
        await callback.answer("Access denied", show_alert=True)
        return
    
    keyboard = InlineKeyboardMarkup(inline_keyboard=[
        [InlineKeyboardButton(text="🔍 Search User", callback_data="admin_search_user")],
        [InlineKeyboardButton(text="🚫 Ban User", callback_data="admin_ban_user")],
        [InlineKeyboardButton(text="✅ Unban User", callback_data="admin_unban_user")],
        [InlineKeyboardButton(text="⭐ Grant Premium", callback_data="admin_grant_premium")],
        [InlineKeyboardButton(text="📋 Recent Users", callback_data="admin_recent_users")],
        [InlineKeyboardButton(text="🔙 Back", callback_data="admin_menu")]
    ])
    
    await callback.message.edit_text(
        "👥 **User Management**\n\nSelect an action:",
        reply_markup=keyboard,
        parse_mode="Markdown"
    )
    await callback.answer()

@router.callback_query(F.data == "admin_reports")
async def show_reports(callback: CallbackQuery):
    """Show recent reports"""
    if not is_admin(callback.from_user.id):
        await callback.answer("Access denied", show_alert=True)
        return
    
    async with db.get_session() as session:
        reports = await session.execute(
            select(Report)
            .order_by(Report.created_at.desc())
            .limit(10)
        )
        reports = reports.scalars().all()
        
        if not reports:
            await callback.message.edit_text(
                "📋 No reports found",
                reply_markup=InlineKeyboardMarkup(inline_keyboard=[
                    [InlineKeyboardButton(text="🔙 Back", callback_data="admin_menu")]
                ])
            )
            return
        
        text = "🚨 **Recent Reports**\n\n"
        for report in reports:
            reporter = await session.get(User, report.reporter_id)
            reported = await session.get(User, report.reported_id)
            
            text += f"📝 Report #{report.id}\n"
            text += f"  From: {reporter.name if reporter else 'Unknown'} ({report.reporter_id})\n"
            text += f"  About: {reported.name if reported else 'Unknown'} ({report.reported_id})\n"
            text += f"  Reason: {report.reason}\n"
            text += f"  Date: {report.created_at.strftime('%Y-%m-%d %H:%M')}\n\n"
        
        keyboard = InlineKeyboardMarkup(inline_keyboard=[
            [InlineKeyboardButton(text="🔄 Refresh", callback_data="admin_reports")],
            [InlineKeyboardButton(text="⚡ Take Action", callback_data="admin_report_action")],
            [InlineKeyboardButton(text="🔙 Back", callback_data="admin_menu")]
        ])
        
        await callback.message.edit_text(text, reply_markup=keyboard, parse_mode="Markdown")
        await callback.answer()

@router.callback_query(F.data == "admin_broadcast")
async def broadcast_menu(callback: CallbackQuery, state: FSMContext):
    """Start broadcast message"""
    if not is_admin(callback.from_user.id):
        await callback.answer("Access denied", show_alert=True)
        return
    
    await callback.message.edit_text(
        "📢 **Broadcast Message**\n\n"
        "Send me the message you want to broadcast to all users.\n"
        "Use /cancel to cancel.",
        parse_mode="Markdown"
    )
    await state.set_state(AdminStates.broadcast_message)
    await callback.answer()

@router.message(AdminStates.broadcast_message)
async def process_broadcast(message: Message, state: FSMContext):
    """Process broadcast message"""
    if not is_admin(message.from_user.id):
        return
    
    if message.text == "/cancel":
        await state.clear()
        await message.answer("Broadcast cancelled")
        return
    
    broadcast_text = message.text
    
    # Confirmation
    keyboard = InlineKeyboardMarkup(inline_keyboard=[
        [InlineKeyboardButton(text="✅ Send", callback_data="confirm_broadcast")],
        [InlineKeyboardButton(text="❌ Cancel", callback_data="cancel_broadcast")]
    ])
    
    await state.update_data(broadcast_text=broadcast_text)
    
    async with db.get_session() as session:
        active_users = await session.scalar(
            select(func.count(User.id)).where(User.is_active == True)
        )
    
    await message.answer(
        f"📢 **Confirm Broadcast**\n\n"
        f"Message will be sent to {active_users} active users.\n\n"
        f"**Message:**\n{broadcast_text}",
        reply_markup=keyboard,
        parse_mode="Markdown"
    )

@router.callback_query(F.data == "confirm_broadcast")
async def confirm_broadcast(callback: CallbackQuery, state: FSMContext):
    """Send broadcast to all users"""
    if not is_admin(callback.from_user.id):
        await callback.answer("Access denied", show_alert=True)
        return
    
    data = await state.get_data()
    broadcast_text = data.get('broadcast_text')
    
    if not broadcast_text:
        await callback.answer("No message to send")
        return
    
    await callback.message.edit_text("📤 Sending broadcast...")
    
    async with db.get_session() as session:
        users = await session.execute(
            select(User).where(User.is_active == True)
        )
        users = users.scalars().all()
        
        sent = 0
        failed = 0
        
        for user in users:
            try:
                # In production, use bot.send_message
                # await bot.send_message(user.telegram_id, broadcast_text)
                sent += 1
            except Exception:
                failed += 1
    
    await callback.message.edit_text(
        f"✅ **Broadcast Complete**\n\n"
        f"Sent: {sent}\n"
        f"Failed: {failed}",
        parse_mode="Markdown"
    )
    await state.clear()
    await callback.answer()

@router.callback_query(F.data == "admin_ban_user")
async def ban_user_start(callback: CallbackQuery, state: FSMContext):
    """Start ban user process"""
    if not is_admin(callback.from_user.id):
        await callback.answer("Access denied", show_alert=True)
        return
    
    await callback.message.edit_text(
        "🚫 **Ban User**\n\n"
        "Send me the Telegram ID of the user to ban.\n"
        "Use /cancel to cancel.",
        parse_mode="Markdown"
    )
    await state.set_state(AdminStates.ban_user)
    await callback.answer()

@router.message(AdminStates.ban_user)
async def process_ban_user(message: Message, state: FSMContext):
    """Process user ban"""
    if not is_admin(message.from_user.id):
        return
    
    if message.text == "/cancel":
        await state.clear()
        await message.answer("Ban cancelled")
        return
    
    try:
        user_id = int(message.text)
    except ValueError:
        await message.answer("Invalid user ID. Please send a number.")
        return
    
    async with db.get_session() as session:
        user = await session.execute(
            select(User).where(User.telegram_id == user_id)
        )
        user = user.scalar_one_or_none()
        
        if not user:
            await message.answer("User not found")
            return
        
        user.is_banned = True
        user.ban_reason = "Admin ban"
        user.banned_at = datetime.utcnow()
        await session.commit()
        
        await message.answer(
            f"✅ User {user.name} ({user_id}) has been banned",
            parse_mode="Markdown"
        )
    
    await state.clear()

@router.callback_query(F.data == "admin_system")
async def system_menu(callback: CallbackQuery):
    """Show system management menu"""
    if not is_admin(callback.from_user.id):
        await callback.answer("Access denied", show_alert=True)
        return
    
    keyboard = InlineKeyboardMarkup(inline_keyboard=[
        [InlineKeyboardButton(text="🗑️ Clean Old Data", callback_data="admin_clean_data")],
        [InlineKeyboardButton(text="🔄 Reset Matches", callback_data="admin_reset_matches")],
        [InlineKeyboardButton(text="📊 Database Info", callback_data="admin_db_info")],
        [InlineKeyboardButton(text="⚙️ Bot Settings", callback_data="admin_settings")],
        [InlineKeyboardButton(text="🔙 Back", callback_data="admin_menu")]
    ])
    
    await callback.message.edit_text(
        "🔧 **System Management**\n\nSelect an option:",
        reply_markup=keyboard,
        parse_mode="Markdown"
    )
    await callback.answer()

@router.callback_query(F.data == "admin_clean_data")
async def clean_old_data(callback: CallbackQuery):
    """Clean old data from database"""
    if not is_admin(callback.from_user.id):
        await callback.answer("Access denied", show_alert=True)
        return
    
    await callback.message.edit_text("🗑️ Cleaning old data...")
    
    async with db.get_session() as session:
        # Delete old ended matches (older than 30 days)
        old_matches = await session.execute(
            select(Match).where(
                and_(
                    Match.status == 'ended',
                    Match.updated_at < datetime.utcnow() - timedelta(days=30)
                )
            )
        )
        old_matches_count = len(old_matches.scalars().all())
        
        await session.execute(
            delete(Match).where(
                and_(
                    Match.status == 'ended',
                    Match.updated_at < datetime.utcnow() - timedelta(days=30)
                )
            )
        )
        
        # Delete old swipes (older than 30 days)
        await session.execute(
            delete(Swipe).where(
                Swipe.created_at < datetime.utcnow() - timedelta(days=30)
            )
        )
        
        await session.commit()
        
        await callback.message.edit_text(
            f"✅ **Cleanup Complete**\n\n"
            f"Deleted {old_matches_count} old matches\n"
            f"Cleaned old swipes",
            parse_mode="Markdown"
        )
    
    await callback.answer()

@router.callback_query(F.data == "admin_menu")
async def back_to_admin_menu(callback: CallbackQuery):
    """Return to admin menu"""
    if not is_admin(callback.from_user.id):
        await callback.answer("Access denied", show_alert=True)
        return
    
    keyboard = InlineKeyboardMarkup(inline_keyboard=[
        [InlineKeyboardButton(text="📊 Statistics", callback_data="admin_stats")],
        [InlineKeyboardButton(text="👥 User Management", callback_data="admin_users")],
        [InlineKeyboardButton(text="🚨 Reports", callback_data="admin_reports")],
        [InlineKeyboardButton(text="📢 Broadcast", callback_data="admin_broadcast")],
        [InlineKeyboardButton(text="🔧 System", callback_data="admin_system")],
        [InlineKeyboardButton(text="❌ Close", callback_data="cancel")]
    ])
    
    await callback.message.edit_text(
        "🔐 **Admin Panel**\n\n"
        "Select an option:",
        reply_markup=keyboard,
        parse_mode="Markdown"
    )
    await callback.answer()

@router.callback_query(F.data == "cancel")
async def cancel_action(callback: CallbackQuery, state: FSMContext):
    """Cancel current action"""
    await state.clear()
    await callback.message.delete()
    await callback.answer("Cancelled")
