from aiogram import Router, F
from aiogram.types import Message, ReplyKeyboardMarkup, KeyboardButton
from aiogram.fsm.context import FSMContext
from aiogram.fsm.state import State, StatesGroup
from sqlalchemy import select, and_, or_, func
from database.database import db
from database.models import User, Swipe, Match
from localization.translations import get_text
from utils.matching import matching_engine
import logging
import json

router = Router()
logger = logging.getLogger(__name__)

class DiscoveryStates(StatesGroup):
    browsing = State()

@router.message(F.text == "🌟 اکتشاف")
async def discovery_mode(message: Message, state: FSMContext):
    """Enter discovery/swipe mode"""
    user_id = message.from_user.id
    
    async with db.get_session() as session:
        # Get user
        result = await session.execute(select(User).where(User.user_id == user_id))
        user = result.scalar_one_or_none()
        
        if not user:
            await message.answer("لطفاً ابتدا ثبت نام کنید: /start")
            return
        
        lang = user.language_code or 'en'
        
        # Get discovery candidates
        candidates = await get_discovery_candidates(session, user.user_id)
        
        if not candidates:
            await message.answer(
                "🌟 کاربران جدیدی برای اکتشاف پیدا نشد!\n\n"
                "ممکن است:\n"
                "• همه کاربران منطقه را دیده باشید\n"
                "• فیلترهای شما خیلی محدود باشد\n\n"
                "بعداً دوباره تلاش کنید."
            )
            return
        
        # Show first candidate
        candidate = candidates[0]
        card_text = create_discovery_card(candidate)
        
        await state.update_data(
            candidates=[c.user_id for c in candidates],
            current_index=0,
            current_candidate=candidate.user_id
        )
        
        
        # کیبورد کاستوم برای اکتشاف
        keyboard = ReplyKeyboardMarkup(
            keyboard=[
                [
                    KeyboardButton(text="💚 لایک"),
                    KeyboardButton(text="❌ رد کردن")
                ],
                [
                    KeyboardButton(text="⭐ سوپر لایک"),
                    KeyboardButton(text="💬 شروع گفتگو")
                ],
                [
                    KeyboardButton(text="➡️ مچ بعدی")
                ],
                [
                    KeyboardButton(text="🔙 بازگشت به منو")
                ]
            ],
            resize_keyboard=True,
            one_time_keyboard=False
        )
        
        # ارسال عکس پروفایل اگر موجود باشد
        profile_photo = None
        if candidate.photos:
            try:
                import json
                photos_list = json.loads(candidate.photos)
                if photos_list and len(photos_list) > 0:
                    profile_photo = photos_list[0]  # اولین عکس
            except:
                profile_photo = None
        
        if profile_photo:
            try:
                await message.answer_photo(
                    photo=profile_photo,
                    caption=f"🌟 <b>حالت اکتشاف</b>\n\n{card_text}",
                    reply_markup=keyboard,
                    parse_mode='HTML'
                )
            except:
                # اگر عکس نتوانست ارسال شود، فقط متن ارسال کن
                await message.answer(
                    f"🌟 <b>حالت اکتشاف</b>\n\n{card_text}",
                    reply_markup=keyboard,
                    parse_mode='HTML'
                )
        else:
            await message.answer(
                f"🌟 <b>حالت اکتشاف</b>\n\n{card_text}",
                reply_markup=keyboard,
                parse_mode='HTML'
            )
        
        await state.set_state(DiscoveryStates.browsing)

# Message handlers برای دکمه‌های کیبورد کاستوم
@router.message(F.text == "💚 لایک")
async def discovery_like_message(message: Message, state: FSMContext):
    """لایک از طریق message"""
    await handle_discovery_action_message(message, state, action=1)

@router.message(F.text == "❌ رد کردن")
async def discovery_reject_message(message: Message, state: FSMContext):
    """رد کردن از طریق message"""
    await handle_discovery_action_message(message, state, action=-1)

@router.message(F.text == "⭐ سوپر لایک")
async def discovery_super_message(message: Message, state: FSMContext):
    """سوپر لایک از طریق message"""
    await handle_discovery_action_message(message, state, action=2)

@router.message(F.text == "💬 شروع گفتگو")
async def discovery_start_chat_message(message: Message, state: FSMContext):
    """شروع گفتگو از طریق message"""
    await start_chat_with_current_candidate(message, state)

@router.message(F.text == "➡️ مچ بعدی", DiscoveryStates.browsing)
async def discovery_next_message(message: Message, state: FSMContext):
    """مچ بعدی از طریق message"""
    await show_next_discovery_candidate_message(message, state)

@router.message(F.text == "🔙 بازگشت به منو")
async def discovery_back_to_menu_message(message: Message, state: FSMContext):
    """بازگشت به منو از اکتشاف"""
    from handlers.registration import get_main_menu_keyboard
    
    # بررسی ادمین بودن
    admin_ids = [7203796090]  # ادمین ID از حافظه
    is_admin = message.from_user.id in admin_ids
    
    await state.clear()
    await message.answer(
        "🏠 <b>منوی اصلی</b>",
        reply_markup=get_main_menu_keyboard('fa', is_admin=is_admin),
        parse_mode='HTML'
    )

async def start_chat_with_current_candidate(message: Message, state: FSMContext):
    """شروع گفتگو با کاندید فعلی"""
    user_id = message.from_user.id
    data = await state.get_data()
    current_candidate_id = data.get('current_candidate')
    
    if not current_candidate_id:
        await message.answer("خطا: کاندید پیدا نشد")
        return
    
    async with db.get_session() as session:
        # چک کردن اینکه آیا مچ وجود دارد
        match_query = select(Match).where(
            or_(
                and_(Match.user_a == user_id, Match.user_b == current_candidate_id),
                and_(Match.user_a == current_candidate_id, Match.user_b == user_id)
            )
        )
        result = await session.execute(match_query)
        match = result.scalar_one_or_none()
        
        if not match:
            # Check if user has liked this candidate
            user_like_result = await session.execute(
                select(Swipe).where(
                    and_(
                        Swipe.from_user == user_id,
                        Swipe.to_user == current_candidate_id,
                        Swipe.action > 0  # لایک یا سوپر لایک
                    )
                )
            )
            user_like = user_like_result.scalar_one_or_none()
            
            if user_like:
                # User has liked, create match and allow chat
                match = Match(
                    user_a=user_id,
                    user_b=current_candidate_id,
                    status=1  # فعال
                )
                session.add(match)
                await session.commit()
                
                # Continue to start chat
            else:
                await message.answer(
                    "❌ <b>گفتگو امکان‌پذیر نیست</b>\n\n"
                    "برای شروع گفتگو، ابتدا باید این کاربر را لایک کنید.\n"
                    "💚 روی دکمه 'لایک' کلیک کنید.",
                    parse_mode='HTML'
                )
                return
        
        # گرفتن اطلاعات کاندید
        candidate = await session.get(User, current_candidate_id)
        if not candidate:
            await message.answer("خطا: کاربر پیدا نشد")
            return
        
        # تغییر وضعیت مچ به فعال
        match.status = 1
        await session.commit()
        
        # اتصال به سیستم گفتگوهای من
        from handlers.my_chats import current_chat_users
        
        # ذخیره اطلاعات گفتگو
        current_chat_users[user_id] = {
            'target_user_id': candidate.user_id,
            'target_user_name': candidate.username or candidate.first_name or "کاربر",
            'match_id': match.id
        }
        
        # نمایش گفتگو
        await message.answer(
            f"💬 <b>گفتگو با {candidate.username or candidate.first_name or 'کاربر'}</b>\n\n"
            f"🎉 شما این کاربر را لایک کرده‌اید!\n"
            f"حالا می‌توانید گفتگو را شروع کنید.\n\n"
            f"💡 برای ارسال پیام روی 'ارسال پیام' کلیک کنید.",
            parse_mode='HTML'
        )
        
        # هدایت به گفتگوهای من
        from handlers.my_chats import view_my_chats_action
        await view_my_chats_action(message)

# توابع کمکی برای message handlers
async def handle_discovery_action_message(message: Message, state: FSMContext, action: int):
    """Handle discovery action from message"""
    user_id = message.from_user.id
    data = await state.get_data()
    current_candidate_id = data.get('current_candidate')
    
    
    if not current_candidate_id:
        action_text = {1: "لایک کردن", 2: "سوپر لایک کردن", -1: "رد کردن"}.get(action, "عمل")
        await message.answer(
            f"❌ <b>کاربری برای {action_text} وجود ندارد</b>\n\n"
            "ابتدا یک کاربر پیدا کنید.\n\n"
            "🌟 روی 'اکتشاف' کلیک کنید تا کاربران جدید پیدا کنید.",
            parse_mode='HTML'
        )
        # بازگشت به حالت اکتشاف
        await discovery_mode(message, state)
        return
    
    async with db.get_session() as session:
        # Get user
        user_result = await session.execute(select(User).where(User.user_id == user_id))
        user = user_result.scalar_one_or_none()
        
        if not user:
            await message.answer("خطا: کاربر پیدا نشد")
            return
        
        # Check if swipe already exists
        existing_swipe = await session.execute(
            select(Swipe).where(
                and_(
                    Swipe.from_user == user.user_id,
                    Swipe.to_user == current_candidate_id
                )
            )
        )
        existing_swipe = existing_swipe.scalar_one_or_none()
        
        if existing_swipe:
            # Update existing swipe
            existing_swipe.action = action
            existing_swipe.created_at = func.now()
        else:
            # Record the new swipe
            swipe = Swipe(
                from_user=user.user_id,
                to_user=current_candidate_id,
                action=action
            )
            session.add(swipe)
        
        # Check for mutual like if this was a like
        if action > 0:  # like or super like
            mutual = await matching_engine.check_mutual_like(session, user.user_id, current_candidate_id)
            
            if mutual:
                # Create match
                match = await matching_engine.create_match(session, user.user_id, current_candidate_id)
                await session.commit()
                
                # Notify about match
                candidate = await session.get(User, current_candidate_id)
                await message.answer(
                    f"🎉 <b>مچ جدید!</b>\n\n"
                    f"شما و {candidate.first_name or candidate.username or 'کاربر'} همدیگر را لایک کردید!\n\n"
                    f"می‌توانید شروع به گفتگو کنید یا ادامه اکتشاف دهید.",
                    parse_mode='HTML'
                )
                
                # Notify the other user
                try:
                    await message.bot.send_message(
                        candidate.user_id,
                        f"🎉 مچ جدید!\n\n"
                        f"کسی شما را لایک کرد! برای دیدن: /matches"
                    )
                except:
                    pass
                
                return
        
        await session.commit()
        
        # Show confirmation message instead of auto-next
        action_text = {1: "لایک", 2: "سوپر لایک", -1: "رد کردن"}.get(action, "عمل")
        await message.answer(
            f"✅ <b>{action_text} ثبت شد!</b>\n\n"
            f"💡 برای دیدن کاربر بعدی روی 'مچ بعدی' کلیک کنید.",
            parse_mode='HTML'
        )

async def show_next_discovery_candidate_message(message: Message, state: FSMContext):
    """Show next candidate from message"""
    data = await state.get_data()
    candidates = data.get('candidates', [])
    current_index = data.get('current_index', 0)
    
    next_index = current_index + 1
    
    if next_index >= len(candidates):
        await message.answer(
            "🎉 <b>تمام کاندیدها دیده شدند!</b>\n\n"
            "همه کاربران موجود را بررسی کردید.\n"
            "بعداً دوباره تلاش کنید تا کاربران جدید پیدا کنید."
        )
        await state.clear()
        return
    
    async with db.get_session() as session:
        candidate_id = candidates[next_index]
        candidate = await session.get(User, candidate_id)
        
        if not candidate:
            # Skip this candidate and try next
            await state.update_data(current_index=next_index)
            await show_next_discovery_candidate_message(message, state)
            return
        
        card_text = create_discovery_card(candidate)
        
        await state.update_data(
            current_index=next_index,
            current_candidate=candidate.user_id
        )
        
        # کیبورد کاستوم برای اکتشاف
        keyboard = ReplyKeyboardMarkup(
            keyboard=[
                [
                    KeyboardButton(text="💚 لایک"),
                    KeyboardButton(text="❌ رد کردن")
                ],
                [
                    KeyboardButton(text="⭐ سوپر لایک"),
                    KeyboardButton(text="💬 شروع گفتگو")
                ],
                [
                    KeyboardButton(text="➡️ مچ بعدی")
                ],
                [
                    KeyboardButton(text="🔙 بازگشت به منو")
                ]
            ],
            resize_keyboard=True,
            one_time_keyboard=False
        )
        
        # ارسال عکس پروفایل اگر موجود باشد
        profile_photo = None
        if candidate.photos:
            try:
                import json
                photos_list = json.loads(candidate.photos)
                if photos_list and len(photos_list) > 0:
                    profile_photo = photos_list[0]  # اولین عکس
            except:
                profile_photo = None
        
        if profile_photo:
            try:
                await message.answer_photo(
                    photo=profile_photo,
                    caption=f"🌟 <b>حالت اکتشاف</b>\n\n{card_text}",
                    reply_markup=keyboard,
                    parse_mode='HTML'
                )
            except:
                # اگر عکس نتوانست ارسال شود، فقط متن ارسال کن
                await message.answer(
                    f"🌟 <b>حالت اکتشاف</b>\n\n{card_text}",
                    reply_markup=keyboard,
                    parse_mode='HTML'
                )
        else:
            await message.answer(
                f"🌟 <b>حالت اکتشاف</b>\n\n{card_text}",
                reply_markup=keyboard,
                parse_mode='HTML'
            )

async def get_discovery_candidates(session, user_id: int, limit: int = 20):
    """Get candidates for discovery mode"""
    from sqlalchemy import select
    from database.models import Preference
    
    # Get user with preferences in one query
    result = await session.execute(
        select(User).where(User.user_id == user_id)
    )
    user = result.scalar_one_or_none()
    
    if not user:
        return []
    
    # Get preferences separately
    pref_result = await session.execute(
        select(Preference).where(Preference.user_id == user_id)
    )
    pref = pref_result.scalar_one_or_none()
    
    if not pref:
        # Create default preferences if not exists
        pref = Preference(
            user_id=user_id,
            age_min=18,
            age_max=50,
            radius_km=50
        )
        session.add(pref)
        await session.commit()
    
    # Get only REJECTED users (not liked users)
    from database.models import Swipe
    rejected_users_result = await session.execute(
        select(Swipe.to_user).where(
            and_(
                Swipe.from_user == user_id,
                Swipe.action == -1  # فقط رد شده‌ها
            )
        )
    )
    rejected_user_ids = [row[0] for row in rejected_users_result.fetchall()]
    
    # Base query for candidates using async select
    from datetime import datetime, timedelta
    
    conditions = [
        User.user_id != user_id,
        User.is_banned == False,
        User.is_active == True
    ]
    
    # Exclude only REJECTED users (liked users can be seen again)
    if rejected_user_ids:
        conditions.append(~User.user_id.in_(rejected_user_ids))
    
    query = select(User).where(and_(*conditions))
    
    # Apply filters
    if pref.gender_pref > 0:
        query = query.where(User.gender == pref.gender_pref)
    
    # Order by recent activity and limit
    query = query.order_by(User.last_active.desc()).limit(limit)
    
    # Execute query
    result = await session.execute(query)
    candidates = result.scalars().all()
    
    return candidates

def create_discovery_card(user: User) -> str:
    """Create discovery card for a user"""
    age = user.age if user.age else "نامشخص"
    
    card = f"💫 <b>{user.username or 'کاربر'}</b>, {age}\n"
    
    if user.city:
        card += f"📍 {user.city}"
        if user.country and user.country != 'XX':
            card += f", {user.country}"
        card += "\n"
    
    if user.bio:
        card += f"\n💭 <i>{user.bio}</i>\n"
    
    if user.interests:
        interests = ", ".join(user.interests[:4])
        card += f"\n🎯 {interests}\n"
    
    return card

@router.message(F.text == "/matches")
async def show_matches(message: Message):
    """Show user's matches"""
    user_id = message.from_user.id
    
    async with db.get_session() as session:
        # Get user
        result = await session.execute(select(User).where(User.user_id == user_id))
        user = result.scalar_one_or_none()
        
        if not user:
            await message.answer("لطفاً ابتدا ثبت نام کنید: /start")
            return
        
        # Get matches
        matches_query = select(Match).where(
            and_(
                or_(Match.user_a == user.user_id, Match.user_b == user.user_id),
                Match.status.in_([0, 1])  # pending or connected
            )
        ).order_by(Match.created_at.desc())
        
        result = await session.execute(matches_query)
        matches = result.scalars().all()
        
        if not matches:
            await message.answer("هنوز مچی ندارید!\n\n🔍 /find برای پیدا کردن مچ\n🌟 /discovery برای اکتشاف")
            return
        
        # Create matches list
        matches_text = "💕 <b>مچ‌های شما:</b>\n\n"
        
        for i, match in enumerate(matches[:10], 1):  # Show max 10
            # Get partner
            partner_id = match.user_b if match.user_a == user.user_id else match.user_a
            partner = await session.get(User, partner_id)
            
            if partner:
                status_emoji = "💬" if match.status == 1 else "⏳"
                matches_text += f"{status_emoji} <b>{partner.first_name or partner.username or 'کاربر'}</b>\n"
                
                if match.status == 1:
                    matches_text += "   گفتگو فعال\n"
                else:
                    matches_text += "   در انتظار شروع گفتگو\n"
                
                matches_text += "\n"
        
        await message.answer(matches_text, parse_mode='HTML')
