"""
Profile Handler - هندلر پروفایل
مدیریت کامل پروفایل کاربر شامل نمایش، ویرایش و عکس‌ها
"""

from aiogram import Router, F, Bot
from aiogram.types import Message, CallbackQuery, InlineKeyboardMarkup, InlineKeyboardButton, ReplyKeyboardMarkup, KeyboardButton
from aiogram.fsm.context import FSMContext
from aiogram.fsm.state import State, StatesGroup
from sqlalchemy import select
from database.database import db
from database.models import User, Preference
import logging
import json

router = Router()
logger = logging.getLogger(__name__)

class ProfileStates(StatesGroup):
    editing_bio = State()
    editing_interests = State()
    editing_age_range = State()
    editing_radius = State()
    editing_gender_preference = State()
    uploading_photo = State()
    editing_name = State()
    editing_age = State()
    editing_gender = State()
    editing_country = State()
    editing_city = State()

def get_profile_keyboard():
    """کیبورد پروفایل اصلی"""
    return ReplyKeyboardMarkup(
        keyboard=[
            [KeyboardButton(text="⚙️ تنظیمات جستجو"), KeyboardButton(text="✏️ ویرایش پروفایل")],
            [KeyboardButton(text="📸 آپلود عکس"), KeyboardButton(text="👁️ مشاهده عکس‌ها")],
            [KeyboardButton(text="🔙 بازگشت به منو")]
        ],
        resize_keyboard=True
    )

def get_edit_profile_keyboard():
    """کیبورد ویرایش پروفایل"""
    return ReplyKeyboardMarkup(
        keyboard=[
            [KeyboardButton(text="🎂 ویرایش سن"), KeyboardButton(text="👤 ویرایش نام")],
            [KeyboardButton(text="🏙️ ویرایش شهر"), KeyboardButton(text="🌍 ویرایش کشور")],
            [KeyboardButton(text="✏️ ویرایش بیو"), KeyboardButton(text="🎯 ویرایش علایق")],
            [KeyboardButton(text="⚧️ ویرایش جنسیت")],
            [KeyboardButton(text="🔙 بازگشت به پروفایل")]
        ],
        resize_keyboard=True
    )

# ==================== نمایش پروفایل ====================
@router.message(F.text == "👤 پروفایل")
async def show_profile(message: Message):
    """نمایش پروفایل کاربر"""
    user_id = message.from_user.id
    
    async with db.get_session() as session:
        result = await session.execute(select(User).where(User.user_id == user_id))
        user = result.scalar_one_or_none()
        
        if not user:
            await message.answer("❌ پروفایل شما یافت نشد. لطفاً ابتدا ثبت‌نام کنید.")
            return
        
        # نمایش سن
        age = user.age if user.age else 18
        
        # نمایش جنسیت
        gender_display = {
            '1': "مرد 👨",
            '2': "زن 👩", 
            '3': "غیره 🏳️‍⚧️",
            1: "مرد 👨",
            2: "زن 👩", 
            3: "غیره 🏳️‍⚧️"
        }.get(user.gender if isinstance(user.gender, int) else str(user.gender), "نامشخص")
        
        # نمایش ترجیحات
        looking_for_display = {
            '1': "مرد 👨",
            '2': "زن 👩",
            '3': "غیره 🏳️‍⚧️",
            1: "مرد 👨",
            2: "زن 👩",
            3: "غیره 🏳️‍⚧️"
        }.get(user.looking_for if isinstance(user.looking_for, int) else str(user.looking_for), "نامشخص")
        
        # ساخت متن پروفایل
        profile = f"👤 <b>پروفایل شما</b>\n\n"
        profile += f"🏷️ <b>نام:</b> {user.first_name or user.username or 'کاربر'}\n"
        profile += f"🎂 <b>سن:</b> {age} ساله\n"
        profile += f"⚧️ <b>جنسیت:</b> {gender_display}\n"
        
        # نمایش کشور و شهر
        if user.country:
            profile += f"🌍 <b>کشور:</b> {user.country}\n"
        if user.city:
            profile += f"🏙️ <b>شهر:</b> {user.city}\n"
        
        profile += f"💕 <b>دنبال:</b> {looking_for_display}\n"
        
        if user.bio:
            profile += f"\n📝 <b>بیو:</b>\n{user.bio}\n"
        
        if user.interests:
            profile += f"\n🎯 <b>علایق:</b>\n{user.interests}\n"
        
        profile += f"\n📊 <b>وضعیت:</b> {'✅ فعال' if user.is_active else '❌ غیرفعال'}"
        profile += f"\n💎 <b>اشتراک:</b> {'⭐ پریمیوم' if user.is_premium else '🆓 رایگان'}"
        
        # بررسی عکس‌ها
        has_photo = False
        main_photo_id = None
        
        if user.photos:
            try:
                photos_data = json.loads(user.photos) if isinstance(user.photos, str) else user.photos
                if photos_data and isinstance(photos_data, list) and len(photos_data) > 0:
                    main_photo_id = photos_data[0]
                    has_photo = True
            except Exception as e:
                logger.error(f"Error parsing photos: {e}")
        
        # ارسال پروفایل با کیبورد
        profile_keyboard = get_profile_keyboard()
        
        if has_photo and main_photo_id:
            try:
                await message.answer_photo(
                    photo=main_photo_id,
                    caption=profile,
                    reply_markup=profile_keyboard,
                    parse_mode='HTML'
                )
            except Exception as e:
                logger.error(f"Error sending photo {main_photo_id}: {e}")
                await message.answer(profile, reply_markup=profile_keyboard, parse_mode='HTML')
        else:
            await message.answer(profile, reply_markup=profile_keyboard, parse_mode='HTML')

# ==================== هندلرهای کیبورد کاستوم ====================
@router.message(F.text == "✏️ ویرایش پروفایل")
async def edit_profile_menu(message: Message):
    """نمایش منوی ویرایش پروفایل"""
    await message.answer(
        "✏️ <b>ویرایش پروفایل</b>\n\n"
        "کدام بخش از پروفایل خود را می‌خواهید ویرایش کنید؟",
        reply_markup=get_edit_profile_keyboard(),
        parse_mode='HTML'
    )

@router.message(F.text == "✏️ ویرایش بیو")
async def edit_bio_keyboard(message: Message, state: FSMContext):
    """شروع ویرایش بیو از کیبورد کاستوم"""
    await message.answer(
        "✏️ بیو جدید خود را بنویسید:\n\n"
        "نکات:\n"
        "• حداکثر 200 کاراکتر\n"
        "• محتوای مناسب\n"
        "• /cancel برای لغو"
    )
    await state.set_state(ProfileStates.editing_bio)

@router.message(F.text == "🎯 ویرایش علایق")
async def edit_interests_keyboard(message: Message, state: FSMContext):
    """شروع ویرایش علایق از کیبورد کاستوم"""
    await message.answer(
        "🎯 علایق جدید خود را با کاما جدا کنید:\n\n"
        "مثال: موسیقی, سفر, ورزش, فیلم, کتاب\n\n"
        "/cancel برای لغو"
    )
    await state.set_state(ProfileStates.editing_interests)

@router.message(F.text == "👤 ویرایش نام")
async def edit_name_keyboard(message: Message, state: FSMContext):
    """شروع ویرایش نام از کیبورد کاستوم"""
    await message.answer(
        "👤 <b>ویرایش نام</b>\n\n"
        "نام جدید خود را وارد کنید:\n\n"
        "نکات:\n"
        "• حداکثر 50 کاراکتر\n"
        "• فقط حروف و اعداد\n"
        "• /cancel برای لغو",
        parse_mode='HTML'
    )
    await state.set_state(ProfileStates.editing_name)

@router.message(F.text == "🎂 ویرایش سن")
async def edit_age_keyboard(message: Message, state: FSMContext):
    """شروع ویرایش سن از کیبورد کاستوم"""
    await message.answer(
        "🎂 <b>ویرایش سن</b>\n\n"
        "سن جدید خود را وارد کنید:\n\n"
        "نکات:\n"
        "• بین 18 تا 99 سال\n"
        "• فقط عدد وارد کنید\n"
        "• /cancel برای لغو",
        parse_mode='HTML'
    )
    await state.set_state(ProfileStates.editing_age)

@router.message(F.text == "🌍 ویرایش کشور")
async def edit_country_keyboard(message: Message, state: FSMContext):
    """شروع ویرایش کشور از کیبورد کاستوم"""
    await message.answer(
        "🌍 <b>ویرایش کشور</b>\n\n"
        "نام کشور خود را وارد کنید:\n\n"
        "مثال: ایران، ترکیه، آلمان، کانادا\n\n"
        "نکات:\n"
        "• حداکثر 50 کاراکتر\n"
        "• /cancel برای لغو",
        parse_mode='HTML'
    )
    await state.set_state(ProfileStates.editing_country)

@router.message(F.text == "🏙️ ویرایش شهر")
async def edit_city_keyboard(message: Message, state: FSMContext):
    """شروع ویرایش شهر از کیبورد کاستوم"""
    await message.answer(
        "🏙️ <b>ویرایش شهر</b>\n\n"
        "نام شهر خود را وارد کنید:\n\n"
        "مثال: تهران، استانبول، برلین، تورنتو\n\n"
        "نکات:\n"
        "• حداکثر 50 کاراکتر\n"
        "• /cancel برای لغو",
        parse_mode='HTML'
    )
    await state.set_state(ProfileStates.editing_city)

@router.message(F.text == "⚧️ ویرایش جنسیت")
async def edit_gender_keyboard(message: Message):
    """شروع ویرایش جنسیت از کیبورد کاستوم"""
    gender_keyboard = ReplyKeyboardMarkup(
        keyboard=[
            [KeyboardButton(text="👨 مرد"), KeyboardButton(text="👩 زن")],
            [KeyboardButton(text="🏳️‍⚧️ غیره")],
            [KeyboardButton(text="🔙 بازگشت به ویرایش")]
        ],
        resize_keyboard=True
    )
    
    await message.answer(
        "⚧️ <b>ویرایش جنسیت</b>\n\n"
        "جنسیت خود را انتخاب کنید:",
        reply_markup=gender_keyboard,
        parse_mode='HTML'
    )

@router.message(F.text == "📸 آپلود عکس")
async def upload_photo_keyboard(message: Message, state: FSMContext):
    """آپلود عکس از کیبورد کاستوم"""
    user_id = message.from_user.id
    
    async with db.get_session() as session:
        result = await session.execute(select(User).where(User.user_id == user_id))
        user = result.scalar_one_or_none()
        
        current_count = 0
        if user and user.photos:
            try:
                photos_data = json.loads(user.photos) if isinstance(user.photos, str) else user.photos
                current_count = len(photos_data) if photos_data else 0
            except:
                current_count = 0
    
    if current_count >= 5:
        await message.answer(
            "📸 <b>حداکثر عکس</b>\n\n"
            "شما قبلاً 5 عکس آپلود کرده‌اید.\n"
            "برای آپلود عکس جدید، ابتدا یکی از عکس‌های قبلی را حذف کنید.",
            parse_mode='HTML'
        )
        return
    
    # ساخت کیبورد انتخاب
    upload_keyboard = InlineKeyboardMarkup(inline_keyboard=[
        [InlineKeyboardButton(
            text="📱 استفاده از عکس پروفایل تلگرام",
            callback_data="use_telegram_profile_photo"
        )],
        [InlineKeyboardButton(
            text="📤 آپلود عکس جدید",
            callback_data="upload_new_photo"
        )]
    ])
    
    await message.answer(
        "📸 <b>آپلود عکس پروفایل</b>\n\n"
        f"📊 تعداد عکس‌های فعلی: {current_count}/5\n\n"
        "💡 یکی از گزینه‌های زیر را انتخاب کنید:",
        reply_markup=upload_keyboard,
        parse_mode='HTML'
    )

@router.message(F.text == "👁️ مشاهده عکس‌ها")
async def view_photos_keyboard(message: Message):
    """مشاهده عکس‌ها از کیبورد کاستوم"""
    user_id = message.from_user.id
    
    async with db.get_session() as session:
        result = await session.execute(select(User).where(User.user_id == user_id))
        user = result.scalar_one_or_none()
        
        if not user or not user.photos:
            await message.answer(
                "📸 <b>عکس‌های شما</b>\n\n"
                "شما هنوز عکسی آپلود نکرده‌اید.",
                parse_mode='HTML'
            )
            return
        
        try:
            photos_data = json.loads(user.photos) if isinstance(user.photos, str) else user.photos
            if not photos_data or not isinstance(photos_data, list):
                raise ValueError("Invalid photos data")
        except:
            await message.answer(
                "📸 <b>عکس‌های شما</b>\n\n"
                "خطا در بارگیری عکس‌ها.",
                parse_mode='HTML'
            )
            return
        
        # ارسال عکس‌ها با دکمه حذف
        for i, photo_id in enumerate(photos_data):
            try:
                # ساخت کیبورد حذف برای هر عکس
                delete_keyboard = InlineKeyboardMarkup(inline_keyboard=[
                    [InlineKeyboardButton(
                        text="🗑️ حذف این عکس",
                        callback_data=f"delete_photo_{i}"
                    )]
                ])
                
                await message.answer_photo(
                    photo=photo_id,
                    caption=f"📸 عکس {i+1} از {len(photos_data)}",
                    reply_markup=delete_keyboard
                )
            except Exception as e:
                logger.error(f"Error sending photo {photo_id}: {e}")
        

@router.message(F.text == "⚙️ تنظیمات جستجو")
async def settings_keyboard(message: Message):
    """تنظیمات جستجو از کیبورد کاستوم"""
    user_id = message.from_user.id
    
    async with db.get_session() as session:
        # بررسی کاربر
        result = await session.execute(select(User).where(User.user_id == user_id))
        user = result.scalar_one_or_none()
        
        if not user:
            await message.answer("❌ کاربر یافت نشد.")
            return
        
        # بررسی تنظیمات موجود
        pref_result = await session.execute(select(Preference).where(Preference.user_id == user_id))
        preference = pref_result.scalar_one_or_none()
        
        if not preference:
            # ایجاد تنظیمات پیش‌فرض
            preference = Preference(
                user_id=user_id,
                age_min=18,
                age_max=35,
                radius_km=50,
                gender_pref=1  # مرد
            )
            session.add(preference)
            await session.commit()
        
        # نمایش تنظیمات فعلی
        gender_text = {
            1: "مرد 👨", 
            2: "زن 👩",
            3: "غیره 🏳️‍⚧️"
        }.get(preference.gender_pref, "نامشخص")
        
        settings_text = f"⚙️ <b>تنظیمات جستجو</b>\n\n"
        settings_text += f"🎂 <b>بازه سنی:</b> {preference.age_min or 18} تا {preference.age_max or 35} سال\n"
        settings_text += f"📍 <b>شعاع جستجو:</b> {preference.radius_km or 50} کیلومتر\n"
        settings_text += f"💕 <b>دنبال چه کسی هستم:</b> {gender_text}\n\n"
        settings_text += "کدام مورد را می‌خواهید تغییر دهید؟"
        
        # کیبورد تنظیمات
        settings_keyboard = ReplyKeyboardMarkup(
            keyboard=[
                [KeyboardButton(text="🎂 تغییر بازه سنی"), KeyboardButton(text="📍 تغییر شعاع")],
                [KeyboardButton(text="💕 دنبال چه کسی هستم")],
                [KeyboardButton(text="🔙 بازگشت به پروفایل")]
            ],
            resize_keyboard=True
        )
        
        await message.answer(
            settings_text,
            reply_markup=settings_keyboard,
            parse_mode='HTML'
        )

# ==================== هندلرهای تنظیمات جستجو ====================
@router.message(F.text == "🎂 تغییر بازه سنی")
async def change_age_range(message: Message, state: FSMContext):
    """تغییر بازه سنی"""
    await message.answer(
        "🎂 <b>تغییر بازه سنی</b>\n\n"
        "بازه سنی مطلوب خود را وارد کنید:\n\n"
        "فرمت: حداقل-حداکثر\n"
        "مثال: 20-30\n"
        "مثال: 25-40\n\n"
        "/cancel برای لغو",
        parse_mode='HTML'
    )
    await state.set_state(ProfileStates.editing_age_range)

@router.message(F.text.in_(["👨 مرد", "👩 زن", "🏳️‍⚧️ غیره"]))
async def save_gender_preference(message: Message):
    """ذخیره جنسیت مطلوب"""
    gender_map = {
        "👨 مرد": 1,
        "👩 زن": 2,
        "🏳️‍⚧️ غیره": 3
    }
    
    selected_gender = gender_map.get(message.text)
    user_id = message.from_user.id
    
    async with db.get_session() as session:
        # بروزرسانی تنظیمات
        pref_result = await session.execute(select(Preference).where(Preference.user_id == user_id))
        preference = pref_result.scalar_one_or_none()
        
        if preference:
            preference.gender_pref = selected_gender
        else:
            preference = Preference(
                user_id=user_id,
                age_min=18,
                age_max=35,
                radius_km=50,
                gender_pref=selected_gender
            )
            session.add(preference)
        
        await session.commit()
        
        await message.answer(
            f"✅ جنسیت مطلوب به '{message.text}' تغییر یافت!",
            reply_markup=get_profile_keyboard()
        )

@router.message(ProfileStates.editing_age_range)
async def save_age_range(message: Message, state: FSMContext):
    """ذخیره بازه سنی"""
    if message.text == "/cancel":
        await message.answer("تغییر بازه سنی لغو شد.")
        await state.clear()
        return
    
    try:
        # پارس کردن ورودی
        age_range = message.text.strip()
        if '-' not in age_range:
            await message.answer("❌ فرمت نادرست! مثال: 20-30")
            return
        
        min_age, max_age = age_range.split('-')
        min_age = int(min_age.strip())
        max_age = int(max_age.strip())
        
        # اعتبارسنجی
        if min_age < 18 or max_age > 99:
            await message.answer("❌ سن باید بین 18 تا 99 سال باشد!")
            return
        
        if min_age >= max_age:
            await message.answer("❌ حداقل سن باید کمتر از حداکثر سن باشد!")
            return
        
        user_id = message.from_user.id
        
        async with db.get_session() as session:
            # بروزرسانی تنظیمات
            pref_result = await session.execute(select(Preference).where(Preference.user_id == user_id))
            preference = pref_result.scalar_one_or_none()
            
            if preference:
                preference.age_min = min_age
                preference.age_max = max_age
            else:
                preference = Preference(
                    user_id=user_id,
                    age_min=min_age,
                    age_max=max_age,
                    radius_km=50,
                    gender_pref=0
                )
                session.add(preference)
            
            await session.commit()
            
            await message.answer(
                f"✅ بازه سنی به {min_age}-{max_age} سال تغییر یافت!",
                reply_markup=get_profile_keyboard()
            )
    
    except ValueError:
        await message.answer("❌ فرمت نادرست! فقط عدد وارد کنید. مثال: 20-30")
        return
    
    await state.clear()

@router.message(F.text == "📍 تغییر شعاع")
async def change_radius(message: Message, state: FSMContext):
    """تغییر شعاع جستجو"""
    await message.answer(
        "📍 <b>تغییر شعاع جستجو</b>\n\n"
        "شعاع جستجو را به کیلومتر وارد کنید:\n\n"
        "مثال: 10\n"
        "مثال: 50\n"
        "مثال: 100\n\n"
        "حداقل: 5 کیلومتر\n"
        "حداکثر: 500 کیلومتر\n\n"
        "/cancel برای لغو",
        parse_mode='HTML'
    )
    await state.set_state(ProfileStates.editing_radius)

@router.message(ProfileStates.editing_radius)
async def save_radius(message: Message, state: FSMContext):
    """ذخیره شعاع جستجو"""
    if message.text == "/cancel":
        await message.answer("تغییر شعاع لغو شد.")
        await state.clear()
        return
    
    try:
        radius = int(message.text.strip())
        
        # اعتبارسنجی
        if radius < 5 or radius > 500:
            await message.answer("❌ شعاع باید بین 5 تا 500 کیلومتر باشد!")
            return
        
        user_id = message.from_user.id
        
        async with db.get_session() as session:
            # بروزرسانی تنظیمات
            pref_result = await session.execute(select(Preference).where(Preference.user_id == user_id))
            preference = pref_result.scalar_one_or_none()
            
            if preference:
                preference.radius_km = radius
            else:
                preference = Preference(
                    user_id=user_id,
                    age_min=18,
                    age_max=35,
                    radius_km=radius,
                    gender_pref=0
                )
                session.add(preference)
            
            await session.commit()
            
            await message.answer(
                f"✅ شعاع جستجو به {radius} کیلومتر تغییر یافت!",
                reply_markup=get_profile_keyboard()
            )
    
    except ValueError:
        await message.answer("❌ فقط عدد وارد کنید! مثال: 50")
        return
    
    await state.clear()

@router.message(F.text == "💕 دنبال چه کسی هستم")
async def change_gender_preference(message: Message):
    """تغییر ترجیح آشنایی"""
    gender_keyboard = ReplyKeyboardMarkup(
        keyboard=[
            [KeyboardButton(text="👨 مرد"), KeyboardButton(text="👩 زن")],
            [KeyboardButton(text="🏳️‍⚧️ غیره")],
            [KeyboardButton(text="🔙 بازگشت به پروفایل")]
        ],
        resize_keyboard=True
    )
    
    await message.answer(
        "💕 <b>دنبال چه کسی هستید؟</b>\n\n"
        "چه جنسیتی را برای آشنایی ترجیح می‌دهید؟",
        reply_markup=gender_keyboard,
        parse_mode='HTML'
    )

@router.message(F.text == "🔙 بازگشت به پروفایل")
async def back_to_profile(message: Message):
    """بازگشت به پروفایل"""
    await message.answer(
        "🔙 بازگشت به پروفایل",
        reply_markup=get_profile_keyboard()
    )

@router.message(F.text == "🔙 بازگشت به ویرایش")
async def back_to_edit_profile(message: Message):
    """بازگشت به منوی ویرایش پروفایل"""
    await message.answer(
        "🔙 بازگشت به ویرایش پروفایل",
        reply_markup=get_edit_profile_keyboard()
    )

@router.message(F.text == "🔙 بازگشت به منو")
async def back_to_main_menu(message: Message):
    """بازگشت به منوی اصلی"""
    from handlers.registration import get_main_menu_keyboard
    
    # بررسی ادمین بودن
    admin_ids = [7203796090]  # ادمین ID از حافظه
    is_admin = message.from_user.id in admin_ids
    
    await message.answer(
        "🏠 بازگشت به منوی اصلی",
        reply_markup=get_main_menu_keyboard('fa', is_admin=is_admin)
    )

# ==================== ویرایش بیو (State Handler) ====================

@router.message(ProfileStates.editing_bio)
async def save_bio(message: Message, state: FSMContext):
    """ذخیره بیو جدید"""
    if message.text == "/cancel":
        await message.answer("ویرایش بیو لغو شد.")
        await state.clear()
        return
    
    new_bio = message.text.strip()
    
    if len(new_bio) > 200:
        await message.answer("بیو نباید بیشتر از 200 کاراکتر باشد. دوباره تلاش کنید:")
        return
    
    user_id = message.from_user.id
    
    async with db.get_session() as session:
        result = await session.execute(select(User).where(User.user_id == user_id))
        user = result.scalar_one_or_none()
        
        if user:
            user.bio = new_bio
            await session.commit()
            await message.answer(
                "✅ بیو با موفقیت به‌روزرسانی شد!\n\n"
                "🔙 برای ادامه ویرایش یا مشاهده پروفایل از دکمه‌های زیر استفاده کنید.",
                reply_markup=get_edit_profile_keyboard()
            )
        else:
            await message.answer("خطا در به‌روزرسانی بیو.")
    
    await state.clear()

# ==================== ویرایش علایق (State Handler) ====================

@router.message(ProfileStates.editing_interests)
async def save_interests(message: Message, state: FSMContext):
    """ذخیره علایق جدید"""
    if message.text == "/cancel":
        await message.answer("ویرایش علایق لغو شد.")
        await state.clear()
        return
    
    interests_text = message.text.strip()
    interests = [i.strip() for i in interests_text.split(',') if i.strip()]
    
    if len(interests) > 10:
        await message.answer("حداکثر 10 علاقه مجاز است. دوباره تلاش کنید:")
        return
    
    user_id = message.from_user.id
    
    async with db.get_session() as session:
        result = await session.execute(select(User).where(User.user_id == user_id))
        user = result.scalar_one_or_none()
        
        if user:
            user.interests = ','.join(interests)
            await session.commit()
            await message.answer(
                f"✅ علایق به‌روزرسانی شد:\n{', '.join(interests)}\n\n"
                "🔙 برای ادامه ویرایش یا مشاهده پروفایل از دکمه‌های زیر استفاده کنید.",
                reply_markup=get_edit_profile_keyboard()
            )
        else:
            await message.answer("خطا در به‌روزرسانی علایق.")
    
    await state.clear()

# ==================== ویرایش نام (State Handler) ====================

@router.message(ProfileStates.editing_name)
async def save_name(message: Message, state: FSMContext):
    """ذخیره نام جدید"""
    if message.text == "/cancel":
        await message.answer("ویرایش نام لغو شد.", reply_markup=get_profile_keyboard())
        await state.clear()
        return
    
    new_name = message.text.strip()
    
    if len(new_name) > 50:
        await message.answer("نام نباید بیشتر از 50 کاراکتر باشد. دوباره تلاش کنید:")
        return
    
    if len(new_name) < 2:
        await message.answer("نام باید حداقل 2 کاراکتر باشد. دوباره تلاش کنید:")
        return
    
    user_id = message.from_user.id
    
    async with db.get_session() as session:
        result = await session.execute(select(User).where(User.user_id == user_id))
        user = result.scalar_one_or_none()
        
        if user:
            user.first_name = new_name
            await session.commit()
            await message.answer(
                f"✅ نام با موفقیت به '{new_name}' تغییر یافت!\n\n"
                "🔙 برای ادامه ویرایش یا مشاهده پروفایل از دکمه‌های زیر استفاده کنید.",
                reply_markup=get_edit_profile_keyboard()
            )
        else:
            await message.answer("خطا در به‌روزرسانی نام.", reply_markup=get_profile_keyboard())
    
    await state.clear()

# ==================== ویرایش سن (State Handler) ====================

@router.message(ProfileStates.editing_age)
async def save_age(message: Message, state: FSMContext):
    """ذخیره سن جدید"""
    if message.text == "/cancel":
        await message.answer("ویرایش سن لغو شد.", reply_markup=get_profile_keyboard())
        await state.clear()
        return
    
    try:
        new_age = int(message.text.strip())
        
        if new_age < 18 or new_age > 99:
            await message.answer("سن باید بین 18 تا 99 سال باشد. دوباره تلاش کنید:")
            return
        
        user_id = message.from_user.id
        
        async with db.get_session() as session:
            result = await session.execute(select(User).where(User.user_id == user_id))
            user = result.scalar_one_or_none()
            if user:
                user.age = new_age
                await session.commit()
                await message.answer(
                    f"✅ سن با موفقیت به {new_age} سال تغییر یافت!\n\n"
                    "🔙 برای ادامه ویرایش یا مشاهده پروفایل از دکمه‌های زیر استفاده کنید.",
                    reply_markup=get_edit_profile_keyboard()
                )
            else:
                await message.answer("خطا در به‌روزرسانی سن.", reply_markup=get_edit_profile_keyboard())
        
        await state.clear()
        
    except ValueError:
        await message.answer("فقط عدد وارد کنید! مثال: 25")

# ==================== ویرایش کشور (State Handler) ====================

@router.message(ProfileStates.editing_country)
async def save_country(message: Message, state: FSMContext):
    """ذخیره کشور جدید"""
    if message.text == "/cancel":
        await message.answer("ویرایش کشور لغو شد.", reply_markup=get_edit_profile_keyboard())
        await state.clear()
        return
    
    new_country = message.text.strip()
    
    if len(new_country) > 50:
        await message.answer("نام کشور نباید بیشتر از 50 کاراکتر باشد. دوباره تلاش کنید:")
        return
    
    if len(new_country) < 2:
        await message.answer("نام کشور باید حداقل 2 کاراکتر باشد. دوباره تلاش کنید:")
        return
    
    user_id = message.from_user.id
    
    async with db.get_session() as session:
        result = await session.execute(select(User).where(User.user_id == user_id))
        user = result.scalar_one_or_none()
        
        if user:
            user.country = new_country
            await session.commit()
            await message.answer(
                f"✅ کشور با موفقیت به '{new_country}' تغییر یافت!\n\n"
                "🔙 برای ادامه ویرایش یا مشاهده پروفایل از دکمه‌های زیر استفاده کنید.",
                reply_markup=get_edit_profile_keyboard()
            )
        else:
            await message.answer("خطا در به‌روزرسانی کشور.", reply_markup=get_edit_profile_keyboard())
    
    await state.clear()

# ==================== ویرایش شهر (State Handler) ====================

@router.message(ProfileStates.editing_city)
async def save_city(message: Message, state: FSMContext):
    """ذخیره شهر جدید"""
    if message.text == "/cancel":
        await message.answer("ویرایش شهر لغو شد.", reply_markup=get_edit_profile_keyboard())
        await state.clear()
        return
    
    new_city = message.text.strip()
    
    if len(new_city) > 50:
        await message.answer("نام شهر نباید بیشتر از 50 کاراکتر باشد. دوباره تلاش کنید:")
        return
    
    if len(new_city) < 2:
        await message.answer("نام شهر باید حداقل 2 کاراکتر باشد. دوباره تلاش کنید:")
        return
    
    user_id = message.from_user.id
    
    async with db.get_session() as session:
        result = await session.execute(select(User).where(User.user_id == user_id))
        user = result.scalar_one_or_none()
        
        if user:
            user.city = new_city
            await session.commit()
            await message.answer(
                f"✅ شهر با موفقیت به '{new_city}' تغییر یافت!\n\n"
                "🔙 برای ادامه ویرایش یا مشاهده پروفایل از دکمه‌های زیر استفاده کنید.",
                reply_markup=get_edit_profile_keyboard()
            )
        else:
            await message.answer("خطا در به‌روزرسانی شهر.", reply_markup=get_edit_profile_keyboard())
    
    await state.clear()

# ==================== ویرایش جنسیت (Handler) ====================

@router.message(F.text.in_(["👨 مرد", "👩 زن", "🏳️‍⚧️ غیره"]))
async def save_gender(message: Message):
    """ذخیره جنسیت جدید"""
    gender_map = {
        "👨 مرد": 1,
        "👩 زن": 2,
        "🏳️‍⚧️ غیره": 3
    }
    
    selected_gender = gender_map.get(message.text)
    user_id = message.from_user.id
    
    async with db.get_session() as session:
        result = await session.execute(select(User).where(User.user_id == user_id))
        user = result.scalar_one_or_none()
        
        if user:
            user.gender = selected_gender
            await session.commit()
            await message.answer(
                f"✅ جنسیت با موفقیت به '{message.text}' تغییر یافت!\n\n"
                "🔙 برای ادامه ویرایش یا مشاهده پروفایل از دکمه‌های زیر استفاده کنید.",
                reply_markup=get_edit_profile_keyboard()
            )
        else:
            await message.answer("خطا در به‌روزرسانی جنسیت.", reply_markup=get_edit_profile_keyboard())

# ==================== مدیریت عکس‌ها (State Handler) ====================

@router.message(ProfileStates.uploading_photo, F.photo)
async def save_uploaded_photo(message: Message, state: FSMContext):
    """ذخیره عکس آپلود شده"""
    photo = message.photo[-1]
    photo_id = photo.file_id
    user_id = message.from_user.id
    
    async with db.get_session() as session:
        result = await session.execute(select(User).where(User.user_id == user_id))
        user = result.scalar_one_or_none()
        
        if not user:
            await message.answer("خطا: کاربر پیدا نشد.")
            await state.clear()
            return
        
        photos_data = []
        if user.photos:
            try:
                photos_data = json.loads(user.photos) if isinstance(user.photos, str) else user.photos
                if not isinstance(photos_data, list):
                    photos_data = []
            except:
                photos_data = []
        
        if len(photos_data) >= 5:
            await message.answer("❌ شما قبلاً 5 عکس آپلود کرده‌اید.")
            await state.clear()
            return
        
        photos_data.append(photo_id)
        user.photos = json.dumps(photos_data)
        
        await session.commit()
        
        await message.answer(
            f"✅ عکس آپلود شد!\n"
            f"📊 تعداد عکس‌ها: {len(photos_data)}/5\n\n"
            "می‌توانید عکس دیگری آپلود کنید یا /done را بزنید.",
            reply_markup=get_profile_keyboard()
        )

@router.message(ProfileStates.uploading_photo)
async def handle_done_or_cancel(message: Message, state: FSMContext):
    """پایان آپلود عکس"""
    if message.text in ["/done", "/cancel"]:
        await state.clear()
        await message.answer(
            "✅ آپلود عکس تمام شد.",
            reply_markup=get_profile_keyboard()
        )
    else:
        await message.answer("لطفاً یک عکس ارسال کنید یا /done را بزنید.")

# ==================== آپلود عکس (Callback Handlers) ====================

@router.callback_query(F.data == "use_telegram_profile_photo")
async def use_telegram_profile_photo_callback(callback: CallbackQuery):
    """استفاده از عکس پروفایل تلگرام"""
    try:
        user_id = callback.from_user.id
        
        # دریافت عکس‌های پروفایل کاربر
        bot = callback.bot
        profile_photos = await bot.get_user_profile_photos(user_id, limit=1)
        
        if not profile_photos.photos:
            await callback.answer("❌ شما عکس پروفایل ندارید!", show_alert=True)
            return
        
        # گرفتن بهترین کیفیت عکس پروفایل
        profile_photo = profile_photos.photos[0][-1]  # آخرین عکس = بهترین کیفیت
        photo_id = profile_photo.file_id
        
        # ذخیره عکس در دیتابیس
        async with db.get_session() as session:
            result = await session.execute(select(User).where(User.user_id == user_id))
            user = result.scalar_one_or_none()
            
            if not user:
                await callback.answer("❌ کاربر پیدا نشد!", show_alert=True)
                return
            
            photos_data = []
            if user.photos:
                try:
                    photos_data = json.loads(user.photos) if isinstance(user.photos, str) else user.photos
                    if not isinstance(photos_data, list):
                        photos_data = []
                except:
                    photos_data = []
            
            if len(photos_data) >= 5:
                await callback.answer("❌ شما قبلاً 5 عکس آپلود کرده‌اید!", show_alert=True)
                return
            
            photos_data.append(photo_id)
            user.photos = json.dumps(photos_data)
            
            await session.commit()
            
            await callback.answer("✅ عکس پروفایل تلگرام اضافه شد!", show_alert=True)
            
            # ویرایش پیام
            await callback.message.edit_text(
                f"✅ <b>عکس پروفایل اضافه شد!</b>\n\n"
                f"📊 تعداد عکس‌ها: {len(photos_data)}/5\n\n"
                f"💡 عکس پروفایل تلگرام شما با موفقیت اضافه شد.",
                parse_mode='HTML'
            )
            
    except Exception as e:
        logger.error(f"Error using telegram profile photo: {e}")
        await callback.answer("❌ خطا در دریافت عکس پروفایل!", show_alert=True)

@router.callback_query(F.data == "upload_new_photo")
async def upload_new_photo_callback(callback: CallbackQuery, state: FSMContext):
    """شروع آپلود عکس جدید"""
    await callback.answer()
    
    await callback.message.edit_text(
        "📸 <b>آپلود عکس جدید</b>\n\n"
        "💡 عکس خود را ارسال کنید یا /cancel برای لغو:",
        parse_mode='HTML'
    )
    
    await state.set_state(ProfileStates.uploading_photo)

# ==================== حذف عکس (Callback Handler) ====================

@router.callback_query(F.data.startswith("delete_photo_"))
async def delete_photo_callback(callback: CallbackQuery):
    """حذف عکس از طریق callback"""
    try:
        # استخراج شماره عکس از callback_data
        photo_index = int(callback.data.split("_")[-1])
        user_id = callback.from_user.id
        
        async with db.get_session() as session:
            result = await session.execute(select(User).where(User.user_id == user_id))
            user = result.scalar_one_or_none()
            
            if not user or not user.photos:
                await callback.answer("❌ عکسی برای حذف یافت نشد!", show_alert=True)
                return
            
            try:
                photos_data = json.loads(user.photos) if isinstance(user.photos, str) else user.photos
                if not isinstance(photos_data, list) or photo_index >= len(photos_data):
                    await callback.answer("❌ عکس یافت نشد!", show_alert=True)
                    return
            except:
                await callback.answer("❌ خطا در بارگیری عکس‌ها!", show_alert=True)
                return
            
            # حذف عکس از لیست
            deleted_photo = photos_data.pop(photo_index)
            
            # به‌روزرسانی دیتابیس
            user.photos = json.dumps(photos_data) if photos_data else None
            await session.commit()
            
            # پاسخ موفقیت‌آمیز
            await callback.answer("✅ عکس حذف شد!", show_alert=True)
            
            # حذف پیام عکس
            try:
                await callback.message.delete()
            except:
                pass
            
            # ارسال پیام تأیید
            remaining_count = len(photos_data)
            await callback.message.answer(
                f"✅ <b>عکس حذف شد!</b>\n\n"
                f"📊 تعداد عکس‌های باقی‌مانده: {remaining_count}/5\n\n"
                f"💡 برای مشاهده عکس‌های به‌روزرسانی شده، دوباره روی '👁️ مشاهده عکس‌ها' کلیک کنید.",
                parse_mode='HTML'
            )
            
    except Exception as e:
        logger.error(f"Error deleting photo: {e}")
        await callback.answer("❌ خطا در حذف عکس!", show_alert=True)

# ==================== پایان فایل ====================
