"""
Settings Handler - هندلر تنظیمات
مدیریت تنظیمات کاربر
"""

from aiogram import Router, F
from aiogram.types import Message, CallbackQuery, InlineKeyboardMarkup, InlineKeyboardButton, ReplyKeyboardMarkup, KeyboardButton
from aiogram.fsm.context import FSMContext
from aiogram.fsm.state import State, StatesGroup
from sqlalchemy import select
from database.database import db
from database.models import User, Preference
from localization.translator import get_text, get_available_languages, get_language_name
import logging

router = Router()
logger = logging.getLogger(__name__)

class SettingsStates(StatesGroup):
    changing_language = State()

@router.message(F.text == "⚙️ تنظیمات")
async def settings_menu(message: Message):
    """نمایش منوی تنظیمات"""
    user_id = message.from_user.id
    
    async with db.get_session() as session:
        result = await session.execute(
            select(User).where(User.user_id == user_id)
        )
        user = result.scalar_one_or_none()
        
        if not user:
            await message.answer("❌ ابتدا باید ثبت‌نام کنید.")
            return
        
        user_language = user.language_code or 'fa'
        text = f"<b>{get_text(user_language, 'settings')}</b>\n\n"
        text += f"<b>{get_text(user_language, 'language')}:</b> {get_language_name(user_language)}"
        
        # کیبورد کاستوم برای تنظیمات
        keyboard = ReplyKeyboardMarkup(
            keyboard=[
                [
                    KeyboardButton(text="👻 حالت ناشناس"),
                    KeyboardButton(text="🌍 تغییر زبان")
                ],
                [
                    KeyboardButton(text="🔒 حریم خصوصی"),
                    KeyboardButton(text="🗑️ حذف حساب")
                ],
                [
                    KeyboardButton(text="📊 آمار من")
                ],
                [
                    KeyboardButton(text="🔙 بازگشت به منو")
                ]
            ],
            resize_keyboard=True,
            one_time_keyboard=False
        )
        
        await message.answer(text, reply_markup=keyboard, parse_mode='HTML')

# Message handlers برای دکمه‌های کیبورد کاستوم

@router.message(F.text == "👻 حالت ناشناس")
async def toggle_invisible_message(message: Message):
    """حالت ناشناس از طریق message"""
    await message.answer(
        "👻 <b>حالت ناشناس</b>\n\n"
        "این قابلیت در حال توسعه است.\n\n"
        "💡 با فعال کردن این حالت:\n"
        "• پروفایل شما در جستجو نمایش داده نمی‌شود\n"
        "• فقط شما می‌توانید دیگران را ببینید\n"
        "• این قابلیت ویژه کاربران پریمیوم است",
        parse_mode='HTML'
    )

@router.message(F.text == "🌍 تغییر زبان")
async def change_language_message(message: Message, state: FSMContext):
    """تغییر زبان از طریق message"""
    await change_language_action(message, state)

@router.message(F.text == "🔒 حریم خصوصی")
async def privacy_settings_message(message: Message):
    """تنظیمات حریم خصوصی از طریق message"""
    await privacy_settings_action(message)


@router.message(F.text == "🗑️ حذف حساب")
async def delete_account_message(message: Message):
    """حذف حساب از طریق message"""
    await delete_account_action(message)

@router.message(F.text == "📊 آمار من")
async def my_statistics_message(message: Message):
    """آمار شخصی از طریق message"""
    await my_statistics_action(message)

@router.message(F.text == "🔙 بازگشت به منو")
async def back_to_main_menu_settings(message: Message):
    """بازگشت به منوی اصلی"""
    from handlers.registration import get_main_menu_keyboard
    
    # بررسی ادمین بودن
    admin_ids = [7203796090]  # ادمین ID از حافظه
    is_admin = message.from_user.id in admin_ids
    
    await message.answer(
        "🏠 بازگشت به منوی اصلی",
        reply_markup=get_main_menu_keyboard('fa', is_admin=is_admin)
    )

# توابع کمکی

async def change_language_action(message: Message, state: FSMContext):
    """عملکرد تغییر زبان"""
    languages = get_available_languages()
    
    # ایجاد کیبورد زبان‌ها (3 تا در هر سطر)
    keyboard_rows = []
    lang_items = list(languages.items())
    
    for i in range(0, len(lang_items), 3):
        row = []
        for j in range(3):
            if i + j < len(lang_items):
                lang_code, lang_name = lang_items[i + j]
                row.append(KeyboardButton(text=lang_name))
        keyboard_rows.append(row)
    
    # اضافه کردن دکمه بازگشت
    keyboard_rows.append([KeyboardButton(text="🔙 بازگشت")])
    
    language_keyboard = ReplyKeyboardMarkup(
        keyboard=keyboard_rows,
        resize_keyboard=True
    )
    
    await message.answer(
        "🌍 <b>انتخاب زبان / Choose Language</b>\n\n"
        "زبان مورد نظر خود را انتخاب کنید:\n"
        "Please select your preferred language:",
        reply_markup=language_keyboard,
        parse_mode='HTML'
    )
    await state.set_state(SettingsStates.changing_language)

async def privacy_settings_action(message: Message):
    """عملکرد تنظیمات حریم خصوصی"""
    await message.answer(
        "🔒 <b>حریم خصوصی</b>\n\n"
        "اطلاعات محفوظ شما:\n"
        "• شماره تلفن\n"
        "• آدرس دقیق\n"
        "• اطلاعات پرداخت\n\n"
        "اطلاعات قابل مشاهده:\n"
        "• نام (اختیاری)\n"
        "• سن\n"
        "• جنسیت\n"
        "• بیو و علایق\n"
        "• عکس‌ها\n\n"
        "💡 می‌توانید هر زمان حساب خود را حذف کنید.",
        parse_mode='HTML'
    )


async def delete_account_action(message: Message):
    """عملکرد حذف حساب"""
    await message.answer(
        "🗑️ <b>حذف حساب کاربری</b>\n\n"
        "⚠️ <b>هشدار:</b> این عمل غیرقابل بازگشت است!\n\n"
        "با حذف حساب:\n"
        "• تمام اطلاعات شما پاک می‌شود\n"
        "• گفتگوهای فعال پایان می‌یابد\n"
        "• مچ‌ها حذف می‌شوند\n\n"
        "💡 اگر مطمئن هستید، با پشتیبانی تماس بگیرید:\n"
        "📱 @ChatMilosSupport",
        parse_mode='HTML'
    )

async def my_statistics_action(message: Message):
    """عملکرد آمار شخصی"""
    user_id = message.from_user.id
    
    async with db.get_session() as session:
        result = await session.execute(select(User).where(User.user_id == user_id))
        user = result.scalar_one_or_none()
        
        if not user:
            await message.answer("❌ کاربر یافت نشد.")
            return
        
        stats_text = "📊 <b>آمار شخصی شما</b>\n\n"
        stats_text += f"👤 <b>نام:</b> {user.first_name or user.username or 'تنظیم نشده'}\n"
        stats_text += f"📅 <b>تاریخ عضویت:</b> {user.created_at.strftime('%Y-%m-%d') if user.created_at else 'نامشخص'}\n"
        stats_text += f"💎 <b>وضعیت:</b> {'⭐ پریمیوم' if user.is_premium else '🆓 رایگان'}\n"
        stats_text += f"✅ <b>وضعیت حساب:</b> {'فعال' if user.is_active else 'غیرفعال'}\n\n"
        stats_text += "📈 <b>فعالیت‌ها:</b>\n"
        stats_text += "• مچ‌ها: در حال محاسبه...\n"
        stats_text += "• لایک‌ها: در حال محاسبه...\n"
        stats_text += "• بازدیدها: در حال محاسبه..."
        
        await message.answer(stats_text, parse_mode='HTML')

# State handlers
@router.message(SettingsStates.changing_language)
async def save_language(message: Message, state: FSMContext):
    """ذخیره زبان جدید"""
    if message.text == "🔙 بازگشت":
        await message.answer("تغییر زبان لغو شد.")
        await state.clear()
        await settings_menu(message)
        return
    
    # پیدا کردن کد زبان بر اساس نام
    languages = get_available_languages()
    selected_lang_code = None
    
    for lang_code, lang_name in languages.items():
        if message.text == lang_name:
            selected_lang_code = lang_code
            break
    
    if not selected_lang_code:
        await message.answer("❌ زبان نامعتبر. لطفاً از دکمه‌های موجود استفاده کنید.")
        return
    
    user_id = message.from_user.id
    
    async with db.get_session() as session:
        result = await session.execute(select(User).where(User.user_id == user_id))
        user = result.scalar_one_or_none()
        
        if user:
            user.language_code = selected_lang_code
            await session.commit()
            
            # پیام تأیید به زبان جدید
            await message.answer(
                f"✅ {get_text(selected_lang_code, 'language')} {get_text(selected_lang_code, 'change_language')} "
                f"{get_language_name(selected_lang_code)} ✅\n\n"
                f"🔄 {get_text(selected_lang_code, 'back_to_menu')}",
                parse_mode='HTML'
            )
        else:
            await message.answer("❌ خطا در تغییر زبان.")
    
    await state.clear()

# State handlers



# ==================== پایان فایل ====================
# تمام callback handlerها حذف شدند - فقط کیبورد کاستوم استفاده می‌شود
