import asyncio
import logging
import os
from dotenv import load_dotenv

from aiogram import Bot, Dispatcher
from aiogram.fsm.storage.memory import MemoryStorage
from aiogram.webhook.aiohttp_server import SimpleRequestHandler, setup_application
from aiohttp import web

from database.database import db
from handlers import registration, matching, safety, discovery, admin, chat, photo_upload, notifications, real_chat, real_matching, profile_handler, help_handler, premium_handler, settings, my_likes, my_chats

# Load environment variables
load_dotenv()

# Configure logging
logging.basicConfig(
    level=logging.INFO,
    format='%(asctime)s - %(name)s - %(levelname)s - %(message)s'
)
logger = logging.getLogger(__name__)

# Bot configuration
BOT_TOKEN = os.getenv('BOT_TOKEN')
BOT_MODE = os.getenv('BOT_MODE', 'polling')
WEBHOOK_URL = os.getenv('WEBHOOK_URL')
WEBHOOK_PATH = '/webhook'
WEB_SERVER_HOST = '0.0.0.0'
WEB_SERVER_PORT = 8443

async def on_startup():
    """Initialize database and other services"""
    await db.init()
    logger.info("Database initialized")

async def on_shutdown():
    """Cleanup resources"""
    await db.close()
    logger.info("Database connection closed")

async def main():
    """Main function to run the bot"""
    if not BOT_TOKEN:
        logger.error("BOT_TOKEN not found in environment variables")
        return

    # Initialize bot and dispatcher
    bot = Bot(token=BOT_TOKEN)
    dp = Dispatcher(storage=MemoryStorage())
    
    # Initialize real systems
    from handlers.real_chat import chat_manager, RealChatManager
    from handlers.real_matching import matching_manager, RealMatchingManager
    
    # Set up real managers
    real_chat.chat_manager = RealChatManager(bot)
    real_matching.matching_manager = RealMatchingManager(bot)
    
    # Include routers (specific handlers first, then general ones)
    dp.include_router(registration.router)
    dp.include_router(real_chat.router)  # Real chat system
    dp.include_router(discovery.router)  # Discovery system (before real_matching)
    dp.include_router(real_matching.router)  # Real matching system
    dp.include_router(profile_handler.router)  # Profile handler
    dp.include_router(photo_upload.router)  # Photo upload functionality
    dp.include_router(my_likes.router)  # My likes handler (before notifications)
    dp.include_router(my_chats.router)  # My chats handler
    dp.include_router(notifications.router)  # Notification system
    dp.include_router(chat.router)  # Fallback chat (for testing)
    dp.include_router(matching.router)  # Fallback matching
    dp.include_router(safety.router)
    dp.include_router(premium_handler.router)  # Premium handler
    dp.include_router(settings.router)  # Settings handler
    dp.include_router(help_handler.router)  # Help handler
    dp.include_router(admin.router)
    
    # Set up startup and shutdown handlers
    dp.startup.register(on_startup)
    dp.shutdown.register(on_shutdown)
    
    if BOT_MODE == 'webhook':
        # Webhook mode
        logger.info("Starting bot in webhook mode")
        
        # Set webhook
        await bot.set_webhook(
            url=f"{WEBHOOK_URL}{WEBHOOK_PATH}",
            drop_pending_updates=True
        )
        
        # Create aiohttp application
        app = web.Application()
        
        # Create webhook handler
        webhook_handler = SimpleRequestHandler(
            dispatcher=dp,
            bot=bot
        )
        
        # Register webhook handler
        webhook_handler.register(app, path=WEBHOOK_PATH)
        
        # Setup application
        setup_application(app, dp, bot=bot)
        
        # Start web server
        runner = web.AppRunner(app)
        await runner.setup()
        site = web.TCPSite(runner, WEB_SERVER_HOST, WEB_SERVER_PORT)
        await site.start()
        
        logger.info(f"Webhook server started on {WEB_SERVER_HOST}:{WEB_SERVER_PORT}")
        
        # Keep the server running
        try:
            await asyncio.Future()  # Run forever
        finally:
            await runner.cleanup()
    else:
        # Polling mode
        logger.info("Starting bot in polling mode")
        await dp.start_polling(bot, skip_updates=True)

if __name__ == '__main__':
    try:
        asyncio.run(main())
    except KeyboardInterrupt:
        logger.info("Bot stopped by user")
    except Exception as e:
        logger.error(f"Bot crashed: {e}")
        raise
