#!/bin/bash

# ChatDate Bot Deployment Script
# This script helps deploy the bot using Docker Compose

set -e

# Colors for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
NC='\033[0m' # No Color

# Function to print colored output
print_message() {
    echo -e "${2}${1}${NC}"
}

# Function to check if command exists
command_exists() {
    command -v "$1" >/dev/null 2>&1
}

# Function to check prerequisites
check_prerequisites() {
    print_message "Checking prerequisites..." "$YELLOW"
    
    if ! command_exists docker; then
        print_message "Error: Docker is not installed" "$RED"
        exit 1
    fi
    
    if ! command_exists docker-compose; then
        print_message "Error: Docker Compose is not installed" "$RED"
        exit 1
    fi
    
    if [ ! -f ".env" ]; then
        print_message "Creating .env file from template..." "$YELLOW"
        cp .env.example .env
        print_message "Please edit .env file with your configuration" "$RED"
        exit 1
    fi
    
    print_message "Prerequisites check passed!" "$GREEN"
}

# Function to deploy in development mode
deploy_dev() {
    print_message "Deploying in development mode (polling)..." "$YELLOW"
    
    # Start services
    docker-compose up -d postgres redis
    
    # Wait for PostgreSQL to be ready
    print_message "Waiting for PostgreSQL to be ready..." "$YELLOW"
    sleep 10
    
    # Initialize database if needed
    if [ "$1" == "--init-db" ]; then
        print_message "Initializing database..." "$YELLOW"
        docker-compose run --rm bot python database/init_db.py
    fi
    
    # Start bot
    docker-compose up -d bot
    
    print_message "Development deployment complete!" "$GREEN"
    print_message "View logs: docker-compose logs -f bot" "$YELLOW"
}

# Function to deploy in production mode
deploy_prod() {
    print_message "Deploying in production mode (webhook)..." "$YELLOW"
    
    # Check for SSL certificates
    if [ ! -d "./ssl" ] || [ ! -f "./ssl/cert.pem" ] || [ ! -f "./ssl/key.pem" ]; then
        print_message "Warning: SSL certificates not found in ./ssl directory" "$RED"
        print_message "Please add cert.pem and key.pem to ./ssl directory for HTTPS" "$YELLOW"
    fi
    
    # Start all services including nginx
    docker-compose --profile webhook up -d
    
    # Wait for PostgreSQL to be ready
    print_message "Waiting for PostgreSQL to be ready..." "$YELLOW"
    sleep 10
    
    # Initialize database if needed
    case "$1" in
        init-db)
            echo "Initializing database..."
            docker-compose exec bot python -c "
from database.database import db
from database.init_db import init_database
import asyncio
asyncio.run(init_database())
"
            echo "Database initialized!"
            ;;
        migrate)
            echo "Running database migrations..."
            docker-compose exec bot python database/migrate.py
            echo "Migrations completed!"
            ;;
    esac
    
    print_message "Production deployment complete!" "$GREEN"
    print_message "View logs: docker-compose logs -f" "$YELLOW"
}

# Function to stop services
stop_services() {
    print_message "Stopping all services..." "$YELLOW"
    docker-compose down
    print_message "Services stopped!" "$GREEN"
}

# Function to update deployment
update_deployment() {
    print_message "Updating deployment..." "$YELLOW"
    
    # Pull latest code
    if command_exists git; then
        git pull origin main
    fi
    
    # Rebuild images
    docker-compose build --no-cache
    
    # Restart services
    docker-compose down
    docker-compose up -d
    
    print_message "Deployment updated!" "$GREEN"
}

# Function to show logs
show_logs() {
    service=$1
    if [ -z "$service" ]; then
        docker-compose logs -f
    else
        docker-compose logs -f "$service"
    fi
}

# Function to backup database
backup_database() {
    print_message "Creating database backup..." "$YELLOW"
    
    timestamp=$(date +%Y%m%d_%H%M%S)
    backup_file="backup_${timestamp}.sql"
    
    docker-compose exec -T postgres pg_dump -U chatdate_user chatdate > "./backups/${backup_file}"
    
    print_message "Database backed up to ./backups/${backup_file}" "$GREEN"
}

# Function to restore database
restore_database() {
    if [ -z "$1" ]; then
        print_message "Error: Please provide backup file path" "$RED"
        exit 1
    fi
    
    print_message "Restoring database from $1..." "$YELLOW"
    
    docker-compose exec -T postgres psql -U chatdate_user chatdate < "$1"
    
    print_message "Database restored!" "$GREEN"
}

# Main script
main() {
    case "$1" in
        dev)
            check_prerequisites
            deploy_dev "$2"
            ;;
        prod)
            check_prerequisites
            deploy_prod "$2"
            ;;
        stop)
            stop_services
            ;;
        update)
            check_prerequisites
            update_deployment
            ;;
        logs)
            show_logs "$2"
            ;;
        backup)
            backup_database
            ;;
        restore)
            restore_database "$2"
            ;;
        *)
            print_message "ChatDate Bot Deployment Script" "$GREEN"
            echo ""
            echo "Usage: $0 {dev|prod|stop|update|logs|backup|restore} [options]"
            echo ""
            echo "Commands:"
            echo "  dev [--init-db]     Deploy in development mode (polling)"
            echo "  prod [--init-db]    Deploy in production mode (webhook)"
            echo "  stop                Stop all services"
            echo "  update              Update and restart deployment"
            echo "  logs [service]      Show logs (optionally for specific service)"
            echo "  backup              Backup database"
            echo "  restore <file>      Restore database from backup"
            echo ""
            echo "Examples:"
            echo "  $0 dev --init-db    # First time development deployment"
            echo "  $0 prod             # Production deployment"
            echo "  $0 logs bot         # Show bot logs"
            echo "  $0 backup           # Create database backup"
            exit 1
            ;;
    esac
}

# Create necessary directories
mkdir -p backups ssl logs

# Run main function
main "$@"
