#!/bin/bash

# ChatDate Bot Management Script
# Administrative tasks and utilities

set -e

# Colors
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m'

# Database connection
DB_CONTAINER="chatdate_postgres"
DB_USER="chatdate_user"
DB_NAME="chatdate"

# Function to execute SQL query
exec_sql() {
    docker exec -i "$DB_CONTAINER" psql -U "$DB_USER" -d "$DB_NAME" -c "$1"
}

# Function to get user statistics
user_stats() {
    echo -e "${BLUE}User Statistics${NC}"
    echo -e "${BLUE}═══════════════${NC}\n"
    
    total=$(docker exec "$DB_CONTAINER" psql -U "$DB_USER" -d "$DB_NAME" -t -c "SELECT COUNT(*) FROM users;" 2>/dev/null || echo "0")
    active=$(docker exec "$DB_CONTAINER" psql -U "$DB_USER" -d "$DB_NAME" -t -c "SELECT COUNT(*) FROM users WHERE is_active = true;" 2>/dev/null || echo "0")
    banned=$(docker exec "$DB_CONTAINER" psql -U "$DB_USER" -d "$DB_NAME" -t -c "SELECT COUNT(*) FROM users WHERE is_banned = true;" 2>/dev/null || echo "0")
    premium=$(docker exec "$DB_CONTAINER" psql -U "$DB_USER" -d "$DB_NAME" -t -c "SELECT COUNT(*) FROM users WHERE is_premium = true;" 2>/dev/null || echo "0")
    
    echo -e "Total Users: ${GREEN}$total${NC}"
    echo -e "Active Users: ${GREEN}$active${NC}"
    echo -e "Banned Users: ${RED}$banned${NC}"
    echo -e "Premium Users: ${YELLOW}$premium${NC}"
    
    echo -e "\n${BLUE}Registration by Language:${NC}"
    exec_sql "SELECT language, COUNT(*) as count FROM users GROUP BY language ORDER BY count DESC;"
    
    echo -e "\n${BLUE}Users by Gender:${NC}"
    exec_sql "SELECT gender, COUNT(*) as count FROM users GROUP BY gender ORDER BY count DESC;"
}

# Function to get match statistics
match_stats() {
    echo -e "${BLUE}Match Statistics${NC}"
    echo -e "${BLUE}════════════════${NC}\n"
    
    total_matches=$(docker exec "$DB_CONTAINER" psql -U "$DB_USER" -d "$DB_NAME" -t -c "SELECT COUNT(*) FROM matches;" 2>/dev/null || echo "0")
    active_chats=$(docker exec "$DB_CONTAINER" psql -U "$DB_USER" -d "$DB_NAME" -t -c "SELECT COUNT(*) FROM matches WHERE status = 'active';" 2>/dev/null || echo "0")
    
    echo -e "Total Matches: ${GREEN}$total_matches${NC}"
    echo -e "Active Chats: ${GREEN}$active_chats${NC}"
    
    echo -e "\n${BLUE}Match Status Distribution:${NC}"
    exec_sql "SELECT status, COUNT(*) as count FROM matches GROUP BY status ORDER BY count DESC;"
    
    echo -e "\n${BLUE}Today's Matches:${NC}"
    exec_sql "SELECT COUNT(*) FROM matches WHERE DATE(created_at) = CURRENT_DATE;"
}

# Function to ban a user
ban_user() {
    user_id=$1
    if [ -z "$user_id" ]; then
        echo -e "${RED}Error: Please provide user ID${NC}"
        exit 1
    fi
    
    echo -e "${YELLOW}Banning user $user_id...${NC}"
    exec_sql "UPDATE users SET is_banned = true, ban_reason = 'Admin ban', banned_at = NOW() WHERE telegram_id = $user_id;"
    echo -e "${GREEN}User $user_id has been banned${NC}"
}

# Function to unban a user
unban_user() {
    user_id=$1
    if [ -z "$user_id" ]; then
        echo -e "${RED}Error: Please provide user ID${NC}"
        exit 1
    fi
    
    echo -e "${YELLOW}Unbanning user $user_id...${NC}"
    exec_sql "UPDATE users SET is_banned = false, ban_reason = NULL, banned_at = NULL WHERE telegram_id = $user_id;"
    echo -e "${GREEN}User $user_id has been unbanned${NC}"
}

# Function to grant premium
grant_premium() {
    user_id=$1
    days=${2:-30}
    
    if [ -z "$user_id" ]; then
        echo -e "${RED}Error: Please provide user ID${NC}"
        exit 1
    fi
    
    echo -e "${YELLOW}Granting $days days premium to user $user_id...${NC}"
    exec_sql "UPDATE users SET is_premium = true, premium_until = NOW() + INTERVAL '$days days' WHERE telegram_id = $user_id;"
    echo -e "${GREEN}Premium granted to user $user_id for $days days${NC}"
}

# Function to view recent reports
view_reports() {
    echo -e "${BLUE}Recent Reports (Last 10)${NC}"
    echo -e "${BLUE}════════════════════════${NC}\n"
    
    exec_sql "SELECT r.id, r.reporter_id, r.reported_id, r.reason, r.created_at 
              FROM reports r 
              ORDER BY r.created_at DESC 
              LIMIT 10;"
}

# Function to clean old data
clean_old_data() {
    days=${1:-30}
    
    echo -e "${YELLOW}Cleaning data older than $days days...${NC}"
    
    # Delete old inactive matches
    exec_sql "DELETE FROM matches WHERE status = 'ended' AND updated_at < NOW() - INTERVAL '$days days';"
    
    # Delete old swipes
    exec_sql "DELETE FROM swipes WHERE created_at < NOW() - INTERVAL '$days days';"
    
    # Delete old reports
    exec_sql "DELETE FROM reports WHERE created_at < NOW() - INTERVAL '90 days';"
    
    echo -e "${GREEN}Old data cleaned${NC}"
}

# Function to export user data
export_user_data() {
    user_id=$1
    if [ -z "$user_id" ]; then
        echo -e "${RED}Error: Please provide user ID${NC}"
        exit 1
    fi
    
    output_file="user_${user_id}_export_$(date +%Y%m%d_%H%M%S).json"
    
    echo -e "${YELLOW}Exporting data for user $user_id...${NC}"
    
    docker exec "$DB_CONTAINER" psql -U "$DB_USER" -d "$DB_NAME" -t -c \
        "SELECT row_to_json(u) FROM users u WHERE telegram_id = $user_id;" > "$output_file"
    
    echo -e "${GREEN}User data exported to $output_file${NC}"
}

# Function to send broadcast message
send_broadcast() {
    message=$1
    if [ -z "$message" ]; then
        echo -e "${RED}Error: Please provide message text${NC}"
        exit 1
    fi
    
    echo -e "${YELLOW}This will send a message to ALL active users. Are you sure? (yes/no)${NC}"
    read -r confirmation
    
    if [ "$confirmation" != "yes" ]; then
        echo -e "${RED}Broadcast cancelled${NC}"
        exit 0
    fi
    
    # Create broadcast record
    exec_sql "INSERT INTO broadcasts (message, sent_at) VALUES ('$message', NOW());"
    
    echo -e "${GREEN}Broadcast message queued for sending${NC}"
    echo -e "${YELLOW}Note: Actual sending requires bot implementation${NC}"
}

# Function to reset user matching
reset_user_matches() {
    user_id=$1
    if [ -z "$user_id" ]; then
        echo -e "${RED}Error: Please provide user ID${NC}"
        exit 1
    fi
    
    echo -e "${YELLOW}Resetting matches for user $user_id...${NC}"
    
    # Delete user's swipes
    exec_sql "DELETE FROM swipes WHERE user_id = $user_id;"
    
    # End active matches
    exec_sql "UPDATE matches SET status = 'ended' WHERE (user1_id = $user_id OR user2_id = $user_id) AND status = 'active';"
    
    echo -e "${GREEN}Matches reset for user $user_id${NC}"
}

# Function to show system info
system_info() {
    echo -e "${BLUE}System Information${NC}"
    echo -e "${BLUE}══════════════════${NC}\n"
    
    # Docker info
    echo -e "${BLUE}Docker Containers:${NC}"
    docker ps --format "table {{.Names}}\t{{.Status}}\t{{.Size}}"
    
    echo -e "\n${BLUE}Docker Volumes:${NC}"
    docker volume ls --format "table {{.Name}}\t{{.Driver}}"
    
    echo -e "\n${BLUE}Database Size:${NC}"
    exec_sql "SELECT pg_database_size('$DB_NAME') / 1024 / 1024 as size_mb;"
    
    echo -e "\n${BLUE}Table Sizes:${NC}"
    exec_sql "SELECT tablename, pg_size_pretty(pg_total_relation_size(tablename::regclass)) as size 
              FROM pg_tables 
              WHERE schemaname = 'public' 
              ORDER BY pg_total_relation_size(tablename::regclass) DESC;"
}

# Main menu
show_menu() {
    echo -e "${BLUE}═══════════════════════════════════════${NC}"
    echo -e "${BLUE}     ChatDate Bot Management${NC}"
    echo -e "${BLUE}═══════════════════════════════════════${NC}"
    echo ""
    echo "Usage: $0 <command> [options]"
    echo ""
    echo "Commands:"
    echo "  user-stats              Show user statistics"
    echo "  match-stats             Show match statistics"
    echo "  ban <user_id>           Ban a user"
    echo "  unban <user_id>         Unban a user"
    echo "  premium <user_id> [days] Grant premium (default 30 days)"
    echo "  reports                 View recent reports"
    echo "  clean [days]            Clean old data (default 30 days)"
    echo "  export <user_id>        Export user data (GDPR)"
    echo "  broadcast <message>     Send broadcast message"
    echo "  reset <user_id>         Reset user matches"
    echo "  system                  Show system information"
    echo ""
    echo "Examples:"
    echo "  $0 user-stats"
    echo "  $0 ban 123456789"
    echo "  $0 premium 123456789 90"
    echo "  $0 clean 60"
}

# Main script
case "$1" in
    user-stats)
        user_stats
        ;;
    match-stats)
        match_stats
        ;;
    ban)
        ban_user "$2"
        ;;
    unban)
        unban_user "$2"
        ;;
    premium)
        grant_premium "$2" "$3"
        ;;
    reports)
        view_reports
        ;;
    clean)
        clean_old_data "$2"
        ;;
    export)
        export_user_data "$2"
        ;;
    broadcast)
        send_broadcast "$2"
        ;;
    reset)
        reset_user_matches "$2"
        ;;
    system)
        system_info
        ;;
    *)
        show_menu
        ;;
esac
