#!/usr/bin/env python3
"""
Test Database Connection for ChatDate Bot
تست اتصال دیتابیس قبل از راه‌اندازی اصلی
"""

import os
import sys
from dotenv import load_dotenv
from urllib.parse import urlparse

# Load environment variables
load_dotenv()

def test_mysql_connection():
    """Test MySQL connection"""
    try:
        import mysql.connector
        from mysql.connector import Error
        
        database_url = os.getenv('DATABASE_URL')
        if not database_url:
            print("❌ DATABASE_URL not found in .env file")
            return False
        
        # Parse database URL
        parsed = urlparse(database_url)
        
        print("🔗 Testing MySQL connection...")
        print(f"Host: {parsed.hostname}")
        print(f"Database: {parsed.path.lstrip('/')}")
        print(f"User: {parsed.username}")
        print("Password: [HIDDEN]")
        print("-" * 40)
        
        # Connect to database
        connection = mysql.connector.connect(
            host=parsed.hostname or 'localhost',
            port=parsed.port or 3306,
            user=parsed.username,
            password=parsed.password,
            database=parsed.path.lstrip('/')
        )
        
        if connection.is_connected():
            cursor = connection.cursor()
            cursor.execute("SELECT VERSION()")
            version = cursor.fetchone()
            
            print("✅ Connection successful!")
            print(f"MySQL Version: {version[0]}")
            
            # Test basic operations
            cursor.execute("SHOW TABLES")
            tables = cursor.fetchall()
            print(f"Existing tables: {len(tables)}")
            
            cursor.close()
            connection.close()
            return True
            
    except ImportError:
        print("❌ mysql-connector-python not installed")
        print("Run: pip install mysql-connector-python")
        return False
    except Error as e:
        print(f"❌ MySQL Error: {e}")
        return False
    except Exception as e:
        print(f"❌ Unexpected error: {e}")
        return False

def main():
    print("=" * 50)
    print("🧪 ChatDate Bot - Database Connection Test")
    print("=" * 50)
    
    if test_mysql_connection():
        print("\n🎉 Database connection test passed!")
        print("✅ You can proceed with bot setup")
    else:
        print("\n❌ Database connection failed!")
        print("🔧 Please check:")
        print("1. Database credentials in .env file")
        print("2. Database server is running")
        print("3. User has proper permissions")
        print("4. mysql-connector-python is installed")

if __name__ == "__main__":
    main()
