import math
import geohash2
from typing import List, Dict, Optional, Tuple
from datetime import datetime, timedelta
from sqlalchemy import and_, or_, func, text, select
from sqlalchemy.ext.asyncio import AsyncSession
from database.models import User, Preference, Match, Swipe

class MatchingEngine:
    def __init__(self):
        # Matching algorithm weights
        self.weights = {
            'gender': 0.4,      # Gender preference match
            'distance': 0.25,   # Geographic proximity
            'activity': 0.15,   # Recent activity
            'language': 0.1,    # Common languages
            'interest': 0.1     # Common interests
        }
    
    def haversine_distance(self, lat1: float, lon1: float, lat2: float, lon2: float) -> float:
        """Calculate distance between two points using Haversine formula"""
        R = 6371.0  # Earth radius in kilometers
        
        lat1_rad = math.radians(lat1)
        lon1_rad = math.radians(lon1)
        lat2_rad = math.radians(lat2)
        lon2_rad = math.radians(lon2)
        
        dlat = lat2_rad - lat1_rad
        dlon = lon2_rad - lon1_rad
        
        a = math.sin(dlat/2)**2 + math.cos(lat1_rad) * math.cos(lat2_rad) * math.sin(dlon/2)**2
        c = 2 * math.asin(math.sqrt(a))
        
        return R * c
    
    def decode_geohash(self, geohash_str: str) -> Tuple[float, float]:
        """Decode geohash to lat, lon"""
        if not geohash_str:
            return None, None
        try:
            return geohash2.decode(geohash_str)
        except:
            return None, None
    
    def encode_geohash(self, lat: float, lon: float, precision: int = 7) -> str:
        """Encode lat, lon to geohash with reduced precision for privacy"""
        return geohash2.encode(lat, lon, precision)
    
    def calculate_match_score(self, user: User, candidate: User, distance_km: Optional[float] = None) -> float:
        """Calculate match score between two users"""
        score = 0.0
        
        # Gender preference match
        gender_score = 0.0
        if user.looking_for == 0 or user.looking_for == candidate.gender:  # Any or specific match
            gender_score = 1.0
        if candidate.looking_for == 0 or candidate.looking_for == user.gender:  # Mutual interest
            gender_score = max(gender_score, 1.0)
        else:
            gender_score = 0.0  # No mutual interest
        
        score += self.weights['gender'] * gender_score
        
        # Distance score
        distance_score = 0.0
        if distance_km is not None:
            max_radius = max(user.radius_km, candidate.radius_km)
            if distance_km <= max_radius:
                distance_score = 1.0 - min(distance_km / max_radius, 1.0)
        else:
            # Same city/country fallback
            if user.city and candidate.city and user.city.lower() == candidate.city.lower():
                distance_score = 0.8
            elif user.country_code and candidate.country_code and user.country_code == candidate.country_code:
                distance_score = 0.4
        
        score += self.weights['distance'] * distance_score
        
        # Activity score (recently active users get higher priority)
        activity_score = 0.0
        if candidate.last_active:
            hours_since_active = (datetime.utcnow() - candidate.last_active).total_seconds() / 3600
            if hours_since_active <= 24:
                activity_score = 1.0
            elif hours_since_active <= 72:
                activity_score = 0.5
            else:
                activity_score = 0.1
        
        score += self.weights['activity'] * activity_score
        
        # Language overlap
        language_score = 0.0
        if user.languages and candidate.languages:
            common_languages = set(user.languages) & set(candidate.languages)
            if common_languages:
                language_score = min(len(common_languages) / max(len(user.languages), len(candidate.languages)), 1.0)
        
        score += self.weights['language'] * language_score
        
        # Interest overlap
        interest_score = 0.0
        if user.interests and candidate.interests:
            common_interests = set(user.interests) & set(candidate.interests)
            if common_interests:
                interest_score = min(len(common_interests) / max(len(user.interests), len(candidate.interests)), 1.0)
        
        score += self.weights['interest'] * interest_score
        
        return min(score, 1.0)  # Cap at 1.0
    
    async def find_matches(self, session: AsyncSession, user_id: int, limit: int = 10) -> List[Dict]:
        """Find potential matches for a user"""
        # Get user and preferences
        user = await session.get(User, user_id)
        if not user or user.is_banned:
            return []
        
        user_pref = user.preferences
        if not user_pref:
            return []
        
        # Calculate age range
        current_year = datetime.now().year
        
        # Build base conditions
        conditions = [
            User.user_id != user_id,
            User.is_banned == False,
            User.last_active >= datetime.utcnow() - timedelta(days=30)
        ]
        
        # Gender filter
        if user_pref.gender_pref > 0:
            conditions.append(User.gender == user_pref.gender_pref)
        
        # Age filter
        if user_pref.age_min and user_pref.age_max:
            birth_year_max = current_year - user_pref.age_min
            birth_year_min = current_year - user_pref.age_max
            conditions.extend([
                User.birth_year >= birth_year_min,
                User.birth_year <= birth_year_max
            ])
        
        # Location filter
        if user_pref.country_code:
            conditions.append(User.country == user_pref.country_code)
        
        if user_pref.city:
            conditions.append(User.city.ilike(f"%{user_pref.city}%"))
        
        # Get excluded user IDs (matches and recent swipes)
        excluded_ids = set()
        
        # Get matched users
        matches_result = await session.execute(
            select(Match.user_a, Match.user_b).where(
                and_(
                    or_(Match.user_a == user_id, Match.user_b == user_id),
                    Match.status.in_([0, 1])
                )
            )
        )
        for match in matches_result:
            excluded_ids.add(match.user_a if match.user_b == user_id else match.user_b)
        
        # Get recently swiped users
        swipes_result = await session.execute(
            select(Swipe.to_user).where(
                and_(
                    Swipe.from_user == user_id,
                    Swipe.created_at >= datetime.utcnow() - timedelta(hours=24)
                )
            )
        )
        for swipe in swipes_result:
            excluded_ids.add(swipe.to_user)
        
        # Add exclusion condition
        if excluded_ids:
            conditions.append(User.user_id.notin_(excluded_ids))
        
        # Execute query
        result = await session.execute(
            select(User).where(and_(*conditions)).limit(limit * 3)
        )
        candidates = result.scalars().all()
        
        # Score candidates
        scored_candidates = []
        
        for candidate in candidates:
            distance_km = None
            if user_lat and user_lon and candidate_lat and candidate_lon:
                distance_km = self.haversine_distance(user_lat, user_lon, candidate_lat, candidate_lon)
                
                # Skip if too far
                max_radius = user_pref.radius_km or user.radius_km or 50
                if distance_km > max_radius:
                    continue
            
            score = self.calculate_match_score(user, candidate, distance_km)
            
            if score > 0.1:  # Minimum threshold
                scored_candidates.append({
                    'user': candidate,
                    'score': score,
                    'distance_km': distance_km
                })
        
        # Sort by score and return top matches
        scored_candidates.sort(key=lambda x: x['score'], reverse=True)
        return scored_candidates[:limit]
    
    async def create_match(self, session: AsyncSession, user_a_id: int, user_b_id: int) -> Optional[Match]:
        """Create a new match between two users"""
        # Check if match already exists
        existing_match = await session.execute(
            select(Match).where(
                or_(
                    and_(Match.user_a == user_a_id, Match.user_b == user_b_id),
                    and_(Match.user_a == user_b_id, Match.user_b == user_a_id)
                )
            )
        )
        
        if existing_match.scalar():
            return None
        
        # Create new match
        match = Match(
            user_a=user_a_id,
            user_b=user_b_id,
            status=0  # pending
        )
        
        session.add(match)
        await session.flush()
        return match
    
    async def check_mutual_like(self, session: AsyncSession, user_a_id: int, user_b_id: int) -> bool:
        """Check if both users liked each other"""
        like_a_to_b = await session.execute(
            select(Swipe).where(
                and_(
                    Swipe.from_user == user_a_id,
                    Swipe.to_user == user_b_id,
                    Swipe.action == 1
                )
            )
        )
        
        like_b_to_a = await session.execute(
            select(Swipe).where(
                and_(
                    Swipe.from_user == user_b_id,
                    Swipe.to_user == user_a_id,
                    Swipe.action == 1
                )
            )
        )
        
        return like_a_to_b.scalar() is not None and like_b_to_a.scalar() is not None

# Global matching engine instance
matching_engine = MatchingEngine()
