"""
Helper برای مدیریت منوی احراز هویت
"""

from database.database import db
from database.models import User
from sqlalchemy import select
import logging

logger = logging.getLogger(__name__)

async def get_user_auth_status(user_id: int) -> tuple[str, bool, bool]:
    """
    دریافت وضعیت احراز هویت کاربر
    
    Returns:
        tuple: (language_code, is_admin, is_guest)
    """
    async with db.get_session() as session:
        result = await session.execute(select(User).where(User.id == user_id))
        user = result.scalar_one_or_none()
        
        if not user:
            return 'fa', False, False
        
        lang = user.language_code or 'fa'
        is_guest = user.is_guest or False
        
        # بررسی ادمین بودن
        import os
        ADMIN_USER_IDS = [int(uid) for uid in os.getenv('ADMIN_USER_IDS', '').split(',') if uid]
        is_admin = user_id in ADMIN_USER_IDS
        
        return lang, is_admin, is_guest

async def update_user_to_guest(user_id: int) -> bool:
    """
    تبدیل کاربر به مهمان (حذف ایمیل و پسورد)
    
    Returns:
        bool: موفقیت عملیات
    """
    try:
        async with db.get_session() as session:
            result = await session.execute(select(User).where(User.id == user_id))
            user = result.scalar_one_or_none()
            
            if not user:
                return False
            
            # تبدیل به مهمان
            user.is_guest = True
            user.email = None
            user.password_hash = None
            user.email_verified = False
            
            await session.commit()
            logger.info(f"User {user_id} converted to guest")
            return True
            
    except Exception as e:
        logger.error(f"Error converting user {user_id} to guest: {e}")
        return False

async def update_user_from_guest(user_id: int, email: str, password_hash: str) -> bool:
    """
    تبدیل کاربر مهمان به کاربر ثبت‌نام شده
    
    Returns:
        bool: موفقیت عملیات
    """
    try:
        async with db.get_session() as session:
            result = await session.execute(select(User).where(User.id == user_id))
            user = result.scalar_one_or_none()
            
            if not user:
                return False
            
            # تبدیل به کاربر ثبت‌نام شده
            user.is_guest = False
            user.email = email
            user.password_hash = password_hash
            user.email_verified = True
            
            await session.commit()
            logger.info(f"User {user_id} converted from guest to registered")
            return True
            
    except Exception as e:
        logger.error(f"Error converting user {user_id} from guest: {e}")
        return False
