#!/usr/bin/env python3
"""سیستم خودکار اتاق گفتگو - ایجاد گروه با Bot API"""

from aiogram import Router, F, Bot
from aiogram.types import Message, InlineKeyboardMarkup, InlineKeyboardButton, CallbackQuery
from aiogram.types import ChatPermissions, ChatMemberAdministrator
from aiogram.fsm.context import FSMContext
from database.database import db
from database.models import User, Match, Swipe
from sqlalchemy import select, and_, or_
from datetime import datetime
import logging
import hashlib
import random
import asyncio

router = Router()
logger = logging.getLogger(__name__)

# ذخیره اطلاعات گروه‌های فعال
active_rooms = {}  # {room_code: {"group_id": id, "user1": id, "user2": id, "created": datetime}}
pending_rooms = {}  # اتاق‌های در انتظار تایید

def generate_room_code() -> str:
    """ایجاد کد 6 رقمی برای اتاق"""
    return str(random.randint(100000, 999999))

async def create_private_chat_group(bot: Bot, user1_id: int, user2_id: int, user1_name: str, user2_name: str):
    """
    ایجاد گروه خصوصی با استفاده از Bot API
    
    Returns: (success, invite_link, room_code, message)
    """
    try:
        room_code = generate_room_code()
        group_title = f"💬 اتاق گفتگو #{room_code}"
        
        # ذخیره اطلاعات اتاق در حالت pending
        pending_rooms[room_code] = {
            "user1": user1_id,
            "user2": user2_id,
            "user1_name": user1_name,
            "user2_name": user2_name,
            "created": datetime.now(),
            "status": "pending"
        }
        
        # ایجاد لینک دعوت خاص برای این گروه
        # توجه: بات نمی‌تواند مستقیماً گروه بسازد، اما می‌تواند لینک خاص بسازد
        
        # روش 1: استفاده از deeplink
        bot_username = (await bot.get_me()).username
        deeplink = f"https://t.me/{bot_username}?startgroup={room_code}"
        
        # پیام راهنما برای کاربران
        invite_message = f"""
🎉 <b>اتاق گفتگو آماده است!</b>

📍 <b>کد اتاق:</b> <code>{room_code}</code>

👥 <b>شرکت‌کنندگان:</b>
• {user1_name}
• {user2_name}

🚀 <b>برای شروع گفتگو:</b>

<b>روش 1 (توصیه شده):</b>
1️⃣ روی لینک زیر کلیک کنید:
{deeplink}

2️⃣ گروه جدید بسازید با نام:
<code>{group_title}</code>

3️⃣ بات به صورت خودکار اضافه می‌شود

4️⃣ کاربر مقابل را به گروه دعوت کنید

<b>روش 2 (دستی):</b>
1️⃣ یک گروه جدید بسازید
2️⃣ بات @{bot_username} را اضافه کنید
3️⃣ دستور زیر را در گروه بزنید:
<code>/activate_{room_code}</code>

⚠️ <b>قوانین اتاق:</b>
• احترام متقابل
• عدم ارسال محتوای نامناسب
• امکان گزارش تخلف با /report

💡 برای خروج: /leave
❓ راهنما: /help
"""
        
        return True, deeplink, room_code, invite_message
        
    except Exception as e:
        logger.error(f"Error creating chat group: {e}")
        return False, None, None, f"خطا در ایجاد اتاق: {str(e)}"

async def send_invitation_to_users(bot: Bot, user1_id: int, user2_id: int, room_code: str, invite_message: str, deeplink: str):
    """ارسال دعوتنامه به هر دو کاربر"""
    
    # کیبورد برای کاربر اول
    keyboard1 = InlineKeyboardMarkup(inline_keyboard=[
        [InlineKeyboardButton(text="🚀 ایجاد گروه", url=deeplink)],
        [InlineKeyboardButton(text="✅ گروه را ساختم", callback_data=f"confirm_group_{room_code}")],
        [InlineKeyboardButton(text="❌ انصراف", callback_data=f"cancel_room_{room_code}")]
    ])
    
    # کیبورد برای کاربر دوم
    keyboard2 = InlineKeyboardMarkup(inline_keyboard=[
        [InlineKeyboardButton(text="⏳ منتظر ایجاد گروه", callback_data=f"waiting_{room_code}")],
        [InlineKeyboardButton(text="❌ انصراف", callback_data=f"cancel_room_{room_code}")]
    ])
    
    try:
        # ارسال به کاربر اول (سازنده گروه)
        await bot.send_message(
            user1_id,
            invite_message + "\n\n💡 <b>شما باید گروه را بسازید.</b>",
            parse_mode='HTML',
            reply_markup=keyboard1
        )
        
        # ارسال به کاربر دوم (منتظر دعوت)
        await bot.send_message(
            user2_id,
            f"🔔 <b>دعوت به اتاق گفتگو</b>\n\n"
            f"📍 کد اتاق: <code>{room_code}</code>\n\n"
            f"⏳ کاربر مقابل در حال ایجاد گروه است.\n"
            f"لطفاً منتظر دعوتنامه باشید...",
            parse_mode='HTML',
            reply_markup=keyboard2
        )
        
        return True
        
    except Exception as e:
        logger.error(f"Error sending invitations: {e}")
        return False

# Handler برای زمانی که بات به گروه اضافه می‌شود
@router.message(F.new_chat_members)
async def on_bot_added_to_group(message: Message, bot: Bot):
    """زمانی که بات به گروه جدید اضافه می‌شود"""
    
    # بررسی اینکه آیا خود بات اضافه شده
    bot_id = (await bot.get_me()).id
    bot_added = any(member.id == bot_id for member in message.new_chat_members)
    
    if not bot_added:
        return
    
    # پیام خوش‌آمدگویی
    await message.answer(
        "👋 <b>سلام! من بات نظارتگر اتاق گفتگو هستم.</b>\n\n"
        "برای فعال‌سازی اتاق، لطفاً کد اتاق را وارد کنید:\n"
        "<code>/activate_کد6رقمی</code>\n\n"
        "مثال: <code>/activate_123456</code>",
        parse_mode='HTML'
    )

# Handler برای فعال‌سازی اتاق
@router.message(F.text.startswith("/activate_"))
async def activate_chat_room(message: Message, bot: Bot):
    """فعال‌سازی اتاق گفتگو در گروه"""
    
    # استخراج کد اتاق
    try:
        room_code = message.text.split("_")[1]
    except:
        await message.answer("❌ فرمت دستور اشتباه است. مثال: /activate_123456")
        return
    
    # بررسی وجود اتاق در pending_rooms
    if room_code not in pending_rooms:
        await message.answer("❌ کد اتاق نامعتبر یا منقضی شده است.")
        return
    
    room_info = pending_rooms[room_code]
    
    # بررسی اینکه کاربر مجاز است
    if message.from_user.id not in [room_info["user1"], room_info["user2"]]:
        await message.answer("❌ شما مجوز فعال‌سازی این اتاق را ندارید.")
        return
    
    # ذخیره اطلاعات گروه
    group_id = message.chat.id
    active_rooms[room_code] = {
        "group_id": group_id,
        "user1": room_info["user1"],
        "user2": room_info["user2"],
        "created": room_info["created"],
        "status": "active"
    }
    
    # حذف از pending
    del pending_rooms[room_code]
    
    # تنظیم دسترسی‌های بات (اگر ادمین باشد)
    try:
        # بررسی وضعیت ادمین بات
        bot_member = await bot.get_chat_member(group_id, bot.id)
        if isinstance(bot_member, ChatMemberAdministrator):
            # تنظیم permissions گروه
            await bot.set_chat_permissions(
                group_id,
                ChatPermissions(
                    can_send_messages=True,
                    can_send_media_messages=True,
                    can_send_polls=False,
                    can_send_other_messages=True,
                    can_add_web_page_previews=True,
                    can_change_info=False,
                    can_invite_users=True,
                    can_pin_messages=False
                )
            )
    except Exception as e:
        logger.warning(f"Could not set group permissions: {e}")
    
    # پیام تایید
    await message.answer(
        f"✅ <b>اتاق گفتگو #{room_code} فعال شد!</b>\n\n"
        f"👥 شرکت‌کنندگان:\n"
        f"• {room_info['user1_name']}\n"
        f"• {room_info['user2_name']}\n\n"
        f"💬 اکنون می‌توانید گفتگو را شروع کنید.\n\n"
        f"📝 <b>دستورات مفید:</b>\n"
        f"/rules - قوانین اتاق\n"
        f"/report - گزارش تخلف\n"
        f"/leave - خروج از اتاق\n"
        f"/help - راهنما",
        parse_mode='HTML'
    )
    
    # اطلاع‌رسانی به کاربر دوم
    try:
        await bot.send_message(
            room_info["user2"],
            f"✅ <b>اتاق گفتگو #{room_code} فعال شد!</b>\n\n"
            f"اکنون می‌توانید به گروه بپیوندید و گفتگو را شروع کنید.",
            parse_mode='HTML'
        )
    except:
        pass

# دستورات گروه
@router.message(F.text == "/rules")
async def show_rules(message: Message):
    """نمایش قوانین اتاق"""
    if message.chat.type not in ['group', 'supergroup']:
        return
    
    await message.answer(
        "📋 <b>قوانین اتاق گفتگو:</b>\n\n"
        "1️⃣ احترام متقابل را رعایت کنید\n"
        "2️⃣ از ارسال محتوای نامناسب خودداری کنید\n"
        "3️⃣ اطلاعات شخصی را به اشتراک نگذارید\n"
        "4️⃣ از تبلیغات و اسپم پرهیز کنید\n"
        "5️⃣ در صورت مشاهده تخلف، گزارش دهید\n\n"
        "⚠️ در صورت نقض قوانین، دسترسی شما محدود خواهد شد.",
        parse_mode='HTML'
    )

@router.message(F.text == "/leave")
async def leave_chat(message: Message, bot: Bot):
    """خروج از اتاق گفتگو"""
    if message.chat.type not in ['group', 'supergroup']:
        return
    
    # پیدا کردن اتاق
    room_code = None
    for code, room in active_rooms.items():
        if room["group_id"] == message.chat.id:
            room_code = code
            break
    
    if not room_code:
        await message.answer("❌ این گروه یک اتاق گفتگو فعال نیست.")
        return
    
    room = active_rooms[room_code]
    
    # بررسی دسترسی کاربر
    if message.from_user.id not in [room["user1"], room["user2"]]:
        await message.answer("❌ شما عضو این اتاق گفتگو نیستید.")
        return
    
    # حذف اتاق از لیست فعال
    del active_rooms[room_code]
    
    await message.answer(
        f"👋 <b>اتاق گفتگو #{room_code} بسته شد.</b>\n\n"
        f"از گفتگوی شما متشکریم!",
        parse_mode='HTML'
    )
    
    # خروج بات از گروه
    try:
        await bot.leave_chat(message.chat.id)
    except:
        pass

# Callback handlers
@router.callback_query(F.data.startswith("confirm_group_"))
async def handle_confirm_group(callback: CallbackQuery):
    """تایید ساخت گروه"""
    room_code = callback.data.split("_")[2]
    
    if room_code in pending_rooms:
        await callback.answer(
            "✅ عالی! اکنون دستور /activate_" + room_code + " را در گروه بزنید.",
            show_alert=True
        )
        await callback.message.edit_reply_markup(reply_markup=None)
    else:
        await callback.answer("❌ اتاق یافت نشد!", show_alert=True)

@router.callback_query(F.data.startswith("cancel_room_"))
async def handle_cancel_room(callback: CallbackQuery):
    """لغو اتاق"""
    room_code = callback.data.split("_")[2]
    
    if room_code in pending_rooms:
        del pending_rooms[room_code]
        await callback.answer("❌ اتاق گفتگو لغو شد.", show_alert=True)
        await callback.message.edit_text(
            "❌ اتاق گفتگو لغو شد.",
            parse_mode='HTML'
        )
    else:
        await callback.answer("اتاق قبلاً لغو شده است.", show_alert=True)

@router.callback_query(F.data.startswith("waiting_"))
async def handle_waiting(callback: CallbackQuery):
    """کاربر منتظر"""
    room_code = callback.data.split("_")[1]
    
    if room_code in pending_rooms:
        await callback.answer(
            "⏳ لطفاً منتظر بمانید تا کاربر مقابل گروه را بسازد...",
            show_alert=True
        )
    elif room_code in active_rooms:
        await callback.answer(
            "✅ اتاق فعال شد! منتظر دعوتنامه باشید.",
            show_alert=True
        )
    else:
        await callback.answer("❌ اتاق یافت نشد!", show_alert=True)
