"""
Bottom Menu Handler - هندلر منوی پایین
مدیریت منوی پایین با دکمه‌های استارت و پشتیبانی
"""

from aiogram import Router, Bot
from aiogram.types import BotCommand, BotCommandScopeDefault, Message
from aiogram.filters import Command
from localization.translator import get_text
import logging

router = Router()
logger = logging.getLogger(__name__)

async def setup_bot_menu(bot: Bot, user_language='fa'):
    """تنظیم منوی پایین بات - فقط برای کاربران ثبت‌نام شده"""
    try:
        # تعریف دستورات منو با پشتیبانی چندزبانه
        from localization.translator import get_text
        
        # دستورات محدود برای کاربران ثبت‌نام نشده (فقط start)
        commands = [
            BotCommand(
                command="start",
                description=f"🚀 {get_text(user_language, 'cmd_start_desc', default='شروع مجدد بات')}"
            )
        ]
        
        # تنظیم دستورات برای همه کاربران
        await bot.set_my_commands(
            commands=commands,
            scope=BotCommandScopeDefault()
        )
        
        logger.info(f"✅ Bot menu commands set successfully for language: {user_language}")
        
    except Exception as e:
        logger.error(f"❌ Error setting bot menu: {e}")

async def update_user_menu(bot: Bot, user_id: int, user_language: str, is_registered: bool = True):
    """تنظیم منوی شخصی‌سازی شده برای کاربر خاص"""
    try:
        from aiogram.types import BotCommandScopeChat
        from localization.translator import get_text
        
        if not is_registered:
            # کاربر ثبت‌نام نشده - فقط start
            commands = [
                BotCommand(
                    command="start",
                    description=f"🚀 {get_text(user_language, 'cmd_start_desc', default='شروع مجدد بات')}"
                )
            ]
        else:
            # کاربر ثبت‌نام شده - منوی کامل
            commands = [
                BotCommand(
                    command="start",
                    description=f"🚀 {get_text(user_language, 'cmd_start_desc', default='شروع مجدد بات')}"
                ),
                BotCommand(
                    command="profile",
                    description=f"👤 {get_text(user_language, 'cmd_profile_desc', default='پروفایل')}"
                ),
                BotCommand(
                    command="settings",
                    description=f"⚙️ {get_text(user_language, 'cmd_settings_desc', default='تنظیمات')}"
                ),
                BotCommand(
                    command="help",
                    description=f"❓ {get_text(user_language, 'cmd_help_desc', default='راهنما')}"
                )
            ]
        
        # تنظیم دستورات برای کاربر خاص
        await bot.set_my_commands(
            commands=commands,
            scope=BotCommandScopeChat(chat_id=user_id)
        )
        
        logger.info(f"✅ User menu updated for user {user_id} (registered: {is_registered}) in language: {user_language}")
        
    except Exception as e:
        logger.error(f"❌ Error updating user menu: {e}")

async def hide_user_menu(bot: Bot, user_id: int):
    """پنهان کردن منوی پایین برای کاربر (در طول ثبت‌نام)"""
    try:
        from aiogram.types import BotCommandScopeChat
        
        # حذف تمام دستورات برای کاربر
        await bot.delete_my_commands(
            scope=BotCommandScopeChat(chat_id=user_id)
        )
        
        logger.info(f"✅ Menu hidden for user {user_id}")
        
    except Exception as e:
        logger.error(f"❌ Error hiding menu: {e}")

async def show_user_menu(bot: Bot, user_id: int, user_language: str):
    """نمایش منوی پایین برای کاربر (بعد از ثبت‌نام)"""
    await update_user_menu(bot, user_id, user_language)

# Handler for /profile command
@router.message(Command('profile'))
async def profile_command(message: Message):
    """هندلر دستور پروفایل"""
    from handlers.profile_handler import show_profile
    await show_profile(message)

# Handler for /settings command
@router.message(Command('settings'))
async def settings_command(message: Message):
    """هندلر دستور تنظیمات"""
    from handlers.settings import settings_menu
    await settings_menu(message)

# Handler for /help command
@router.message(Command('help'))
async def help_command(message: Message):
    """هندلر دستور راهنما"""
    from handlers.help_handler import help_menu
    await help_menu(message)
