from aiogram import Router, F
from aiogram.types import Message, CallbackQuery, InlineKeyboardMarkup, InlineKeyboardButton
from aiogram.fsm.context import FSMContext
from aiogram.fsm.state import State, StatesGroup
from sqlalchemy import select, and_
from database.database import db
from database.models import User, Match, Swipe
from localization.translator import get_text
from localization.language_helper import get_user_with_language
from utils.matching import matching_engine
import logging
import asyncio

router = Router()
logger = logging.getLogger(__name__)

class MatchingStates(StatesGroup):
    viewing_match = State()
    in_chat = State()

@router.message(F.text.contains("🔍 یافتن مچ"))
@router.message(F.text.contains("/find"))
async def find_match(message: Message, state: FSMContext):
    """Find and show a potential match"""
    user_id = message.from_user.id
    
    async with db.get_session() as session:
        # Get user
        result = await session.execute(select(User).where(User.id == user_id))
        user = result.scalar_one_or_none()
        
        if not user:
            await message.answer("لطفاً ابتدا ثبت نام کنید: /start")
            return
        
        lang = user.language_code or 'en'
        
        # Check if user is already in a chat
        active_match = await session.execute(
            select(Match).where(
                and_(
                    or_(Match.user_a == user.user_id, Match.user_b == user.user_id),
                    Match.status == 1  # connected
                )
            )
        )
        active_match = active_match.scalar_one_or_none()
        
        if active_match:
            await message.answer("شما در حال حاضر در یک گفتگو هستید. ابتدا آن را پایان دهید: /stop")
            return
        
        await message.answer(get_text(lang, 'searching_match'))
        
        # Find matches
        matches = await matching_engine.find_matches(session, user.user_id, limit=5)
        
        if not matches:
            await message.answer(
                "متأسفانه مچ مناسبی پیدا نشد 😔\n\nممکن است:\n• فیلترهای شما خیلی محدود باشد\n• کاربران کمی در منطقه شما باشند\n\nبعداً دوباره تلاش کنید یا فیلترها را تغییر دهید."
            )
            return
        
        # Show first match
        match_data = matches[0]
        candidate = match_data['user']
        score = match_data['score']
        distance = match_data.get('distance_km')
        
        # Create match card
        card_text = create_match_card(candidate, distance, lang)
        
        # Store match candidates in state
        await state.update_data(
            candidates=[m['user'].id for m in matches],
            current_index=0,
            current_candidate_id=candidate.id
        )
        
        await message.answer(
            card_text,
            reply_markup=get_match_keyboard(lang),
            parse_mode='HTML'
        )
        await state.set_state(MatchingStates.viewing_match)

def create_match_card(user: User, distance_km: float = None, lang: str = 'en') -> str:
    """Create a match card text"""
    age = 2024 - user.birth_year if user.birth_year else "نامشخص"
    
    card = f"💫 <b>مچ جدید!</b>\n\n"
    card += f"👤 <b>{user.handle}</b>\n"
    card += f"🎂 {age} ساله\n"
    
    if user.city:
        card += f"📍 {user.city}"
        if user.country_code and user.country_code != 'XX':
            card += f", {user.country_code}"
        if distance_km:
            card += f" ({distance_km:.1f} کیلومتر)"
        card += "\n"
    
    if user.bio:
        card += f"\n💭 {user.bio}\n"
    
    if user.interests:
        interests_text = ", ".join(user.interests[:5])  # Show max 5 interests
        card += f"\n🎯 {interests_text}\n"
    
    card += f"\n⭐ آخرین فعالیت: "
    if user.last_active:
        hours_ago = (datetime.utcnow() - user.last_active).total_seconds() / 3600
        if hours_ago < 1:
            card += "همین الان"
        elif hours_ago < 24:
            card += f"{int(hours_ago)} ساعت پیش"
        else:
            card += f"{int(hours_ago/24)} روز پیش"
    else:
        card += "نامشخص"
    
    return card

def get_match_keyboard(lang: str):
    """Get match action keyboard"""
    return InlineKeyboardMarkup(inline_keyboard=[
        [
            InlineKeyboardButton(text="💬 شروع گفتگو", callback_data="match_chat"),
            InlineKeyboardButton(text="⏭️ مچ بعدی", callback_data="match_next")
        ],
        [
            InlineKeyboardButton(text="❤️ لایک", callback_data="match_like"),
            InlineKeyboardButton(text="👎 پس", callback_data="match_pass")
        ],
        [InlineKeyboardButton(text="⚙️ تغییر فیلترها", callback_data="match_filters")]
    ])

@router.callback_query(F.data == "match_chat")
async def start_match_chat(callback: CallbackQuery, state: FSMContext):
    """Start chat with current match"""
    user_id = callback.from_user.id
    data = await state.get_data()
    candidate_id = data.get('current_candidate_id')
    
    if not candidate_id:
        await callback.answer("خطا: مچ پیدا نشد")
        return
    
    async with db.get_session() as session:
        # Get users
        user_result = await session.execute(select(User).where(User.id == user_id))
        user = user_result.scalar_one_or_none()
        
        candidate_result = await session.execute(select(User).where(User.id == candidate_id))
        candidate = candidate_result.scalar_one_or_none()
        
        if not user or not candidate:
            await callback.answer("خطا: کاربر پیدا نشد")
            return
        
        # Create match
        match = await matching_engine.create_match(session, user.user_id, candidate.user_id)
        if not match:
            await callback.answer("این مچ قبلاً ایجاد شده است")
            return
        
        # Update match status to connected
        match.status = 1
        await session.commit()
        
        lang = user.language_code or 'en'
        
        # Notify both users
        await callback.message.edit_text(
            get_text(lang, 'chat_started'),
            reply_markup=get_chat_keyboard(lang)
        )
        
        # Notify the other user
        try:
            candidate_lang = candidate.language_code or 'en'
            await callback.bot.send_message(
                candidate.user_id,
                f"💬 گفتگوی جدید شروع شد!\n\n{get_text(candidate_lang, 'chat_started')}",
                reply_markup=get_chat_keyboard(candidate_lang)
            )
        except Exception as e:
            logger.warning(f"Could not notify candidate {candidate.user_id}: {e}")
        
        # Update state
        await state.update_data(
            match_id=match.id,
            partner_id=candidate.user_id,
            partner_tg_id=candidate.user_id
        )
        await state.set_state(MatchingStates.in_chat)
    
    await callback.answer()

@router.callback_query(F.data == "match_next")
async def next_match(callback: CallbackQuery, state: FSMContext):
    """Show next match candidate"""
    data = await state.get_data()
    candidates = data.get('candidates', [])
    current_index = data.get('current_index', 0)
    
    if current_index + 1 >= len(candidates):
        await callback.answer("تمام مچ‌ها نمایش داده شد")
        await callback.message.edit_text("تمام مچ‌های موجود نمایش داده شد.\n\n/find برای جستجوی مجدد")
        await state.clear()
        return
    
    new_index = current_index + 1
    candidate_id = candidates[new_index]
    
    async with db.get_session() as session:
        result = await session.execute(select(User).where(User.id == candidate_id))
        candidate = result.scalar_one_or_none()
        
        if not candidate:
            await callback.answer("خطا: کاربر پیدا نشد")
            return
        
        user_result = await session.execute(select(User).where(User.id == callback.from_user.id))
        user = user_result.scalar_one_or_none()
        lang = user.language_code if user and user.language_code else 'en'
        
        card_text = create_match_card(candidate, None, lang)
        
        await state.update_data(
            current_index=new_index,
            current_candidate_id=candidate_id
        )
        
        await callback.message.edit_text(
            card_text,
            reply_markup=get_match_keyboard(lang),
            parse_mode='HTML'
        )
    
    await callback.answer()

@router.callback_query(F.data == "match_like")
async def like_match(callback: CallbackQuery, state: FSMContext):
    """Like current match"""
    user_id = callback.from_user.id
    data = await state.get_data()
    candidate_id = data.get('current_candidate_id')
    
    async with db.get_session() as session:
        user_result = await session.execute(select(User).where(User.id == user_id))
        user = user_result.scalar_one_or_none()
        
        if not user or not candidate_id:
            await callback.answer("خطا")
            return
        
        # Record swipe
        swipe = Swipe(
            from_user=user.user_id,
            to_user=candidate_id,
            action=1  # like
        )
        session.add(swipe)
        
        # Check for mutual like
        mutual = await matching_engine.check_mutual_like(session, user.user_id, candidate_id)
        
        if mutual:
            # Create match
            match = await matching_engine.create_match(session, user.user_id, candidate_id)
            await session.commit()
            
            await callback.answer("🎉 مچ متقابل! می‌توانید شروع به گفتگو کنید.")
            await callback.message.edit_text(
                "🎉 مچ متقابل!\n\nهر دو شما همدیگر را لایک کردید. می‌توانید شروع به گفتگو کنید:",
                reply_markup=InlineKeyboardMarkup(inline_keyboard=[
                    [InlineKeyboardButton(text="💬 شروع گفتگو", callback_data="match_chat")]
                ])
            )
        else:
            await session.commit()
            await callback.answer("❤️ لایک شد!")
            # Show next match
            await next_match(callback, state)

@router.callback_query(F.data == "match_pass")
async def pass_match(callback: CallbackQuery, state: FSMContext):
    """Pass on current match"""
    user_id = callback.from_user.id
    data = await state.get_data()
    candidate_id = data.get('current_candidate_id')
    
    async with db.get_session() as session:
        user_result = await session.execute(select(User).where(User.id == user_id))
        user = user_result.scalar_one_or_none()
        
        if not user or not candidate_id:
            await callback.answer("خطا")
            return
        
        # Record swipe
        swipe = Swipe(
            from_user=user.user_id,
            to_user=candidate_id,
            action=-1  # pass
        )
        session.add(swipe)
        await session.commit()
    
    await callback.answer("👎 رد شد")
    # Show next match
    await next_match(callback, state)

def get_chat_keyboard(lang: str):
    """Get in-chat keyboard"""
    return InlineKeyboardMarkup(inline_keyboard=[
        [
            InlineKeyboardButton(text="❌ پایان گفتگو", callback_data="chat_end"),
            InlineKeyboardButton(text="🚨 گزارش", callback_data="chat_report")
        ],
        [
            InlineKeyboardButton(text="💣 خودتخریب", callback_data="chat_destruct"),
            InlineKeyboardButton(text="🧊 یخ‌شکن", callback_data="chat_icebreaker")
        ]
    ])

# Chat message forwarding
@router.message(MatchingStates.in_chat)
async def forward_chat_message(message: Message, state: FSMContext):
    """Forward messages between matched users"""
    data = await state.get_data()
    partner_tg_id = data.get('partner_tg_id')
    match_id = data.get('match_id')
    
    if not partner_tg_id or not match_id:
        await message.answer("خطا: گفتگو پیدا نشد")
        return
    
    # Check if match is still active
    async with db.get_session() as session:
        match_result = await session.execute(select(Match).where(Match.id == match_id))
        match = match_result.scalar_one_or_none()
        
        if not match or match.status != 1:
            await message.answer("این گفتگو پایان یافته است.")
            await state.clear()
            return
    
    # Forward message to partner
    try:
        if message.text:
            await message.bot.send_message(partner_tg_id, message.text)
        elif message.sticker:
            await message.bot.send_sticker(partner_tg_id, message.sticker.file_id)
        elif message.voice:
            await message.bot.send_voice(partner_tg_id, message.voice.file_id)
        elif message.photo:
            # Add NSFW warning for photos
            await message.bot.send_photo(
                partner_tg_id, 
                message.photo[-1].file_id,
                caption="⚠️ تصویر دریافت شد - لطفاً محتوای مناسب ارسال کنید"
            )
        else:
            await message.answer("این نوع پیام پشتیبانی نمی‌شود.")
            return
            
    except Exception as e:
        logger.error(f"Error forwarding message: {e}")
        await message.answer("خطا در ارسال پیام. ممکن است طرف مقابل ربات را مسدود کرده باشد.")

@router.callback_query(F.data == "chat_end")
async def end_chat(callback: CallbackQuery, state: FSMContext):
    """End current chat"""
    data = await state.get_data()
    match_id = data.get('match_id')
    partner_tg_id = data.get('partner_tg_id')
    
    if not match_id:
        await callback.answer("گفتگو پیدا نشد")
        return
    
    async with db.get_session() as session:
        match_result = await session.execute(select(Match).where(Match.id == match_id))
        match = match_result.scalar_one_or_none()
        
        if match:
            match.status = 2  # ended
            match.ended_at = func.now()
            match.ended_by = callback.from_user.id
            await session.commit()
    
    # Notify both users
    await callback.message.edit_text("👋 گفتگو پایان یافت.\n\n/find برای مچ جدید")
    
    if partner_tg_id:
        try:
            await callback.bot.send_message(
                partner_tg_id,
                "👋 طرف مقابل گفتگو را پایان داد.\n\n/find برای مچ جدید"
            )
        except:
            pass
    
    await state.clear()
    await callback.answer()

@router.message(F.text.contains("/stop"))
async def stop_command(message: Message, state: FSMContext):
    """Stop current chat via command"""
    current_state = await state.get_state()
    
    if current_state == MatchingStates.in_chat:
        data = await state.get_data()
        match_id = data.get('match_id')
        partner_tg_id = data.get('partner_tg_id')
        
        if match_id:
            async with db.get_session() as session:
                match_result = await session.execute(select(Match).where(Match.id == match_id))
                match = match_result.scalar_one_or_none()
                
                if match:
                    match.status = 2  # ended
                    match.ended_at = func.now()
                    match.ended_by = message.from_user.id
                    await session.commit()
            
            await message.answer("👋 گفتگو پایان یافت.\n\n/find برای مچ جدید")
            
            if partner_tg_id:
                try:
                    await message.bot.send_message(
                        partner_tg_id,
                        "👋 طرف مقابل گفتگو را پایان داد.\n\n/find برای مچ جدید"
                    )
                except:
                    pass
        
        await state.clear()
    else:
        await message.answer("شما در حال حاضر در گفتگو نیستید.")

from datetime import datetime
from sqlalchemy import func, or_
