#!/usr/bin/env python3
"""Matching actions handlers - عملیات لایک، دیس لایک، سوپر لایک و چت"""

import logging
from aiogram import Router, F, Bot
from aiogram.types import Message, ReplyKeyboardMarkup, KeyboardButton, InlineKeyboardMarkup, InlineKeyboardButton
from aiogram.fsm.context import FSMContext
from sqlalchemy import select, and_, or_
from database.database import db
from database.models import User, Match
from database.simple_models import Swipe, Notification
from handlers.profile_features import increment_likes_received
from datetime import datetime, timedelta

router = Router()
logger = logging.getLogger(__name__)

# Import shared data from display module
from .matching_display import current_candidate, show_candidate, user_candidates, current_candidate_index
from .matching_filters import MatchingStates

# ==================== عملیات لایک ====================
@router.message(MatchingStates.viewing, F.text == "❤️ لایک")
async def handle_like(message: Message, state: FSMContext):
    """ثبت لایک برای کاربر - منطق کامل"""
    user_id = message.from_user.id
    
    if user_id not in current_candidate or not current_candidate[user_id]:
        await message.answer("❌ کاربری برای لایک کردن وجود ندارد.")
        return
    
    # current_candidate حالا فقط user_id را ذخیره می‌کند
    target_user_id = current_candidate[user_id]
    
    if not target_user_id or not isinstance(target_user_id, int):
        await message.answer("❌ خطا در شناسایی کاربر.")
        return
    
    async with db.get_session() as session:
        # دریافت مجدد target_user در session جدید
        target_result = await session.execute(
            select(User).where(User.id == target_user_id)
        )
        target_user = target_result.scalar_one_or_none()
        
        if not target_user:
            await message.answer("❌ کاربر مورد نظر یافت نشد.")
            return
        # بررسی لایک قبلی
        existing_swipe_result = await session.execute(
            select(Swipe).where(
                and_(
                    Swipe.from_user == user_id,
                    Swipe.to_user == target_user.user_id
                )
            )
        )
        existing_swipe = existing_swipe_result.scalar_one_or_none()
        
        if existing_swipe:
            if existing_swipe.action == 1:
                await message.answer(
                    "❤️ شما قبلاً این کاربر را لایک کرده‌اید!\n\n"
                    "💡 نکته: می‌توانید با سوپر لایک، چت فوری را فعال کنید.",
                    parse_mode='HTML'
                )
                return
            elif existing_swipe.action == 2:
                await message.answer(
                    "💎 شما قبلاً این کاربر را سوپر لایک کرده‌اید!\n\n"
                    "سوپر لایک قوی‌ترین نوع علاقه است و چت فوری فعال شده.\n"
                    "💬 می‌توانید از دکمه چت/گفتگو استفاده کنید.",
                    parse_mode='HTML'
                )
                return
            # اگر action == -1 (دیس لایک) بود، اجازه لایک مجدد می‌دهیم
        
        # ثبت لایک
        swipe = Swipe(
            from_user=user_id,
            to_user=target_user.user_id,
            action=1,  # لایک
            created_at=datetime.now()
        )
        session.add(swipe)
        
        # افزایش شمارنده لایک‌های دریافتی
        await increment_likes_received(target_user.user_id)
        
        # بررسی لایک متقابل (مچ)
        mutual_swipe = await session.execute(
            select(Swipe).where(
                and_(
                    Swipe.from_user == target_user.user_id,
                    Swipe.to_user == user_id,
                    Swipe.action.in_([1, 2])  # لایک یا سوپر لایک
                )
            )
        )
        
        is_match = mutual_swipe.scalar_one_or_none() is not None
        
        if is_match:
            # ایجاد مچ
            match = Match(
                user_a=min(user_id, target_user.user_id),
                user_b=max(user_id, target_user.user_id),
                status=1  # متصل
            )
            session.add(match)
            
            # ایجاد نوتیفیکیشن برای هر دو کاربر
            notification1 = Notification(
                user_id=user_id,
                type='match',
                title='🎉 مچ جدید!',
                message=f'شما با {target_user.username or target_user.first_name or "کاربری"} مچ شدید!',
                related_user_id=target_user.user_id
            )
            notification2 = Notification(
                user_id=target_user.user_id,
                type='match',
                title='🎉 مچ جدید!',
                message=f'شما با کاربری مچ شدید!',
                related_user_id=user_id
            )
            session.add(notification1)
            session.add(notification2)
        
        await session.commit()
        
        # نمایش پیام مناسب
        if is_match:
            await message.answer(
                "🎉🎉 <b>تبریک! شما مچ شدید!</b> 🎉🎉\n\n"
                f"شما و {target_user.username or target_user.first_name or 'این کاربر'} "
                "همدیگر را لایک کردید!\n\n"
                "💬 اکنون می‌توانید با هم چت کنید.\n"
                "برای شروع گفتگو روی دکمه <b>💬 چت/گفتگو</b> کلیک کنید.",
                parse_mode='HTML'
            )
        else:
            await message.answer(
                "❤️ <b>شما این فرد را لایک کردید!</b>\n\n"
                "منتظر باشید تا او نیز شما را لایک کند تا گفتگو فعال شود.\n"
                "⏳ چت غیرفعال است تا مچ ایجاد شود.",
                parse_mode='HTML'
            )
    
    # بدون رفتن به کاندید بعدی - کاربر فعلی باقی می‌ماند
    # await show_next_candidate(message, state)

# ==================== عملیات سوپر لایک ====================
@router.message(MatchingStates.viewing, F.text == "💎 سوپر لایک")
async def handle_super_like(message: Message, state: FSMContext):
    """ثبت سوپر لایک - چت فوری فعال می‌شود"""
    user_id = message.from_user.id
    
    if user_id not in current_candidate or not current_candidate[user_id]:
        await message.answer("❌ کاربری برای سوپر لایک کردن وجود ندارد.")
        return
    
    # current_candidate حالا فقط user_id را ذخیره می‌کند
    target_user_id = current_candidate[user_id]
    
    if not target_user_id or not isinstance(target_user_id, int):
        await message.answer("❌ خطا در شناسایی کاربر.")
        return
    
    async with db.get_session() as session:
        # دریافت مجدد target_user در session جدید
        target_result = await session.execute(
            select(User).where(User.id == target_user_id)
        )
        target_user = target_result.scalar_one_or_none()
        
        if not target_user:
            await message.answer("❌ کاربر مورد نظر یافت نشد.")
            return
        # بررسی لایک قبلی
        existing_swipe_result = await session.execute(
            select(Swipe).where(
                and_(
                    Swipe.from_user == user_id,
                    Swipe.to_user == target_user.user_id
                )
            )
        )
        existing_swipe = existing_swipe_result.scalar_one_or_none()
        
        if existing_swipe:
            if existing_swipe.action == 2:
                # قبلاً سوپر لایک زده
                await message.answer("❌ شما قبلاً این کاربر را سوپر لایک کرده‌اید!")
                return
            elif existing_swipe.action == 1:
                # لایک عادی را به سوپر لایک ارتقا می‌دهیم
                existing_swipe.action = 2
                existing_swipe.created_at = datetime.now()
                
                await message.answer(
                    "⬆️ <b>ارتقا به سوپر لایک!</b> 💎\n\n"
                    "لایک شما به سوپر لایک ارتقا یافت.\n"
                    "✅ چت فوری فعال شد!\n"
                    "💬 می‌توانید بلافاصله گفتگو را شروع کنید.",
                    parse_mode='HTML'
                )
                
                # ارسال نوتیفیکیشن سوپر لایک (با type='like' چون super_like در ENUM نیست)
                notification = Notification(
                    user_id=target_user.user_id,
                    type='like',  # استفاده از 'like' به جای 'super_like'
                    title='سوپر لایک جدید',  # اضافه کردن title
                    message=f"💎 {message.from_user.username or message.from_user.first_name or 'کسی'} به شما سوپر لایک داد!",
                    created_at=datetime.now()
                )
                session.add(notification)
                await session.commit()
                return
            else:
                # دیس لایک قبلی - اجازه سوپر لایک
                pass
        
        # ثبت سوپر لایک
        swipe = Swipe(
            from_user=user_id,
            to_user=target_user.user_id,
            action=2,  # سوپر لایک
            created_at=datetime.now()
        )
        session.add(swipe)
        
        # افزایش شمارنده لایک‌های دریافتی
        await increment_likes_received(target_user.user_id)
        
        # بررسی لایک متقابل
        reverse_swipe = await session.execute(
            select(Swipe).where(
                and_(
                    Swipe.from_user == target_user.user_id,
                    Swipe.to_user == user_id,
                    Swipe.action.in_([1, 2])  # لایک یا سوپر لایک
                )
            )
        )
        
        is_match = reverse_swipe.scalar_one_or_none() is not None
        
        if is_match:
            # ایجاد مچ
            match = Match(
                user_a=min(user_id, target_user.user_id),
                user_b=max(user_id, target_user.user_id),
                created_at=datetime.now()
            )
            session.add(match)
            
            # ارسال نوتیفیکیشن به هر دو کاربر
            notification1 = Notification(
                user_id=user_id,
                type='match',
                message=f"🎆 مچ جدید با {target_user.username or target_user.first_name or 'کاربر'}! (سوپر لایک)",
                created_at=datetime.now()
            )
            notification2 = Notification(
                user_id=target_user.user_id,
                type='match',
                message=f"🎆 مچ جدید با {message.from_user.username or message.from_user.first_name or 'کاربر'}! (سوپر لایک)",
                created_at=datetime.now()
            )
            session.add(notification1)
            session.add(notification2)
        else:
            # ارسال نوتیفیکیشن سوپر لایک (با type='like' چون super_like در ENUM نیست)
            notification = Notification(
                user_id=target_user.user_id,
                type='like',  # استفاده از 'like' به جای 'super_like'
                title='سوپر لایک جدید',  # اضافه کردن title
                message=f"💎 {message.from_user.username or message.from_user.first_name or 'کسی'} به شما سوپر لایک داد!",
                created_at=datetime.now()
            )
            session.add(notification)
        
        await session.commit()
        
        # نمایش پیام مناسب
        if is_match:
            await message.answer(
                "🎆🎆 <b>تبریک! مچ جدید! (سوپر لایک)</b> 🎆🎆\n\n"
                f"شما و {target_user.username or target_user.first_name or 'این کاربر'} "
                "همدیگر را لایک کردید!\n\n"
                "👬 حالا می‌توانید با هم گفتگو کنید.\n"
                "برای شروع گفتگو از منوی <b>گفتگوهای من</b> استفاده کنید.",
                parse_mode='HTML'
            )
        else:
            await message.answer(
                "💎 <b>سوپر لایک ارسال شد!</b> 💎\n\n"
                "✅ چت فعال شد! این یک سوپر لایک بود و امکان گفتگو فوری فراهم شد.\n"
                "💬 برای شروع گفتگو روی دکمه <b>💬 چت/گفتگو</b> کلیک کنید.",
                parse_mode='HTML'
            )
    
    # بدون رفتن به کاندید بعدی - کاربر فعلی باقی می‌ماند
    # await show_next_candidate(message, state)

# ==================== عملیات دیس لایک ====================
@router.message(MatchingStates.viewing, F.text == "👎 دیس لایک")
async def handle_dislike(message: Message, state: FSMContext):
    """دیس لایک - پروفایل رد شده و دیگر نمایش داده نمی‌شود"""
    user_id = message.from_user.id
    
    if user_id not in current_candidate or not current_candidate[user_id]:
        await message.answer("❌ کاربری برای رد کردن وجود ندارد.")
        return
    
    # current_candidate حالا فقط user_id را ذخیره می‌کند
    target_user_id = current_candidate[user_id]
    
    if not target_user_id or not isinstance(target_user_id, int):
        await message.answer("❌ خطا در شناسایی کاربر.")
        return
    
    async with db.get_session() as session:
        # دریافت مجدد target_user در session جدید
        target_result = await session.execute(
            select(User).where(User.id == target_user_id)
        )
        target_user = target_result.scalar_one_or_none()
        
        if not target_user:
            await message.answer("❌ کاربر مورد نظر یافت نشد.")
            return
        # بررسی وجود swipe قبلی
        existing_swipe_result = await session.execute(
            select(Swipe).where(
                and_(
                    Swipe.from_user == user_id,
                    Swipe.to_user == target_user_id  # استفاده از target_user_id
                )
            )
        )
        existing_swipe = existing_swipe_result.scalar_one_or_none()
        
        if existing_swipe:
            # آپدیت swipe موجود به دیس لایک
            existing_swipe.action = -1
            existing_swipe.created_at = datetime.now()
        else:
            # ثبت دیس لایک جدید
            swipe = Swipe(
                from_user=user_id,
                to_user=target_user_id,  # استفاده از target_user_id
                action=-1,  # دیس لایک (برای حذف دائمی از لیست)
                created_at=datetime.now()
            )
            session.add(swipe)
        
        await session.commit()
    
    # حذف کاربر دیس لایک شده از لیست کاندیدها
    if user_id in user_candidates:
        candidates = user_candidates[user_id]
        # حذف کاربر رد شده از لیست (candidates حالا لیست ID است)
        # استفاده از target_user_id که قبلاً تعریف شده، نه target_user.user_id
        user_candidates[user_id] = [c for c in candidates if c != target_user_id]
        
        # اگر لیست خالی شد
        if not user_candidates[user_id]:
            await message.answer(
                "👎 <b>پروفایل رد شد</b>\n\n"
                "❌ دیگر کاربری برای نمایش وجود ندارد.",
                parse_mode='HTML'
            )
            await state.clear()
            return
        
        # تنظیم مجدد ایندکس اگر لازم باشد
        current_index = current_candidate_index.get(user_id, 0)
        if current_index >= len(user_candidates[user_id]):
            current_candidate_index[user_id] = 0
    
    await message.answer(
        "👎 <b>پروفایل رد شد</b>\n\n"
        "این پروفایل دیگر نمایش داده نخواهد شد.",
        parse_mode='HTML'
    )
    
    # نمایش کاندید بعدی
    await show_next_candidate(message, state)

# ==================== درخواست چت ====================
@router.message(MatchingStates.viewing, F.text == "💬 چت/گفتگو")
async def handle_chat_request(message: Message, state: FSMContext):
    """منطق کامل چت/گفتگو"""
    user_id = message.from_user.id
    
    if user_id not in current_candidate or not current_candidate[user_id]:
        await message.answer("❌ کاربری برای چت وجود ندارد.")
        return
    
    # current_candidate حالا فقط user_id را ذخیره می‌کند
    target_user_id = current_candidate[user_id]
    
    if not target_user_id or not isinstance(target_user_id, int):
        await message.answer("❌ خطا در شناسایی کاربر.")
        return
    
    async with db.get_session() as session:
        # دریافت مجدد target_user در session جدید
        target_result = await session.execute(
            select(User).where(User.id == target_user_id)
        )
        target_user = target_result.scalar_one_or_none()
        
        if not target_user:
            await message.answer("❌ کاربر مورد نظر یافت نشد.")
            return
        
        # بررسی لایک یا سوپر لایک کاربر
        user_swipe = await session.execute(
            select(Swipe).where(
                and_(
                    Swipe.from_user == user_id,
                    Swipe.to_user == target_user.user_id,
                    Swipe.action.in_([1, 2])  # لایک یا سوپر لایک
                )
            )
        )
        user_has_liked = user_swipe.scalar_one_or_none()
        
        # اگر کاربر هیچ لایکی نزده
        if not user_has_liked:
            await message.answer(
                "⚠️ <b>برای شروع گفتگو، ابتدا باید این شخص را لایک کنید.</b>\n\n"
                "❤️ روی دکمه لایک کلیک کنید (نیاز به مچ دارد) یا\n"
                "💎 از سوپر لایک استفاده کنید (چت فوری فعال می‌شود)",
                parse_mode='HTML'
            )
            return
        
        # بررسی نوع لایک
        is_super_like = user_has_liked.action == 2
        
        # اگر سوپر لایک زده، چت فوری فعال است (بدون نیاز به مچ)
        if is_super_like:
            # ایجاد دکمه ورود به اتاق
            from aiogram.types import ReplyKeyboardMarkup, KeyboardButton
            room_keyboard = ReplyKeyboardMarkup(
                keyboard=[
                    [KeyboardButton(text="🚪 ورود به اتاق گفتگو")],
                    [KeyboardButton(text="🔙 بازگشت")]
                ],
                resize_keyboard=True,
                one_time_keyboard=True
            )
            
            await message.answer(
                "✅ <b>گفتگو آماده است!</b>\n\n"
                "شما سوپر لایک زده‌اید و چت فوری فعال شد.\n"
                "برای ورود به اتاق گفتگو، روی دکمه زیر کلیک کنید.",
                reply_markup=room_keyboard,
                parse_mode='HTML'
            )
            
            # ذخیره اطلاعات برای ایجاد اتاق
            await state.update_data(chat_target=target_user.user_id)
            return
        
        # بررسی وجود مچ (هر دو طرف لایک زده باشند)
        match_result = await session.execute(
            select(Match).where(
                or_(
                    and_(Match.user_a == user_id, Match.user_b == target_user.user_id),
                    and_(Match.user_a == target_user.user_id, Match.user_b == user_id)
                )
            )
        )
        existing_match = match_result.scalar_one_or_none()
        
        if existing_match:
            # مچ وجود دارد، چت فعال است
            from aiogram.types import ReplyKeyboardMarkup, KeyboardButton
            room_keyboard = ReplyKeyboardMarkup(
                keyboard=[
                    [KeyboardButton(text="🚪 ورود به اتاق گفتگو")],
                    [KeyboardButton(text="🔙 بازگشت")]
                ],
                resize_keyboard=True,
                one_time_keyboard=True
            )
            
            await message.answer(
                "✅ <b>گفتگو آماده است!</b>\n\n"
                "شما و این کاربر مچ شده‌اید.\n"
                "برای ورود به اتاق گفتگو، روی دکمه زیر کلیک کنید.",
                reply_markup=room_keyboard,
                parse_mode='HTML'
            )
            
            # ذخیره اطلاعات برای ایجاد اتاق
            await state.update_data(chat_target=target_user.user_id)
        else:
            # فقط کاربر لایک زده، طرف مقابل هنوز لایک نکرده
            await message.answer(
                "⏳ <b>شما این فرد را لایک کرده‌اید.</b>\n\n"
                "منتظر باشید تا او نیز شما را لایک کند تا گفتگو فعال شود.\n"
                "❌ ارسال پیام غیرفعال است.\n\n"
                "💡 نکته: با سوپر لایک می‌توانید چت فوری را فعال کنید!",
                parse_mode='HTML'
            )

# ==================== ورود به اتاق گفتگو ====================
@router.message(F.text == "🚪 ورود به اتاق گفتگو")
async def handle_enter_chat_room(message: Message, state: FSMContext, bot: Bot):
    """ایجاد گروه تلگرام برای گفتگو خصوصی"""
    user_id = message.from_user.id
    
    # دریافت اطلاعات ذخیره شده
    data = await state.get_data()
    target_user_id = data.get('chat_target')
    
    if not target_user_id:
        await message.answer("❌ خطا در ایجاد اتاق گفتگو. لطفاً دوباره تلاش کنید.")
        return
    
    async with db.get_session() as session:
        # دریافت اطلاعات کاربران
        user_result = await session.execute(
            select(User).where(User.id == user_id)
        )
        user = user_result.scalar_one_or_none()
        
        target_result = await session.execute(
            select(User).where(User.id == target_user_id)
        )
        target_user = target_result.scalar_one_or_none()
        
        if not target_user or not user:
            await message.answer("❌ کاربر مورد نظر یافت نشد.")
            return
        
        # بررسی مجوز چت (مچ یا سوپر لایک)
        can_chat = False
        chat_reason = ""
        
        # بررسی سوپر لایک
        super_like_check = await session.execute(
            select(Swipe).where(
                and_(
                    Swipe.from_user == user_id,
                    Swipe.to_user == target_user_id,
                    Swipe.action == 2  # سوپر لایک
                )
            )
        )
        has_super_like = super_like_check.scalar_one_or_none()
        
        if has_super_like:
            can_chat = True
            chat_reason = "سوپر لایک 💎"
        else:
            # بررسی مچ
            match_check = await session.execute(
                select(Match).where(
                    or_(
                        and_(Match.user_a == user_id, Match.user_b == target_user_id),
                        and_(Match.user_a == target_user_id, Match.user_b == user_id)
                    )
                )
            )
            has_match = match_check.scalar_one_or_none()
            
            if has_match:
                can_chat = True
                chat_reason = "مچ ❤️"
        
        if not can_chat:
            await message.answer(
                "❌ <b>شما مجوز گفتگو با این کاربر را ندارید.</b>\n\n"
                "برای فعال شدن چت:\n"
                "• هر دو طرف باید همدیگر را لایک کنند (مچ)\n"
                "• یا از سوپر لایک استفاده کنید",
                parse_mode='HTML'
            )
            return
        
        # نام کاربران برای نمایش
        user_name = user.username or user.first_name or f"کاربر {user_id}"
        target_name = target_user.username or target_user.first_name or f"کاربر {target_user_id}"
        
        try:
            # استفاده از سیستم خودکار جدید
            from .auto_chat_room import create_private_chat_group, send_invitation_to_users
            
            # ایجاد گروه و دریافت لینک
            success, deeplink, room_code, invite_message = await create_private_chat_group(
                bot, user_id, target_user_id, user_name, target_name
            )
            
            if success:
                # ارسال دعوتنامه به هر دو کاربر
                invitation_sent = await send_invitation_to_users(
                    bot, user_id, target_user_id, room_code, invite_message, deeplink
                )
                
                if invitation_sent:
                    await message.answer(
                        f"✅ <b>اتاق گفتگو آماده شد!</b>\n\n"
                        f"🆔 کد اتاق: <code>{room_code}</code>\n\n"
                        f"📨 دستورالعمل برای شما و {target_name} ارسال شد.\n"
                        f"لطفاً پیام دریافتی را چک کنید و مراحل را دنبال کنید.",
                        parse_mode='HTML'
                    )
                else:
                    await message.answer(
                        "⚠️ اتاق ایجاد شد ولی خطا در ارسال دعوتنامه.\n"
                        f"کد اتاق: <code>{room_code}</code>",
                        parse_mode='HTML'
                    )
                
                # بازگشت به حالت نمایش پروفایل
                await state.set_state(MatchingStates.viewing)
            else:
                await message.answer(
                    f"❌ خطا در ایجاد اتاق گفتگو:\n{invite_message}\n\n"
                    "لطفاً دوباره تلاش کنید.",
                    parse_mode='HTML'
                )
                
        except Exception as e:
            logger.error(f"Error in chat room creation: {e}")
            await message.answer(
                "❌ خطا در ایجاد اتاق گفتگو.\n"
                "لطفاً دوباره تلاش کنید.",
                parse_mode='HTML'
            )

@router.message(F.text == "🔙 بازگشت")
async def handle_back_from_chat(message: Message, state: FSMContext):
    """بازگشت از منوی چت به نمایش پروفایل"""
    # بازگشت به نمایش پروفایل فعلی
    user_id = message.from_user.id
    
    if user_id in current_candidate and current_candidate[user_id]:
        await show_candidate(message, state, current_candidate[user_id])
    else:
        await message.answer("بازگشت به منوی اصلی...")
        await state.clear()

# ==================== Helper Functions ====================
async def show_next_candidate(message: Message, state: FSMContext):
    """نمایش کاندید بعدی"""
    user_id = message.from_user.id
    
    if user_id not in user_candidates:
        await message.answer("❌ لیست کاندیدها خالی است. جستجوی جدید را امتحان کنید.")
        return
    
    candidates = user_candidates[user_id]
    current_index = current_candidate_index.get(user_id, 0)
    
    # به کاندید بعدی برو
    next_index = current_index + 1
    
    if next_index >= len(candidates):
        # وقتی پروفایل‌ها تمام شد، از اول شروع کن (بدون پیام)
        next_index = 0
        current_candidate_index[user_id] = 0
    
    # نمایش کاندید بعدی یا اولین کاندید
    current_candidate_index[user_id] = next_index
    
    # user_candidates حالا فقط ID ها را دارد
    candidate_id = candidates[next_index]  # همیشه int است
    await show_candidate(message, state, candidate_id)

# ==================== بازگشت به منو ====================
@router.message(MatchingStates.viewing, F.text == "🔙 بازگشت به منو")
async def back_to_menu_from_viewing(message: Message, state: FSMContext):
    """بازگشت به منوی اصلی از حالت viewing"""
    from handlers.registration import get_main_menu_keyboard
    from localization.language_helper import get_user_with_language
    from localization.translator import get_text
    import os
    
    user, user_language = await get_user_with_language(message.from_user.id)
    
    if not user:
        await message.answer("❌ ابتدا باید ثبت‌نام کنید.")
        return
    
    # بررسی ادمین
    ADMIN_USER_IDS = [int(uid) for uid in os.getenv('ADMIN_USER_IDS', '').split(',') if uid]
    is_admin = message.from_user.id in ADMIN_USER_IDS
    
    # پاک کردن state و داده‌ها
    await state.clear()
    
    user_id = message.from_user.id
    if user_id in current_candidate:
        del current_candidate[user_id]
    if user_id in user_candidates:
        del user_candidates[user_id]
    if user_id in current_candidate_index:
        del current_candidate_index[user_id]
    
    # Import and clear filters
    from .matching_filters import user_filters
    if user_id in user_filters:
        del user_filters[user_id]
    
    await message.answer(
        f"🏠 {get_text(user_language, 'back_to_menu', default='بازگشت به منوی اصلی')}",
        reply_markup=get_main_menu_keyboard(user_language, is_admin=is_admin)
    )

# ==================== دکمه بعدی ====================
@router.message(MatchingStates.viewing, F.text == "➡️ بعدی")
async def handle_next(message: Message, state: FSMContext):
    """نمایش پروفایل بعدی بدون رد کردن پروفایل فعلی"""
    user_id = message.from_user.id
    
    # نمایش کاندید بعدی بدون ثبت هیچ اکشنی
    await show_next_candidate(message, state)

# Export
__all__ = ['router']
