"""
Menu Handler - هندلر منوی اصلی
مدیریت منوی اصلی با پشتیبانی چندزبانه
"""

from aiogram import Router, F
from aiogram.types import Message
from sqlalchemy import select
from database.database import db
from database.models import User
from localization.language_helper import get_user_with_language
from localization.translator import get_text
from handlers.registration import get_main_menu_keyboard
import os
import logging

router = Router()
logger = logging.getLogger(__name__)

# Universal menu handlers that work with any language
@router.message(F.text.contains("یافتن مچ") | F.text.contains("Find Match") | F.text.contains("Eşleşme Bul") | F.text.contains("العثور على مطابقة") | F.text.contains("Найти пару"))
async def find_match_universal(message: Message):
    """Universal find match handler"""
    user, user_language = await get_user_with_language(message.from_user.id)
    
    if not user:
        await message.answer(f"❌ {get_text(user_language, 'profile_not_found', default='پروفایل شما یافت نشد. لطفاً ابتدا ثبت‌نام کنید.')}")
        return
    
    # Import here to avoid circular imports
    from handlers.matching import find_matches
    await find_matches(message)

@router.message(F.text.contains("گفتگوهای من") | F.text.contains("My Chats") | F.text.contains("Sohbetlerim") | F.text.contains("محادثاتي") | F.text.contains("Мои чаты"))
async def my_chats_universal(message: Message):
    """Universal my chats handler"""
    user, user_language = await get_user_with_language(message.from_user.id)
    
    if not user:
        await message.answer(f"❌ {get_text(user_language, 'profile_not_found', default='پروفایل شما یافت نشد. لطفاً ابتدا ثبت‌نام کنید.')}")
        return
    
    # Import here to avoid circular imports
    from handlers.my_chats import show_my_chats
    await show_my_chats(message)

@router.message(F.text.contains("💚 لایک‌های من") | F.text.contains("💚 My Likes") | F.text.contains("💚 Beğenilerim") | F.text.contains("💚 إعجاباتي") | F.text.contains("💚 Мои лайки"))
async def my_likes_universal(message: Message):
    """Universal my likes handler"""
    user, user_language = await get_user_with_language(message.from_user.id)
    
    if not user:
        await message.answer(f"❌ {get_text(user_language, 'profile_not_found', default='پروفایل شما یافت نشد. لطفاً ابتدا ثبت‌نام کنید.')}")
        return
    
    # Import here to avoid circular imports
    from handlers.my_likes import my_likes_main
    await my_likes_main(message)

@router.message(F.text.contains("🌟 اکتشاف") | F.text.contains("🌟 Discovery") | F.text.contains("🌟 Keşfet") | F.text.contains("🌟 اكتشاف") | F.text.contains("🌟 Обзор"))
async def discovery_universal(message: Message):
    """Universal discovery handler"""
    user, user_language = await get_user_with_language(message.from_user.id)
    
    if not user:
        await message.answer(f"❌ {get_text(user_language, 'profile_not_found', default='پروفایل شما یافت نشد. لطفاً ابتدا ثبت‌نام کنید.')}")
        return
    
    # Import here to avoid circular imports
    from handlers.discovery import start_discovery
    await start_discovery(message)

# Premium and Help handlers are now handled by their dedicated handlers
# These handlers have been moved to premium_handler.py and help_handler.py

# Refresh menu command
@router.message(F.text == "/menu")
async def refresh_menu(message: Message):
    """Refresh main menu with user's language"""
    user_id = message.from_user.id
    
    # Check if user is admin
    ADMIN_USER_IDS = [int(uid) for uid in os.getenv('ADMIN_USER_IDS', '').split(',') if uid]
    is_admin = user_id in ADMIN_USER_IDS
    
    user, user_language = await get_user_with_language(user_id)
    
    if not user:
        await message.answer(f"❌ {get_text(user_language, 'profile_not_found', default='پروفایل شما یافت نشد. لطفاً ابتدا ثبت‌نام کنید.')}")
        return
    
    menu_text = f"🏠 <b>{get_text(user_language, 'main_menu', default='منوی اصلی')}</b>\n\n"
    menu_text += f"{get_text(user_language, 'welcome_back', default='خوش آمدید!')} {user.first_name or user.username or get_text(user_language, 'user', default='کاربر')}"
    
    await message.answer(
        menu_text,
        reply_markup=get_main_menu_keyboard(user_language, is_admin=is_admin),
        parse_mode='HTML'
    )
