from aiogram import Router, F
from aiogram.types import Message, ReplyKeyboardMarkup, KeyboardButton
from aiogram.fsm.context import FSMContext
from aiogram.fsm.state import State, StatesGroup
from sqlalchemy import select, and_, or_, desc
import logging
import json
import asyncio
from datetime import datetime, timedelta

from database.database import db
from database.models import User, Match
from database.models import ChatMessage

logger = logging.getLogger(__name__)
router = Router()

# States برای مدیریت گفتگو
class ChatStates(StatesGroup):
    in_chat_with_user = State()
    sending_message = State()
    video_chat_active = State()    # چت تصویری فعال

# متغیر global برای ذخیره کاربر فعلی در گفتگو
current_chat_users = {}

# متغیر global برای ردیابی کاربران آنلاین در چت
# ساختار: {user_id: {'target_user_id': int, 'match_id': int, 'last_activity': datetime}}
active_chat_sessions = {}

# متغیر global برای مدیریت چت تصویری
# ساختار: {user_id: {'target_user_id': int, 'target_user_name': str, 'match_id': int}}
video_chat_sessions = {}

# تابع حذف پیام بعد از مدت زمان مشخص
async def delete_message_after_delay(message_obj, delay_seconds: int):
    """حذف پیام بعد از مدت زمان مشخص"""
    try:
        await asyncio.sleep(delay_seconds)
        await message_obj.delete()
    except Exception as e:
        logger.warning(f"Could not delete message: {e}")

# تابع بررسی آنلاین بودن طرف مقابل در چت
def is_user_online_in_chat(user_id: int, target_user_id: int) -> bool:
    """بررسی اینکه آیا طرف مقابل در چت آنلاین است"""
    if target_user_id in active_chat_sessions:
        session = active_chat_sessions[target_user_id]
        # بررسی اینکه آیا با همین کاربر در چت است
        if session.get('target_user_id') == user_id:
            # بررسی اینکه آیا در 30 ثانیه گذشته فعال بوده
            time_diff = datetime.now() - session.get('last_activity', datetime.now())
            return time_diff.total_seconds() < 30
    return False

# تابع به‌روزرسانی وضعیت آنلاین
def update_chat_activity(user_id: int, target_user_id: int, match_id: int):
    """به‌روزرسانی وضعیت فعالیت کاربر در چت"""
    active_chat_sessions[user_id] = {
        'target_user_id': target_user_id,
        'match_id': match_id,
        'last_activity': datetime.now()
    }

# تابع قدیمی حذف شد - حالا از send_direct_chat_message استفاده می‌کنیم

# تابع ارسال نوتیفیکیشن عادی
async def send_notification(target_user_id: int, sender_name: str):
    """ارسال نوتیفیکیشن عادی برای کاربران آفلاین"""
    try:
        from aiogram import Bot
        bot_token = "8145304128:AAFkpxUUUpI68vYeQWdVo0JdxnmLbWXd_MQ"
        bot = Bot(token=bot_token)
        
        await bot.send_message(
            chat_id=target_user_id,
            text=f"🔴 <b>پیام جدید از {sender_name}</b>\n\n"
                 f"برای پاسخ، به گفتگوهای من بروید.",
            parse_mode='HTML'
        )
        return True
    except Exception as e:
        logger.error(f"Error sending notification: {e}")
        return False

# تابع ارسال پیام مستقیم به محیط چت (مثل تلگرام)
async def send_direct_chat_message(sender_id: int, target_user_id: int, sender_name: str, content: str, message_type: str = "text", original_message=None):
    """ارسال مستقیم پیام به محیط چت طرف مقابل (real-time reply)"""
    try:
        from aiogram import Bot
        bot_token = "8145304128:AAFkpxUUUpI68vYeQWdVo0JdxnmLbWXd_MQ"
        bot = Bot(token=bot_token)
        
        # ارسال پیام بر اساس نوع - به صورت forward/copy
        if message_type == "text":
            await bot.send_message(
                chat_id=target_user_id,
                text=f"👤 <b>{sender_name}:</b> {content}",
                parse_mode='HTML'
            )
        elif message_type == "photo":
            if original_message and original_message.photo:
                # Forward کردن عکس اصلی
                await bot.copy_message(
                    chat_id=target_user_id,
                    from_chat_id=sender_id,
                    message_id=original_message.message_id,
                    caption=f"👤 <b>{sender_name}</b>",
                    parse_mode='HTML'
                )
            else:
                await bot.send_photo(
                    chat_id=target_user_id,
                    photo=content,
                    caption=f"👤 <b>{sender_name}</b>",
                    parse_mode='HTML'
                )
        elif message_type == "voice":
            if original_message and original_message.voice:
                await bot.copy_message(
                    chat_id=target_user_id,
                    from_chat_id=sender_id,
                    message_id=original_message.message_id,
                    caption=f"👤 <b>{sender_name}</b>",
                    parse_mode='HTML'
                )
            else:
                await bot.send_voice(
                    chat_id=target_user_id,
                    voice=content,
                    caption=f"👤 <b>{sender_name}</b>",
                    parse_mode='HTML'
                )
        elif message_type == "sticker":
            if original_message and original_message.sticker:
                # Forward کردن استیکر اصلی
                await bot.copy_message(
                    chat_id=target_user_id,
                    from_chat_id=sender_id,
                    message_id=original_message.message_id
                )
                await bot.send_message(
                    chat_id=target_user_id,
                    text=f"👤 <b>{sender_name}</b>",
                    parse_mode='HTML'
                )
            else:
                await bot.send_sticker(
                    chat_id=target_user_id,
                    sticker=content
                )
                await bot.send_message(
                    chat_id=target_user_id,
                    text=f"👤 <b>{sender_name}</b>",
                    parse_mode='HTML'
                )
        elif message_type == "audio":
            if original_message and original_message.audio:
                await bot.copy_message(
                    chat_id=target_user_id,
                    from_chat_id=sender_id,
                    message_id=original_message.message_id,
                    caption=f"👤 <b>{sender_name}</b>",
                    parse_mode='HTML'
                )
            else:
                await bot.send_audio(
                    chat_id=target_user_id,
                    audio=content,
                    caption=f"👤 <b>{sender_name}</b>",
                    parse_mode='HTML'
                )
        elif message_type == "animation" or message_type == "gif":
            if original_message and (original_message.animation or original_message.document):
                await bot.copy_message(
                    chat_id=target_user_id,
                    from_chat_id=sender_id,
                    message_id=original_message.message_id,
                    caption=f"👤 <b>{sender_name}</b>",
                    parse_mode='HTML'
                )
            else:
                await bot.send_animation(
                    chat_id=target_user_id,
                    animation=content,
                    caption=f"👤 <b>{sender_name}</b>",
                    parse_mode='HTML'
                )
        elif message_type == "document":
            if original_message and original_message.document:
                await bot.copy_message(
                    chat_id=target_user_id,
                    from_chat_id=sender_id,
                    message_id=original_message.message_id,
                    caption=f"👤 <b>{sender_name}</b>",
                    parse_mode='HTML'
                )
            else:
                await bot.send_document(
                    chat_id=target_user_id,
                    document=content,
                    caption=f"👤 <b>{sender_name}</b>",
                    parse_mode='HTML'
                )
        else:
            await bot.send_message(
                chat_id=target_user_id,
                text=f"👤 <b>{sender_name}:</b> پیام",
                parse_mode='HTML'
            )
        return True
    except Exception as e:
        logger.error(f"Error sending direct chat message: {e}")
        return False

@router.message(F.text.contains("💬 گفتگوهای من") & ~F.text.contains("راهنمای"))
async def view_my_chats_message(message: Message):
    """نمایش گفتگوهای من از طریق message"""
    await view_my_chats_action(message)

async def view_my_chats_action(message: Message):
    """عملکرد نمایش گفتگوهای من"""
    try:
        current_user_id = message.from_user.id
        
        async with db.get_session() as session:
            # Get matches where user is involved
            matches_result = await session.execute(
                select(Match).where(
                    and_(
                        Match.status == 1,  # Connected matches only
                        or_(Match.user_a == current_user_id, Match.user_b == current_user_id)
                    )
                ).order_by(desc(Match.created_at))
            )
            matches = matches_result.scalars().all()
            
            if not matches:
                # کیبورد کاستوم برای حالت بدون گفتگو
                no_chats_keyboard = ReplyKeyboardMarkup(
                    keyboard=[
                        [
                            KeyboardButton(text="🔍 یافتن مچ"),
                            KeyboardButton(text="🌟 اکتشاف")
                        ],
                        [
                            KeyboardButton(text="💚 لایک‌های من"),
                            KeyboardButton(text="👤 پروفایل")
                        ],
                        [
                            KeyboardButton(text="🔙 بازگشت به منو")
                        ]
                    ],
                    resize_keyboard=True,
                    one_time_keyboard=False
                )
                
                await message.answer(
                    "💬 <b>گفتگوهای من</b>\n\n"
                    "شما هنوز گفتگویی ندارید.\n\n"
                    "💡 برای شروع گفتگو:\n"
                    "• روی 'یافتن مچ' کلیک کنید\n"
                    "• کسانی را که دوست دارید لایک کنید\n"
                    "• اگر آن‌ها هم شما را لایک کنند = مچ! 🎉\n"
                    "• سپس می‌توانید با هم گفتگو کنید",
                    reply_markup=no_chats_keyboard,
                    parse_mode='HTML'
                )
                return
                
            # Build chat list
            text = "💬 <b>گفتگوهای من</b>\n\n"
            chat_buttons = []
            
            for i, match in enumerate(matches[:8]):  # Show max 8 chats
                # Get the other user
                other_user_id = match.user_b if match.user_a == current_user_id else match.user_a
                
                # Get other user info
                user_result = await session.execute(
                    select(User).where(User.id == other_user_id)
                )
                other_user = user_result.scalar_one_or_none()
                
                if not other_user:
                    continue
                    
                # User info
                user_name = other_user.username or other_user.first_name or "کاربر"
                
                # Get recent message
                recent_msg_result = await session.execute(
                    select(ChatMessage).where(
                        ChatMessage.match_id == match.id
                    ).order_by(desc(ChatMessage.created_at)).limit(1)
                )
                recent_msg = recent_msg_result.scalar_one_or_none()
                
                # Recent message preview
                if recent_msg:
                    # نمایش پیش‌نمایش بر اساس نوع پیام
                    if recent_msg.message_type == "text":
                        msg_content = recent_msg.content or "پیام متنی"
                        msg_preview = msg_content[:25] + "..." if len(msg_content) > 25 else msg_content
                    elif recent_msg.message_type == "photo":
                        msg_preview = "📸 عکس"
                    elif recent_msg.message_type == "voice":
                        msg_preview = "🎤 پیام صوتی"
                    elif recent_msg.message_type == "sticker":
                        msg_preview = "😄 استیکر"
                    elif recent_msg.message_type == "audio":
                        msg_preview = "🎵 موزیک"
                    elif recent_msg.message_type == "video":
                        msg_preview = "🎬 ویدیو"
                    elif recent_msg.message_type == "video_note":
                        msg_preview = "📹 ویدیو پیام"
                    elif recent_msg.message_type == "animation" or recent_msg.message_type == "gif":
                        msg_preview = "🎞️ گیف"
                    elif recent_msg.message_type == "document":
                        msg_preview = "📄 فایل"
                    else:
                        msg_preview = "💬 پیام"
                else:
                    msg_preview = "هنوز پیامی نیست"
                    
                text += f"{i+1}. 👤 <b>{user_name}</b>\n"
                text += f"   💭 {msg_preview}\n\n"
                
                # Add button for this chat (1 per row for better readability)
                chat_buttons.append([KeyboardButton(text=f"💬 {user_name[:15]}")])
            
            # کیبورد کاستوم برای گفتگوها
            chats_keyboard = ReplyKeyboardMarkup(
                keyboard=chat_buttons + [
                    [
                        KeyboardButton(text="🔄 بروزرسانی"),
                        KeyboardButton(text="🔍 یافتن مچ جدید")
                    ],
                    [
                        KeyboardButton(text="🔙 بازگشت به منو")
                    ]
                ],
                resize_keyboard=True,
                one_time_keyboard=False
            )
            
            await message.answer(text, reply_markup=chats_keyboard, parse_mode='HTML')
        
    except Exception as e:
        logger.error(f"Error viewing chats: {e}")
        await message.answer(
            "❌ <b>خطا در نمایش گفتگوها</b>\n\n"
            "لطفاً دوباره تلاش کنید.",
            parse_mode='HTML'
        )

# Message handlers برای دکمه‌های کیبورد کاستوم
@router.message(F.text.contains("🔄 بروزرسانی"))
async def refresh_chats_message(message: Message):
    """بروزرسانی گفتگوها از طریق message"""
    await view_my_chats_action(message)

@router.message(F.text.contains("🔍 یافتن مچ جدید"))
async def find_new_match_from_chats(message: Message):
    """یافتن مچ جدید از گفتگوها"""
    await message.answer(
        "🔍 <b>یافتن مچ جدید</b>\n\n"
        "برای یافتن مچ جدید، از منوی اصلی روی دکمه 'یافتن مچ' کلیک کنید.\n\n"
        "💡 نکات برای مچ بهتر:\n"
        "• پروفایل خود را کامل کنید\n"
        "• عکس‌های جذاب آپلود کنید\n"
        "• بیو جالب بنویسید\n"
        "• فعال باشید و دیگران را لایک کنید",
        parse_mode='HTML'
    )

@router.message(F.text.contains("🔙 بازگشت به منو"))
async def back_to_main_menu_chats(message: Message):
    """بازگشت به منوی اصلی از گفتگوها"""
    from handlers.registration import get_main_menu_keyboard
    from localization.language_helper import get_user_with_language
    from localization.translator import get_text
    import os
    
    user, user_language = await get_user_with_language(message.from_user.id)
    
    if not user:
        await message.answer(f"❌ {get_text(user_language, 'must_register_first', default='ابتدا باید ثبت‌نام کنید.')}")
        return
    
    # بررسی ادمین بودن
    ADMIN_USER_IDS = [int(uid) for uid in os.getenv('ADMIN_USER_IDS', '').split(',') if uid]
    is_admin = message.from_user.id in ADMIN_USER_IDS
    
    await message.answer(
        f"🏠 {get_text(user_language, 'back_to_menu', default='بازگشت به منوی اصلی')}",
        reply_markup=get_main_menu_keyboard(user_language, is_admin=is_admin)
    )

# Handler برای کلیک روی نام کاربران (شروع گفتگو)
@router.message(F.text.startswith("💬 "))
async def start_chat_with_user(message: Message):
    """شروع گفتگو با کاربر انتخاب شده"""
    user_name = message.text.replace("💬 ", "").strip()
    current_user_id = message.from_user.id
    
    async with db.get_session() as session:
        # پیدا کردن مچ مربوط به این کاربر
        matches_result = await session.execute(
            select(Match).where(
                and_(
                    Match.status == 1,
                    or_(Match.user_a == current_user_id, Match.user_b == current_user_id)
                )
            ).order_by(desc(Match.created_at))
        )
        matches = matches_result.scalars().all()
        
        target_match = None
        target_user = None
        
        for match in matches:
            other_user_id = match.user_b if match.user_a == current_user_id else match.user_a
            
            user_result = await session.execute(
                select(User).where(User.id == other_user_id)
            )
            other_user = user_result.scalar_one_or_none()
            
            if other_user:
                full_name = other_user.username or other_user.first_name or "کاربر"
                button_name = full_name[:15]
                
                # مقایسه دقیق‌تر
                if button_name == user_name or full_name == user_name:
                    target_match = match
                    target_user = other_user
                    break
        
        if not target_match or not target_user:
            await message.answer(
                f"❌ <b>کاربر یافت نشد</b>\n\n"
                f"جستجو برای: '{user_name}'\n"
                f"لطفاً از لیست گفتگوها انتخاب کنید.",
                parse_mode='HTML'
            )
            return
        
        # ذخیره اطلاعات کاربر فعلی
        current_chat_users[current_user_id] = {
            'target_user_id': target_user.user_id,
            'target_user_name': target_user.username or target_user.first_name or "کاربر",
            'match_id': target_match.id
        }
        
        # نمایش اطلاعات گفتگو
        user_display_name = target_user.username or target_user.first_name or "کاربر"
        
        # گرفتن آخرین پیام‌ها (بدون محدودیت)
        recent_messages_result = await session.execute(
            select(ChatMessage).where(
                ChatMessage.match_id == target_match.id
            ).order_by(desc(ChatMessage.created_at))
        )
        recent_messages = recent_messages_result.scalars().all()
        
        # کیبورد گفتگو
        chat_keyboard = ReplyKeyboardMarkup(
            keyboard=[
                [
                    KeyboardButton(text="✍️ ارسال پیام"),
                    KeyboardButton(text="📹 چت تصویری")
                ],
                [
                    KeyboardButton(text="👤 مشاهده پروفایل"),
                    KeyboardButton(text="🚨 گزارش")
                ],
                [
                    KeyboardButton(text="🚫 مسدود کردن"),
                    KeyboardButton(text="🔙 بازگشت به گفتگوها")
                ]
            ],
            resize_keyboard=True,
            one_time_keyboard=False
        )
        
        chat_text = f"💬 <b>گفتگو با {user_display_name}</b>\n\n"
        
        if recent_messages:
            chat_text += "📝 <b>آخرین پیام‌ها:</b>\n\n"
            for msg in reversed(recent_messages):
                sender = "شما" if msg.sender_id == current_user_id else user_display_name
                
                # نمایش پیام بر اساس نوع
                if msg.message_type == "text":
                    content = msg.content or "پیام متنی"
                elif msg.message_type == "photo":
                    content = "📸 عکس"
                elif msg.message_type == "voice":
                    content = "🎤 پیام صوتی"
                elif msg.message_type == "sticker":
                    content = "😄 استیکر"
                elif msg.message_type == "audio":
                    content = "🎵 موزیک"
                elif msg.message_type == "video":
                    content = "🎬 ویدیو"
                elif msg.message_type == "video_note":
                    content = "📹 ویدیو پیام"
                elif msg.message_type == "animation" or msg.message_type == "gif":
                    content = "🎞️ گیف"
                elif msg.message_type == "document":
                    content = "📄 فایل"
                else:
                    content = "💬 پیام"
                
                chat_text += f"• <b>{sender}:</b> {content}\n"
        else:
            chat_text += "هنوز پیامی رد و بدل نشده است.\n\n"
            chat_text += "💡 اولین پیام را شما ارسال کنید!"
        
        await message.answer(
            chat_text,
            reply_markup=chat_keyboard,
            parse_mode='HTML'
        )

@router.message(F.text.contains("🔙 بازگشت به گفتگوها"))
async def back_to_chats_list(message: Message):
    """بازگشت به لیست گفتگوها"""
    await view_my_chats_action(message)

@router.message(F.text.contains("✍️ ارسال پیام"))
async def send_message_prompt(message: Message, state: FSMContext):
    """شروع ارسال پیام"""
    user_id = message.from_user.id
    
    # بررسی وجود کاربر فعلی در گفتگو
    if user_id not in current_chat_users:
        await message.answer(
            "❌ <b>خطا</b>\n\n"
            "لطفاً ابتدا یک گفتگو را انتخاب کنید.",
            parse_mode='HTML'
        )
        return
    
    target_user_id = current_chat_users[user_id]['target_user_id']
    target_user_name = current_chat_users[user_id]['target_user_name']
    match_id = current_chat_users[user_id]['match_id']
    
    # کیبورد ساده برای چت
    chat_keyboard = ReplyKeyboardMarkup(
        keyboard=[
            [KeyboardButton(text="🔙 بازگشت"), KeyboardButton(text="🚫 پایان گفتگو")]
        ],
        resize_keyboard=True,
        one_time_keyboard=False
    )
    
    await message.answer(
        f"💬 <b>چت با {target_user_name}</b>\n\n"
        f"📝 پیام خود را بنویسید...",
        reply_markup=chat_keyboard,
        parse_mode='HTML'
    )
    
    # ذخیره اطلاعات در state
    await state.update_data(
        target_user_id=target_user_id,
        target_user_name=target_user_name,
        match_id=match_id
    )
    
    # ثبت وضعیت آنلاین در چت
    update_chat_activity(user_id, target_user_id, match_id)
    
    await state.set_state(ChatStates.sending_message)

@router.message(F.text.contains("👤 مشاهده پروفایل"))
async def view_user_profile_from_chat(message: Message):
    """مشاهده پروفایل کاربر از گفتگو"""
    user_id = message.from_user.id
    
    # بررسی وجود کاربر فعلی در گفتگو
    if user_id not in current_chat_users:
        await message.answer(
            "❌ <b>خطا</b>\n\n"
            "لطفاً ابتدا یک گفتگو را انتخاب کنید.",
            parse_mode='HTML'
        )
        return
    
    target_user_id = current_chat_users[user_id]['target_user_id']
    target_user_name = current_chat_users[user_id]['target_user_name']
    
    async with db.get_session() as session:
        # گرفتن اطلاعات کامل کاربر
        result = await session.execute(
            select(User).where(User.id == target_user_id)
        )
        target_user = result.scalar_one_or_none()
        
        if not target_user:
            await message.answer(
                "❌ <b>کاربر یافت نشد</b>\n\n"
                "این کاربر ممکن است حساب خود را حذف کرده باشد.",
                parse_mode='HTML'
            )
            return
        
        # ساخت پروفایل
        age = target_user.age if target_user.age else "نامشخص"
        gender_display = {
            1: 'مرد',
            2: 'زن',
            3: 'غیره',
            '1': 'مرد',
            '2': 'زن',
            '3': 'غیره'
        }.get(target_user.gender, 'نامشخص')
        
        profile_text = f"👤 <b>پروفایل {target_user_name}</b>\n\n"
        profile_text += f"🎂 <b>سن:</b> {age} ساله\n"
        profile_text += f"⚧️ <b>جنسیت:</b> {gender_display}\n"
        
        if target_user.city:
            profile_text += f"📍 <b>شهر:</b> {target_user.city}\n"
        
        if target_user.bio:
            profile_text += f"\n💭 <b>بیو:</b>\n<i>{target_user.bio}</i>\n"
        
        if target_user.interests:
            interests_list = target_user.interests.split(',') if isinstance(target_user.interests, str) else []
            if interests_list and any(i.strip() for i in interests_list):
                interests = ", ".join([i.strip() for i in interests_list if i.strip()])
                profile_text += f"\n🎯 <b>علایق:</b> {interests}\n"
        
        # نمایش عکس اصلی اگر موجود باشد
        has_photo = False
        main_photo_id = None
        
        if target_user.photos:
            try:
                photos_data = json.loads(target_user.photos) if isinstance(target_user.photos, str) else target_user.photos
                if photos_data and isinstance(photos_data, list) and len(photos_data) > 0:
                    photo_item = photos_data[0]
                    main_photo_id = photo_item["file_id"] if isinstance(photo_item, dict) and "file_id" in photo_item else (photo_item if isinstance(photo_item, str) else str(photo_item))
                    has_photo = True
            except Exception as e:
                logger.error(f"Error parsing photos: {e}")
        
        # ارسال پروفایل
        if has_photo and main_photo_id:
            try:
                await message.answer_photo(
                    photo=main_photo_id,
                    caption=profile_text,
                    parse_mode='HTML'
                )
            except Exception as e:
                logger.error(f"Error sending photo {main_photo_id}: {e}")
                await message.answer(profile_text, parse_mode='HTML')
        else:
            await message.answer(profile_text, parse_mode='HTML')

@router.message(F.text.contains("🚫 مسدود کردن"))
async def block_user_from_chat(message: Message):
    """مسدود کردن کاربر از گفتگو"""
    user_id = message.from_user.id
    
    # بررسی وجود کاربر فعلی در گفتگو
    if user_id not in current_chat_users:
        await message.answer(
            "❌ <b>خطا</b>\n\n"
            "لطفاً ابتدا یک گفتگو را انتخاب کنید.",
            parse_mode='HTML'
        )
        return
    
    target_user_name = current_chat_users[user_id]['target_user_name']
    target_user_id = current_chat_users[user_id]['target_user_id']
    match_id = current_chat_users[user_id]['match_id']
    
    # کیبورد تأیید
    confirm_keyboard = ReplyKeyboardMarkup(
        keyboard=[
            [
                KeyboardButton(text="✅ تأیید مسدود کردن"),
                KeyboardButton(text="❌ لغو")
            ],
            [
                KeyboardButton(text="🔙 بازگشت به گفتگوها")
            ]
        ],
        resize_keyboard=True,
        one_time_keyboard=False
    )
    
    await message.answer(
        f"🚫 <b>مسدود کردن {target_user_name}</b>\n\n"
        f"⚠️ <b>هشدار:</b> با مسدود کردن این کاربر:\n\n"
        f"• گفتگو فوراً پایان می‌یابد\n"
        f"• تمام پیام‌ها حذف می‌شوند\n"
        f"• دیگر نمی‌توانید با هم ارتباط برقرار کنید\n"
        f"• کاربر شما را در جستجو نخواهد دید\n\n"
        f"آیا مطمئن هستید؟",
        reply_markup=confirm_keyboard,
        parse_mode='HTML'
    )

@router.message(F.text.contains("🚨 گزارش"))
async def report_user_from_chat(message: Message):
    """گزارش کاربر از گفتگو"""
    user_id = message.from_user.id
    
    # بررسی وجود کاربر فعلی در گفتگو
    if user_id not in current_chat_users:
        await message.answer(
            "❌ <b>خطا</b>\n\n"
            "لطفاً ابتدا یک گفتگو را انتخاب کنید.",
            parse_mode='HTML'
        )
        return
    
    target_user_name = current_chat_users[user_id]['target_user_name']
    target_user_id = current_chat_users[user_id]['target_user_id']
    
    # کیبورد انتخاب دلیل گزارش
    report_keyboard = ReplyKeyboardMarkup(
        keyboard=[
            [
                KeyboardButton(text="🚫 هرزنامه/اسپم"),
                KeyboardButton(text="😡 رفتار نامناسب")
            ],
            [
                KeyboardButton(text="🔞 محتوای نامناسب"),
                KeyboardButton(text="💰 کلاهبرداری")
            ],
            [
                KeyboardButton(text="👤 پروفایل جعلی"),
                KeyboardButton(text="🔗 سایر موارد")
            ],
            [
                KeyboardButton(text="❌ لغو گزارش")
            ]
        ],
        resize_keyboard=True,
        one_time_keyboard=False
    )
    
    await message.answer(
        f"🚨 <b>گزارش {target_user_name}</b>\n\n"
        f"دلیل گزارش خود را انتخاب کنید:\n\n"
        f"⚠️ <b>توجه:</b>\n"
        f"• گزارش‌های کاذب منجر به مسدود شدن حساب شما می‌شود\n"
        f"• تمام گزارشات توسط تیم پشتیبانی بررسی می‌شوند\n"
        f"• اطلاعات شما محرمانه باقی می‌ماند",
        reply_markup=report_keyboard,
        parse_mode='HTML'
    )

# ==================== State Handlers ====================

@router.message(ChatStates.sending_message)
async def handle_sending_message(message: Message, state: FSMContext):
    """مدیریت ارسال پیام"""
    user_id = message.from_user.id
    
    # مدیریت دکمه‌های کنترلی (فقط برای پیام‌های متنی)
    if message.text:
        if message.text == "🔙 بازگشت":
            # پاک کردن وضعیت آنلاین
            if user_id in active_chat_sessions:
                del active_chat_sessions[user_id]
            await state.clear()
            await back_to_chats_list(message)
            return
        
        if message.text == "🚫 پایان گفتگو":
            # پاک کردن وضعیت آنلاین
            if user_id in active_chat_sessions:
                del active_chat_sessions[user_id]
            await state.clear()
            await message.answer("🚫 گفتگو پایان یافت")
            await back_to_chats_list(message)
            return
    
    # گرفتن اطلاعات از state
    data = await state.get_data()
    
    # اگر اطلاعات در state نیست، از current_chat_users استفاده کن
    if data and 'target_user_id' in data:
        target_user_id = data['target_user_id']
        target_user_name = data['target_user_name']
        match_id = data['match_id']
    elif user_id in current_chat_users:
        target_user_id = current_chat_users[user_id]['target_user_id']
        target_user_name = current_chat_users[user_id]['target_user_name']
        match_id = current_chat_users[user_id]['match_id']
        
        # ذخیره در state برای پیام‌های بعدی
        await state.update_data(
            target_user_id=target_user_id,
            target_user_name=target_user_name,
            match_id=match_id
        )
    else:
        await message.answer(
            "❌ <b>خطا: جلسه گفتگو یافت نشد</b>\n\n"
            "لطفاً دوباره از گفتگوهای من وارد شوید.",
            parse_mode='HTML'
        )
        await state.clear()
        await view_my_chats_action(message)
        return
    
    # بررسی اینکه آیا پیام محتوای معتبری دارد
    if not (message.text or message.photo or message.voice or message.sticker or 
            message.audio or message.document or message.animation):
        await message.answer("❌ نوع پیام پشتیبانی نمی‌شود.")
        return
    
    # فیلتر ویدیو - ویدیوها فقط در چت تصویری مجاز هستند
    if message.video or message.video_note:
        await message.answer(
            "❌ <b>ویدیو در چت عادی مجاز نیست!</b>\n\n"
            f"📹 برای ارسال ویدیو، از <b>چت تصویری</b> استفاده کنید\n"
            f"🎥 ویدیوهای دایره‌ای فقط در چت تصویری قابل ارسال هستند\n\n"
            f"💡 برای ورود به چت تصویری، دکمه <b>📹 چت تصویری</b> را بزنید",
            parse_mode='HTML'
        )
        return
    
    # ذخیره پیام در دیتابیس
    async with db.get_session() as session:
        try:
            # تشخیص نوع پیام
            message_type = "text"
            content = message.text
            
            if message.photo:
                message_type = "photo"
                content = message.photo[-1].file_id
            elif message.voice:
                message_type = "voice"
                content = message.voice.file_id
            elif message.sticker:
                message_type = "sticker"
                content = message.sticker.file_id
            elif message.audio:
                message_type = "audio"
                content = message.audio.file_id
            elif message.document:
                # بررسی اینکه آیا سند یک گیف است
                if message.document.mime_type and "gif" in message.document.mime_type:
                    message_type = "gif"
                else:
                    message_type = "document"
                content = message.document.file_id
            elif message.animation:
                message_type = "animation"
                content = message.animation.file_id
            
            # ایجاد پیام جدید
            new_message = ChatMessage(
                match_id=match_id,
                sender_id=user_id,
                message_type=message_type,
                content=content
            )
            session.add(new_message)
            await session.commit()
            
            # به‌روزرسانی وضعیت فعالیت فرستنده
            update_chat_activity(user_id, target_user_id, match_id)
            
            # گرفتن اطلاعات کاربران
            sender_result = await session.execute(
                select(User).where(User.id == user_id)
            )
            sender = sender_result.scalar_one_or_none()
            sender_name = sender.first_name if sender else "کاربر"
            
            target_result = await session.execute(
                select(User).where(User.id == target_user_id)
            )
            target_user = target_result.scalar_one_or_none()
            target_user_name = target_user.first_name if target_user else "کاربر"
            
            # پیام خودم نباید با برچسب "شما:" نمایش داده شود - مثل تلگرام عادی
            
            # ارسال پیام به طرف مقابل
            try:
                # بررسی اینکه آیا طرف مقابل در چت آنلاین است
                if is_user_online_in_chat(user_id, target_user_id):
                    # طرف مقابل آنلاین است - ارسال مستقیم پیام
                    success = await send_direct_chat_message(
                        user_id, target_user_id, sender_name, content, message_type, message
                    )
                    if success:
                        # تأیید ارسال real-time
                        confirm_msg = await message.answer("✅ ارسال شد (آنلاین)")
                        asyncio.create_task(delete_message_after_delay(confirm_msg, 2))
                    else:
                        confirm_msg = await message.answer("✅ ذخیره شد")
                        asyncio.create_task(delete_message_after_delay(confirm_msg, 2))
                else:
                    # طرف مقابل آفلاین است - فقط نوتیفیکیشن بفرست
                    await send_notification(target_user_id, sender_name)
                    confirm_msg = await message.answer("✅ ارسال شد")
                    asyncio.create_task(delete_message_after_delay(confirm_msg, 2))
                    
            except Exception as send_error:
                logger.warning(f"Could not send message to target user: {send_error}")
                confirm_msg = await message.answer("✅ ذخیره شد")
                asyncio.create_task(delete_message_after_delay(confirm_msg, 2))
            
            # state را پاک نکنیم تا بتوان پیام‌های بعدی ارسال کرد
            
        except Exception as e:
            logger.error(f"Error saving message: {e}")
            await message.answer(
                "❌ خطا در ارسال پیام. لطفاً دوباره تلاش کنید.",
                parse_mode='HTML'
            )

# ==================== Handler قدیمی /end حذف شد ====================
# حالا از دکمه‌های "🔙 بازگشت" و "🚫 پایان گفتگو" استفاده می‌کنیم

# ==================== Confirmation Handlers ====================

@router.message(F.text.contains("✅ تأیید مسدود کردن"))
async def confirm_block_user(message: Message):
    """تأیید مسدود کردن کاربر"""
    user_id = message.from_user.id
    
    if user_id not in current_chat_users:
        await message.answer("❌ خطا: کاربر هدف یافت نشد.")
        return
    
    target_user_id = current_chat_users[user_id]['target_user_id']
    target_user_name = current_chat_users[user_id]['target_user_name']
    match_id = current_chat_users[user_id]['match_id']
    
    async with db.get_session() as session:
        try:
            # تغییر وضعیت مچ به مسدود شده
            match_result = await session.execute(
                select(Match).where(Match.id == match_id)
            )
            match = match_result.scalar_one_or_none()
            
            if match:
                match.status = 0  # blocked
                await session.commit()
                
                # پاک کردن از حافظه
                del current_chat_users[user_id]
                
                await message.answer(
                    f"🚫 <b>{target_user_name} مسدود شد</b>\n\n"
                    f"✅ گفتگو پایان یافت\n"
                    f"✅ کاربر مسدود شد\n"
                    f"✅ دیگر نمی‌تواند با شما ارتباط برقرار کند\n\n"
                    f"📞 در صورت نیاز با پشتیبانی تماس بگیرید: @ChatMilosSupport",
                    parse_mode='HTML'
                )
                
                await view_my_chats_action(message)
            else:
                await message.answer("❌ خطا در مسدود کردن کاربر.")
                
        except Exception as e:
            logger.error(f"Error blocking user: {e}")
            await message.answer("❌ خطا در مسدود کردن کاربر.")

# ==================== Report Handlers ====================

@router.message(F.text.in_([
    "🚫 هرزنامه/اسپم", "😡 رفتار نامناسب", "🔞 محتوای نامناسب",
    "💰 کلاهبرداری", "👤 پروفایل جعلی", "🔗 سایر موارد"
]))
async def handle_report_reason(message: Message):
    """مدیریت دلیل گزارش"""
    user_id = message.from_user.id
    
    if user_id not in current_chat_users:
        await message.answer("❌ خطا: کاربر هدف یافت نشد.")
        return
    
    target_user_id = current_chat_users[user_id]['target_user_id']
    target_user_name = current_chat_users[user_id]['target_user_name']
    report_reason = message.text
    
    # ذخیره گزارش (شبیه‌سازی)
    try:
        # اطلاع به ادمین
        admin_id = 7203796090  # از حافظه
        admin_message = (
            f"🚨 <b>گزارش جدید</b>\n\n"
            f"👤 <b>گزارش‌دهنده:</b> {user_id}\n"
            f"🎯 <b>کاربر گزارش شده:</b> {target_user_name} ({target_user_id})\n"
            f"📋 <b>دلیل:</b> {report_reason}\n"
            f"📅 <b>تاریخ:</b> {datetime.now().strftime('%Y-%m-%d %H:%M:%S')}\n\n"
            f"لطفاً بررسی کنید."
        )
        
        # ارسال به ادمین (شبیه‌سازی)
        logger.info(f"Report sent to admin: {admin_message}")
        
        await message.answer(
            f"✅ <b>گزارش ثبت شد</b>\n\n"
            f"گزارش شما در مورد {target_user_name} با دلیل '{report_reason}' ثبت شد.\n\n"
            f"🔍 <b>مراحل بعدی:</b>\n"
            f"• تیم پشتیبانی گزارش را بررسی می‌کند\n"
            f"• در صورت تأیید، اقدامات لازم انجام می‌شود\n"
            f"• نتیجه از طریق نوتیفیکیشن اطلاع‌رسانی می‌شود\n\n"
            f"📞 پشتیبانی: @ChatMilosSupport\n"
            f"🆔 شماره گزارش: {user_id}{target_user_id}",
            parse_mode='HTML'
        )
        
        await view_my_chats_action(message)
        
    except Exception as e:
        logger.error(f"Error handling report: {e}")
        await message.answer("❌ خطا در ثبت گزارش. لطفاً دوباره تلاش کنید.")

@router.message(F.text.contains("❌ لغو گزارش"))
async def cancel_report(message: Message):
    """لغو گزارش"""
    await message.answer("❌ گزارش لغو شد.")
    await back_to_chats_list(message)

@router.message(F.text.contains("❌ لغو"))
async def cancel_action(message: Message):
    """لغو عمل"""
    await message.answer("❌ عمل لغو شد.")
    await back_to_chats_list(message)

# ==================== Video Chat Handlers ====================

async def delete_message_after_delay(message, delay_seconds: int):
    """حذف پیام بعد از تأخیر مشخص"""
    try:
        await asyncio.sleep(delay_seconds)
        await message.delete()
    except Exception as e:
        logger.error(f"Error deleting message: {e}")

async def show_unread_videos(message: Message, user_id: int, target_user_id: int, match_id: int):
    """نمایش ویدیوهای دیده‌نشده"""
    try:
        async with db.get_session() as session:
            # جستجوی ویدیوهای دیده‌نشده از طرف مقابل
            from sqlalchemy import select, and_
            unread_videos_result = await session.execute(
                select(ChatMessage).where(
                    and_(
                        ChatMessage.match_id == match_id,
                        ChatMessage.sender_id == target_user_id,
                        ChatMessage.message_type == "video_note",
                        ChatMessage.is_read == False
                    )
                ).order_by(ChatMessage.created_at)
            )
            unread_videos = unread_videos_result.scalars().all()
            
            if unread_videos:
                # نمایش تعداد ویدیوهای دیده‌نشده
                count = len(unread_videos)
                await message.answer(
                    f"📹 <b>{count} ویدیو دیده‌نشده دارید!</b>\n\n"
                    f"🎥 در حال نمایش ویدیوها...",
                    parse_mode='HTML'
                )
                
                # ارسال تمام ویدیوهای دیده‌نشده
                for video_msg in unread_videos:
                    try:
                        await message.answer_video_note(
                            video_note=video_msg.content
                        )
                        
                        # علامت‌گذاری به عنوان خوانده شده
                        video_msg.is_read = True
                        
                    except Exception as e:
                        logger.error(f"Error showing unread video: {e}")
                
                # ذخیره تغییرات
                await session.commit()
                
                # پیام اتمام نمایش
                completion_msg = await message.answer(
                    f"✅ <b>تمام ویدیوها نمایش داده شد</b>\n\n"
                    f"📱 حالا می‌توانید پاسخ دهید",
                    parse_mode='HTML'
                )
                
                # حذف پیام بعد از 3 ثانیه
                asyncio.create_task(delete_message_after_delay(completion_msg, 3))
                
    except Exception as e:
        logger.error(f"Error showing unread videos: {e}")

@router.message(F.text.contains("📹 چت تصویری"))
async def start_video_chat(message: Message, state: FSMContext):
    """شروع چت تصویری"""
    user_id = message.from_user.id
    
    # بررسی وجود کاربر فعلی در گفتگو
    if user_id not in current_chat_users:
        await message.answer(
            "❌ <b>خطا</b>\n\n"
            "لطفاً ابتدا یک گفتگو را انتخاب کنید.",
            parse_mode='HTML'
        )
        return
    
    target_user_id = current_chat_users[user_id]['target_user_id']
    target_user_name = current_chat_users[user_id]['target_user_name']
    match_id = current_chat_users[user_id]['match_id']
    
    # ذخیره اطلاعات چت تصویری
    video_chat_sessions[user_id] = {
        'target_user_id': target_user_id,
        'target_user_name': target_user_name,
        'match_id': match_id
    }
    
    # بررسی ویدیوهای دیده‌نشده
    await show_unread_videos(message, user_id, target_user_id, match_id)
    
    # کیبورد چت تصویری
    video_chat_keyboard = ReplyKeyboardMarkup(
        keyboard=[
            [KeyboardButton(text="🔙 بازگشت به چت عادی")]
        ],
        resize_keyboard=True,
        one_time_keyboard=False
    )
    
    await message.answer(
        f"📹 <b>چت تصویری با {target_user_name}</b>\n\n"
        f"🎥 <b>حالت دایره‌ای فعال</b>\n\n"
        f"📝 <b>راهنما:</b>\n"
        f"• فقط ویدیوهای دایره‌ای ارسال کنید\n"
        f"• پیام متنی و عکس مجاز نیست\n"
        f"• برای ضبط ویدیو دایره‌ای، دکمه ضبط را نگه دارید\n\n"
        f"🎬 آماده برای ارسال ویدیو دایره‌ای هستید!",
        reply_markup=video_chat_keyboard,
        parse_mode='HTML'
    )
    
    await state.set_state(ChatStates.video_chat_active)

@router.message(ChatStates.video_chat_active)
async def handle_video_chat_message(message: Message, state: FSMContext):
    """مدیریت پیام‌های چت تصویری"""
    user_id = message.from_user.id
    
    # بررسی دکمه بازگشت
    if message.text == "🔙 بازگشت به چت عادی":
        # حذف از session چت تصویری
        if user_id in video_chat_sessions:
            del video_chat_sessions[user_id]
        
        await state.clear()
        await message.answer(
            "✅ <b>بازگشت به چت عادی</b>\n\n"
            "حالا می‌توانید پیام‌های معمولی ارسال کنید.",
            parse_mode='HTML'
        )
        await view_my_chats_action(message)
        return
    
    # بررسی وجود session چت تصویری
    if user_id not in video_chat_sessions:
        await message.answer("❌ خطا در چت تصویری. لطفاً دوباره شروع کنید.")
        await state.clear()
        return
    
    session_info = video_chat_sessions[user_id]
    target_user_id = session_info['target_user_id']
    target_user_name = session_info['target_user_name']
    match_id = session_info['match_id']
    
    # بررسی نوع پیام - فقط ویدیو مجاز است
    if message.video_note:
        # ویدیو دایره‌ای - مجاز
        try:
            # ذخیره در دیتابیس
            async with db.get_session() as session:
                new_message = ChatMessage(
                    match_id=match_id,
                    sender_id=user_id,
                    message_type="video_note",
                    content=message.video_note.file_id,
                    is_read=False
                )
                session.add(new_message)
                await session.commit()
            
            # ارسال تأیید ساده به فرستنده
            confirmation_msg = await message.answer(
                "✅ <b>ویدیو ارسال شد!</b>",
                parse_mode='HTML'
            )
            
            # حذف پیام تأیید بعد از 2 ثانیه
            asyncio.create_task(delete_message_after_delay(confirmation_msg, 2))
            
            # ارسال اطلاع‌رسانی به طرف مقابل
            try:
                from aiogram import Bot
                bot_token = "8145304128:AAFkpxUUUpI68vYeQWdVo0JdxnmLbWXd_MQ"
                bot = Bot(token=bot_token)
                
                # ارسال پیام اطلاع‌رسانی به طرف مقابل
                sender_name = message.from_user.first_name or "کاربر"
                await bot.send_message(
                    chat_id=target_user_id,
                    text=f"📹 <b>ویدیو جدید از {sender_name}</b>\n\n"
                         f"💡 برای پاسخ به گفتگوهای من بروید",
                    parse_mode='HTML'
                )
                
            except Exception as e:
                logger.error(f"Error sending notification to target user: {e}")
                
        except Exception as e:
            logger.error(f"Error saving video message: {e}")
            await message.answer("❌ خطا در ارسال ویدیو. لطفاً دوباره تلاش کنید.")
            
    elif message.video:
        # ویدیو معمولی - غیرمجاز در چت تصویری
        await message.answer(
            "❌ <b>فقط ویدیو دایره‌ای مجاز است!</b>\n\n"
            f"🎥 لطفاً ویدیو دایره‌ای ارسال کنید\n"
            f"📱 برای ضبط ویدیو دایره‌ای، دکمه ضبط را نگه دارید",
            parse_mode='HTML'
        )
        
    elif message.text:
        # پیام متنی - غیرمجاز
        await message.answer(
            "❌ <b>پیام متنی در چت تصویری مجاز نیست!</b>\n\n"
            f"📹 فقط ویدیو دایره‌ای ارسال کنید\n"
            f"💬 برای پیام متنی، به چت عادی بازگردید",
            parse_mode='HTML'
        )
        
    elif message.photo:
        # عکس - غیرمجاز
        await message.answer(
            "❌ <b>عکس در چت تصویری مجاز نیست!</b>\n\n"
            f"📹 فقط ویدیو دایره‌ای ارسال کنید\n"
            f"🖼️ برای ارسال عکس، به چت عادی بازگردید",
            parse_mode='HTML'
        )
        
    elif message.voice:
        # پیام صوتی - غیرمجاز
        await message.answer(
            "❌ <b>پیام صوتی در چت تصویری مجاز نیست!</b>\n\n"
            f"📹 فقط ویدیو دایره‌ای ارسال کنید\n"
            f"🎤 برای پیام صوتی، به چت عادی بازگردید",
            parse_mode='HTML'
        )
        
    else:
        # سایر انواع فایل - غیرمجاز
        await message.answer(
            "❌ <b>این نوع فایل در چت تصویری مجاز نیست!</b>\n\n"
            f"📹 فقط ویدیو دایره‌ای ارسال کنید\n"
            f"📁 برای سایر فایل‌ها، به چت عادی بازگردید",
            parse_mode='HTML'
        )

# ==================== پایان فایل ====================
# تمام callback handlerها حذف شدند - فقط کیبورد کاستوم استفاده می‌شود
