"""
Profile Handler - هندلر پروفایل
مدیریت کامل پروفایل کاربر شامل نمایش، ویرایش و عکس‌ها
"""

from aiogram.types import Message, CallbackQuery, InlineKeyboardMarkup, InlineKeyboardButton, ReplyKeyboardMarkup, KeyboardButton, FSInputFile
from aiogram.fsm.context import FSMContext
from aiogram.fsm.state import State, StatesGroup
from sqlalchemy import select, and_, func, update
from sqlalchemy.exc import SQLAlchemyError
from aiogram import Router, F
from datetime import datetime
import logging
import os
import json
import io
from typing import Optional
from database.database import db
from database.models import User, Swipe
from localization.translator import get_text
from localization.language_helper import get_user_with_language, get_gender_display

router = Router()
logger = logging.getLogger(__name__)

# Helper function to get user data safely
async def get_user_data_safe(user_id: int):
    """دریافت داده‌های کاربر به صورت ایمن بدون detached object"""
    async with db.get_session() as session:
        result = await session.execute(select(User).where(User.id == user_id))
        user = result.scalar_one_or_none()
        
        if not user:
            return None, 'fa'
        
        # استخراج داده‌های مورد نیاز
        user_data = {
            'user_id': user.user_id,
            'username': user.username,
            'first_name': user.first_name,
            'age': user.age,
            'gender': user.gender,
            'country': user.country,
            'city': user.city,
            'bio': user.bio,
            'interests': user.interests,
            'photos': user.photos,
            'language_code': user.language_code or 'fa',
            'is_active': user.is_active,
            'is_premium': user.is_premium if hasattr(user, 'is_premium') else False,
            'looking_for': user.looking_for if hasattr(user, 'looking_for') else None
        }
        
        return user_data, user_data['language_code']

def get_default_profile_photo(gender: str):
    """دریافت مسیر عکس پیش‌فرض بر اساس جنسیت"""
    # gender: "1" = مرد، "2" = زن، "3" = غیره
    base_path = os.path.join(os.path.dirname(os.path.dirname(__file__)), 'assets')
    
    if gender == "1":
        photo_path = os.path.join(base_path, 'default_male.jpg')
    elif gender == "2":
        photo_path = os.path.join(base_path, 'default_female.jpg')
    else:
        photo_path = os.path.join(base_path, 'default_other.jpg')
    
    # اگر فایل وجود نداشت، از عکس پیش‌فرض عمومی استفاده کن
    if not os.path.exists(photo_path):
        photo_path = os.path.join(base_path, 'default_other.jpg')
    
    return photo_path if os.path.exists(photo_path) else None

class ProfileStates(StatesGroup):
    editing_bio = State()
    editing_interests = State()
    uploading_photo = State()
    editing_name = State()
    editing_age = State()
    editing_gender = State()
    editing_country = State()
    editing_city = State()

def get_profile_keyboard(language_code: str = 'fa'):
    """کیبورد پروفایل اصلی - ساده شده"""
    return ReplyKeyboardMarkup(
        keyboard=[
            [
                KeyboardButton(text=f"✏️ {get_text(language_code, 'edit_profile', default='ویرایش پروفایل')}"),
                KeyboardButton(text=f"📸 {get_text(language_code, 'profile_photos', default='عکس‌های پروفایل')}")
            ],
            [
                KeyboardButton(text=f"📊 {get_text(language_code, 'profile_stats', default='آمار پروفایل')}")
            ],
            [
                KeyboardButton(text=f"🔙 {get_text(language_code, 'back_to_menu', default='بازگشت به منو')}")
            ]
        ],
        resize_keyboard=True
    )

def get_edit_profile_keyboard(language_code: str = 'fa'):
    """کیبورد ویرایش پروفایل"""
    return ReplyKeyboardMarkup(
        keyboard=[
            [KeyboardButton(text=f"🎂 {get_text(language_code, 'edit_age', default='ویرایش سن')}"), KeyboardButton(text=f"👤 {get_text(language_code, 'edit_name', default='ویرایش نام')}")],
            [KeyboardButton(text=f"🏙️ {get_text(language_code, 'edit_city', default='ویرایش شهر')}"), KeyboardButton(text=f"🌍 {get_text(language_code, 'edit_country', default='ویرایش کشور')}")],
            [KeyboardButton(text=f"✏️ {get_text(language_code, 'edit_bio', default='ویرایش بیو')}"), KeyboardButton(text=f"🎯 {get_text(language_code, 'edit_interests', default='ویرایش علایق')}")],
            [KeyboardButton(text=f"⚧️ {get_text(language_code, 'edit_gender', default='ویرایش جنسیت')}")],
            [KeyboardButton(text=f"🔙 {get_text(language_code, 'back_to_profile', default='بازگشت به پروفایل')}")]
        ],
        resize_keyboard=True
    )


def get_interests_keyboard(language_code: str = 'fa', selected_interests: list = None):
    """کیبورد انتخاب علایق - Custom Keyboard"""
    if selected_interests is None:
        selected_interests = []
    
    # دریافت لیست علایق از ترجمه‌ها
    interests_dict = get_text(language_code, 'interests_list', default={})
    
    # اگر ترجمه موجود نباشد، از فارسی استفاده کن
    if not interests_dict:
        interests_dict = {
            'sports': 'ورزش',
            'music': 'موسیقی',
            'movies': 'فیلم',
            'books': 'کتاب',
            'travel': 'سفر',
            'cooking': 'آشپزی',
            'art': 'هنر',
            'photography': 'عکاسی',
            'gaming': 'بازی',
            'fitness': 'تناسب اندام',
            'nature': 'طبیعت',
            'technology': 'فناوری',
            'fashion': 'مد',
            'dancing': 'رقص',
            'reading': 'مطالعه',
            'writing': 'نویسندگی',
            'swimming': 'شنا',
            'hiking': 'کوهنوردی',
            'cycling': 'دوچرخه‌سواری',
            'yoga': 'یوگا',
            'meditation': 'مدیتیشن',
            'volunteering': 'داوطلبی',
            'learning': 'یادگیری',
            'socializing': 'اجتماعی بودن',
            'pets': 'حیوانات خانگی',
            'gardening': 'باغبانی',
            'shopping': 'خرید',
            'coffee': 'قهوه',
            'wine': 'شراب',
            'beer': 'آبجو',
            'nightlife': 'زندگی شبانه',
            'board_games': 'بازی‌های رومیزی',
            'video_games': 'بازی‌های ویدیویی',
            'anime': 'انیمه',
            'comedy': 'کمدی',
            'theater': 'تئاتر',
            'concerts': 'کنسرت',
            'festivals': 'جشنواره‌ها',
            'beach': 'ساحل',
            'mountains': 'کوه',
            'camping': 'کمپینگ',
            'fishing': 'ماهیگیری',
            'running': 'دویدن',
            'basketball': 'بسکتبال',
            'football': 'فوتبال',
            'tennis': 'تنیس',
            'golf': 'گلف',
            'skiing': 'اسکی',
            'surfing': 'موج‌سواری',
            'martial_arts': 'هنرهای رزمی',
            'boxing': 'بوکس',
            'chess': 'شطرنج',
            'meditation_yoga': 'مراقبه و یوگا',
            'volunteering_charity': 'کارهای خیریه'
        }
    
    # ساخت کیبورد custom
    keyboard = []
    row = []
    
    for key, value in interests_dict.items():
        # اگر علاقه انتخاب شده باشد، علامت ✅ اضافه کن
        if key in selected_interests:
            button_text = f"✅ {value}"
        else:
            button_text = value
        
        row.append(KeyboardButton(text=button_text))
        
        # هر 3 دکمه، یک ردیف جدید
        if len(row) == 3:
            keyboard.append(row)
            row = []
    
    # اگر دکمه‌ای باقی مانده، اضافه کن
    if row:
        keyboard.append(row)
    
    # اگر علاقه‌ای انتخاب شده، دکمه تایید اضافه کن
    if selected_interests:
        keyboard.append([
            KeyboardButton(text=f"✅ {get_text(language_code, 'confirm_interests', default='تایید علایق')} ({len(selected_interests)})")
        ])
    
    # دکمه لغو
    keyboard.append([
        KeyboardButton(text=f"❌ {get_text(language_code, 'cancel', default='لغو')}")
    ])
    
    return ReplyKeyboardMarkup(keyboard=keyboard, resize_keyboard=True)

def get_countries_keyboard(language_code: str = 'fa'):
    """کیبورد انتخاب کشور - Custom Keyboard"""
    
    # دریافت لیست کشورها از ترجمه‌ها
    countries_dict = get_text(language_code, 'countries_list', default={})
    
    # اگر ترجمه موجود نباشد، از فارسی استفاده کن (194 کشور)
    if not countries_dict:
        countries_dict = {
            # آسیا
            'afghanistan': 'افغانستان', 'armenia': 'ارمنستان', 'azerbaijan': 'آذربایجان', 'bahrain': 'بحرین',
            'bangladesh': 'بنگلادش', 'bhutan': 'بوتان', 'brunei': 'برونئی', 'cambodia': 'کامبوج',
            'china': 'چین', 'cyprus': 'قبرس', 'georgia': 'گرجستان', 'india': 'هند',
            'indonesia': 'اندونزی', 'iran': 'ایران', 'iraq': 'عراق', 'israel': 'اسرائیل',
            'japan': 'ژاپن', 'jordan': 'اردن', 'kazakhstan': 'قزاقستان', 'kuwait': 'کویت',
            'kyrgyzstan': 'قرقیزستان', 'laos': 'لائوس', 'lebanon': 'لبنان', 'malaysia': 'مالزی',
            'maldives': 'مالدیو', 'mongolia': 'مغولستان', 'myanmar': 'میانمار', 'nepal': 'نپال',
            'north_korea': 'کره شمالی', 'oman': 'عمان', 'pakistan': 'پاکستان', 'palestine': 'فلسطین',
            'philippines': 'فیلیپین', 'qatar': 'قطر', 'saudi_arabia': 'عربستان سعودی', 'singapore': 'سنگاپور',
            'south_korea': 'کره جنوبی', 'sri_lanka': 'سری‌لانکا', 'syria': 'سوریه', 'tajikistan': 'تاجیکستان',
            'thailand': 'تایلند', 'timor_leste': 'تیمور شرقی', 'turkey': 'ترکیه', 'turkmenistan': 'ترکمنستان',
            'uae': 'امارات متحده عربی', 'uzbekistan': 'ازبکستان', 'vietnam': 'ویتنام', 'yemen': 'یمن',
            
            # اروپا
            'albania': 'آلبانی', 'andorra': 'آندورا', 'austria': 'اتریش', 'belarus': 'بلاروس',
            'belgium': 'بلژیک', 'bosnia_herzegovina': 'بوسنی و هرزگوین', 'bulgaria': 'بلغارستان', 'croatia': 'کرواسی',
            'czech_republic': 'جمهوری چک', 'denmark': 'دانمارک', 'estonia': 'استونی', 'finland': 'فنلاند',
            'france': 'فرانسه', 'germany': 'آلمان', 'greece': 'یونان', 'hungary': 'مجارستان',
            'iceland': 'ایسلند', 'ireland': 'ایرلند', 'italy': 'ایتالیا', 'latvia': 'لتونی',
            'liechtenstein': 'لیختن‌اشتاین', 'lithuania': 'لیتوانی', 'luxembourg': 'لوکزامبورگ', 'malta': 'مالت',
            'moldova': 'مولداوی', 'monaco': 'موناکو', 'montenegro': 'مونته‌نگرو', 'netherlands': 'هلند',
            'north_macedonia': 'مقدونیه شمالی', 'norway': 'نروژ', 'poland': 'لهستان', 'portugal': 'پرتغال',
            'romania': 'رومانی', 'russia': 'روسیه', 'san_marino': 'سان‌مارینو', 'serbia': 'صربستان',
            'slovakia': 'اسلواکی', 'slovenia': 'اسلوونی', 'spain': 'اسپانیا', 'sweden': 'سوئد',
            'switzerland': 'سوئیس', 'ukraine': 'اوکراین', 'uk': 'انگلستان', 'vatican': 'واتیکان',
            
            # آفریقا
            'algeria': 'الجزایر', 'angola': 'آنگولا', 'benin': 'بنین', 'botswana': 'بوتسوانا',
            'burkina_faso': 'بورکینافاسو', 'burundi': 'بوروندی', 'cameroon': 'کامرون', 'cape_verde': 'کیپ‌ورد',
            'central_african_republic': 'جمهوری آفریقای مرکزی', 'chad': 'چاد', 'comoros': 'کومور', 'congo': 'کنگو',
            'congo_dr': 'جمهوری دموکراتیک کنگو', 'djibouti': 'جیبوتی', 'egypt': 'مصر', 'equatorial_guinea': 'گینه استوایی',
            'eritrea': 'اریتره', 'eswatini': 'اسواتینی', 'ethiopia': 'اتیوپی', 'gabon': 'گابن',
            'gambia': 'گامبیا', 'ghana': 'غنا', 'guinea': 'گینه', 'guinea_bissau': 'گینه بیسائو',
            'ivory_coast': 'ساحل عاج', 'kenya': 'کنیا', 'lesotho': 'لسوتو', 'liberia': 'لیبریا',
            'libya': 'لیبی', 'madagascar': 'ماداگاسکار', 'malawi': 'مالاوی', 'mali': 'مالی',
            'mauritania': 'موریتانی', 'mauritius': 'موریس', 'morocco': 'مراکش', 'mozambique': 'موزامبیک',
            'namibia': 'نامیبیا', 'niger': 'نیجر', 'nigeria': 'نیجریه', 'rwanda': 'رواندا',
            'sao_tome_principe': 'سائوتومه و پرنسیپ', 'senegal': 'سنگال', 'seychelles': 'سیشل', 'sierra_leone': 'سیرالئون',
            'somalia': 'سومالی', 'south_africa': 'آفریقای جنوبی', 'south_sudan': 'سودان جنوبی', 'sudan': 'سودان',
            'tanzania': 'تانزانیا', 'togo': 'توگو', 'tunisia': 'تونس', 'uganda': 'اوگاندا',
            'zambia': 'زامبیا', 'zimbabwe': 'زیمبابوه',
            
            # آمریکای شمالی
            'antigua_barbuda': 'آنتیگوا و باربودا', 'bahamas': 'باهاما', 'barbados': 'باربادوس', 'belize': 'بلیز',
            'canada': 'کانادا', 'costa_rica': 'کاستاریکا', 'cuba': 'کوبا', 'dominica': 'دومینیکا',
            'dominican_republic': 'جمهوری دومینیکن', 'el_salvador': 'السالوادور', 'grenada': 'گرنادا', 'guatemala': 'گواتمالا',
            'haiti': 'هائیتی', 'honduras': 'هندوراس', 'jamaica': 'جامائیکا', 'mexico': 'مکزیک',
            'nicaragua': 'نیکاراگوئه', 'panama': 'پاناما', 'saint_kitts_nevis': 'سنت کیتس و نویس', 'saint_lucia': 'سنت لوسیا',
            'saint_vincent_grenadines': 'سنت وینسنت و گرنادین', 'trinidad_tobago': 'ترینیداد و توباگو', 'usa': 'آمریکا',
            
            # آمریکای جنوبی
            'argentina': 'آرژانتین', 'bolivia': 'بولیوی', 'brazil': 'برزیل', 'chile': 'شیلی',
            'colombia': 'کلمبیا', 'ecuador': 'اکوادور', 'guyana': 'گویان', 'paraguay': 'پاراگوئه',
            'peru': 'پرو', 'suriname': 'سورینام', 'uruguay': 'اروگوئه', 'venezuela': 'ونزوئلا',
            
            # اقیانوسیه
            'australia': 'استرالیا', 'fiji': 'فیجی', 'kiribati': 'کیریباتی', 'marshall_islands': 'جزایر مارشال',
            'micronesia': 'میکرونزی', 'nauru': 'نائورو', 'new_zealand': 'نیوزیلند', 'palau': 'پالائو',
            'papua_new_guinea': 'پاپوآ گینه نو', 'samoa': 'ساموآ', 'solomon_islands': 'جزایر سلیمان', 'tonga': 'تونگا',
            'tuvalu': 'تووالو', 'vanuatu': 'وانواتو'
        }
    
    # ساخت کیبورد custom
    keyboard = []
    row = []
    
    for key, value in countries_dict.items():
        row.append(KeyboardButton(text=value))
        
        # هر 3 دکمه، یک ردیف جدید
        if len(row) == 3:
            keyboard.append(row)
            row = []
    
    # اگر دکمه‌ای باقی مانده، اضافه کن
    if row:
        keyboard.append(row)
    
    # دکمه لغو
    keyboard.append([
        KeyboardButton(text=f"❌ {get_text(language_code, 'cancel', default='لغو')}")
    ])
    
    return ReplyKeyboardMarkup(keyboard=keyboard, resize_keyboard=True)

# ==================== نمایش پروفایل ====================
@router.message((F.text.contains("پروفایل") | F.text.contains("Profile") | F.text.contains("Profil") | F.text.contains("الملف الشخصي") | F.text.contains("Профиль") | F.text.contains("Perfil") | F.text.contains("प्रोफाइल") | F.text.contains("Profilo") | F.text.contains("Профіль")) & ~F.text.contains("📸") & ~F.text.contains("ویرایش") & ~F.text.contains("Edit") & ~F.text.contains("Düzenle") & ~F.text.contains("تعديل") & ~F.text.contains("Редактировать") & ~F.text.contains("Editar") & ~F.text.contains("संपादित") & ~F.text.contains("Modifica") & ~F.text.contains("Modifier") & ~F.text.contains("өңдеу") & ~F.text.contains("bearbeiten") & ~F.text.contains("Edytuj") & ~F.text.contains("Редагувати") & ~F.text.contains("آمار") & ~F.text.contains("Stats") & ~F.text.contains("حریم") & ~F.text.contains("Privacy") & ~F.text.contains("راهنمای") & ~F.text.contains("Guide") & ~F.text.contains("Guida") & ~F.text.contains("Guía") & ~F.text.contains("İstatistik") & ~F.text.contains("إحصائيات") & ~F.text.contains("Статистика") & ~F.text.contains("Estadísticas") & ~F.text.contains("आँकड़े") & ~F.text.contains("Statistiche") & ~F.text.contains("Statistiques") & ~F.text.contains("Estatísticas") & ~F.text.contains("Statistik") & ~F.text.contains("статистикасы") & ~F.text.contains("statistiken") & ~F.text.contains("Statystyki"))
async def show_profile(message: Message):
    """نمایش پروفایل کاربر"""
    user_id = message.from_user.id
    
    # دریافت تازه کاربر از دیتابیس در session جدید
    async with db.get_session() as session:
        from sqlalchemy import select
        from database.models import User
        
        result = await session.execute(
            select(User).where(User.id == user_id)
        )
        user = result.scalar_one_or_none()
        
        if not user:
            await message.answer("❌ پروفایل شما یافت نشد. لطفاً ابتدا ثبت‌نام کنید.")
            return
        
        # دریافت زبان کاربر
        user_language = user.language_code or 'fa'
        
        # نمایش سن
        age = user.age if user.age else 18
        
        # نمایش جنسیت با ترجمه
        gender_display = get_gender_display(user_language, user.gender)
        
        # نمایش "دنبال چه کسی" با مقادیر صحیح
        if user.looking_for == '0' or user.looking_for == 0:
            looking_for_display = get_text(user_language, 'looking_for_any', default='💑 فرقی نمی‌کند')
        elif user.looking_for == '1' or user.looking_for == 1:
            looking_for_display = get_text(user_language, 'looking_for_male', default='👨 مرد')
        elif user.looking_for == '2' or user.looking_for == 2:
            looking_for_display = get_text(user_language, 'looking_for_female', default='👩 زن')
        else:
            looking_for_display = get_text(user_language, 'unknown', default='نامشخص')
        
        # ساخت متن پروفایل با ترجمه
        profile = f"{get_text(user_language, 'profile')}\n\n"
        profile += f"🏷️ <b>{get_text(user_language, 'name', default='نام')}:</b> {user.first_name or user.username or get_text(user_language, 'user', default='کاربر')}\n"
        profile += f"🎂 <b>{get_text(user_language, 'age', default='سن')}:</b> {age} {get_text(user_language, 'years_old', default='ساله')}\n"
        profile += f"⚧️ <b>{get_text(user_language, 'gender', default='جنسیت')}:</b> {gender_display}\n"
        
        # نمایش کشور و شهر با ترجمه
        if user.country:
            # استفاده مستقیم از نام کامل کشور (که در دیتابیس ذخیره شده)
            # چون در ثبت‌نام و ویرایش، نام کامل ذخیره می‌شود نه کد
            country_display = user.country
            
            profile += f"🌍 <b>{get_text(user_language, 'country', default='کشور')}:</b> {country_display}\n"
        
        if user.city:
            # ترجمه شهر به زبان کاربر (اگر موجود باشد)
            from utils.geo_helper import get_city_name
            city_display = get_city_name(user.city, user_language)
            profile += f"🏙️ <b>{get_text(user_language, 'city', default='شهر')}:</b> {city_display}\n"
        
        profile += f"💕 <b>{get_text(user_language, 'looking_for')}:</b> {looking_for_display}\n"
        
        if user.bio:
            profile += f"\n📝 <b>{get_text(user_language, 'bio', default='بیو')}:</b>\n{user.bio}\n"
        
        if user.interests:
            # تبدیل علایق به نام‌های قابل خواندن با ترجمه
            try:
                interests_dict = get_text(user_language, 'interests_list', default={})
                interests_keys = []
                
                # Parse کردن علایق (JSON یا comma-separated)
                if isinstance(user.interests, str):
                    if user.interests.startswith('['):
                        # فرمت JSON
                        interests_keys = json.loads(user.interests)
                    elif ',' in user.interests:
                        # فرمت comma-separated (قدیمی)
                        interests_keys = [k.strip() for k in user.interests.split(',')]
                    else:
                        # یک علاقه
                        interests_keys = [user.interests]
                
                # ترجمه علایق به زبان کاربر
                interests_names = []
                for key in interests_keys:
                    if key and key in interests_dict:
                        interests_names.append(interests_dict[key])
                    elif key:
                        interests_names.append(key)
                
                if interests_names:
                    interests_display = ', '.join(interests_names)
                    profile += f"\n🎯 <b>{get_text(user_language, 'interests', default='علایق')}:</b>\n{interests_display}\n"
            except Exception as e:
                logger.error(f"Error parsing interests: {e}")
                # در صورت خطا، نمایش ساده
                if user.interests:
                    profile += f"\n🎯 <b>{get_text(user_language, 'interests', default='علایق')}:</b>\n{user.interests}\n"
        
        active_status = get_text(user_language, 'active', default='فعال') if user.is_active else get_text(user_language, 'inactive', default='غیرفعال')
        premium_status = get_text(user_language, 'premium', default='پریمیوم') if user.is_premium else get_text(user_language, 'free', default='رایگان')
        
        profile += f"\n📊 <b>{get_text(user_language, 'status', default='وضعیت')}:</b> {'✅' if user.is_active else '❌'} {active_status}"
        profile += f"\n💎 <b>{get_text(user_language, 'subscription', default='اشتراک')}:</b> {'⭐' if user.is_premium else '🆓'} {premium_status}"
        
        # بررسی عکس‌ها
        has_photos = False
        photos_list = []
        
        if user.photos:
            try:
                photos_data = json.loads(user.photos) if isinstance(user.photos, str) else user.photos
                if photos_data and isinstance(photos_data, list) and len(photos_data) > 0:
                    photos_list = photos_data
                    has_photos = True
            except Exception as e:
                logger.error(f"Error parsing photos: {e}")
        
        # ذخیره gender برای استفاده خارج از session
        user_gender = user.gender
    
    # ارسال پروفایل با کیبورد چندزبانه (خارج از with block)
    profile_keyboard = get_profile_keyboard(user_language)
    
    if has_photos and photos_list:
        try:
            if len(photos_list) == 1:
                # یک عکس - ارسال معمولی
                await message.answer_photo(
                    photo=photos_list[0],
                    caption=profile,
                    reply_markup=profile_keyboard,
                    parse_mode='HTML'
                )
            elif len(photos_list) >= 2:
                # بیش از یک عکس - ارسال به صورت media group
                from aiogram.types import InputMediaPhoto
                
                media_group = []
                for i, photo_id in enumerate(photos_list[:10]):  # حداکثر 10 عکس (محدودیت تلگرام)
                    if i == 0:
                        # فقط اولین عکس با caption پروفایل
                        media_group.append(InputMediaPhoto(media=photo_id, caption=profile, parse_mode='HTML'))
                    else:
                        # بقیه عکس‌ها بدون caption
                        media_group.append(InputMediaPhoto(media=photo_id))
                
                await message.answer_media_group(media=media_group)
                
                # ارسال فقط کیبورد با کاراکتر نامرئی
                await message.answer(
                    get_text(user_language, 'choose_option', default='گزینه مورد نظر را انتخاب کنید:'),
                    reply_markup=profile_keyboard
                )
            else:
                await message.answer(profile, reply_markup=profile_keyboard, parse_mode='HTML')
        except Exception as e:
            logger.error(f"Error sending photos: {e}")
            await message.answer(profile, reply_markup=profile_keyboard, parse_mode='HTML')
    else:
        # عکس ندارد - نمایش عکس پیش‌فرض بر اساس جنسیت
        default_photo_path = get_default_profile_photo(user_gender)
        
        if default_photo_path:
            try:
                await message.answer_photo(
                    photo=FSInputFile(default_photo_path),
                    caption=profile,
                    reply_markup=profile_keyboard,
                    parse_mode='HTML'
                )
            except Exception as e:
                logger.error(f"Error sending default photo: {e}")
                await message.answer(profile, reply_markup=profile_keyboard, parse_mode='HTML')
        else:
            await message.answer(profile, reply_markup=profile_keyboard, parse_mode='HTML')

# ==================== هندلرهای کیبورد کاستوم ====================

# Handler های عکس (باید قبل از بقیه باشند)
@router.message(F.text.contains("📤"))
async def upload_photo_keyboard(message: Message, state: FSMContext):
    """آپلود عکس از کیبورد کاستوم"""
    user_id = message.from_user.id
    
    # Clear any existing state
    await state.clear()
    
    logger.info(f"Upload photo handler called for user {user_id}")
    
    # دریافت زبان کاربر
    user_data, user_language = await get_user_data_safe(user_id)
    
    async with db.get_session() as session:
        result = await session.execute(select(User).where(User.id == user_id))
        user = result.scalar_one_or_none()
        
        current_count = 0
        if user and user_data['photos']:
            try:
                photos_data = json.loads(user_data['photos']) if isinstance(user_data['photos'], str) else user_data['photos']
                current_count = len(photos_data) if photos_data else 0
            except:
                current_count = 0
    
    if current_count >= 5:
        max_photos_message = get_text(user_language, 'max_photos_message', default='شما قبلاً 5 عکس آپلود کرده‌اید.\nبرای آپلود عکس جدید، ابتدا یکی از عکس‌های قبلی را حذف کنید.')
        await message.answer(
            f"📸 <b>{get_text(user_language, 'max_photos_reached', default='حداکثر عکس')}</b>\n\n"
            f"{max_photos_message}",
            parse_mode='HTML'
        )
        return
    
    # ساخت کیبورد انتخاب چندزبانه
    upload_keyboard = InlineKeyboardMarkup(inline_keyboard=[
        [InlineKeyboardButton(
            text=f"📱 {get_text(user_language, 'use_telegram_photo', default='استفاده از عکس پروفایل تلگرام')}",
            callback_data="use_telegram_profile_photo"
        )],
        [InlineKeyboardButton(
            text=f"📤 {get_text(user_language, 'upload_new_photo', default='آپلود عکس جدید')}",
            callback_data="upload_new_photo"
        )]
    ])
    
    current_photos_text = get_text(user_language, 'current_photos_count', default='تعداد عکس‌های فعلی: {current}/5').format(current=current_count)
    
    await message.answer(
        f"📤 <b>{get_text(user_language, 'upload_profile_photo', default='آپلود عکس پروفایل')}</b>\n\n"
        f"📊 {current_photos_text}\n\n"
        f"💡 {get_text(user_language, 'choose_upload_option', default='یکی از گزینه‌های زیر را انتخاب کنید:')}",
        reply_markup=upload_keyboard,
        parse_mode='HTML'
    )

@router.message(F.text.contains("👁️"))
async def view_photos_keyboard(message: Message):
    """مشاهده عکس‌ها از کیبورد کاستوم"""
    user_id = message.from_user.id
    
    # دریافت زبان کاربر
    user_data, user_language = await get_user_data_safe(user_id)
    
    async with db.get_session() as session:
        result = await session.execute(select(User).where(User.id == user_id))
        user = result.scalar_one_or_none()
        
        if not user or not user_data['photos']:
            await message.answer(
                f"📸 <b>{get_text(user_language, 'your_photos', default='عکس‌های شما')}</b>\n\n"
                f"{get_text(user_language, 'no_photos_uploaded', default='شما هنوز عکسی آپلود نکرده‌اید.')}",
                parse_mode='HTML'
            )
            return
        
        try:
            photos_data = json.loads(user_data['photos']) if isinstance(user_data['photos'], str) else user_data['photos']
            if not photos_data or not isinstance(photos_data, list):
                raise ValueError("Invalid photos data")
        except Exception as e:
            logger.error(f"Error parsing photos for viewing: {e}")
            await message.answer(
                f"📸 <b>{get_text(user_language, 'your_photos', default='عکس‌های شما')}</b>\n\n"
                f"{get_text(user_language, 'error_loading_photos', default='خطا در بارگیری عکس‌ها.')}",
                parse_mode='HTML'
            )
            return
        
        # ارسال عکس‌ها با دکمه حذف
        for i, photo_id in enumerate(photos_data):
            try:
                # ساخت کیبورد حذف برای هر عکس
                delete_keyboard = InlineKeyboardMarkup(inline_keyboard=[
                    [InlineKeyboardButton(
                        text=f"🗑️ {get_text(user_language, 'delete_this_photo', default='حذف این عکس')}",
                        callback_data=f"delete_photo_{i}"
                    )]
                ])
                
                caption_text = get_text(user_language, 'photo_x_of_y', default='عکس {current} از {total}').format(
                    current=i+1, 
                    total=len(photos_data)
                )
                
                await message.answer_photo(
                    photo=photo_id,
                    caption=f"📸 {caption_text}",
                    reply_markup=delete_keyboard
                )
            except Exception as e:
                logger.error(f"Error sending photo {photo_id}: {e}")
        
        # دکمه بازگشت به پروفایل
        back_keyboard = ReplyKeyboardMarkup(
            keyboard=[
                [KeyboardButton(text=f"🔙 {get_text(user_language, 'back_to_profile', default='بازگشت به پروفایل')}")]
            ],
            resize_keyboard=True
        )
        
        await message.answer(
            f"✅ {get_text(user_language, 'photos_displayed', default='تمام عکس‌های شما نمایش داده شد.')}",
            reply_markup=back_keyboard
        )

@router.message(F.text.contains("📸"))
async def profile_photos_menu(message: Message):
    """نمایش منوی عکس‌های پروفایل"""
    user_data, user_language = await get_user_data_safe(message.from_user.id)
    
    if not user_data:
        await message.answer(f"❌ {get_text(user_language, 'user_not_found', default='کاربر یافت نشد')}")
        return
    
    # کیبورد عکس‌های پروفایل (2 ردیف)
    photos_keyboard = ReplyKeyboardMarkup(
        keyboard=[
            [
                KeyboardButton(text=f"👁️ {get_text(user_language, 'view_photos', default='مشاهده عکس‌ها')}"),
                KeyboardButton(text=f"📤 {get_text(user_language, 'upload_photo', default='آپلود عکس')}")
            ],
            [
                KeyboardButton(text=f"🔙 {get_text(user_language, 'back_to_profile', default='بازگشت به پروفایل')}")
            ]
        ],
        resize_keyboard=True
    )
    
    await message.answer(
        get_text(user_language, 'choose_photo_option', default='لطفاً یکی از موارد زیر را انتخاب کنید:'),
        reply_markup=photos_keyboard
    )

@router.message(F.text.contains("ویرایش پروفایل") | F.text.contains("Edit Profile") | F.text.contains("Profil Düzenle") | F.text.contains("تعديل الملف الشخصي") | F.text.contains("Редактировать профиль") | F.text.contains("Editar perfil") | F.text.contains("Редагувати профіль") | F.text.contains("प्रोफाइल संपादित करें") | F.text.contains("Modifica profilo") | F.text.contains("Modifier le profil") | F.text.contains("Editar perfil") | F.text.contains("Edit profil") | F.text.contains("Профильді өңдеу") | F.text.contains("Profil bearbeiten") | F.text.contains("Edytuj profil"))
async def edit_profile_menu(message: Message):
    """نمایش منوی ویرایش پروفایل"""
    user_data, user_language = await get_user_data_safe(message.from_user.id)
    
    if not user_data:
        await message.answer(f"❌ {get_text(user_language, 'profile_not_found', default='پروفایل شما یافت نشد. لطفاً ابتدا ثبت‌نام کنید.')}")
        return
    
    await message.answer(
        f"{get_text(user_language, 'edit_profile', default='✏️ ویرایش پروفایل')}\n\n"
        f"{get_text(user_language, 'edit_profile_menu', default='کدام بخش از پروفایل خود را می‌خواهید ویرایش کنید؟')}",
        reply_markup=get_edit_profile_keyboard(user_language),
        parse_mode='HTML'
    )

@router.message(F.text.contains("ویرایش بیو") | F.text.contains("Edit Bio") | F.text.contains("Biyografi Düzenle") | F.text.contains("تعديل النبذة") | F.text.contains("Редактировать био") | F.text.contains("Editar biografía") | F.text.contains("Редагувати біо") | F.text.contains("बायो संपादित करें") | F.text.contains("Modifica bio") | F.text.contains("Modifier la bio") | F.text.contains("Editar biografia") | F.text.contains("Edit Bio") | F.text.contains("Биографияны өңдеу") | F.text.contains("Bio bearbeiten") | F.text.contains("Edytuj bio"))
async def edit_bio_keyboard(message: Message, state: FSMContext):
    """شروع ویرایش بیو از کیبورد کاستوم"""
    user_data, user_language = await get_user_data_safe(message.from_user.id)
    
    if not user_data:
        await message.answer(f"❌ {get_text(user_language, 'profile_not_found', default='پروفایل شما یافت نشد. لطفاً ابتدا ثبت‌نام کنید.')}")
        return
    
    # کیبورد لغو
    cancel_keyboard = ReplyKeyboardMarkup(
        keyboard=[
            [KeyboardButton(text=f"❌ {get_text(user_language, 'cancel', default='لغو')}")]
        ],
        resize_keyboard=True,
        one_time_keyboard=False
    )
    
    await message.answer(
        f"<b>{get_text(user_language, 'edit_bio', default='✏️ ویرایش بیو')}</b>\n\n"
        f"{get_text(user_language, 'enter_new_bio', default='بیو جدید خود را بنویسید')}:\n\n"
        f"{get_text(user_language, 'notes_title', default='نکات:')}\n"
        f"• {get_text(user_language, 'max_characters', default='حداکثر {{count}} کاراکتر').format(count=200)}\n"
        f"• {get_text(user_language, 'appropriate_content', default='محتوای مناسب')}",
        reply_markup=cancel_keyboard,
        parse_mode='HTML'
    )
    await state.set_state(ProfileStates.editing_bio)

@router.message(F.text.contains("ویرایش علایق") | F.text.contains("Edit Interests") | F.text.contains("İlgi Alanları Düzenle") | F.text.contains("تعديل الاهتمامات") | F.text.contains("Редактировать интересы") | F.text.contains("Editar intereses") | F.text.contains("Редагувати інтереси") | F.text.contains("रुचियां संपादित करें") | F.text.contains("Modifica interessi") | F.text.contains("Modifier les intérêts") | F.text.contains("Editar interesses") | F.text.contains("Edit Minat") | F.text.contains("Қызығушылықтарды өңдеу") | F.text.contains("Interessen bearbeiten") | F.text.contains("Edytuj zainteresowania"))
async def edit_interests_keyboard(message: Message, state: FSMContext):
    """شروع ویرایش علایق از کیبورد کاستوم"""
    # دریافت کاربر در session جدید
    user_language = 'fa'
    current_interests = []
    
    async with db.get_session() as session:
        result = await session.execute(select(User).where(User.id == message.from_user.id))
        user = result.scalar_one_or_none()
        
        if not user:
            await message.answer(f"❌ {get_text(user_language, 'profile_not_found', default='پروفایل شما یافت نشد. لطفاً ابتدا ثبت‌نام کنید.')}")
            return
        
        user_language = user.language_code or 'fa'
        
        # ذخیره علایق فعلی کاربر در state
        if user.interests:
            try:
                current_interests = json.loads(user.interests) if isinstance(user.interests, str) else user.interests
            except:
                current_interests = user.interests.split(', ') if user.interests else []
    
    await state.update_data(selected_interests=current_interests)
    
    # نمایش کیبورد انتخاب علایق
    keyboard = get_interests_keyboard(user_language, current_interests)
    
    await message.answer(
        f"<b>{get_text(user_language, 'edit_interests', default='🎯 ویرایش علایق')}</b>\n\n"
        f"{get_text(user_language, 'select_interests', default='علایق خود را انتخاب کنید:')}",
        reply_markup=keyboard,
        parse_mode='HTML'
    )
    await state.set_state(ProfileStates.editing_interests)

@router.message(F.text.contains("ویرایش نام") | F.text.contains("Edit Name") | F.text.contains("İsim Düzenle") | F.text.contains("تعديل الاسم") | F.text.contains("Редактировать имя") | F.text.contains("Editar nombre") | F.text.contains("Редагувати ім'я") | F.text.contains("नाम संपादित करें") | F.text.contains("Modifica nome") | F.text.contains("Modifier le nom") | F.text.contains("Editar nome") | F.text.contains("Edit Nama") | F.text.contains("Атын өңдеу") | F.text.contains("Namen bearbeiten") | F.text.contains("Edytuj imię"))
async def edit_name_keyboard(message: Message, state: FSMContext):
    """شروع ویرایش نام از کیبورد کاستوم"""
    user_data, user_language = await get_user_data_safe(message.from_user.id)
    
    if not user_data:
        await message.answer(f"❌ {get_text(user_language, 'profile_not_found', default='پروفایل شما یافت نشد. لطفاً ابتدا ثبت‌نام کنید.')}")
        return
    
    # کیبورد لغو
    cancel_keyboard = ReplyKeyboardMarkup(
        keyboard=[
            [KeyboardButton(text=f"❌ {get_text(user_language, 'cancel', default='لغو')}")]
        ],
        resize_keyboard=True,
        one_time_keyboard=False
    )
    
    await message.answer(
        f"👤 <b>{get_text(user_language, 'edit_name', default='ویرایش نام')}</b>\n\n"
        f"{get_text(user_language, 'enter_new_name', default='نام جدید خود را وارد کنید')}:\n\n"
        f"{get_text(user_language, 'notes_title', default='نکات:')}\n"
        f"• {get_text(user_language, 'max_characters', default='حداکثر {{count}} کاراکتر').format(count=50)}\n"
        f"• {get_text(user_language, 'only_letters_numbers', default='فقط حروف و اعداد')}",
        reply_markup=cancel_keyboard,
        parse_mode='HTML'
    )
    await state.set_state(ProfileStates.editing_name)

@router.message(F.text.contains("ویرایش سن") | F.text.contains("Edit Age") | F.text.contains("Yaş Düzenle") | F.text.contains("تعديل العمر") | F.text.contains("Редактировать возраст") | F.text.contains("Editar edad") | F.text.contains("Редагувати вік") | F.text.contains("उम्र संपादित करें") | F.text.contains("Modifica età") | F.text.contains("Modifier l'âge") | F.text.contains("Editar idade") | F.text.contains("Edit Usia") | F.text.contains("Жасты өңдеу") | F.text.contains("Alter bearbeiten") | F.text.contains("Edytuj wiek"))
async def edit_age_keyboard(message: Message, state: FSMContext):
    """شروع ویرایش سن از کیبورد کاستوم"""
    user_data, user_language = await get_user_data_safe(message.from_user.id)
    
    if not user_data:
        await message.answer(f"❌ {get_text(user_language, 'profile_not_found', default='پروفایل شما یافت نشد. لطفاً ابتدا ثبت‌نام کنید.')}")
        return
    
    # کیبورد لغو
    cancel_keyboard = ReplyKeyboardMarkup(
        keyboard=[
            [KeyboardButton(text=f"❌ {get_text(user_language, 'cancel', default='لغو')}")]
        ],
        resize_keyboard=True,
        one_time_keyboard=False
    )
    
    await message.answer(
        f"🎂 <b>{get_text(user_language, 'edit_age', default='ویرایش سن')}</b>\n\n"
        f"{get_text(user_language, 'enter_new_age', default='سن جدید خود را وارد کنید')}:\n\n"
        f"{get_text(user_language, 'notes_title', default='نکات:')}\n"
        f"• {get_text(user_language, 'age_range_note', default='بین 18 تا 99 سال')}\n"
        f"• {get_text(user_language, 'only_numbers', default='فقط عدد وارد کنید')}",
        reply_markup=cancel_keyboard,
        parse_mode='HTML'
    )
    await state.set_state(ProfileStates.editing_age)

@router.message(F.text.contains("ویرایش کشور") | F.text.contains("Edit Country") | F.text.contains("Ülke Düzenle") | F.text.contains("تعديل البلد") | F.text.contains("Редактировать страну") | F.text.contains("Editar país") | F.text.contains("Редагувати країну") | F.text.contains("देश संपादित करें") | F.text.contains("Modifica paese") | F.text.contains("Modifier le pays") | F.text.contains("Editar país") | F.text.contains("Edit Negara") | F.text.contains("Елді өңдеу") | F.text.contains("Land bearbeiten") | F.text.contains("Edytuj kraj"))
async def edit_country_keyboard(message: Message, state: FSMContext):
    """شروع ویرایش کشور - ورودی دستی با fuzzy matching"""
    user_data, user_language = await get_user_data_safe(message.from_user.id)
    
    if not user_data:
        await message.answer(f"❌ {get_text(user_language, 'profile_not_found', default='پروفایل شما یافت نشد. لطفاً ابتدا ثبت‌نام کنید.')}")
        return
    
    # کیبورد لغو
    from aiogram.types import ReplyKeyboardMarkup, KeyboardButton
    cancel_keyboard = ReplyKeyboardMarkup(
        keyboard=[
            [KeyboardButton(text=f"❌ {get_text(user_language, 'cancel', default='لغو')}")]
        ],
        resize_keyboard=True
    )
    
    await message.answer(
        f"<b>{get_text(user_language, 'edit_country', default='🌍 ویرایش کشور')}</b>\n\n"
        f"{get_text(user_language, 'enter_country', default='نام کشور خود را بنویسید:')}\n\n"
        f"💡 {get_text(user_language, 'country_example', default='مثال: ایران، ترکیه، آلمان، کانادا')}",
        reply_markup=cancel_keyboard,
        parse_mode='HTML'
    )
    await state.set_state(ProfileStates.editing_country)

@router.message(F.text.contains("ویرایش شهر") | F.text.contains("Edit City") | F.text.contains("Şehir Düzenle") | F.text.contains("تعديل المدينة") | F.text.contains("Редактировать город") | F.text.contains("Editar ciudad") | F.text.contains("Редагувати місто") | F.text.contains("शहर संपादित करें") | F.text.contains("Modifica città") | F.text.contains("Modifier la ville") | F.text.contains("Editar cidade") | F.text.contains("Edit Kota") | F.text.contains("Қаланы өңдеу") | F.text.contains("Stadt bearbeiten") | F.text.contains("Edytuj miasto"))
async def edit_city_keyboard(message: Message, state: FSMContext):
    """شروع ویرایش شهر از کیبورد کاستوم"""
    user_data, user_language = await get_user_data_safe(message.from_user.id)
    
    if not user_data:
        await message.answer(f"❌ {get_text(user_language, 'profile_not_found', default='پروفایل شما یافت نشد. لطفاً ابتدا ثبت‌نام کنید.')}")
        return
    
    # کیبورد لغو
    cancel_keyboard = ReplyKeyboardMarkup(
        keyboard=[
            [KeyboardButton(text=f"❌ {get_text(user_language, 'cancel', default='لغو')}")]
        ],
        resize_keyboard=True,
        one_time_keyboard=False
    )
    
    await message.answer(
        f"🏙️ <b>{get_text(user_language, 'edit_city', default='ویرایش شهر')}</b>\n\n"
        f"{get_text(user_language, 'enter_new_city', default='نام شهر خود را وارد کنید')}:\n\n"
        f"{get_text(user_language, 'city_example', default='مثال: تهران، استانبول، برلین، تورنتو')}",
        reply_markup=cancel_keyboard,
        parse_mode='HTML'
    )
    await state.set_state(ProfileStates.editing_city)

@router.message(F.text.contains("ویرایش جنسیت") | F.text.contains("Edit Gender") | F.text.contains("Cinsiyet Düzenle") | F.text.contains("تعديل الجنس") | F.text.contains("Редактировать пол") | F.text.contains("Editar Género") | F.text.contains("Редагувати стать") | F.text.contains("लिंग संपादित करें") | F.text.contains("Modifica Genere") | F.text.contains("Modifier le Genre") | F.text.contains("Editar Gênero") | F.text.contains("Edit Jenis Kelamin") | F.text.contains("Жынысын өңдеу") | F.text.contains("Geschlecht bearbeiten") | F.text.contains("Edytuj Płeć"))
async def edit_gender_keyboard(message: Message, state: FSMContext):
    """شروع ویرایش جنسیت از کیبورد کاستوم"""
    user_data, user_language = await get_user_data_safe(message.from_user.id)
    
    if not user_data:
        await message.answer(f"❌ {get_text(user_language, 'profile_not_found', default='پروفایل شما یافت نشد. لطفاً ابتدا ثبت‌نام کنید.')}")
        return
    
    # تنظیم state برای ویرایش جنسیت پروفایل
    await state.set_state(ProfileStates.editing_gender)
    
    gender_keyboard = ReplyKeyboardMarkup(
        keyboard=[
            [KeyboardButton(text=f"👨 {get_text(user_language, 'gender_male', default='مرد')}"), 
             KeyboardButton(text=f"👩 {get_text(user_language, 'gender_female', default='زن')}")],
            [KeyboardButton(text=f"🏳️‍⚧️ {get_text(user_language, 'gender_other', default='غیره')}")],
            [KeyboardButton(text=f"🔙 {get_text(user_language, 'back_to_edit', default='بازگشت به ویرایش')}")]
        ],
        resize_keyboard=True
    )
    
    await message.answer(
        f"⚧️ <b>{get_text(user_language, 'edit_gender', default='ویرایش جنسیت')}</b>\n\n"
        f"{get_text(user_language, 'select_gender', default='جنسیت خود را انتخاب کنید')}:",
        reply_markup=gender_keyboard,
        parse_mode='HTML'
    )


@router.message(F.text.contains("بازگشت به پروفایل") | F.text.contains("Back to Profile") | F.text.contains("Profile'e Dön") | F.text.contains("العودة للملف الشخصي") | F.text.contains("Назад к профилю") | F.text.contains("Volver al Perfil") | F.text.contains("प्रोफाइल पर वापस") | F.text.contains("Torna al Profilo") | F.text.contains("Retour au Profil") | F.text.contains("Voltar ao Perfil") | F.text.contains("Kembali ke Profil") | F.text.contains("Профильге оралу") | F.text.contains("Zurück zum Profil") | F.text.contains("Powrót do Profilu") | F.text.contains("Назад до профілю"))
async def back_to_profile(message: Message):
    """بازگشت به پروفایل"""
    user_data, user_language = await get_user_data_safe(message.from_user.id)
    await message.answer(
        f"🔙 {get_text(user_language, 'back_to_profile', default='بازگشت به پروفایل')}",
        reply_markup=get_profile_keyboard(user_language)
    )

@router.message(F.text.contains("بازگشت به ویرایش") | F.text.contains("Back to Edit") | F.text.contains("Düzenleme'ye Dön") | F.text.contains("العودة للتعديل") | F.text.contains("Назад к редактированию") | F.text.contains("Volver a Editar") | F.text.contains("संपादन पर वापस") | F.text.contains("Torna alla Modifica") | F.text.contains("Retour à la Modification") | F.text.contains("Voltar à Edição") | F.text.contains("Kembali ke Edit") | F.text.contains("Өңдеуге оралу") | F.text.contains("Zurück zur Bearbeitung") | F.text.contains("Powrót do Edycji") | F.text.contains("Назад до редагування"))
async def back_to_edit_profile(message: Message):
    """بازگشت به منوی ویرایش پروفایل"""
    user_data, user_language = await get_user_data_safe(message.from_user.id)
    
    await message.answer(
        f"🔙 {get_text(user_language, 'back_to_edit', default='بازگشت به ویرایش پروفایل')}",
        reply_markup=get_edit_profile_keyboard(user_language)
    )

@router.message(F.text.contains("بازگشت به منو") | F.text.contains("Back to Menu") | F.text.contains("Menü'ye Dön") | F.text.contains("العودة للقائمة") | F.text.contains("Назад в меню") | F.text.contains("Volver al Menú") | F.text.contains("Назад до меню") | F.text.contains("मेनू पर वापस") | F.text.contains("Torna al Menu") | F.text.contains("Retour au Menu") | F.text.contains("Voltar ao Menu") | F.text.contains("Kembali ke Menu") | F.text.contains("Мәзірге оралу") | F.text.contains("Zurück zum Menü") | F.text.contains("Powrót do Menu"))
async def back_to_main_menu(message: Message):
    """بازگشت به منوی اصلی"""
    from handlers.registration import get_main_menu_keyboard
    from localization.language_helper import get_user_with_language
    from localization.translator import get_text
    import os
    
    user_data, user_language = await get_user_data_safe(message.from_user.id)
    
    if not user_data:
        await message.answer(f"❌ {get_text(user_language, 'must_register_first', default='ابتدا باید ثبت‌نام کنید.')}")
        return
    
    # بررسی ادمین بودن
    ADMIN_USER_IDS = [int(uid) for uid in os.getenv('ADMIN_USER_IDS', '').split(',') if uid]
    is_admin = message.from_user.id in ADMIN_USER_IDS
    
    await message.answer(
        f"🏠 {get_text(user_language, 'back_to_menu', default='بازگشت به منوی اصلی')}",
        reply_markup=get_main_menu_keyboard(user_language, is_admin=is_admin)
    )

# ==================== ویرایش بیو (State Handler) ====================

@router.message(ProfileStates.editing_bio)
async def save_bio(message: Message, state: FSMContext):
    """ذخیره بیو جدید"""
    user_data, user_language = await get_user_data_safe(message.from_user.id)
    
    # بررسی لغو
    if message.text == "/cancel" or message.text.startswith("❌") or get_text(user_language, 'cancel', default='لغو') in message.text:
        await message.answer(
            get_text(user_language, 'bio_edit_cancelled', default='ویرایش بیو لغو شد.'),
            reply_markup=get_edit_profile_keyboard(user_language)
        )
        await state.clear()
        return
    
    new_bio = message.text.strip()
    
    if len(new_bio) > 200:
        await message.answer(get_text(user_language, 'bio_too_long', default='بیو نباید بیشتر از 200 کاراکتر باشد. دوباره تلاش کنید:'))
        return
    
    user_id = message.from_user.id
    
    async with db.get_session() as session:
        result = await session.execute(select(User).where(User.id == user_id))
        user = result.scalar_one_or_none()
        
        if user:
            user.bio = new_bio
            await session.commit()
            await message.answer(
                f"✅ {get_text(user_language, 'bio_updated_success', default='بیو با موفقیت به‌روزرسانی شد!')}\n\n"
                f"🔙 {get_text(user_language, 'continue_editing_message', default='برای ادامه ویرایش یا مشاهده پروفایل از دکمه‌های زیر استفاده کنید.')}",
                reply_markup=get_edit_profile_keyboard(user_language)
            )
        else:
            await message.answer(get_text(user_language, 'bio_update_error', default='خطا در به‌روزرسانی بیو.'))
    
    await state.clear()

# ==================== ویرایش علایق (حذف شده - جایگزین با کیبورد انتخابی) ====================

# ==================== ویرایش نام (State Handler) ====================

@router.message(ProfileStates.editing_name)
async def save_name(message: Message, state: FSMContext):
    """ذخیره نام جدید"""
    user_data, user_language = await get_user_data_safe(message.from_user.id)
    
    # بررسی لغو
    if message.text == "/cancel" or message.text.startswith("❌") or get_text(user_language, 'cancel', default='لغو') in message.text:
        await message.answer(
            get_text(user_language, 'name_edit_cancelled', default='ویرایش نام لغو شد.'), 
            reply_markup=get_edit_profile_keyboard(user_language)
        )
        await state.clear()
        return
    
    new_name = message.text.strip()
    
    if len(new_name) > 50:
        await message.answer(get_text(user_language, 'name_too_long', default='نام نباید بیشتر از 50 کاراکتر باشد. دوباره تلاش کنید:'))
        return
    
    if len(new_name) < 2:
        await message.answer(get_text(user_language, 'name_too_short', default='نام باید حداقل 2 کاراکتر باشد. دوباره تلاش کنید:'))
        return
    
    user_id = message.from_user.id
    user_data, user_language = await get_user_data_safe(user_id)
    
    async with db.get_session() as session:
        result = await session.execute(select(User).where(User.id == user_id))
        user = result.scalar_one_or_none()
        
        if user:
            user.first_name = new_name
            await session.commit()
            await message.answer(
                f"✅ {get_text(user_language, 'name_updated_success', default='نام با موفقیت تغییر یافت!')}\n\n"
                f"🔙 {get_text(user_language, 'continue_editing_message', default='برای ادامه ویرایش یا مشاهده پروفایل از دکمه‌های زیر استفاده کنید.')}",
                reply_markup=get_edit_profile_keyboard(user_language)
            )
        else:
            await message.answer(get_text(user_language, 'name_update_error', default='خطا در به‌روزرسانی نام.'), reply_markup=get_profile_keyboard(user_language))
    
    await state.clear()

# ==================== ویرایش سن (State Handler) ====================

@router.message(ProfileStates.editing_age)
async def save_age(message: Message, state: FSMContext):
    """ذخیره سن جدید"""
    user_data, user_language = await get_user_data_safe(message.from_user.id)
    
    # بررسی لغو
    if message.text == "/cancel" or message.text.startswith("❌") or get_text(user_language, 'cancel', default='لغو') in message.text:
        await message.answer(
            get_text(user_language, 'age_edit_cancelled', default='ویرایش سن لغو شد.'), 
            reply_markup=get_edit_profile_keyboard(user_language)
        )
        await state.clear()
        return
    
    try:
        new_age = int(message.text.strip())
        
        if new_age < 18 or new_age > 99:
            await message.answer(get_text(user_language, 'age_range_error', default='سن باید بین 18 تا 99 سال باشد. دوباره تلاش کنید:'))
            return
        
        user_id = message.from_user.id
        user_data, user_language = await get_user_data_safe(user_id)
        
        async with db.get_session() as session:
            result = await session.execute(select(User).where(User.id == user_id))
            user = result.scalar_one_or_none()
            if user:
                user.age = new_age
                await session.commit()
                await message.answer(
                    f"✅ {get_text(user_language, 'age_updated_success', default='سن با موفقیت تغییر یافت!')}\n\n"
                    f"🔙 {get_text(user_language, 'continue_editing_message', default='برای ادامه ویرایش یا مشاهده پروفایل از دکمه‌های زیر استفاده کنید.')}",
                    reply_markup=get_edit_profile_keyboard(user_language)
                )
            else:
                await message.answer(get_text(user_language, 'age_update_error', default='خطا در به‌روزرسانی سن.'), reply_markup=get_edit_profile_keyboard(user_language))
        
        await state.clear()
        
    except ValueError:
        await message.answer(get_text(user_language, 'only_numbers', default='فقط عدد وارد کنید'))

# ==================== ویرایش کشور (State Handler) ====================

@router.message(ProfileStates.editing_country)
async def save_country(message: Message, state: FSMContext):
    """ذخیره کشور با fuzzy matching و پیشنهادات"""
    user_data, user_language = await get_user_data_safe(message.from_user.id)
    
    # بررسی لغو
    if message.text.startswith("❌") or get_text(user_language, 'cancel', default='لغو') in message.text:
        await message.answer(
            get_text(user_language, 'country_cancelled', default='انتخاب کشور لغو شد.'),
            reply_markup=get_edit_profile_keyboard(user_language)
        )
        await state.clear()
        return
    
    country_input = message.text.strip()
    
    # استفاده از fuzzy matching برای پیدا کردن کشور
    from utils.geo_helper import suggest_countries
    
    suggestions = suggest_countries(country_input, lang=user_language, limit=3)
    
    if not suggestions:
        await message.answer(
            f"❌ {get_text(user_language, 'country_not_found', default='کشوری یافت نشد. لطفاً دوباره تلاش کنید.')}\n\n"
            f"💡 {get_text(user_language, 'country_example', default='مثال: ایران، ترکیه، آلمان، کانادا')}"
        )
        return
    
    # بررسی امتیاز - اگر امتیاز اولی بالای 90 باشد، مستقیم قبول کن
    if suggestions[0]['score'] >= 90:
        selected = suggestions[0]
        
        user_id = message.from_user.id
        async with db.get_session() as session:
            result = await session.execute(select(User).where(User.id == user_id))
            user_obj = result.scalar_one_or_none()
            
            if user_obj:
                user_obj.country = selected['name']
                user_obj.country_code = selected['code']
                await session.commit()
                
                await message.answer(
                    f"✅ <b>{get_text(user_language, 'country_updated', default='کشور شما بروزرسانی شد!')}</b>\n\n"
                    f"🌍 <b>{get_text(user_language, 'selected_country', default='کشور انتخاب شده')}:</b> {selected['name']}",
                    reply_markup=get_edit_profile_keyboard(user_language),
                    parse_mode='HTML'
                )
            else:
                await message.answer(get_text(user_language, 'country_update_error', default='خطا در به‌روزرسانی کشور.'), reply_markup=get_edit_profile_keyboard(user_language))
        
        await state.clear()
    else:
        # نمایش پیشنهادات
        from aiogram.types import ReplyKeyboardMarkup, KeyboardButton
        
        keyboard_buttons = []
        for i, sug in enumerate(suggestions, 1):
            emoji = ['1️⃣', '2️⃣', '3️⃣'][i-1]
            keyboard_buttons.append([KeyboardButton(text=f"{emoji} {sug['name']}")])  
        keyboard_buttons.append([KeyboardButton(text=f"❌ {get_text(user_language, 'cancel', default='لغو')}")])
        
        keyboard = ReplyKeyboardMarkup(keyboard=keyboard_buttons, resize_keyboard=True)
        
        # ذخیره پیشنهادات در state
        await state.update_data(country_suggestions=suggestions)
        
        await message.answer(
            f"🔍 {get_text(user_language, 'did_you_mean', default='منظور شما این است؟')}\n\n"
            f"{get_text(user_language, 'select_from_suggestions', default='یکی از موارد زیر را انتخاب کنید:')}",
            reply_markup=keyboard
        )
        
        # منتظر انتخاب از پیشنهادات
        # اگر کاربر یکی از پیشنهادات را انتخاب کرد، دوباره به این handler می‌آید
        # و این بار باید از state داده‌ها را بخوانیم
        data = await state.get_data()
        if 'country_suggestions' in data:
            # کاربر در حال انتخاب از پیشنهادات است
            selected_text = message.text.replace("1️⃣", "").replace("2️⃣", "").replace("3️⃣", "").strip()
            
            for sug in data['country_suggestions']:
                if sug['name'] == selected_text:
                    user_id = message.from_user.id
                    async with db.get_session() as session:
                        result = await session.execute(select(User).where(User.id == user_id))
                        user_obj = result.scalar_one_or_none()
                        
                        if user_obj:
                            user_obj.country = sug['name']
                            user_obj.country_code = sug['code']
                            await session.commit()
                            
                            await message.answer(
                                f"✅ <b>{get_text(user_language, 'country_updated', default='کشور شما بروزرسانی شد!')}</b>\n\n"
                                f"🌍 <b>{get_text(user_language, 'selected_country', default='کشور انتخاب شده')}:</b> {sug['name']}",
                                reply_markup=get_edit_profile_keyboard(user_language),
                                parse_mode='HTML'
                            )
                        else:
                            await message.answer(get_text(user_language, 'country_update_error', default='خطا در به‌روزرسانی کشور.'), reply_markup=get_edit_profile_keyboard(user_language))
                    
                    await state.clear()
                    return

# ==================== ویرایش شهر (State Handler) ====================

@router.message(ProfileStates.editing_city)
async def save_city(message: Message, state: FSMContext):
    """ذخیره شهر جدید با fuzzy matching"""
    user_data, user_language = await get_user_data_safe(message.from_user.id)
    
    # بررسی لغو
    if message.text == "/cancel" or message.text.startswith("❌") or get_text(user_language, 'cancel', default='لغو') in message.text:
        await message.answer(
            get_text(user_language, 'city_edit_cancelled', default='ویرایش شهر لغو شد.'), 
            reply_markup=get_edit_profile_keyboard(user_language)
        )
        await state.clear()
        return
    
    new_city = message.text.strip()
    
    if len(new_city) > 50:
        await message.answer(get_text(user_language, 'city_too_long', default='نام شهر نباید بیشتر از 50 کاراکتر باشد. دوباره تلاش کنید:'))
        return
    
    if len(new_city) < 2:
        await message.answer(get_text(user_language, 'city_too_short', default='نام شهر باید حداقل 2 کاراکتر باشد. دوباره تلاش کنید:'))
        return
    
    # استفاده از fuzzy matching برای پیدا کردن شهر
    from utils.geo_helper import suggest_cities
    
    # دریافت country_code کاربر
    user_id = message.from_user.id
    async with db.get_session() as session:
        result = await session.execute(select(User).where(User.id == user_id))
        user_obj = result.scalar_one_or_none()
        country_code = getattr(user_obj, 'country_code', None) if user_obj else None
    
    # جستجوی شهر با fuzzy matching
    suggestions = suggest_cities(new_city, country_code=country_code, lang=user_language, limit=3)
    
    if not suggestions:
        await message.answer(
            f"❌ {get_text(user_language, 'city_not_found', default='شهری یافت نشد. لطفاً دوباره تلاش کنید.')}\n\n"
            f"💡 {get_text(user_language, 'city_example', default='مثال: تهران، استانبول، برلین، تورنتو')}"
        )
        return
    
    # بررسی امتیاز - اگر امتیاز اولی بالای 90 باشد، مستقیم قبول کن
    if suggestions[0]['score'] >= 90:
        selected = suggestions[0]
        
        async with db.get_session() as session:
            result = await session.execute(select(User).where(User.id == user_id))
            user_obj = result.scalar_one_or_none()
            
            if user_obj:
                user_obj.city = selected['name']
                await session.commit()
                
                await message.answer(
                    f"✅ <b>{get_text(user_language, 'city_updated_success', default='شهر شما بروزرسانی شد!')}</b>\n\n"
                    f"🏙️ <b>{get_text(user_language, 'selected_city', default='شهر انتخاب شده')}:</b> {selected['name']}",
                    reply_markup=get_edit_profile_keyboard(user_language),
                    parse_mode='HTML'
                )
            else:
                await message.answer(get_text(user_language, 'city_update_error', default='خطا در به‌روزرسانی شهر.'), reply_markup=get_edit_profile_keyboard(user_language))
        
        await state.clear()
    else:
        # نمایش پیشنهادات
        from aiogram.types import ReplyKeyboardMarkup, KeyboardButton
        
        keyboard_buttons = []
        for i, sug in enumerate(suggestions, 1):
            emoji = ['1️⃣', '2️⃣', '3️⃣'][i-1]
            keyboard_buttons.append([KeyboardButton(text=f"{emoji} {sug['name']}")])  
        keyboard_buttons.append([KeyboardButton(text=f"❌ {get_text(user_language, 'cancel', default='لغو')}")])
        
        keyboard = ReplyKeyboardMarkup(keyboard=keyboard_buttons, resize_keyboard=True)
        
        # ذخیره پیشنهادات در state
        await state.update_data(city_suggestions=suggestions)
        
        await message.answer(
            f"🔍 {get_text(user_language, 'did_you_mean', default='منظور شما این است؟')}\n\n"
            f"{get_text(user_language, 'select_from_suggestions', default='یکی از موارد زیر را انتخاب کنید:')}",
            reply_markup=keyboard
        )
        
        # منتظر انتخاب از پیشنهادات
        data = await state.get_data()
        if 'city_suggestions' in data:
            # کاربر در حال انتخاب از پیشنهادات است
            selected_text = message.text.replace("1️⃣", "").replace("2️⃣", "").replace("3️⃣", "").strip()
            
            for sug in data['city_suggestions']:
                if sug['name'] == selected_text:
                    user_id = message.from_user.id
                    async with db.get_session() as session:
                        result = await session.execute(select(User).where(User.id == user_id))
                        user_obj = result.scalar_one_or_none()
                        
                        if user_obj:
                            user_obj.city = sug['name']
                            await session.commit()
                            
                            await message.answer(
                                f"✅ <b>{get_text(user_language, 'city_updated_success', default='شهر شما بروزرسانی شد!')}</b>\n\n"
                                f"🏙️ <b>{get_text(user_language, 'selected_city', default='شهر انتخاب شده')}:</b> {sug['name']}",
                                reply_markup=get_edit_profile_keyboard(user_language),
                                parse_mode='HTML'
                            )
                        else:
                            await message.answer(get_text(user_language, 'city_update_error', default='خطا در به‌روزرسانی شهر.'), reply_markup=get_edit_profile_keyboard(user_language))
                    
                    await state.clear()
                    return

# ==================== ویرایش جنسیت (Handler) ====================

@router.message(ProfileStates.editing_gender)
async def save_gender(message: Message, state: FSMContext):
    """ذخیره جنسیت جدید - منطق مشابه ثبت‌نام"""
    user_data, user_language = await get_user_data_safe(message.from_user.id)
    
    # بررسی لغو
    if message.text == "/cancel" or message.text.startswith("❌") or get_text(user_language, 'cancel', default='لغو') in message.text:
        await state.clear()
        await message.answer(
            get_text(user_language, 'gender_edit_cancelled', default='ویرایش جنسیت لغو شد.'),
            reply_markup=get_edit_profile_keyboard(user_language)
        )
        return
    
    # تشخیص جنسیت از متن (مشابه ثبت‌نام)
    gender_male = get_text(user_language, 'gender_male', default='مرد')
    gender_female = get_text(user_language, 'gender_female', default='زن')
    gender_other = get_text(user_language, 'gender_other', default='غیره')
    
    if gender_male in message.text or "مرد" in message.text or "Male" in message.text or "Erkek" in message.text or "رجل" in message.text or "Мужчина" in message.text or "Hombre" in message.text or "Чоловік" in message.text or "पुरुष" in message.text or "Uomo" in message.text or "Homme" in message.text or "Homem" in message.text or "Pria" in message.text or "Ер адам" in message.text or "Mann" in message.text or "Mężczyzna" in message.text:
        selected_gender = 1
    elif gender_female in message.text or "زن" in message.text or "Female" in message.text or "Kadın" in message.text or "امرأة" in message.text or "Женщина" in message.text or "Mujer" in message.text or "Жінка" in message.text or "महिला" in message.text or "Donna" in message.text or "Femme" in message.text or "Mulher" in message.text or "Wanita" in message.text or "Әйел" in message.text or "Frau" in message.text or "Kobieta" in message.text:
        selected_gender = 2
    elif gender_other in message.text or "غیره" in message.text or "Other" in message.text or "Diğer" in message.text or "آخر" in message.text or "Другое" in message.text or "Otro" in message.text or "Інше" in message.text or "अन्य" in message.text or "Altro" in message.text or "Autre" in message.text or "Outro" in message.text or "Lainnya" in message.text or "Басқа" in message.text or "Andere" in message.text or "Inne" in message.text:
        selected_gender = 3
    else:
        # اگر تشخیص داده نشد، دوباره کیبورد نمایش بده
        await message.answer(
            get_text(user_language, 'select_gender', default='جنسیت خود را انتخاب کنید:'),
            reply_markup=get_gender_keyboard(user_language)
        )
        return
    
    user_id = message.from_user.id
    
    async with db.get_session() as session:
        result = await session.execute(select(User).where(User.id == user_id))
        user = result.scalar_one_or_none()
        
        if user:
            user.gender = selected_gender
            await session.commit()
            await message.answer(
                f"✅ {get_text(user_language, 'gender_updated_success', default='جنسیت با موفقیت تغییر یافت!')}\n\n"
                f"{get_text(user_language, 'continue_editing_message', default='برای ادامه ویرایش یا مشاهده پروفایل از دکمه‌های زیر استفاده کنید.')}",
                reply_markup=get_edit_profile_keyboard(user_language)
            )
        else:
            await message.answer(
                get_text(user_language, 'gender_update_error', default='خطا در به‌روزرسانی جنسیت.'), 
                reply_markup=get_edit_profile_keyboard(user_language)
            )
    
    await state.clear()

# ==================== مدیریت عکس‌ها (State Handler) ====================

@router.message(ProfileStates.uploading_photo, F.photo)
async def save_uploaded_photo(message: Message, state: FSMContext):
    """ذخیره عکس آپلود شده"""
    photo = message.photo[-1]
    photo_id = photo.file_id
    user_id = message.from_user.id
    
    # دریافت کاربر و زبان
    user_data, user_language = await get_user_data_safe(user_id)
    
    if not user_data:
        await message.answer(f"❌ {get_text(user_language, 'user_not_found', default='خطا: کاربر پیدا نشد.')}")
        await state.clear()
        return
    
    async with db.get_session() as session:
        result = await session.execute(select(User).where(User.id == user_id))
        user = result.scalar_one_or_none()
        
        if not user:
            await message.answer(f"❌ {get_text(user_language, 'user_not_found', default='خطا: کاربر پیدا نشد.')}")
            await state.clear()
            return
        
        photos_data = []
        if user_data['photos']:
            try:
                photos_data = json.loads(user_data['photos']) if isinstance(user_data['photos'], str) else user_data['photos']
                if not isinstance(photos_data, list):
                    photos_data = []
            except Exception as e:
                logger.error(f"Error parsing existing photos: {e}")
                photos_data = []
        
        if len(photos_data) >= 5:
            await message.answer(f"❌ {get_text(user_language, 'max_photos_reached_message', default='شما قبلاً 5 عکس آپلود کرده‌اید.')}")
            await state.clear()
            return
        
        photos_data.append(photo_id)
        user.photos = json.dumps(photos_data)
        
        # اضافه کردن user به session برای update
        session.add(user)
        await session.commit()
        
        # کیبورد مشاهده عکس‌ها
        from aiogram.types import ReplyKeyboardMarkup, KeyboardButton
        photos_keyboard = ReplyKeyboardMarkup(
            keyboard=[
                [
                    KeyboardButton(text=f"👁️ {get_text(user_language, 'view_photos', default='مشاهده عکس‌ها')}"),
                    KeyboardButton(text=f"📤 {get_text(user_language, 'upload_photo', default='آپلود عکس')}")
                ],
                [
                    KeyboardButton(text=f"🔙 {get_text(user_language, 'back_to_profile', default='بازگشت به پروفایل')}")
                ]
            ],
            resize_keyboard=True
        )
        
        await message.answer(
            f"✅ {get_text(user_language, 'photo_uploaded_success', default='عکس آپلود شد!')}\n"
            f"📊 {get_text(user_language, 'photo_count_status', default='تعداد عکس‌ها: {count}/5').format(count=len(photos_data))}\n\n"
            f"{get_text(user_language, 'view_updated_photos', default='مشاهده عکس‌های به‌روزرسانی شده')}",
            reply_markup=photos_keyboard
        )
        
        # پاک کردن state بعد از آپلود موفق
        await state.clear()

@router.message(ProfileStates.uploading_photo, F.content_type.in_({'video', 'document', 'sticker', 'voice', 'video_note', 'animation'}))
async def handle_invalid_file_type(message: Message, state: FSMContext):
    """مدیریت فایل‌های غیرعکس"""
    user_data, user_language = await get_user_data_safe(message.from_user.id)
    
    # کیبورد لغو
    from aiogram.types import ReplyKeyboardMarkup, KeyboardButton
    cancel_keyboard = ReplyKeyboardMarkup(
        keyboard=[
            [KeyboardButton(text=f"❌ {get_text(user_language, 'cancel', default='لغو')}")]
        ],
        resize_keyboard=True
    )
    
    await message.answer(
        f"❌ {get_text(user_language, 'invalid_file_type', default='فایل ارسالی نامعتبر است. لطفاً فقط عکس ارسال کنید.')}",
        reply_markup=cancel_keyboard
    )

@router.message(ProfileStates.uploading_photo)
async def handle_done_or_cancel(message: Message, state: FSMContext):
    """پایان آپلود عکس یا درخواست ارسال عکس"""
    user_data, user_language = await get_user_data_safe(message.from_user.id)
    
    # بررسی لغو
    if message.text in ["/done", "/cancel"] or message.text.startswith("❌") or get_text(user_language, 'cancel', default='لغو') in message.text:
        await state.clear()
        await message.answer(
            f"✅ {get_text(user_language, 'upload_finished', default='آپلود عکس تمام شد.')}",
            reply_markup=get_profile_keyboard(user_language)
        )
    else:
        # کیبورد لغو
        from aiogram.types import ReplyKeyboardMarkup, KeyboardButton
        cancel_keyboard = ReplyKeyboardMarkup(
            keyboard=[
                [KeyboardButton(text=f"❌ {get_text(user_language, 'cancel', default='لغو')}")]
            ],
            resize_keyboard=True
        )
        
        await message.answer(
            f"📸 {get_text(user_language, 'please_send_photo', default='لطفاً یک عکس ارسال کنید یا روی دکمه لغو کلیک کنید.')}",
            reply_markup=cancel_keyboard
        )

# ==================== آپلود عکس (Callback Handlers) ====================

@router.callback_query(F.data == "use_telegram_profile_photo")
async def use_telegram_profile_photo_callback(callback: CallbackQuery):
    """استفاده از عکس پروفایل تلگرام"""
    try:
        user_id = callback.from_user.id
        user_data, user_language = await get_user_data_safe(user_id)
        
        # دریافت عکس‌های پروفایل کاربر
        bot = callback.bot
        profile_photos = await bot.get_user_profile_photos(user_id, limit=1)
        
        if not profile_photos.photos:
            await callback.answer(f"❌ {get_text(user_language, 'no_profile_photo', default='شما عکس پروفایل ندارید!')}", show_alert=True)
            return
        
        # گرفتن بهترین کیفیت عکس پروفایل
        profile_photo = profile_photos.photos[0][-1]  # آخرین عکس = بهترین کیفیت
        photo_id = profile_photo.file_id
        
        # ذخیره عکس در دیتابیس
        async with db.get_session() as session:
            result = await session.execute(select(User).where(User.id == user_id))
            user = result.scalar_one_or_none()
            
            if not user:
                await callback.answer(f"❌ {get_text(user_language, 'user_not_found', default='کاربر پیدا نشد!')}", show_alert=True)
                return
            
            photos_data = []
            if user.photos:
                try:
                    photos_data = json.loads(user.photos) if isinstance(user.photos, str) else user.photos
                    if not isinstance(photos_data, list):
                        photos_data = []
                except:
                    photos_data = []
            
            if len(photos_data) >= 5:
                await callback.answer(f"❌ {get_text(user_language, 'max_photos_reached', default='شما قبلاً 5 عکس آپلود کرده‌اید!')}", show_alert=True)
                return
            
            photos_data.append(photo_id)
            user.photos = json.dumps(photos_data)
            session.add(user)
            
            await session.commit()
            
            user_data, user_language = await get_user_data_safe(callback.from_user.id)
            await callback.answer(f"✅ {get_text(user_language, 'photo_added_success', default='عکس پروفایل تلگرام اضافه شد!')}", show_alert=True)
            
            # ویرایش پیام
            await callback.message.edit_text(
                f"✅ <b>{get_text(user_language, 'profile_photo_added_success', default='عکس پروفایل اضافه شد!')}</b>\n\n"
                f"📊 {get_text(user_language, 'photo_count_status', default='تعداد عکس‌ها: {count}/5').format(count=len(photos_data))}\n\n"
                f"💡 {get_text(user_language, 'profile_photo_added_detail', default='عکس پروفایل تلگرام شما با موفقیت اضافه شد.')}",
                parse_mode='HTML'
            )
            
    except Exception as e:
        logger.error(f"Error using telegram profile photo: {e}")
        user_data, user_language = await get_user_data_safe(callback.from_user.id)
        await callback.answer(f"❌ {get_text(user_language, 'error_getting_photo', default='خطا در دریافت عکس پروفایل!')}", show_alert=True)

@router.callback_query(F.data == "upload_new_photo")
async def upload_new_photo_callback(callback: CallbackQuery, state: FSMContext):
    """شروع آپلود عکس جدید"""
    await callback.answer()
    
    user_data, user_language = await get_user_data_safe(callback.from_user.id)
    
    # کیبورد لغو
    cancel_keyboard = ReplyKeyboardMarkup(
        keyboard=[
            [KeyboardButton(text=f"❌ {get_text(user_language, 'cancel', default='لغو')}")]
        ],
        resize_keyboard=True,
        one_time_keyboard=False
    )
    
    await callback.message.edit_text(
        f"📸 <b>{get_text(user_language, 'upload_new_photo', default='آپلود عکس جدید')}</b>\n\n"
        f"{get_text(user_language, 'send_your_photo', default='عکس خود را ارسال کنید یا گزینه ایکون سنجاق بزنید تا عکس مورد نظر انتخاب شود')}",
        parse_mode='HTML'
    )
    
    # ارسال پیام جداگانه با کیبورد
    await callback.message.answer(
        f"💡 {get_text(user_language, 'photo_upload_instruction', default='عکس خود را ارسال کنید')}",
        reply_markup=cancel_keyboard
    )
    
    await state.set_state(ProfileStates.uploading_photo)

# ==================== حذف عکس (Callback Handler) ====================

@router.callback_query(F.data.startswith("delete_photo_"))
async def delete_photo_callback(callback: CallbackQuery):
    """حذف عکس از طریق callback"""
    try:
        # استخراج شماره عکس از callback_data
        photo_index = int(callback.data.split("_")[-1])
        user_id = callback.from_user.id
        
        # دریافت زبان کاربر
        user_data, user_language = await get_user_data_safe(user_id)
        
        async with db.get_session() as session:
            result = await session.execute(select(User).where(User.id == user_id))
            user = result.scalar_one_or_none()
            
            if not user or not user_data['photos']:
                await callback.answer(
                    f"❌ {get_text(user_language, 'no_photos_uploaded', default='عکسی برای حذف یافت نشد!')}",
                    show_alert=True
                )
                return
            
            try:
                photos_data = json.loads(user_data['photos']) if isinstance(user_data['photos'], str) else user_data['photos']
                if not isinstance(photos_data, list) or photo_index >= len(photos_data):
                    await callback.answer(
                        f"❌ {get_text(user_language, 'error_loading_photos', default='عکس یافت نشد!')}",
                        show_alert=True
                    )
                    return
            except:
                await callback.answer(
                    f"❌ {get_text(user_language, 'error_loading_photos', default='خطا در بارگیری عکس‌ها!')}",
                    show_alert=True
                )
                return
            
            # حذف عکس از لیست
            deleted_photo = photos_data.pop(photo_index)
            
            # به‌روزرسانی دیتابیس
            user.photos = json.dumps(photos_data) if photos_data else None
            session.add(user)
            await session.commit()
            
            # پاسخ موفقیت‌آمیز
            await callback.answer(
                f"✅ {get_text(user_language, 'photo_deleted_success', default='عکس حذف شد!')}",
                show_alert=True
            )
            
            # حذف پیام عکس
            try:
                await callback.message.delete()
            except:
                pass
            
            # ارسال پیام تأیید
            remaining_count = len(photos_data)
            await callback.message.answer(
                f"✅ <b>{get_text(user_language, 'photo_deleted_success', default='عکس حذف شد!')}</b>\n\n"
                f"📊 {get_text(user_language, 'remaining_photos_count', default='تعداد عکس‌های باقی‌مانده: {count}/5').format(count=remaining_count)}\n\n"
                f"💡 {get_text(user_language, 'view_updated_photos', default='برای مشاهده عکس‌های به‌روزرسانی شده، دوباره روی مشاهده عکس‌ها کلیک کنید.')}",
                parse_mode='HTML'
            )
            
    except Exception as e:
        logger.error(f"Error deleting photo: {e}")
        user_data, user_language = await get_user_data_safe(callback.from_user.id)
        await callback.answer(f"❌ {get_text(user_language, 'error_deleting_photo', default='خطا در حذف عکس!')}", show_alert=True)

# ==================== انتخاب علایق (Custom Keyboard Handler) ====================

@router.message(ProfileStates.editing_interests)
async def handle_interest_selection(message: Message, state: FSMContext):
    """پردازش انتخاب علایق از custom keyboard"""
    user_data, user_language = await get_user_data_safe(message.from_user.id)
    
    # بررسی لغو
    if message.text.startswith("❌") or get_text(user_language, 'cancel', default='لغو') in message.text:
        await message.answer(
            f"❌ {get_text(user_language, 'interests_cancelled', default='انتخاب علایق لغو شد.')}",
            reply_markup=get_edit_profile_keyboard(user_language)
        )
        await state.clear()
        return
    
    # دریافت علایق انتخاب شده از state
    data = await state.get_data()
    selected_interests = data.get('selected_interests', [])
    
    # بررسی تایید - چک کردن اینکه آیا متن شامل "تایید علایق" یا "Confirm Interests" است
    confirm_keywords = ['تایید علایق', 'Confirm Interests', 'İlgi Alanlarını Onayla', 'تأكيد الاهتمامات', 'Подтвердить интересы', 'Confirmar Intereses', 'Підтвердити інтереси', 'रुचियों की पुष्टि करें', 'Conferma Interessi', 'Confirmer les Intérêts', 'Confirmar Interesses', 'Konfirmasi Minat', 'Қызығушылықтарды растау', 'Interessen bestätigen', 'Potwierdź Zainteresowania']
    
    if message.text.startswith("✅") and any(keyword in message.text for keyword in confirm_keywords):
        
        if not selected_interests:
            await message.answer(
                get_text(user_language, 'no_interests_selected', default='هیچ علاقه‌ای انتخاب نشده است')
            )
            return
        
        # ذخیره علایق در دیتابیس
        async with db.get_session() as session:
            result = await session.execute(select(User).where(User.id == user_data['user_id']))
            user_obj = result.scalar_one_or_none()
            
            if user_obj:
                # تبدیل علایق به JSON
                user_obj.interests = json.dumps(selected_interests)
                await session.commit()
                
                # دریافت نام علایق برای نمایش
                interests_dict = get_text(user_language, 'interests_list', default={})
                selected_names = []
                for interest in selected_interests:
                    if interest in interests_dict:
                        selected_names.append(interests_dict[interest])
                    else:
                        selected_names.append(interest)
                
                # پیام تایید
                interests_text = ', '.join(selected_names)
                await message.answer(
                    f"✅ <b>{get_text(user_language, 'interests_updated', default='علایق شما بروزرسانی شد!')}</b>\n\n"
                    f"🎯 <b>{get_text(user_language, 'selected_interests', default='علایق انتخاب شده')}:</b>\n{interests_text}",
                    reply_markup=get_edit_profile_keyboard(user_language),
                    parse_mode='HTML'
                )
        
        await state.clear()
        return
    
    # پردازش انتخاب علاقه
    interests_dict = get_text(user_language, 'interests_list', default={})
    
    # پیدا کردن کلید علاقه از متن دکمه
    selected_key = None
    button_text = message.text.replace("✅ ", "")  # حذف علامت تیک
    
    for key, value in interests_dict.items():
        if value == button_text:
            selected_key = key
            break
    
    if selected_key:
        # اگر علاقه انتخاب شده، حذف کن، وگرنه اضافه کن
        if selected_key in selected_interests:
            selected_interests.remove(selected_key)
        else:
            # بررسی محدودیت 6 علاقه
            if len(selected_interests) >= 6:
                max_interests_messages = {
                    'fa': '❌ حداکثر 6 علاقه می‌توانید انتخاب کنید.\n\n💡 برای تایید نهایی لطفاً به پایین کیبورد اسکرول کنید و دکمه "✅ تایید علایق" را بزنید.',
                    'en': '❌ You can select maximum 6 interests.\n\n💡 To confirm, please scroll down the keyboard and press "✅ Confirm Interests".',
                    'tr': '❌ Maksimum 6 ilgi alanı seçebilirsiniz.\n\n💡 Onaylamak için lütfen klavyeyi aşağı kaydırın ve "✅ İlgi Alanlarını Onayla" düğmesine basın.',
                    'ar': '❌ يمكنك اختيار 6 اهتمامات كحد أقصى.\n\n💡 للتأكيد، يرجى التمرير لأسفل لوحة المفاتيح والضغط على "✅ تأكيد الاهتمامات".',
                    'ru': '❌ Вы можете выбрать максимум 6 интересов.\n\n💡 Для подтверждения прокрутите клавиатуру вниз и нажмите "✅ Подтвердить интересы".',
                    'es': '❌ Puedes seleccionar máximo 6 intereses.\n\n💡 Para confirmar, desplázate hacia abajo en el teclado y presiona "✅ Confirmar Intereses".',
                    'uk': '❌ Ви можете вибрати максимум 6 інтересів.\n\n💡 Для підтвердження прокрутіть клавіатуру вниз і натисніть "✅ Підтвердити інтереси".',
                    'hi': '❌ आप अधिकतम 6 रुचियां चुन सकते हैं।\n\n💡 पुष्टि करने के लिए, कृपया कीबोर्ड को नीचे स्क्रॉल करें और "✅ रुचियों की पुष्टि करें" दबाएं।',
                    'it': '❌ Puoi selezionare massimo 6 interessi.\n\n💡 Per confermare, scorri verso il basso la tastiera e premi "✅ Conferma Interessi".',
                    'fr': '❌ Vous pouvez sélectionner maximum 6 intérêts.\n\n💡 Pour confirmer, faites défiler le clavier vers le bas et appuyez sur "✅ Confirmer les Intérêts".',
                    'pt': '❌ Você pode selecionar no máximo 6 interesses.\n\n💡 Para confirmar, role o teclado para baixo e pressione "✅ Confirmar Interesses".',
                    'id': '❌ Anda dapat memilih maksimal 6 minat.\n\n💡 Untuk mengonfirmasi, gulir keyboard ke bawah dan tekan "✅ Konfirmasi Minat".',
                    'kk': '❌ Сіз максимум 6 қызығушылық таңдай аласыз.\n\n💡 Растау үшін пернетақтаны төмен айналдырыңыз және "✅ Қызығушылықтарды растау" басыңыз.',
                    'de': '❌ Sie können maximal 6 Interessen auswählen.\n\n💡 Zum Bestätigen scrollen Sie bitte die Tastatur nach unten und drücken Sie "✅ Interessen bestätigen".',
                    'pl': '❌ Możesz wybrać maksymalnie 6 zainteresowań.\n\n💡 Aby potwierdzić, przewiń klawiaturę w dół i naciśnij "✅ Potwierdź Zainteresowania".'
                }
                await message.answer(max_interests_messages.get(user_language, max_interests_messages['en']))
                return
            
            selected_interests.append(selected_key)
        
        # بروزرسانی state
        await state.update_data(selected_interests=selected_interests)
        
        # بروزرسانی کیبورد بدون پیام اضافی
        new_keyboard = get_interests_keyboard(user_language, selected_interests)
        
        await message.answer(
            "✅" if selected_key in selected_interests else "❌",
            reply_markup=new_keyboard
        )

# ==================== آمار پروفایل ====================

async def get_profile_stats(user_id: int) -> dict:
    """دریافت آمار پروفایل"""
    async with db.get_session() as session:
        result = await session.execute(
            select(User).where(User.id == user_id)
        )
        user = result.scalar_one_or_none()
        
        if not user:
            return {'views': 0, 'likes_received': 0, 'likes_sent': 0}
        
        # شمارش لایک‌های ارسالی
        likes_sent_result = await session.execute(
            select(func.count(Swipe.id)).where(
                and_(Swipe.from_user == user_id, Swipe.action == 1)
            )
        )
        likes_sent = likes_sent_result.scalar() or 0
        
        return {
            'views': user.profile_views or 0,
            'likes_received': user.likes_received or 0,
            'likes_sent': likes_sent
        }

@router.message(F.text.contains("آمار پروفایل") | F.text.contains("Profile Stats") | F.text.contains("Profil İstatistikleri") | F.text.contains("إحصائيات الملف") | F.text.contains("Статистика профиля") | F.text.contains("Estadísticas del perfil") | F.text.contains("Статистика профілю") | F.text.contains("प्रोफ़ाइल आँकड़े") | F.text.contains("Statistiche profilo") | F.text.contains("Statistiques du profil") | F.text.contains("Estatísticas do perfil") | F.text.contains("Statistik Profil") | F.text.contains("Профиль статистикасы") | F.text.contains("Profilstatistiken") | F.text.contains("Statystyki profilu"))
async def show_profile_stats(message: Message):
    """نمایش آمار پروفایل"""
    user_data, user_language = await get_user_data_safe(message.from_user.id)
    
    if not user_data:
        await message.answer(
            get_text(user_language, 'profile_not_found', default='پروفایل یافت نشد')
        )
        return
    
    stats = await get_profile_stats(user_data['user_id'])
    
    stats_text = (
        f"<b>{get_text(user_language, 'profile_stats', default='📊 آمار پروفایل')}</b>\n\n"
        f"👁️ <b>{get_text(user_language, 'profile_views', default='بازدید پروفایل')}:</b> "
        f"{get_text(user_language, 'views_count', default='{count} بازدید').format(count=stats['views'])}\n\n"
        f"❤️ <b>{get_text(user_language, 'likes_received', default='لایک دریافتی')}:</b> "
        f"{get_text(user_language, 'likes_count', default='{count} لایک').format(count=stats['likes_received'])}\n\n"
        f"💕 <b>{get_text(user_language, 'likes_sent', default='لایک ارسالی')}:</b> "
        f"{get_text(user_language, 'likes_count', default='{count} لایک').format(count=stats['likes_sent'])}"
    )
    
    await message.answer(stats_text, parse_mode='HTML')


# ==================== پایان فایل ====================
