"""
Registration Handler - هندلر ثبت‌نام
سناریوی کامل ثبت‌نام با کیبورد کاستوم (بدون inline button)

ترتیب مراحل:
1. انتخاب زبان (متن پیش‌فرض انگلیسی)
2. پیام خوش‌آمدگویی + دکمه ادامه
3. قوانین و حریم خصوصی
4. انتخاب جنسیت
5. دنبال چه کسی هستید
6. نام
7. سن (18-80)
8. روش انتخاب کشور (دستی یا لوکیشن)
9. بیو (اختیاری)
10. علایق (کیبورد کاستوم مثل پروفایل)
11. پایان ثبت‌نام
"""

from aiogram import Router, F, Bot
from aiogram.types import Message, ReplyKeyboardMarkup, KeyboardButton, ReplyKeyboardRemove, InputFile, FSInputFile
from aiogram.fsm.context import FSMContext
from aiogram.fsm.state import State, StatesGroup
from sqlalchemy import select
from database.database import db
from database.models import User, Preference
from localization.translator import get_text
from localization.translations import LANGUAGE_CODES
from utils.geo_helper import suggest_countries, suggest_cities, get_country_name, get_city_name
from handlers.auth_handler import (
    is_valid_email, is_valid_password, hash_password, verify_password,
    generate_verification_code, send_verification_email, verify_code,
    get_password_requirements
)
import os
import json
from utils.matching import matching_engine
import logging
from datetime import datetime

router = Router()
logger = logging.getLogger(__name__)

class RegistrationStates(StatesGroup):
    language = State()
    welcome = State()
    terms = State()
    auth_choice = State()  # انتخاب ثبت‌نام یا ورود مهمان
    signup_or_login = State()  # انتخاب ثبت‌نام یا ورود
    email_input = State()  # وارد کردن ایمیل
    verification_code = State()  # وارد کردن کد تأیید
    password_input = State()  # وارد کردن رمز عبور
    password_confirm = State()  # تأیید رمز عبور
    login_password = State()  # رمز عبور برای ورود
    logout_confirm = State()  # تأیید خروج از حساب
    gender = State()
    looking_for = State()
    name = State()
    age = State()
    country_method = State()
    country = State()
    country_confirm = State()  # تایید کشور از پیشنهادها
    city = State()
    city_confirm = State()  # تایید شهر از پیشنهادها
    bio = State()
    interests = State()
    photo_method = State()
    photo_upload = State()
    photo_confirm = State()

# ==================== Helper Functions ====================

def get_language_keyboard():
    """کیبورد انتخاب زبان - 3 زبان در هر سطر"""
    lang_items = list(LANGUAGE_CODES.items())
    keyboard_rows = []
    
    for i in range(0, len(lang_items), 3):
        row = []
        for j in range(3):
            if i + j < len(lang_items):
                lang_code, lang_name = lang_items[i + j]
                row.append(KeyboardButton(text=lang_name))
        keyboard_rows.append(row)
    
    return ReplyKeyboardMarkup(
        keyboard=keyboard_rows,
        resize_keyboard=True,
        one_time_keyboard=True
    )

def get_continue_keyboard(lang: str):
    """کیبورد ادامه"""
    continue_texts = {
        'fa': '▶️ ادامه',
        'en': '▶️ Continue',
        'tr': '▶️ Devam',
        'ar': '▶️ متابعة',
        'ru': '▶️ Продолжить',
        'es': '▶️ Continuar',
        'uk': '▶️ Продовжити',
        'hi': '▶️ जारी रखें',
        'it': '▶️ Continua',
        'fr': '▶️ Continuer',
        'pt': '▶️ Continuar',
        'id': '▶️ Lanjutkan',
        'kk': '▶️ Жалғастыру',
        'de': '▶️ Weiter',
        'pl': '▶️ Kontynuuj'
    }
    return ReplyKeyboardMarkup(
        keyboard=[
            [KeyboardButton(text=continue_texts.get(lang, '▶️ Continue'))]
        ],
        resize_keyboard=True,
        one_time_keyboard=True
    )

def get_yes_no_keyboard(lang: str):
    """کیبورد بله/خیر"""
    yes_texts = {
        'fa': '✅ بله',
        'en': '✅ Yes',
        'tr': '✅ Evet',
        'ar': '✅ نعم',
        'ru': '✅ Да',
        'es': '✅ Sí',
        'uk': '✅ Так',
        'hi': '✅ हाँ',
        'it': '✅ Sì',
        'fr': '✅ Oui',
        'pt': '✅ Sim',
        'id': '✅ Ya',
        'kk': '✅ Иә',
        'de': '✅ Ja',
        'pl': '✅ Tak'
    }
    no_texts = {
        'fa': '❌ خیر',
        'en': '❌ No',
        'tr': '❌ Hayır',
        'ar': '❌ لا',
        'ru': '❌ Нет',
        'es': '❌ No',
        'uk': '❌ Ні',
        'hi': '❌ नहीं',
        'it': '❌ No',
        'fr': '❌ Non',
        'pt': '❌ Não',
        'id': '❌ Tidak',
        'kk': '❌ Жоқ',
        'de': '❌ Nein',
        'pl': '❌ Nie'
    }
    return ReplyKeyboardMarkup(
        keyboard=[
            [
                KeyboardButton(text=yes_texts.get(lang, '✅ Yes')),
                KeyboardButton(text=no_texts.get(lang, '❌ No'))
            ]
        ],
        resize_keyboard=True,
        one_time_keyboard=True
    )

def get_gender_keyboard(lang: str):
    """کیبورد انتخاب جنسیت - یک ردیف سه تایی"""
    return ReplyKeyboardMarkup(
        keyboard=[
            [

                KeyboardButton(text=f"👩 {get_text(lang, 'gender_female')}"),
                KeyboardButton(text=f"🏳️‍⚧️ {get_text(lang, 'gender_other')}"),
                KeyboardButton(text=f"👨 {get_text(lang, 'gender_male')}")
            ]
        ],
        resize_keyboard=True,
        one_time_keyboard=True
    )

def get_looking_for_keyboard(lang: str):
    """کیبورد انتخاب دنبال چه کسی - مرد و زن یک ردیف"""
    return ReplyKeyboardMarkup(
        keyboard=[
            [
                KeyboardButton(text=f"👨 {get_text(lang, 'looking_for_male')}"),
                KeyboardButton(text=f"👩 {get_text(lang, 'looking_for_female')}")
            ],
            [
                KeyboardButton(text=f"💑 {get_text(lang, 'looking_for_any')}")
            ]
        ],
        resize_keyboard=True,
        one_time_keyboard=True
    )

def get_country_method_keyboard(lang: str):
    """کیبورد انتخاب روش وارد کردن کشور - یک ردیف"""
    manual_texts = {
        'fa': '✍️ دستی',
        'en': '✍️ Manual',
        'tr': '✍️ Manuel',
        'ar': '✍️ يدوي',
        'ru': '✍️ Вручную',
        'es': '✍️ Manual',
        'uk': '✍️ Вручну',
        'hi': '✍️ मैनुअल',
        'it': '✍️ Manuale',
        'fr': '✍️ Manuel',
        'pt': '✍️ Manual',
        'id': '✍️ Manual',
        'kk': '✍️ Қолмен',
        'de': '✍️ Manuell',
        'pl': '✍️ Ręcznie'
    }
    location_texts = {
        'fa': '📍 لوکیشن',
        'en': '📍 Location',
        'tr': '📍 Konum',
        'ar': '📍 الموقع',
        'ru': '📍 Местоположение',
        'es': '📍 Ubicación',
        'uk': '📍 Місцезнаходження',
        'hi': '📍 स्थान',
        'it': '📍 Posizione',
        'fr': '📍 Localisation',
        'pt': '📍 Localização',
        'id': '📍 Lokasi',
        'kk': '📍 Орналасқан жері',
        'de': '📍 Standort',
        'pl': '📍 Lokalizacja'
    }
    return ReplyKeyboardMarkup(
        keyboard=[
            [
                KeyboardButton(text=manual_texts.get(lang, '✍️ Manual')),
                KeyboardButton(text=location_texts.get(lang, '📍 Location'), request_location=True)
            ]
        ],
        resize_keyboard=True,
        one_time_keyboard=True
    )

def get_suggestions_keyboard(suggestions, lang='fa'):
    """کیبورد پیشنهادات (3 گزینه + لغو)"""
    keyboard = []
    
    for i, suggestion in enumerate(suggestions[:3], 1):
        emoji = ['1️⃣', '2️⃣', '3️⃣'][i-1]
        name = suggestion.get('name', suggestion.get('original', ''))
        keyboard.append([KeyboardButton(text=f"{emoji} {name}")])
    
    cancel_texts = {
        'fa': '❌ لغو و ورود مجدد',
        'en': '❌ Cancel and Re-enter',
        'tr': '❌ İptal ve Tekrar Gir',
        'ar': '❌ إلغاء وإعادة الإدخال',
        'ru': '❌ Отмена и повторный ввод',
        'es': '❌ Cancelar y volver a ingresar',
        'uk': '❌ Скасувати та ввести знову',
        'hi': '❌ रद्द करें और फिर से दर्ज करें',
        'it': '❌ Annulla e reinserisci',
        'fr': '❌ Annuler et ressaisir',
        'pt': '❌ Cancelar e reinserir',
        'id': '❌ Batal dan Masukkan Lagi',
        'kk': '❌ Болдырмау және қайта енгізу',
        'de': '❌ Abbrechen und erneut eingeben',
        'pl': '❌ Anuluj i wprowadź ponownie'
    }
    keyboard.append([KeyboardButton(text=cancel_texts.get(lang, cancel_texts['en']))])
    
    return ReplyKeyboardMarkup(keyboard=keyboard, resize_keyboard=True)

def get_skip_keyboard(lang: str):
    """کیبورد رد کردن"""
    return ReplyKeyboardMarkup(
        keyboard=[
            [KeyboardButton(text=f"⏭️ {get_text(lang, 'pass', default='رد کردن')}")] 
        ],
        resize_keyboard=True,
        one_time_keyboard=True
    )

def get_photo_method_keyboard(lang: str):
    """کیبورد انتخاب روش آپلود عکس"""
    photo_methods = {
        'fa': ['📱 استفاده از عکس پروفایل تلگرام', '📤 آپلود عکس جدید', '⏭️ رد کردن'],
        'en': ['📱 Use Telegram Profile Photo', '📤 Upload New Photo', '⏭️ Skip'],
        'tr': ['📱 Telegram Profil Fotoğrafını Kullan', '📤 Yeni Fotoğraf Yükle', '⏭️ Atla'],
        'ar': ['📱 استخدام صورة الملف الشخصي', '📤 تحميل صورة جديدة', '⏭️ تخطي'],
        'ru': ['📱 Использовать фото профиля', '📤 Загрузить новое фото', '⏭️ Пропустить'],
        'es': ['📱 Usar foto de perfil', '📤 Subir nueva foto', '⏭️ Omitir'],
        'uk': ['📱 Використати фото профілю Telegram', '📤 Завантажити нове фото', '⏭️ Пропустити'],
        'hi': ['📱 टेलीग्राम प्रोफ़ाइल फ़ोटो का उपयोग करें', '📤 नई फ़ोटो अपलोड करें', '⏭️ छोड़ें'],
        'it': ['📱 Usa foto profilo Telegram', '📤 Carica nuova foto', '⏭️ Salta'],
        'fr': ['📱 Utiliser la photo de profil Telegram', '📤 Télécharger une nouvelle photo', '⏭️ Passer'],
        'pt': ['📱 Usar foto de perfil do Telegram', '📤 Carregar nova foto', '⏭️ Pular'],
        'id': ['📱 Gunakan Foto Profil Telegram', '📤 Unggah Foto Baru', '⏭️ Lewati'],
        'kk': ['📱 Telegram профиль фотосын пайдалану', '📤 Жаңа фото жүктеу', '⏭️ Өткізу'],
        'de': ['📱 Telegram-Profilfoto verwenden', '📤 Neues Foto hochladen', '⏭️ Überspringen'],
        'pl': ['📱 Użyj zdjęcia profilu Telegram', '📤 Prześlij nowe zdjęcie', '⏭️ Pomiń']
    }
    
    buttons = photo_methods.get(lang, photo_methods['en'])
    
    return ReplyKeyboardMarkup(
        keyboard=[
            [KeyboardButton(text=buttons[0])],
            [KeyboardButton(text=buttons[1])],
            [KeyboardButton(text=buttons[2])]
        ],
        resize_keyboard=True,
        one_time_keyboard=True
    )

def get_photo_confirm_keyboard(lang: str):
    """کیبورد تایید/حذف عکس"""
    confirm_texts = {
        'fa': ['✅ تایید', '❌ حذف'],
        'en': ['✅ Confirm', '❌ Delete'],
        'tr': ['✅ Onayla', '❌ Sil'],
        'ar': ['✅ تأكيد', '❌ حذف'],
        'ru': ['✅ Подтвердить', '❌ Удалить'],
        'es': ['✅ Confirmar', '❌ Eliminar'],
        'uk': ['✅ Підтвердити', '❌ Видалити'],
        'hi': ['✅ पुष्टि करें', '❌ हटाएं'],
        'it': ['✅ Conferma', '❌ Elimina'],
        'fr': ['✅ Confirmer', '❌ Supprimer'],
        'pt': ['✅ Confirmar', '❌ Excluir'],
        'id': ['✅ Konfirmasi', '❌ Hapus'],
        'kk': ['✅ Растау', '❌ Жою'],
        'de': ['✅ Bestätigen', '❌ Löschen'],
        'pl': ['✅ Potwierdź', '❌ Usuń']
    }
    
    buttons = confirm_texts.get(lang, confirm_texts['en'])
    
    return ReplyKeyboardMarkup(
        keyboard=[
            [KeyboardButton(text=buttons[0])],
            [KeyboardButton(text=buttons[1])]
        ],
        resize_keyboard=True,
        one_time_keyboard=True
    )

def get_default_photo_path(gender: int):
    """دریافت مسیر عکس پیش‌فرض بر اساس جنسیت"""
    # 1 = مرد، 2 = زن، 3 = غیره
    if gender == 1:
        return "assets/default_male.jpg"
    elif gender == 2:
        return "assets/default_female.jpg"
    else:
        return "assets/default_other.jpg"

def get_interests_keyboard(lang: str, selected_interests: list = None):
    """کیبورد انتخاب علایق - مشابه پروفایل"""
    if selected_interests is None:
        selected_interests = []
    
    interests_dict = get_text(lang, 'interests_list', default={})
    
    if not interests_dict:
        interests_dict = {
            'sports': 'ورزش',
            'music': 'موسیقی',
            'movies': 'فیلم',
            'books': 'کتاب',
            'travel': 'سفر',
            'cooking': 'آشپزی',
            'art': 'هنر',
            'photography': 'عکاسی',
            'gaming': 'بازی',
            'fitness': 'تناسب اندام',
            'nature': 'طبیعت',
            'technology': 'فناوری',
            'fashion': 'مد',
            'dancing': 'رقص',
            'reading': 'مطالعه',
            'writing': 'نویسندگی',
            'swimming': 'شنا',
            'hiking': 'کوهنوردی',
            'cycling': 'دوچرخه‌سواری',
            'yoga': 'یوگا',
            'meditation': 'مدیتیشن',
            'volunteering': 'داوطلبی',
            'learning': 'یادگیری',
            'socializing': 'اجتماعی بودن',
            'pets': 'حیوانات خانگی',
            'gardening': 'باغبانی',
            'shopping': 'خرید',
            'coffee': 'قهوه',
            'wine': 'شراب',
            'beer': 'آبجو',
            'nightlife': 'زندگی شبانه',
            'board_games': 'بازی‌های رومیزی',
            'video_games': 'بازی‌های ویدیویی',
            'anime': 'انیمه',
            'comedy': 'کمدی',
            'theater': 'تئاتر',
            'concerts': 'کنسرت',
            'festivals': 'جشنواره‌ها',
            'beach': 'ساحل',
            'mountains': 'کوه',
            'camping': 'کمپینگ',
            'fishing': 'ماهیگیری',
            'running': 'دویدن',
            'basketball': 'بسکتبال',
            'football': 'فوتبال',
            'tennis': 'تنیس',
            'golf': 'گلف',
            'skiing': 'اسکی',
            'surfing': 'موج‌سواری',
            'martial_arts': 'هنرهای رزمی',
            'boxing': 'بوکس',
            'chess': 'شطرنج',
            'meditation_yoga': 'مراقبه و یوگا',
            'volunteering_charity': 'کارهای خیریه'
        }
    
    keyboard = []
    row = []
    
    for key, value in interests_dict.items():  # تمام علایق موجود
        if key in selected_interests:
            button_text = f"✅ {value}"
        else:
            button_text = value
        
        row.append(KeyboardButton(text=button_text))
        
        if len(row) == 3:
            keyboard.append(row)
            row = []
    
    if row:
        keyboard.append(row)
    
    # دکمه تایید و رد کردن
    if selected_interests:
        confirm_text = get_text(lang, 'confirm_interests', default='تایید علایق')
        keyboard.append([
            KeyboardButton(text=f"✅ {confirm_text} ({len(selected_interests)})")
        ])
    
    keyboard.append([
        KeyboardButton(text=f"⏭️ {get_text(lang, 'pass', default='رد کردن')}")
    ])
    
    return ReplyKeyboardMarkup(keyboard=keyboard, resize_keyboard=True)

def get_main_menu_keyboard(lang: str, is_admin: bool = False, is_guest: bool = False):
    """منوی اصلی - با دکمه‌های ورود/خروج"""
    if is_admin:
        # منوی ادمین (بدون تغییر)
        return ReplyKeyboardMarkup(
            keyboard=[
                [KeyboardButton(text="🔐 پنل ادمین"), KeyboardButton(text="📊 آمار سریع")],
                [KeyboardButton(text="👥 مدیریت کاربران"), KeyboardButton(text="🚨 گزارشات")],
                [KeyboardButton(text="📢 پیام همگانی"), KeyboardButton(text="🔧 تنظیمات سیستم")],
                [KeyboardButton(text="🧪 حالت تست"), KeyboardButton(text="❓ راهنما")]
            ],
            resize_keyboard=True
        )
    else:
        # تعیین متن دکمه ورود/خروج بر اساس وضعیت کاربر
        auth_button_texts = {
            'fa': '🔐 ورود به حساب کاربری' if is_guest else '🚪 خروج از حساب',
            'en': '🔐 Login to Account' if is_guest else '🚪 Logout',
            'tr': '🔐 Hesaba Giriş' if is_guest else '🚪 Çıkış',
            'ar': '🔐 تسجيل الدخول' if is_guest else '🚪 تسجيل الخروج',
            'ru': '🔐 Войти в аккаунт' if is_guest else '🚪 Выйти',
            'es': '🔐 Iniciar sesión' if is_guest else '🚪 Cerrar sesión',
            'uk': '🔐 Увійти в акаунт' if is_guest else '🚪 Вийти',
            'hi': '🔐 खाते में लॉगिन करें' if is_guest else '🚪 लॉगआउट',
            'it': '🔐 Accedi all\'account' if is_guest else '🚪 Esci',
            'fr': '🔐 Se connecter' if is_guest else '🚪 Se déconnecter',
            'pt': '🔐 Entrar na conta' if is_guest else '🚪 Sair',
            'id': '🔐 Masuk ke Akun' if is_guest else '🚪 Keluar',
            'kk': '🔐 Аккаунтқа кіру' if is_guest else '🚪 Шығу',
            'de': '🔐 Anmelden' if is_guest else '🚪 Abmelden',
            'pl': '🔐 Zaloguj się' if is_guest else '🚪 Wyloguj'
        }
        
        auth_button = auth_button_texts.get(lang, auth_button_texts['en'])
        
        # منوی کاربر عادی - با دکمه ورود/خروج
        return ReplyKeyboardMarkup(
            keyboard=[
                [
                    KeyboardButton(text=f"🔍 {get_text(lang, 'find_match', default='یافتن مچ')}"),
                    KeyboardButton(text=f"💚 {get_text(lang, 'my_likes', default='لایک‌های من')}")
                ],
                [
                    KeyboardButton(text=f"💬 {get_text(lang, 'my_chats', default='گفتگوهای من')}"),
                    KeyboardButton(text=f"⭐ {get_text(lang, 'premium', default='پریمیوم')}")
                ],
                [
                    KeyboardButton(text=auth_button)
                ]
            ],
            resize_keyboard=True
        )

# ==================== Registration Flow ====================

@router.message(F.text == "/start")
async def start_command(message: Message, state: FSMContext):
    """مرحله 1: شروع ثبت‌نام یا ورود به منوی اصلی"""
    user_id = message.from_user.id
    
    # بررسی ادمین بودن
    ADMIN_USER_IDS = [int(uid) for uid in os.getenv('ADMIN_USER_IDS', '').split(',') if uid]
    is_admin = user_id in ADMIN_USER_IDS
    
    # بررسی وجود کاربر
    async with db.get_session() as session:
        result = await session.execute(select(User).where(User.id == user_id))
        existing_user = result.scalar_one_or_none()
        
        if existing_user:
            # کاربر وجود دارد، نمایش منوی اصلی
            lang = existing_user.language_code or 'fa'
            is_guest = existing_user.is_guest or False
            
            # فعال کردن منوی کامل برای کاربر ثبت‌نام شده
            from handlers.bottom_menu import update_user_menu
            await update_user_menu(message.bot, user_id, lang, is_registered=True)
            
            if is_admin:
                await message.answer(
                    get_text(lang, 'reg_admin_welcome'),
                    reply_markup=get_main_menu_keyboard(lang, is_admin=True, is_guest=False)
                )
            else:
                await message.answer(
                    get_text(lang, 'welcome'),
                    reply_markup=get_main_menu_keyboard(lang, is_admin=False, is_guest=is_guest)
                )
            return
    
    # کاربر جدید - شروع ثبت‌نام
    # پنهان کردن منوی پایین در طول ثبت‌نام
    from handlers.bottom_menu import hide_user_menu
    await hide_user_menu(message.bot, user_id)
    
    # مرحله 1: انتخاب زبان (متن پیش‌فرض انگلیسی)
    await message.answer(
        "🌍 <b>Choose Language</b>\n\n"
        "Please select your preferred language:",
        reply_markup=get_language_keyboard(),
        parse_mode='HTML'
    )
    await state.set_state(RegistrationStates.language)

@router.message(RegistrationStates.language)
async def language_selected(message: Message, state: FSMContext):
    """مرحله 2: انتخاب زبان"""
    # پیدا کردن کد زبان از نام
    selected_lang = None
    for code, name in LANGUAGE_CODES.items():
        if name == message.text:
            selected_lang = code
            break
    
    if not selected_lang:
        await message.answer(get_text('en', 'reg_select_language_error'))
        return
    
    await state.update_data(language=selected_lang)
    
    # مرحله 3: پیام خوش‌آمدگویی با دکمه ادامه
    welcome_messages = {
        'fa': '👋 به ChatMilos خوش آمدید!\n\n💫 جایی که افراد واقعی با هم آشنا می‌شوند\n\n🌟 آماده‌اید تا دنیای جدیدی از آشنایی‌ها را تجربه کنید؟',
        'en': '👋 Welcome to ChatMilos!\n\n💫 Where real people meet\n\n🌟 Ready to experience a new world of connections?',
        'tr': '👋 ChatMilos\'a hoş geldiniz!\n\n💫 Gerçek insanların buluştuğu yer\n\n🌟 Yeni bir tanışma dünyasını deneyimlemeye hazır mısınız?',
        'ar': '👋 مرحباً بك في ChatMilos!\n\n💫 حيث يلتقي الأشخاص الحقيقيون\n\n🌟 هل أنت مستعد لتجربة عالم جديد من الاتصالات؟',
        'ru': '👋 Добро пожаловать в ChatMilos!\n\n💫 Где встречаются настоящие люди\n\n🌟 Готовы испытать новый мир знакомств?',
        'es': '👋 ¡Bienvenido a ChatMilos!\n\n💫 Donde las personas reales se encuentran\n\n🌟 ¿Listo para experimentar un nuevo mundo de conexiones?',
        'uk': '👋 Ласкаво просимо до ChatMilos!\n\n💫 Де зустрічаються справжні люди\n\n🌟 Готові відкрити новий світ знайомств?',
        'hi': '👋 ChatMilos में आपका स्वागत है!\n\n💫 जहां असली लोग मिलते हैं\n\n🌟 क्या आप कनेक्शन की नई दुनिया का अनुभव करने के लिए तैयार हैं?',
        'it': '👋 Benvenuto su ChatMilos!\n\n💫 Dove le persone reali si incontrano\n\n🌟 Pronto a sperimentare un nuovo mondo di connessioni?',
        'fr': '👋 Bienvenue sur ChatMilos!\n\n💫 Où les vraies personnes se rencontrent\n\n🌟 Prêt à découvrir un nouveau monde de connexions?',
        'pt': '👋 Bem-vindo ao ChatMilos!\n\n💫 Onde pessoas reais se encontram\n\n🌟 Pronto para experimentar um novo mundo de conexões?',
        'id': '👋 Selamat datang di ChatMilos!\n\n💫 Tempat orang-orang nyata bertemu\n\n🌟 Siap merasakan dunia koneksi baru?',
        'kk': '👋 ChatMilos-қа қош келдіңіз!\n\n💫 Нақты адамдар кездесетін жер\n\n🌟 Жаңа танысу әлеміне дайынсыз ба?',
        'de': '👋 Willkommen bei ChatMilos!\n\n💫 Wo echte Menschen sich treffen\n\n🌟 Bereit, eine neue Welt der Verbindungen zu erleben?',
        'pl': '👋 Witamy w ChatMilos!\n\n💫 Gdzie spotykają się prawdziwi ludzie\n\n🌟 Gotowy doświadczyć nowego świata połączeń?'
    }
    
    welcome_text = welcome_messages.get(selected_lang, welcome_messages['en'])
    
    await message.answer(
        welcome_text,
        reply_markup=get_continue_keyboard(selected_lang)
    )
    await state.set_state(RegistrationStates.welcome)

@router.message(RegistrationStates.welcome)
async def welcome_continue(message: Message, state: FSMContext):
    """مرحله 4: دکمه ادامه و نمایش قوانین"""
    data = await state.get_data()
    lang = data.get('language', 'fa')
    
    # نمایش قوانین و حریم خصوصی
    terms_keyboard = ReplyKeyboardMarkup(
        keyboard=[[KeyboardButton(text=f"✅ {get_text(lang, 'accept_terms', default='قبول می‌کنم')}")]],
        resize_keyboard=True,
        one_time_keyboard=True
    )
    
    await message.answer(
        get_text(lang, 'terms_text'),
        reply_markup=terms_keyboard,
        parse_mode='HTML'
    )
    await state.set_state(RegistrationStates.terms)

@router.message(RegistrationStates.terms)
async def terms_accepted(message: Message, state: FSMContext):
    """مرحله 5: قبول قوانین - نمایش گزینه‌های ثبت‌نام/ورود"""
    data = await state.get_data()
    lang = data.get('language', 'fa')
    
    accept_term = get_text(lang, 'accept_terms')
    if "✅" not in message.text and accept_term not in message.text:
        await message.answer(get_text(lang, 'reg_accept_terms_prompt').format(accept_term=accept_term))
        return
    
    # نمایش گزینه‌های ثبت‌نام/ورود مهمان
    auth_choice_texts = {
        'fa': 'برای ادامه، یکی از گزینه‌های زیر را انتخاب کنید:',
        'en': 'To continue, choose one of the following options:',
        'tr': 'Devam etmek için aşağıdaki seçeneklerden birini seçin:',
        'ar': 'للمتابعة، اختر أحد الخيارات التالية:'
    }
    
    signup_texts = {
        'fa': '🟢 ثبت‌نام',
        'en': '🟢 Sign Up',
        'tr': '🟢 Kayıt Ol',
        'ar': '🟢 التسجيل'
    }
    
    guest_texts = {
        'fa': '🟡 ورود مهمان',
        'en': '🟡 Guest Login',
        'tr': '🟡 Misafir Girişi',
        'ar': '🟡 دخول الضيف'
    }
    
    auth_keyboard = ReplyKeyboardMarkup(
        keyboard=[
            [KeyboardButton(text=signup_texts.get(lang, signup_texts['en']))],
            [KeyboardButton(text=guest_texts.get(lang, guest_texts['en']))]
        ],
        resize_keyboard=True,
        one_time_keyboard=True
    )
    
    await message.answer(
        auth_choice_texts.get(lang, auth_choice_texts['en']),
        reply_markup=auth_keyboard
    )
    await state.set_state(RegistrationStates.auth_choice)

@router.message(RegistrationStates.auth_choice)
async def auth_choice_selected(message: Message, state: FSMContext):
    """مرحله 6: انتخاب ثبت‌نام یا ورود مهمان"""
    data = await state.get_data()
    lang = data.get('language', 'fa')
    
    # بررسی ورود مهمان
    guest_keywords = ['مهمان', 'Guest', 'Misafir', 'ضيف', '🟡']
    if any(keyword in message.text for keyword in guest_keywords):
        # ورود مهمان - مستقیماً به انتخاب جنسیت
        await state.update_data(is_guest=True)
        await message.answer(
            get_text(lang, 'select_gender'),
            reply_markup=get_gender_keyboard(lang)
        )
        await state.set_state(RegistrationStates.gender)
        return
    
    # ثبت‌نام - نمایش گزینه ثبت‌نام/ورود
    signup_login_texts = {
        'fa': 'لطفاً یکی از گزینه‌های زیر را انتخاب کنید:',
        'en': 'Please choose one of the following options:',
        'tr': 'Lütfen aşağıdaki seçeneklerden birini seçin:',
        'ar': 'الرجاء اختيار أحد الخيارات التالية:'
    }
    
    signup_btn_texts = {
        'fa': '📝 ثبت‌نام',
        'en': '📝 Sign Up',
        'tr': '📝 Kayıt Ol',
        'ar': '📝 التسجيل'
    }
    
    login_btn_texts = {
        'fa': '🔑 ورود',
        'en': '🔑 Login',
        'tr': '🔑 Giriş',
        'ar': '🔑 تسجيل الدخول'
    }
    
    signup_login_keyboard = ReplyKeyboardMarkup(
        keyboard=[
            [KeyboardButton(text=signup_btn_texts.get(lang, signup_btn_texts['en']))],
            [KeyboardButton(text=login_btn_texts.get(lang, login_btn_texts['en']))]
        ],
        resize_keyboard=True,
        one_time_keyboard=True
    )
    
    await message.answer(
        signup_login_texts.get(lang, signup_login_texts['en']),
        reply_markup=signup_login_keyboard
    )
    await state.set_state(RegistrationStates.signup_or_login)

@router.message(RegistrationStates.signup_or_login)
async def signup_or_login_selected(message: Message, state: FSMContext):
    """مرحله 7: انتخاب ثبت‌نام یا ورود"""
    data = await state.get_data()
    lang = data.get('language', 'fa')
    user_id = message.from_user.id
    
    # بررسی دکمه بازگشت به منوی اصلی
    back_keywords = ['بازگشت', 'Back', 'Dön', 'العودة', 'Вернуться', 'Volver', 'Повернутися', 'वापस', 'Torna', 'Retour', 'Voltar', 'Kembali', 'оралу', 'Zurück', 'Powrót', '🔙']
    if any(keyword in message.text for keyword in back_keywords):
        # بازگشت به منوی اصلی
        async with db.get_session() as session:
            result = await session.execute(select(User).where(User.id == user_id))
            user = result.scalar_one_or_none()
            
            if user:
                back_to_menu_msg = {
                    'fa': '🏠 بازگشت به منوی اصلی',
                    'en': '🏠 Back to Main Menu',
                    'tr': '🏠 Ana Menüye Dönüldü',
                    'ar': '🏠 العودة إلى القائمة الرئيسية',
                    'ru': '🏠 Возврат в главное меню',
                    'es': '🏠 Volver al menú principal',
                    'uk': '🏠 Повернення до головного меню',
                    'hi': '🏠 मुख्य मेनू पर वापस',
                    'it': '🏠 Ritorno al menu principale',
                    'fr': '🏠 Retour au menu principal',
                    'pt': '🏠 Voltar ao menu principal',
                    'id': '🏠 Kembali ke Menu Utama',
                    'kk': '🏠 Басты мәзірге оралу',
                    'de': '🏠 Zurück zum Hauptmenü',
                    'pl': '🏠 Powrót do menu głównego'
                }
                
                await message.answer(
                    back_to_menu_msg.get(lang, back_to_menu_msg['en']),
                    reply_markup=get_main_menu_keyboard(lang, is_admin=False, is_guest=user.is_guest)
                )
                await state.clear()
                return
    
    # بررسی ورود
    login_keywords = ['ورود', 'Login', 'Giriş', 'الدخول', '🔑']
    is_login = any(keyword in message.text for keyword in login_keywords)
    
    await state.update_data(is_login=is_login)
    
    # درخواست ایمیل
    email_prompt_texts = {
        'fa': 'لطفاً ایمیل خود را وارد کنید:',
        'en': 'Please enter your email:',
        'tr': 'Lütfen e-postanızı girin:',
        'ar': 'الرجاء إدخال بريدك الإلكتروني:'
    }
    
    await message.answer(
        email_prompt_texts.get(lang, email_prompt_texts['en']),
        reply_markup=ReplyKeyboardRemove()
    )
    await state.set_state(RegistrationStates.email_input)

@router.message(RegistrationStates.email_input)
async def email_entered(message: Message, state: FSMContext):
    """مرحله 8: وارد کردن ایمیل"""
    data = await state.get_data()
    lang = data.get('language', 'fa')
    is_login = data.get('is_login', False)
    
    email = message.text.strip().lower()
    
    # بررسی معتبر بودن ایمیل
    if not is_valid_email(email):
        invalid_email_texts = {
            'fa': '❌ ایمیل نامعتبر است. لطفاً یک ایمیل صحیح وارد کنید.',
            'en': '❌ Invalid email. Please enter a valid email address.',
            'tr': '❌ Geçersiz e-posta. Lütfen geçerli bir e-posta adresi girin.',
            'ar': '❌ بريد إلكتروني غير صالح. الرجاء إدخال عنوان بريد إلكتروني صحيح.'
        }
        await message.answer(invalid_email_texts.get(lang, invalid_email_texts['en']))
        return
    
    # بررسی وجود ایمیل در دیتابیس
    async with db.get_session() as session:
        result = await session.execute(select(User).where(User.email == email))
        existing_user = result.scalar_one_or_none()
        
        if is_login:
            # حالت ورود
            if not existing_user:
                not_registered_texts = {
                    'fa': '❌ این ایمیل در سیستم ثبت نشده است. لطفاً ابتدا ثبت‌نام کنید.',
                    'en': '❌ This email is not registered. Please sign up first.',
                    'tr': '❌ Bu e-posta kayıtlı değil. Lütfen önce kaydolun.',
                    'ar': '❌ هذا البريد الإلكتروني غير مسجل. الرجاء التسجيل أولاً.'
                }
                await message.answer(not_registered_texts.get(lang, not_registered_texts['en']))
                return
            
            # درخواست رمز عبور
            await state.update_data(email=email, user_id_for_login=existing_user.user_id)
            
            password_prompt_texts = {
                'fa': 'لطفاً رمز عبور خود را وارد کنید:',
                'en': 'Please enter your password:',
                'tr': 'Lütfen şifrenizi girin:',
                'ar': 'الرجاء إدخال كلمة المرور الخاصة بك:'
            }
            
            await message.answer(password_prompt_texts.get(lang, password_prompt_texts['en']))
            await state.set_state(RegistrationStates.login_password)
        else:
            # حالت ثبت‌نام
            # بررسی اینکه آیا این ایمیل قبلاً توسط کاربر دیگری ثبت شده
            if existing_user:
                already_registered_texts = {
                    'fa': '❌ این ایمیل قبلاً ثبت شده است. لطفاً وارد شوید یا از ایمیل دیگری استفاده کنید.',
                    'en': '❌ This email is already registered. Please login or use a different email.',
                    'tr': '❌ Bu e-posta zaten kayıtlı. Lütfen giriş yapın veya farklı bir e-posta kullanın.',
                    'ar': '❌ هذا البريد الإلكتروني مسجل بالفعل. الرجاء تسجيل الدخول أو استخدام بريد إلكتروني مختلف.'
                }
                await message.answer(already_registered_texts.get(lang, already_registered_texts['en']))
                return
            
            # بررسی اینکه آیا این user_id قبلاً با ایمیل دیگری ثبت‌نام کرده
            user_id = message.from_user.id
            result_user_id = await session.execute(select(User).where(User.id == user_id))
            existing_user_with_same_id = result_user_id.scalar_one_or_none()
            
            if existing_user_with_same_id and not existing_user_with_same_id.is_guest:
                # کاربر قبلاً با ایمیل دیگری ثبت‌نام کرده
                already_has_account_texts = {
                    'fa': f'⚠️ شما قبلاً با ایمیل <b>{existing_user_with_same_id.email}</b> ثبت‌نام کرده‌اید.\n\n'
                          f'💡 برای ثبت‌نام با ایمیل جدید، ابتدا باید از حساب فعلی خارج شوید.\n\n'
                          f'آیا می‌خواهید از حساب فعلی خارج شوید و با ایمیل <b>{email}</b> ثبت‌نام کنید؟',
                    'en': f'⚠️ You have already registered with email <b>{existing_user_with_same_id.email}</b>.\n\n'
                          f'💡 To register with a new email, you must first logout from your current account.\n\n'
                          f'Do you want to logout and register with <b>{email}</b>?',
                    'tr': f'⚠️ Zaten <b>{existing_user_with_same_id.email}</b> ile kayıt oldunuz.\n\n'
                          f'💡 Yeni bir e-posta ile kayıt olmak için önce mevcut hesabınızdan çıkış yapmalısınız.\n\n'
                          f'<b>{email}</b> ile çıkış yapmak ve kayıt olmak istiyor musunuz?',
                    'ar': f'⚠️ لقد قمت بالتسجيل بالفعل باستخدام البريد الإلكتروني <b>{existing_user_with_same_id.email}</b>.\n\n'
                          f'💡 للتسجيل باستخدام بريد إلكتروني جديد، يجب عليك أولاً تسجيل الخروج من حسابك الحالي.\n\n'
                          f'هل تريد تسجيل الخروج والتسجيل باستخدام <b>{email}</b>؟'
                }
                
                await message.answer(
                    already_has_account_texts.get(lang, already_has_account_texts['en']),
                    reply_markup=get_yes_no_keyboard(lang),
                    parse_mode='HTML'
                )
                await state.update_data(new_email=email, switch_account=True)
                await state.set_state(RegistrationStates.logout_confirm)
                return
            
            # ارسال کد تأیید
            verification_code = generate_verification_code()
            await send_verification_email(email, verification_code, lang)
            await state.update_data(email=email)
            
            code_sent_texts = {
                'fa': f'✅ کد تأیید به ایمیل {email} ارسال شد.\n\n💡 لطفاً کد 6 رقمی را وارد کنید:\n\n⚠️ اگر ایمیل را دریافت نکردید، پوشه spam را بررسی کنید.',
                'en': f'✅ Verification code sent to {email}.\n\n💡 Please enter the 6-digit code:\n\n⚠️ If you didn\'t receive the email, check your spam folder.',
                'tr': f'✅ Doğrulama kodu {email} adresine gönderildi.\n\n💡 Lütfen 6 haneli kodu girin:\n\n⚠️ E-postayı almadıysanız spam klasörünüzü kontrol edin.',
                'ar': f'✅ تم إرسال رمز التحقق إلى {email}.\n\n💡 الرجاء إدخال الرمز المكون من 6 أرقام:\n\n⚠️ إذا لم تتلق البريد الإلكتروني، تحقق من مجلد البريد العشوائي.'
            }
            
            await message.answer(code_sent_texts.get(lang, code_sent_texts['en']))
            await state.set_state(RegistrationStates.verification_code)

@router.message(RegistrationStates.login_password)
async def login_password_entered(message: Message, state: FSMContext):
    """مرحله 9: ورود رمز عبور برای ورود"""
    data = await state.get_data()
    lang = data.get('language', 'fa')
    user_id_for_login = data.get('user_id_for_login')
    
    password = message.text.strip()
    
    # بررسی رمز عبور
    async with db.get_session() as session:
        result = await session.execute(select(User).where(User.id == user_id_for_login))
        user = result.scalar_one_or_none()
        
        if not user or not verify_password(password, user.password_hash):
            wrong_password_texts = {
                'fa': '❌ رمز عبور اشتباه است. لطفاً دوباره تلاش کنید.',
                'en': '❌ Wrong password. Please try again.',
                'tr': '❌ Yanlış şifre. Lütfen tekrar deneyin.',
                'ar': '❌ كلمة مرور خاطئة. يرجى المحاولة مرة أخرى.'
            }
            await message.answer(wrong_password_texts.get(lang, wrong_password_texts['en']))
            return
        
        # ورود موفق - تبدیل از مهمان به کاربر وارد شده
        user.is_guest = False
        await session.commit()
        
        login_success_texts = {
            'fa': '✅ ورود موفقیت‌آمیز! خوش آمدید.',
            'en': '✅ Login successful! Welcome back.',
            'tr': '✅ Giriş başarılı! Hoş geldiniz.',
            'ar': '✅ تسجيل الدخول ناجح! مرحباً بعودتك.'
        }
        
        await message.answer(
            login_success_texts.get(lang, login_success_texts['en']),
            reply_markup=get_main_menu_keyboard(lang, is_admin=False, is_guest=False)
        )
        
        # فعال کردن منوی کامل
        from handlers.bottom_menu import update_user_menu
        await update_user_menu(message.bot, user_id_for_login, lang, is_registered=True)
        
        await state.clear()

@router.message(RegistrationStates.verification_code)
async def verification_code_entered(message: Message, state: FSMContext):
    """مرحله 10: وارد کردن کد تأیید"""
    data = await state.get_data()
    lang = data.get('language', 'fa')
    email = data.get('email')
    
    code = message.text.strip()
    
    # بررسی کد تأیید
    if not verify_code(email, code):
        wrong_code_texts = {
            'fa': '❌ کد تأیید اشتباه یا منقضی شده است. لطفاً دوباره تلاش کنید.',
            'en': '❌ Wrong or expired verification code. Please try again.',
            'tr': '❌ Yanlış veya süresi dolmuş doğrulama kodu. Lütfen tekrar deneyin.',
            'ar': '❌ رمز تحقق خاطئ أو منتهي الصلاحية. يرجى المحاولة مرة أخرى.'
        }
        await message.answer(wrong_code_texts.get(lang, wrong_code_texts['en']))
        return
    
    # کد صحیح - درخواست رمز عبور
    code_verified_texts = {
        'fa': '✅ ایمیل تأیید شد!\n\nلطفاً رمز عبور خود را وارد کنید.\n\n' + get_password_requirements(lang),
        'en': '✅ Email verified!\n\nPlease enter your password.\n\n' + get_password_requirements('en'),
        'tr': '✅ E-posta doğrulandı!\n\nLütfen şifrenizi girin.\n\n' + get_password_requirements('tr'),
        'ar': '✅ تم التحقق من البريد الإلكتروني!\n\nالرجاء إدخال كلمة المرور الخاصة بك.\n\n' + get_password_requirements('ar')
    }
    
    await message.answer(code_verified_texts.get(lang, code_verified_texts['en']))
    await state.set_state(RegistrationStates.password_input)

@router.message(RegistrationStates.password_input)
async def password_entered(message: Message, state: FSMContext):
    """مرحله 11: وارد کردن رمز عبور"""
    data = await state.get_data()
    lang = data.get('language', 'fa')
    
    password = message.text.strip()
    
    # بررسی معتبر بودن رمز عبور
    is_valid, error_message = is_valid_password(password)
    if not is_valid:
        error_texts = {
            'fa': f'❌ {error_message}\n\n{get_password_requirements(lang)}',
            'en': f'❌ {error_message}\n\n{get_password_requirements("en")}',
            'tr': f'❌ {error_message}\n\n{get_password_requirements("tr")}',
            'ar': f'❌ {error_message}\n\n{get_password_requirements("ar")}'
        }
        await message.answer(error_texts.get(lang, error_texts['en']))
        return
    
    # ذخیره رمز عبور
    await state.update_data(password=password)
    
    # درخواست تأیید رمز عبور
    confirm_password_texts = {
        'fa': 'لطفاً رمز عبور را دوباره وارد کنید:',
        'en': 'Please enter your password again:',
        'tr': 'Lütfen şifrenizi tekrar girin:',
        'ar': 'الرجاء إدخال كلمة المرور مرة أخرى:'
    }
    
    await message.answer(confirm_password_texts.get(lang, confirm_password_texts['en']))
    await state.set_state(RegistrationStates.password_confirm)

@router.message(RegistrationStates.password_confirm)
async def password_confirmed(message: Message, state: FSMContext):
    """مرحله 12: تأیید رمز عبور"""
    data = await state.get_data()
    lang = data.get('language', 'fa')
    password = data.get('password')
    
    password_confirm = message.text.strip()
    
    # بررسی تطابق رمزهای عبور
    if password != password_confirm:
        mismatch_texts = {
            'fa': '❌ رمزهای عبور مطابقت ندارند. لطفاً دوباره رمز عبور را وارد کنید:',
            'en': '❌ Passwords do not match. Please enter your password again:',
            'tr': '❌ Şifreler eşleşmiyor. Lütfen şifrenizi tekrar girin:',
            'ar': '❌ كلمات المرور غير متطابقة. الرجاء إدخال كلمة المرور مرة أخرى:'
        }
        await message.answer(mismatch_texts.get(lang, mismatch_texts['en']))
        await state.set_state(RegistrationStates.password_input)
        return
    
    # رمزهای عبور مطابقت دارند - ادامه به انتخاب جنسیت
    password_match_texts = {
        'fa': '✅ رمز عبور تأیید شد!',
        'en': '✅ Password confirmed!',
        'tr': '✅ Şifre onaylandı!',
        'ar': '✅ تم تأكيد كلمة المرور!'
    }
    
    await message.answer(password_match_texts.get(lang, password_match_texts['en']))
    
    # ادامه به انتخاب جنسیت
    await message.answer(
        get_text(lang, 'select_gender'),
        reply_markup=get_gender_keyboard(lang)
    )
    await state.set_state(RegistrationStates.gender)

@router.message(RegistrationStates.gender)
async def gender_selected(message: Message, state: FSMContext):
    """مرحله 6: انتخاب جنسیت"""
    data = await state.get_data()
    lang = data.get('language', 'fa')
    
    # تشخیص جنسیت از متن
    gender_male = get_text(lang, 'gender_male')
    gender_female = get_text(lang, 'gender_female')
    gender_other = get_text(lang, 'gender_other')
    
    if gender_male in message.text or "مرد" in message.text or "Male" in message.text or "Erkek" in message.text:
        gender = 1
    elif gender_female in message.text or "زن" in message.text or "Female" in message.text or "Kadın" in message.text:
        gender = 2
    elif gender_other in message.text or "غیره" in message.text or "Other" in message.text or "Diğer" in message.text:
        gender = 3
    else:
        await message.answer(get_text(lang, 'select_gender'), reply_markup=get_gender_keyboard(lang))
        return
    
    await state.update_data(gender=gender)
    
    # نمایش انتخاب دنبال چه کسی
    await message.answer(
        get_text(lang, 'select_looking_for'),
        reply_markup=get_looking_for_keyboard(lang)
    )
    await state.set_state(RegistrationStates.looking_for)

@router.message(RegistrationStates.looking_for)
async def looking_for_selected(message: Message, state: FSMContext):
    """مرحله 7: انتخاب دنبال چه کسی"""
    data = await state.get_data()
    lang = data.get('language', 'fa')
    
    # تشخیص ترجیح از متن
    looking_male = get_text(lang, 'looking_for_male')
    looking_female = get_text(lang, 'looking_for_female')
    looking_any = get_text(lang, 'looking_for_any')
    
    if looking_male in message.text or "مرد" in message.text or "Male" in message.text:
        looking_for = 1
    elif looking_female in message.text or "زن" in message.text or "Female" in message.text:
        looking_for = 2
    elif looking_any in message.text or "فرقی" in message.text or "Anyone" in message.text or "Any" in message.text:
        looking_for = 0
    else:
        await message.answer(get_text(lang, 'select_looking_for'), reply_markup=get_looking_for_keyboard(lang))
        return
    
    await state.update_data(looking_for=looking_for)
    
    # درخواست نام
    await message.answer(
        get_text(lang, 'enter_name'),
        reply_markup=ReplyKeyboardRemove()
    )
    await state.set_state(RegistrationStates.name)

@router.message(RegistrationStates.name)
async def name_entered(message: Message, state: FSMContext):
    """مرحله 8: وارد کردن نام"""
    data = await state.get_data()
    lang = data.get('language', 'fa')
    
    name = message.text.strip()
    if len(name) < 2:
        await message.answer(get_text(lang, 'reg_name_too_short'))
        return
    
    await state.update_data(name=name)
    
    # درخواست سن
    await message.answer(get_text(lang, 'enter_age'))
    await state.set_state(RegistrationStates.age)

@router.message(RegistrationStates.age)
async def age_entered(message: Message, state: FSMContext):
    """مرحله 9: وارد کردن سن"""
    data = await state.get_data()
    lang = data.get('language', 'fa')
    
    try:
        age = int(message.text)
        if age < 18 or age > 80:
            await message.answer(get_text(lang, 'age_invalid'))
            return
        
        await state.update_data(age=age)
        
        # روش انتخاب کشور
        country_method_texts = {
            'fa': 'روش انتخاب کشور خود را مشخص کنید:',
            'en': 'Choose how to select your country:',
            'tr': 'Ülkenizi seçme yöntemini belirleyin:',
            'ar': 'حدد طريقة اختيار بلدك:',
            'ru': 'Выберите способ выбора вашей страны:',
            'es': 'Elige cómo seleccionar tu país:',
            'uk': 'Виберіть спосіб вибору вашої країни:',
            'hi': 'अपना देश चुनने का तरीका चुनें:',
            'it': 'Scegli come selezionare il tuo paese:',
            'fr': 'Choisissez comment sélectionner votre pays:',
            'pt': 'Escolha como selecionar seu país:',
            'id': 'Pilih cara memilih negara Anda:',
            'kk': 'Еліңізді таңдау әдісін таңдаңыз:',
            'de': 'Wählen Sie, wie Sie Ihr Land auswählen möchten:',
            'pl': 'Wybierz sposób wyboru swojego kraju:'
        }
        
        await message.answer(
            country_method_texts.get(lang, 'Choose how to select your country:'),
            reply_markup=get_country_method_keyboard(lang)
        )
        await state.set_state(RegistrationStates.country_method)
        
    except ValueError:
        await message.answer(get_text(lang, 'reg_age_number_format'))

@router.message(RegistrationStates.country_method, F.location)
async def location_received_for_country(message: Message, state: FSMContext):
    """مرحله 10: دریافت لوکیشن برای کشور و شهر"""
    data = await state.get_data()
    lang = data.get('language', 'fa')
    
    lat = message.location.latitude
    lon = message.location.longitude
    
    # Encode to geohash
    geohash = matching_engine.encode_geohash(lat, lon, precision=6)
    
    # Reverse Geocoding برای دریافت کشور و شهر
    country_name = "Unknown"
    city_name = "Unknown"
    country_code = "XX"
    
    try:
        import aiohttp
        async with aiohttp.ClientSession() as session:
            # استفاده از Nominatim API (رایگان)
            url = f"https://nominatim.openstreetmap.org/reverse?format=json&lat={lat}&lon={lon}&accept-language={lang}"
            headers = {'User-Agent': 'ChatDate/1.0'}
            
            async with session.get(url, headers=headers) as response:
                if response.status == 200:
                    data_geo = await response.json()
                    address = data_geo.get('address', {})
                    
                    # دریافت کشور
                    country_name = address.get('country', 'Unknown')
                    country_code = address.get('country_code', 'XX').upper()
                    
                    # دریافت شهر (با اولویت)
                    city_name = (
                        address.get('city') or 
                        address.get('town') or 
                        address.get('village') or 
                        address.get('municipality') or 
                        address.get('county') or 
                        'Unknown'
                    )
                    
                    # حذف کلمات اضافی مثل "شهر"، "City"، "Şehir" و غیره
                    city_prefixes = ['شهر ', 'شهرستان ', 'City of ', 'Şehir ', 'Stadt ', 'Ville de ', 'Cidade de ']
                    for prefix in city_prefixes:
                        if city_name.startswith(prefix):
                            city_name = city_name[len(prefix):]
                            break
                    
                    logger.info(f"Geocoded: {country_name} ({country_code}), {city_name}")
    except Exception as e:
        logger.error(f"Geocoding error: {e}")
    
    await state.update_data(
        latitude=lat,
        longitude=lon,
        geohash=geohash,
        country=country_name,
        country_code=country_code,
        city=city_name
    )
    
    # نمایش کشور و شهر شناسایی شده
    location_confirmed_texts = {
        'fa': f"✅ موقعیت شما شناسایی شد:\n\n🌍 کشور: <b>{country_name}</b>\n🏙️ شهر: <b>{city_name}</b>",
        'en': f"✅ Your location has been identified:\n\n🌍 Country: <b>{country_name}</b>\n🏙️ City: <b>{city_name}</b>",
        'tr': f"✅ Konumunuz tanımlandı:\n\n🌍 Ülke: <b>{country_name}</b>\n🏙️ Şehir: <b>{city_name}</b>",
        'ar': f"✅ تم تحديد موقعك:\n\n🌍 البلد: <b>{country_name}</b>\n🏙️ المدينة: <b>{city_name}</b>",
        'ru': f"✅ Ваше местоположение определено:\n\n🌍 Страна: <b>{country_name}</b>\n🏙️ Город: <b>{city_name}</b>",
        'es': f"✅ Se ha identificado tu ubicación:\n\n🌍 País: <b>{country_name}</b>\n🏙️ Ciudad: <b>{city_name}</b>",
        'uk': f"✅ Ваше місцезнаходження визначено:\n\n🌍 Країна: <b>{country_name}</b>\n🏙️ Місто: <b>{city_name}</b>",
        'hi': f"✅ आपका स्थान पहचाना गया:\n\n🌍 देश: <b>{country_name}</b>\n🏙️ शहर: <b>{city_name}</b>",
        'it': f"✅ La tua posizione è stata identificata:\n\n🌍 Paese: <b>{country_name}</b>\n🏙️ Città: <b>{city_name}</b>",
        'fr': f"✅ Votre emplacement a été identifié:\n\n🌍 Pays: <b>{country_name}</b>\n🏙️ Ville: <b>{city_name}</b>",
        'pt': f"✅ Sua localização foi identificada:\n\n🌍 País: <b>{country_name}</b>\n🏙️ Cidade: <b>{city_name}</b>",
        'id': f"✅ Lokasi Anda telah diidentifikasi:\n\n🌍 Negara: <b>{country_name}</b>\n🏙️ Kota: <b>{city_name}</b>",
        'kk': f"✅ Сіздің орналасқан жеріңіз анықталды:\n\n🌍 Ел: <b>{country_name}</b>\n🏙️ Қала: <b>{city_name}</b>",
        'de': f"✅ Ihr Standort wurde identifiziert:\n\n🌍 Land: <b>{country_name}</b>\n🏙️ Stadt: <b>{city_name}</b>",
        'pl': f"✅ Twoja lokalizacja została zidentyfikowana:\n\n🌍 Kraj: <b>{country_name}</b>\n🏙️ Miasto: <b>{city_name}</b>"
    }
    
    await message.answer(
        location_confirmed_texts.get(lang, location_confirmed_texts['en']),
        parse_mode='HTML'
    )
    
    # درخواست بیو
    await message.answer(
        get_text(lang, 'enter_bio'),
        reply_markup=get_skip_keyboard(lang)
    )
    await state.set_state(RegistrationStates.bio)

@router.message(RegistrationStates.country_method)
async def country_method_selected(message: Message, state: FSMContext):
    """مرحله 10: انتخاب روش دستی"""
    data = await state.get_data()
    lang = data.get('language', 'fa')
    
    if "دستی" in message.text or "Manual" in message.text or "Manuel" in message.text or "✍️" in message.text:
        # وارد کردن دستی کشور
        country_prompts = {
            'fa': 'نام کشور خود را وارد کنید:',
            'en': 'Enter your country name:',
            'tr': 'Ülke adınızı girin:',
            'ar': 'أدخل اسم بلدك:',
            'ru': 'Введите название вашей страны:',
            'es': 'Ingresa el nombre de tu país:',
            'uk': 'Введіть назву вашої країни:',
            'hi': 'अपने देश का नाम दर्ज करें:',
            'it': 'Inserisci il nome del tuo paese:',
            'fr': 'Entrez le nom de votre pays:',
            'pt': 'Digite o nome do seu país:',
            'id': 'Masukkan nama negara Anda:',
            'kk': 'Еліңіздің атын енгізіңіз:',
            'de': 'Geben Sie Ihren Ländernamen ein:',
            'pl': 'Wprowadź nazwę swojego kraju:'
        }
        
        await message.answer(
            country_prompts.get(lang, 'Enter your country name:'),
            reply_markup=ReplyKeyboardRemove()
        )
        await state.set_state(RegistrationStates.country)
    else:
        await message.answer(
            get_text(lang, 'reg_select_option'),
            reply_markup=get_country_method_keyboard(lang)
        )

@router.message(RegistrationStates.country)
async def country_entered(message: Message, state: FSMContext):
    """مرحله 11: وارد کردن کشور با پیشنهاد هوشمند"""
    data = await state.get_data()
    lang = data.get('language', 'fa')
    
    country_input = message.text.strip()
    
    # پیشنهاد کشورها
    suggestions = suggest_countries(country_input, lang=lang, limit=3)
    
    if not suggestions:
        await message.answer(
            get_text(lang, 'reg_country_not_found'),
            reply_markup=ReplyKeyboardRemove()
        )
        return
    
    # بررسی امتیاز - اگر امتیاز اولی بالای 90 باشد، مستقیم قبول کن
    if suggestions[0]['score'] >= 90:
        await state.update_data(
            country=suggestions[0]['name'],
            country_code=suggestions[0]['code']
        )
        
        # ادامه به شهر
        city_prompts = {
            'fa': f"✅ کشور: <b>{suggestions[0]['name']}</b>\n\nحالا نام شهر خود را وارد کنید:",
            'en': f"✅ Country: <b>{suggestions[0]['name']}</b>\n\nNow enter your city name:",
            'tr': f"✅ Ülke: <b>{suggestions[0]['name']}</b>\n\nŞimdi şehir adınızı girin:",
            'ar': f"✅ البلد: <b>{suggestions[0]['name']}</b>\n\nالآن أدخل اسم مدينتك:",
            'ru': f"✅ Страна: <b>{suggestions[0]['name']}</b>\n\nТеперь введите название вашего города:",
            'es': f"✅ País: <b>{suggestions[0]['name']}</b>\n\nAhora ingresa el nombre de tu ciudad:",
            'uk': f"✅ Країна: <b>{suggestions[0]['name']}</b>\n\nТепер введіть назву вашого міста:",
            'hi': f"✅ देश: <b>{suggestions[0]['name']}</b>\n\nअब अपने शहर का नाम दर्ज करें:",
            'it': f"✅ Paese: <b>{suggestions[0]['name']}</b>\n\nOra inserisci il nome della tua città:",
            'fr': f"✅ Pays: <b>{suggestions[0]['name']}</b>\n\nMaintenant entrez le nom de votre ville:",
            'pt': f"✅ País: <b>{suggestions[0]['name']}</b>\n\nAgora insira o nome da sua cidade:",
            'id': f"✅ Negara: <b>{suggestions[0]['name']}</b>\n\nSekarang masukkan nama kota Anda:",
            'kk': f"✅ Ел: <b>{suggestions[0]['name']}</b>\n\nЕнді қалаңыздың атын енгізіңіз:",
            'de': f"✅ Land: <b>{suggestions[0]['name']}</b>\n\nGeben Sie jetzt Ihren Stadtnamen ein:",
            'pl': f"✅ Kraj: <b>{suggestions[0]['name']}</b>\n\nTeraz wprowadź nazwę swojego miasta:"
        }
        
        await message.answer(
            city_prompts.get(lang, city_prompts['en']),
            reply_markup=ReplyKeyboardRemove(),
            parse_mode='HTML'
        )
        await state.set_state(RegistrationStates.city)
        return
    
    # نمایش پیشنهادات
    prompt_texts = {
        'fa': f"🤔 آیا منظور شما یکی از این‌هاست؟\n\nشما نوشتید: <b>{country_input}</b>",
        'en': f"🤔 Did you mean one of these?\n\nYou entered: <b>{country_input}</b>",
        'tr': f"🤔 Bunlardan birini mi kastediyorsunuz?\n\nYazdığınız: <b>{country_input}</b>",
        'ar': f"🤔 هل تقصد أحد هذه؟\n\nكتبت: <b>{country_input}</b>",
        'ru': f"🤔 Вы имели в виду одно из этих?\n\nВы ввели: <b>{country_input}</b>",
        'es': f"🤔 ¿Quisiste decir uno de estos?\n\nEscribiste: <b>{country_input}</b>",
        'uk': f"🤔 Ви мали на увазі одне з цих?\n\nВи ввели: <b>{country_input}</b>",
        'hi': f"🤔 क्या आपका मतलब इनमें से एक था?\n\nआपने दर्ज किया: <b>{country_input}</b>",
        'it': f"🤔 Intendevi uno di questi?\n\nHai inserito: <b>{country_input}</b>",
        'fr': f"🤔 Vouliez-vous dire l'un de ceux-ci?\n\nVous avez entré: <b>{country_input}</b>",
        'pt': f"🤔 Você quis dizer um destes?\n\nVocê inseriu: <b>{country_input}</b>",
        'id': f"🤔 Apakah Anda maksud salah satu dari ini?\n\nAnda memasukkan: <b>{country_input}</b>",
        'kk': f"🤔 Сіз осылардың бірін айтқыңыз келді ме?\n\nСіз енгіздіңіз: <b>{country_input}</b>",
        'de': f"🤔 Meinten Sie eines davon?\n\nSie haben eingegeben: <b>{country_input}</b>",
        'pl': f"🤔 Czy chodziło Ci o jedno z tych?\n\nWpisałeś: <b>{country_input}</b>"
    }
    
    await message.answer(
        prompt_texts.get(lang, prompt_texts['en']),
        reply_markup=get_suggestions_keyboard(suggestions, lang),
        parse_mode='HTML'
    )
    
    # ذخیره پیشنهادها در state
    await state.update_data(country_suggestions=suggestions)
    await state.set_state(RegistrationStates.country_confirm)

@router.message(RegistrationStates.country_confirm)
async def country_confirmed(message: Message, state: FSMContext):
    """تایید انتخاب کشور از پیشنهادها"""
    data = await state.get_data()
    lang = data.get('language', 'fa')
    suggestions = data.get('country_suggestions', [])
    
    # بررسی لغو
    if "❌" in message.text or "لغو" in message.text or "Cancel" in message.text or "İptal" in message.text or "Отмена" in message.text or "Cancelar" in message.text:
        await message.answer(
            get_text(lang, 'reg_reenter_country'),
            reply_markup=ReplyKeyboardRemove()
        )
        await state.set_state(RegistrationStates.country)
        return
    
    # انتخاب گزینه
    selected = None
    if "1️⃣" in message.text and len(suggestions) >= 1:
        selected = suggestions[0]
    elif "2️⃣" in message.text and len(suggestions) >= 2:
        selected = suggestions[1]
    elif "3️⃣" in message.text and len(suggestions) >= 3:
        selected = suggestions[2]
    
    if not selected:
        await message.answer(
            get_text(lang, 'reg_select_from_options'),
            reply_markup=get_suggestions_keyboard(suggestions, lang)
        )
        return
    
    # ذخیره کشور
    await state.update_data(
        country=selected['name'],
        country_code=selected['code']
    )
    
    # ادامه به شهر
    city_prompts = {
        'fa': f"✅ کشور: <b>{selected['name']}</b>\n\nحالا نام شهر خود را وارد کنید:",
        'en': f"✅ Country: <b>{selected['name']}</b>\n\nNow enter your city name:",
        'tr': f"✅ Ülke: <b>{selected['name']}</b>\n\nŞimdi şehir adınızı girin:",
        'ar': f"✅ البلد: <b>{selected['name']}</b>\n\nالآن أدخل اسم مدينتك:",
        'ru': f"✅ Страна: <b>{selected['name']}</b>\n\nТеперь введите название вашего города:",
        'es': f"✅ País: <b>{selected['name']}</b>\n\nAhora ingresa el nombre de tu ciudad:",
        'uk': f"✅ Країна: <b>{selected['name']}</b>\n\nТепер введіть назву вашого міста:",
        'hi': f"✅ देश: <b>{selected['name']}</b>\n\nअब अपने शहर का नाम दर्ज करें:",
        'it': f"✅ Paese: <b>{selected['name']}</b>\n\nOra inserisci il nome della tua città:",
        'fr': f"✅ Pays: <b>{selected['name']}</b>\n\nMaintenant entrez le nom de votre ville:",
        'pt': f"✅ País: <b>{selected['name']}</b>\n\nAgora insira o nome da sua cidade:",
        'id': f"✅ Negara: <b>{selected['name']}</b>\n\nSekarang masukkan nama kota Anda:",
        'kk': f"✅ Ел: <b>{selected['name']}</b>\n\nЕнді қалаңыздың атын енгізіңіз:",
        'de': f"✅ Land: <b>{selected['name']}</b>\n\nGeben Sie jetzt Ihren Stadtnamen ein:",
        'pl': f"✅ Kraj: <b>{selected['name']}</b>\n\nTeraz wprowadź nazwę swojego miasta:"
    }
    
    await message.answer(
        city_prompts.get(lang, city_prompts['en']),
        reply_markup=ReplyKeyboardRemove(),
        parse_mode='HTML'
    )
    await state.set_state(RegistrationStates.city)

@router.message(RegistrationStates.city)
async def city_entered(message: Message, state: FSMContext):
    """مرحله 12: وارد کردن شهر با پیشنهاد هوشمند"""
    data = await state.get_data()
    lang = data.get('language', 'fa')
    country_code = data.get('country_code', None)
    
    city_input = message.text.strip()
    
    # پیشنهاد شهرها
    suggestions = suggest_cities(city_input, country_code=country_code, lang=lang, limit=3)
    
    if not suggestions:
        # اگر پیشنهادی نبود، همان ورودی را قبول کن
        await state.update_data(city=city_input)
        
        # ادامه به بیو
        await message.answer(
            get_text(lang, 'enter_bio'),
            reply_markup=get_skip_keyboard(lang)
        )
        await state.set_state(RegistrationStates.bio)
        return
    
    # بررسی امتیاز - اگر امتیاز اولی بالای 90 باشد، مستقیم قبول کن
    if suggestions[0]['score'] >= 90:
        await state.update_data(city=suggestions[0]['name'])
        
        # ادامه به بیو
        await message.answer(
            get_text(lang, 'enter_bio'),
            reply_markup=get_skip_keyboard(lang)
        )
        await state.set_state(RegistrationStates.bio)
        return
    
    # نمایش پیشنهادات
    prompt_texts = {
        'fa': f"🤔 آیا منظور شما یکی از این شهرهاست؟\n\nشما نوشتید: <b>{city_input}</b>",
        'en': f"🤔 Did you mean one of these cities?\n\nYou entered: <b>{city_input}</b>",
        'tr': f"🤔 Bu şehirlerden birini mi kastediyorsunuz?\n\nYazdığınız: <b>{city_input}</b>",
        'ar': f"🤔 هل تقصد إحدى هذه المدن؟\n\nكتبت: <b>{city_input}</b>",
        'ru': f"🤔 Вы имели в виду один из этих городов?\n\nВы ввели: <b>{city_input}</b>",
        'es': f"🤔 ¿Quisiste decir una de estas ciudades?\n\nEscribiste: <b>{city_input}</b>",
        'uk': f"🤔 Ви мали на увазі одне з цих міст?\n\nВи ввели: <b>{city_input}</b>",
        'hi': f"🤔 क्या आपका मतलब इनमें से एक शहर था?\n\nआपने दर्ज किया: <b>{city_input}</b>",
        'it': f"🤔 Intendevi una di queste città?\n\nHai inserito: <b>{city_input}</b>",
        'fr': f"🤔 Vouliez-vous dire l'une de ces villes?\n\nVous avez entré: <b>{city_input}</b>",
        'pt': f"🤔 Você quis dizer uma destas cidades?\n\nVocê inseriu: <b>{city_input}</b>",
        'id': f"🤔 Apakah Anda maksud salah satu dari kota ini?\n\nAnda memasukkan: <b>{city_input}</b>",
        'kk': f"🤔 Сіз осы қалалардың бірін айтқыңыз келді ме?\n\nСіз енгіздіңіз: <b>{city_input}</b>",
        'de': f"🤔 Meinten Sie eine dieser Städte?\n\nSie haben eingegeben: <b>{city_input}</b>",
        'pl': f"🤔 Czy chodziło Ci o jedno z tych miast?\n\nWpisałeś: <b>{city_input}</b>"
    }
    
    await message.answer(
        prompt_texts.get(lang, prompt_texts['en']),
        reply_markup=get_suggestions_keyboard(suggestions, lang),
        parse_mode='HTML'
    )
    
    # ذخیره پیشنهادها و ورودی اصلی
    await state.update_data(city_suggestions=suggestions, city_input=city_input)
    await state.set_state(RegistrationStates.city_confirm)

@router.message(RegistrationStates.city_confirm)
async def city_confirmed(message: Message, state: FSMContext):
    """تایید انتخاب شهر از پیشنهادها"""
    data = await state.get_data()
    lang = data.get('language', 'fa')
    suggestions = data.get('city_suggestions', [])
    city_input = data.get('city_input', '')
    
    # بررسی لغو - درخواست ورود مجدد شهر
    if "❌" in message.text or "لغو" in message.text or "Cancel" in message.text or "İptal" in message.text or "Отмена" in message.text or "Cancelar" in message.text:
        # درخواست ورود مجدد شهر
        city_prompts = {
            'fa': 'لطفاً نام شهر خود را دوباره وارد کنید:',
            'en': 'Please enter your city name again:',
            'tr': 'Lütfen şehir adınızı tekrar girin:',
            'ar': 'يرجى إدخال اسم مدينتك مرة أخرى:',
            'ru': 'Пожалуйста, введите название вашего города снова:',
            'es': 'Por favor ingresa el nombre de tu ciudad nuevamente:',
            'uk': 'Будь ласка, введіть назву вашого міста знову:',
            'hi': 'कृपया अपने शहर का नाम फिर से दर्ज करें:',
            'it': 'Per favore inserisci di nuovo il nome della tua città:',
            'fr': 'Veuillez entrer à nouveau le nom de votre ville:',
            'pt': 'Por favor digite o nome da sua cidade novamente:',
            'id': 'Silakan masukkan nama kota Anda lagi:',
            'kk': 'Қалаңыздың атын қайта енгізіңіз:',
            'de': 'Bitte geben Sie Ihren Stadtnamen erneut ein:',
            'pl': 'Proszę ponownie wprowadzić nazwę swojego miasta:'
        }
        
        await message.answer(
            city_prompts.get(lang, 'Please enter your city name again:'),
            reply_markup=ReplyKeyboardRemove()
        )
        await state.set_state(RegistrationStates.city)
        return
    
    # انتخاب گزینه
    selected = None
    if "1️⃣" in message.text and len(suggestions) >= 1:
        selected = suggestions[0]
    elif "2️⃣" in message.text and len(suggestions) >= 2:
        selected = suggestions[1]
    elif "3️⃣" in message.text and len(suggestions) >= 3:
        selected = suggestions[2]
    
    if not selected:
        await message.answer(
            get_text(lang, 'reg_select_from_options'),
            reply_markup=get_suggestions_keyboard(suggestions, lang)
        )
        return
    
    # ذخیره شهر
    await state.update_data(city=selected['name'])
    
    # ادامه به بیو
    await message.answer(
        get_text(lang, 'enter_bio'),
        reply_markup=get_skip_keyboard(lang)
    )
    await state.set_state(RegistrationStates.bio)

@router.message(RegistrationStates.bio)
async def bio_entered(message: Message, state: FSMContext):
    """مرحله 13: وارد کردن بیو"""
    data = await state.get_data()
    lang = data.get('language', 'fa')
    
    # بررسی رد کردن
    if "⏭️" in message.text or "رد کردن" in message.text or "Skip" in message.text or get_text(lang, 'pass', default='رد کردن') in message.text:
        bio = None
    else:
        bio = message.text.strip()
        if len(bio) > 200:
            await message.answer(get_text(lang, 'bio_too_long'))
            return
    
    await state.update_data(bio=bio)
    
    # درخواست علایق با کیبورد کاستوم
    await message.answer(
        get_text(lang, 'select_interests'),
        reply_markup=get_interests_keyboard(lang, [])
    )
    await state.set_state(RegistrationStates.interests)

@router.message(RegistrationStates.interests)
async def interests_selected(message: Message, state: FSMContext):
    """مرحله 14: انتخاب علایق"""
    data = await state.get_data()
    lang = data.get('language', 'fa')
    selected_interests = data.get('selected_interests', [])
    
    confirm_text = get_text(lang, 'confirm_interests', default='تایید علایق')
    
    # بررسی رد کردن یا تایید
    if "⏭️" in message.text or "رد کردن" in message.text or "Skip" in message.text or get_text(lang, 'pass', default='رد کردن') in message.text:
        # رد کردن علایق - ذخیره لیست خالی و رفتن به مرحله عکس
        await state.update_data(selected_interests=[])
        await ask_for_photo(message, state)
        return
    elif "✅" in message.text and (confirm_text in message.text or "تایید" in message.text or "Confirm" in message.text):
        # تایید علایق - رفتن به مرحله عکس
        await state.update_data(selected_interests=selected_interests)
        await ask_for_photo(message, state)
        return
    
    # انتخاب/لغو انتخاب علاقه
    interests_dict = get_text(lang, 'interests_list', default={})
    
    if not interests_dict:
        interests_dict = {
            'sports': 'ورزش', 'music': 'موسیقی', 'movies': 'فیلم', 'books': 'کتاب',
            'travel': 'سفر', 'cooking': 'آشپزی', 'art': 'هنر', 'photography': 'عکاسی',
            'gaming': 'بازی', 'fitness': 'تناسب اندام', 'nature': 'طبیعت', 'technology': 'فناوری',
            'fashion': 'مد', 'dancing': 'رقص', 'reading': 'مطالعه', 'writing': 'نویسندگی'
        }
    
    # پیدا کردن علاقه انتخاب شده
    selected_key = None
    for key, value in interests_dict.items():
        if value in message.text.replace('✅ ', ''):
            selected_key = key
            break
    
    if selected_key:
        if selected_key in selected_interests:
            # لغو انتخاب
            selected_interests.remove(selected_key)
        else:
            # انتخاب جدید
            if len(selected_interests) >= 6:
                max_interests_messages = {
                    'fa': '❌ حداکثر 6 علاقه می‌توانید انتخاب کنید.\n\n💡 برای تایید نهایی لطفاً به پایین کیبورد اسکرول کنید و دکمه "✅ تایید علایق" را بزنید.',
                    'en': '❌ You can select maximum 6 interests.\n\n💡 To confirm, please scroll down the keyboard and press "✅ Confirm Interests".',
                    'tr': '❌ Maksimum 6 ilgi alanı seçebilirsiniz.\n\n💡 Onaylamak için lütfen klavyeyi aşağı kaydırın ve "✅ İlgi Alanlarını Onayla" düğmesine basın.',
                    'ar': '❌ يمكنك اختيار 6 اهتمامات كحد أقصى.\n\n💡 للتأكيد، يرجى التمرير لأسفل لوحة المفاتيح والضغط على "✅ تأكيد الاهتمامات".',
                    'ru': '❌ Вы можете выбрать максимум 6 интересов.\n\n💡 Для подтверждения прокрутите клавиатуру вниз и нажмите "✅ Подтвердить интересы".',
                    'es': '❌ Puedes seleccionar máximo 6 intereses.\n\n💡 Para confirmar, desplázate hacia abajo en el teclado y presiona "✅ Confirmar Intereses".',
                    'uk': '❌ Ви можете вибрати максимум 6 інтересів.\n\n💡 Для підтвердження прокрутіть клавіатуру вниз і натисніть "✅ Підтвердити інтереси".',
                    'hi': '❌ आप अधिकतम 6 रुचियां चुन सकते हैं।\n\n💡 पुष्टि करने के लिए, कृपया कीबोर्ड को नीचे स्क्रॉल करें और "✅ रुचियों की पुष्टि करें" बटन दबाएं।',
                    'it': '❌ Puoi selezionare massimo 6 interessi.\n\n💡 Per confermare, scorri la tastiera verso il basso e premi "✅ Conferma Interessi".',
                    'fr': '❌ Vous pouvez sélectionner maximum 6 intérêts.\n\n💡 Pour confirmer, faites défiler le clavier vers le bas et appuyez sur "✅ Confirmer les Intérêts".',
                    'pt': '❌ Você pode selecionar no máximo 6 interesses.\n\n💡 Para confirmar, role o teclado para baixo e pressione "✅ Confirmar Interesses".',
                    'id': '❌ Anda dapat memilih maksimum 6 minat.\n\n💡 Untuk mengonfirmasi, gulir keyboard ke bawah dan tekan "✅ Konfirmasi Minat".',
                    'kk': '❌ Сіз максимум 6 қызығушылық таңдай аласыз.\n\n💡 Растау үшін пернетақтаны төмен айналдырып, "✅ Қызығушылықтарды растау" батырмасын басыңыз.',
                    'de': '❌ Sie können maximal 6 Interessen auswählen.\n\n💡 Zum Bestätigen scrollen Sie die Tastatur nach unten und drücken Sie "✅ Interessen bestätigen".',
                    'pl': '❌ Możesz wybrać maksymalnie 6 zainteresowań.\n\n💡 Aby potwierdzić, przewiń klawiaturę w dół i naciśnij "✅ Potwierdź Zainteresowania".'
                }
                await message.answer(max_interests_messages.get(lang, max_interests_messages['en']))
                return
            
            selected_interests.append(selected_key)
        
        await state.update_data(selected_interests=selected_interests)
        
        # به‌روزرسانی کیبورد بدون پیام اضافی
        await message.answer(
            "✅" if selected_key in selected_interests else "❌",
            reply_markup=get_interests_keyboard(lang, selected_interests)
        )

async def ask_for_photo(message: Message, state: FSMContext):
    """درخواست آپلود عکس پروفایل"""
    data = await state.get_data()
    lang = data.get('language', 'fa')
    
    photo_prompts = {
        'fa': 'برای آپلود عکس پروفایل کدام گزینه را انتخاب می‌کنید؟',
        'en': 'Which option do you choose to upload profile photo?',
        'tr': 'Profil fotoğrafı yüklemek için hangi seçeneği seçiyorsunuz?',
        'ar': 'أي خيار تختار لتحميل صورة الملف الشخصي؟',
        'ru': 'Какой вариант вы выбираете для загрузки фото профиля?',
        'es': '¿Qué opción eliges para subir la foto de perfil?',
        'uk': 'Який варіант ви обираєте для завантаження фото профілю?',
        'hi': 'प्रोफ़ाइल फ़ोटो अपलोड करने के लिए आप कौन सा विकल्प चुनते हैं?',
        'it': 'Quale opzione scegli per caricare la foto del profilo?',
        'fr': 'Quelle option choisissez-vous pour télécharger la photo de profil?',
        'pt': 'Qual opção você escolhe para carregar a foto do perfil?',
        'id': 'Opsi mana yang Anda pilih untuk mengunggah foto profil?',
        'kk': 'Профиль фотосын жүктеу үшін қай нұсқаны таңдайсыз?',
        'de': 'Welche Option wählen Sie zum Hochladen des Profilfotos?',
        'pl': 'Którą opcję wybierasz do przesłania zdjęcia profilu?'
    }
    
    await message.answer(
        photo_prompts.get(lang, photo_prompts['en']),
        reply_markup=get_photo_method_keyboard(lang)
    )
    await state.set_state(RegistrationStates.photo_method)

@router.message(RegistrationStates.photo_method)
async def photo_method_selected(message: Message, state: FSMContext):
    """مرحله 15: انتخاب روش آپلود عکس"""
    data = await state.get_data()
    lang = data.get('language', 'fa')
    
    # بررسی رد کردن
    if "رد" in message.text or "Skip" in message.text or "Atla" in message.text or "⏭️" in message.text:
        # رد کردن عکس - ذخیره عکس پیش‌فرض و تکمیل ثبت‌نام
        await state.update_data(profile_photo=None)
        await complete_registration(message, state)
        return
    
    # استفاده از عکس پروفایل تلگرام
    if "📱" in message.text or "Telegram" in message.text or "تلگرام" in message.text:
        try:
            # دریافت عکس پروفایل کاربر
            from aiogram import Bot
            bot = message.bot
            
            photos = await bot.get_user_profile_photos(message.from_user.id, limit=1)
            
            if photos.total_count > 0:
                # دریافت بهترین کیفیت
                photo = photos.photos[0][-1]  # آخرین عکس = بهترین کیفیت
                file_id = photo.file_id
                
                # ذخیره file_id و نمایش عکس
                await state.update_data(temp_photo=file_id)
                
                # ارسال عکس با دکمه‌های تایید/حذف
                await bot.send_photo(
                    chat_id=message.chat.id,
                    photo=file_id,
                    caption=get_text(lang, 'reg_profile_photo_confirm'),
                    reply_markup=get_photo_confirm_keyboard(lang)
                )
                await state.set_state(RegistrationStates.photo_confirm)
            else:
                await message.answer(
                    get_text(lang, 'reg_no_profile_photo'),
                    reply_markup=get_photo_method_keyboard(lang)
                )
        except Exception as e:
            logger.error(f"Error getting profile photo: {e}")
            await message.answer(
                get_text(lang, 'reg_profile_photo_error'),
                reply_markup=get_photo_method_keyboard(lang)
            )
        return
    
    # آپلود عکس جدید
    if "📤" in message.text or "Upload" in message.text or "آپلود" in message.text or "Yükle" in message.text:
        upload_prompts = {
            'fa': 'لطفاً عکس خود را ارسال کنید:',
            'en': 'Please send your photo:',
            'tr': 'Lütfen fotoğrafınızı gönderin:',
            'ar': 'يرجى إرسال صورتك:',
            'ru': 'Пожалуйста, отправьте свое фото:',
            'es': 'Por favor envía tu foto:',
            'uk': 'Будь ласка, надішліть своє фото:',
            'hi': 'कृपया अपनी फ़ोटो भेजें:',
            'it': 'Per favore invia la tua foto:',
            'fr': 'Veuillez envoyer votre photo:',
            'pt': 'Por favor envie sua foto:',
            'id': 'Silakan kirim foto Anda:',
            'kk': 'Өтінемін, фотосуретіңізді жіберіңіз:',
            'de': 'Bitte senden Sie Ihr Foto:',
            'pl': 'Proszę wyślij swoje zdjęcie:'
        }
        
        await message.answer(
            upload_prompts.get(lang, upload_prompts['en']),
            reply_markup=ReplyKeyboardRemove()
        )
        await state.set_state(RegistrationStates.photo_upload)
        return
    
    # گزینه نامعتبر
    await message.answer(
        get_text(lang, 'reg_select_available_option'),
        reply_markup=get_photo_method_keyboard(lang)
    )

@router.message(RegistrationStates.photo_upload, F.photo)
async def photo_uploaded(message: Message, state: FSMContext):
    """مرحله 16: دریافت عکس آپلود شده"""
    data = await state.get_data()
    lang = data.get('language', 'fa')
    
    # دریافت بهترین کیفیت عکس
    photo = message.photo[-1]
    file_id = photo.file_id
    
    # ذخیره file_id
    await state.update_data(temp_photo=file_id)
    
    # ارسال مجدد عکس با دکمه‌های تایید/حذف
    await message.bot.send_photo(
        chat_id=message.chat.id,
        photo=file_id,
        caption=get_text(lang, 'reg_photo_confirm'),
        reply_markup=get_photo_confirm_keyboard(lang)
    )
    await state.set_state(RegistrationStates.photo_confirm)

@router.message(RegistrationStates.photo_upload, F.content_type.in_({'video', 'document', 'sticker', 'voice', 'video_note', 'animation'}))
async def handle_invalid_file_type_registration(message: Message, state: FSMContext):
    """مدیریت فایل‌های غیرعکس در ثبت‌نام"""
    data = await state.get_data()
    lang = data.get('language', 'fa')
    
    await message.answer(
        f"❌ {get_text(lang, 'invalid_file_type', default='فایل ارسالی نامعتبر است. لطفاً فقط عکس ارسال کنید.')}\n\n"
        f"📸 {get_text(lang, 'please_send_photo', default='لطفاً یک عکس ارسال کنید.')}",
        reply_markup=get_photo_method_keyboard(lang)
    )

@router.message(RegistrationStates.photo_confirm)
async def photo_confirm_handler(message: Message, state: FSMContext):
    """مرحله 17: تایید یا حذف عکس"""
    data = await state.get_data()
    lang = data.get('language', 'fa')
    
    # تایید عکس
    if "✅" in message.text or "تایید" in message.text or "Confirm" in message.text or "Onayla" in message.text:
        # ذخیره عکس و تکمیل ثبت‌نام
        temp_photo = data.get('temp_photo')
        if temp_photo:
            await state.update_data(profile_photo=temp_photo)
        
        await complete_registration(message, state)
        return
    
    # حذف عکس
    if "❌" in message.text or "حذف" in message.text or "Delete" in message.text or "Sil" in message.text:
        # بازگشت به انتخاب روش
        await ask_for_photo(message, state)
        return
    
    # گزینه نامعتبر
    await message.answer(
        get_text(lang, 'reg_select_button'),
        reply_markup=get_photo_confirm_keyboard(lang)
    )

async def complete_registration(message: Message, state: FSMContext):
    """تکمیل ثبت‌نام و ذخیره در دیتابیس"""
    data = await state.get_data()
    lang = data.get('language', 'fa')
    selected_interests = data.get('selected_interests', [])
    
    # ذخیره کاربر در دیتابیس
    async with db.get_session() as session:
        try:
            # ذخیره علایق به صورت JSON برای سازگاری با profile_handler
            interests_str = json.dumps(selected_interests) if selected_interests else ''
            
            # دریافت عکس پروفایل
            profile_photo = data.get('profile_photo')
            photos_list = []
            if profile_photo:
                photos_list = [profile_photo]
            
            # دریافت اطلاعات احراز هویت
            email = data.get('email')
            password = data.get('password')
            is_guest = data.get('is_guest', False)
            
            # ایجاد کاربر
            user = User(
                user_id=message.from_user.id,
                username=message.from_user.username or f"user_{message.from_user.id}",
                first_name=data.get('name'),
                last_name=message.from_user.last_name,
                gender=str(data.get('gender')),
                age=data.get('age'),
                language_code=lang,
                country=data.get('country', ''),  # نام کامل کشور (مثلاً "ایران")
                country_code=data.get('country_code', ''),  # کد کشور (مثلاً "IR")
                city=data.get('city', ''),
                location_lat=str(data.get('latitude', 0)) if data.get('latitude') else None,
                location_lon=str(data.get('longitude', 0)) if data.get('longitude') else None,
                looking_for=str(data.get('looking_for', 0)),
                bio=data.get('bio', ''),
                interests=interests_str,
                photos=json.dumps(photos_list) if photos_list else None,
                email=email,
                password_hash=hash_password(password) if password else None,
                is_guest=is_guest,
                email_verified=True if email else False
            )
            
            session.add(user)
            await session.flush()
            
            # ایجاد تنظیمات جستجو
            preference = Preference(
                user_id=message.from_user.id,
                age_min=18,
                age_max=45,
                gender_pref=data.get('looking_for', 0),
                radius_km=50,
                country_code=data.get('country_code', ''),
                city=data.get('city', '')
            )
            
            session.add(preference)
            await session.commit()
            
            # بررسی ادمین بودن
            ADMIN_USER_IDS = [int(uid) for uid in os.getenv('ADMIN_USER_IDS', '').split(',') if uid]
            is_admin = message.from_user.id in ADMIN_USER_IDS
            
            # پایان ثبت‌نام
            completion_messages = {
                'fa': '🎉 <b>پروفایل شما تکمیل شد!</b>',
                'en': '🎉 <b>Your profile is complete!</b>',
                'tr': '🎉 <b>Profiliniz tamamlandı!</b>',
                'ar': '🎉 <b>اكتمل ملفك الشخصي!</b>',
                'ru': '🎉 <b>Ваш профиль завершен!</b>',
                'es': '🎉 <b>¡Tu perfil está completo!</b>',
                'uk': '🎉 <b>Ваш профіль завершено!</b>',
                'hi': '🎉 <b>आपकी प्रोफ़ाइल पूर्ण है!</b>',
                'it': '🎉 <b>Il tuo profilo è completo!</b>',
                'fr': '🎉 <b>Votre profil est complet!</b>',
                'pt': '🎉 <b>Seu perfil está completo!</b>',
                'id': '🎉 <b>Profil Anda lengkap!</b>',
                'kk': '🎉 <b>Сіздің профиліңіз толық!</b>',
                'de': '🎉 <b>Ihr Profil ist vollständig!</b>',
                'pl': '🎉 <b>Twój profil jest kompletny!</b>'
            }
            
            await message.answer(
                completion_messages.get(lang, completion_messages['en']),
                parse_mode='HTML'
            )
            
            # نمایش منوی پایین بعد از ثبت‌نام
            from handlers.bottom_menu import show_user_menu
            await show_user_menu(message.bot, message.from_user.id, lang)
            
            if is_admin:
                await message.answer(
                    get_text(lang, 'reg_admin_identified'),
                    reply_markup=get_main_menu_keyboard(lang, is_admin=True, is_guest=False)
                )
            else:
                await message.answer(
                    get_text(lang, 'reg_success_emoji'),
                    reply_markup=get_main_menu_keyboard(lang, is_admin=False, is_guest=is_guest)
                )
            
            await state.clear()
            logger.info(f"User {message.from_user.id} registered successfully")
            
        except Exception as e:
            import traceback
            error_details = traceback.format_exc()
            logger.error(f"Registration error for user {message.from_user.id}: {e}\n{error_details}")
            
            await message.answer(
                get_text(lang, 'reg_error_message').format(error=str(e)[:100])
            )
            await state.clear()

# ==================== Auth Management (Login/Logout) ====================

@router.message(F.text.contains("ورود به حساب") | F.text.contains("Login to Account") | 
                F.text.contains("Hesaba Giriş") | F.text.contains("تسجيل الدخول") |
                F.text.contains("Войти в аккаунт") | F.text.contains("Iniciar sesión") |
                F.text.contains("Увійти в акаунт") | F.text.contains("खाते में लॉगिन") |
                F.text.contains("Accedi all'account") | F.text.contains("Se connecter") |
                F.text.contains("Entrar na conta") | F.text.contains("Masuk ke Akun") |
                F.text.contains("Аккаунтқа кіру") | F.text.contains("Anmelden") |
                F.text.contains("Zaloguj się"))
async def login_to_account_handler(message: Message, state: FSMContext):
    """ورود به حساب کاربری برای کاربران مهمان"""
    user_id = message.from_user.id
    
    async with db.get_session() as session:
        result = await session.execute(select(User).where(User.id == user_id))
        user = result.scalar_one_or_none()
        
        if not user:
            await message.answer("❌ کاربر یافت نشد. لطفاً ابتدا ثبت‌نام کنید.")
            return
        
        lang = user.language_code or 'fa'
        
        # بررسی اینکه آیا کاربر مهمان است
        if not user.is_guest:
            already_logged_in_texts = {
                'fa': '✅ شما قبلاً وارد حساب کاربری خود شده‌اید.',
                'en': '✅ You are already logged in to your account.',
                'tr': '✅ Zaten hesabınıza giriş yaptınız.',
                'ar': '✅ لقد قمت بالفعل بتسجيل الدخول إلى حسابك.',
                'ru': '✅ Вы уже вошли в свой аккаунт.',
                'es': '✅ Ya has iniciado sesión en tu cuenta.',
                'uk': '✅ Ви вже увійшли в свій акаунт.',
                'hi': '✅ आप पहले से ही अपने खाते में लॉग इन हैं।',
                'it': '✅ Hai già effettuato l\'accesso al tuo account.',
                'fr': '✅ Vous êtes déjà connecté à votre compte.',
                'pt': '✅ Você já está conectado à sua conta.',
                'id': '✅ Anda sudah masuk ke akun Anda.',
                'kk': '✅ Сіз қазірдің өзінде аккаунтқа кірдіңіз.',
                'de': '✅ Sie sind bereits in Ihrem Konto angemeldet.',
                'pl': '✅ Jesteś już zalogowany na swoje konto.'
            }
            await message.answer(already_logged_in_texts.get(lang, already_logged_in_texts['en']))
            return
        
        # کاربر مهمان است - نمایش گزینه‌های ثبت‌نام/ورود
        login_choice_texts = {
            'fa': '🔐 <b>ورود به حساب کاربری</b>\n\nلطفاً یکی از گزینه‌های زیر را انتخاب کنید:',
            'en': '🔐 <b>Login to Account</b>\n\nPlease choose one of the following options:',
            'tr': '🔐 <b>Hesaba Giriş</b>\n\nLütfen aşağıdaki seçeneklerden birini seçin:',
            'ar': '🔐 <b>تسجيل الدخول</b>\n\nالرجاء اختيار أحد الخيارات التالية:',
            'ru': '🔐 <b>Вход в аккаунт</b>\n\nПожалуйста, выберите один из вариантов:',
            'es': '🔐 <b>Iniciar sesión</b>\n\nPor favor, elija una de las siguientes opciones:',
            'uk': '🔐 <b>Вхід в акаунт</b>\n\nБудь ласка, виберіть один з варіантів:',
            'hi': '🔐 <b>लॉगिन</b>\n\nकृपया निम्नलिखित विकल्पों में से एक चुनें:',
            'it': '🔐 <b>Accedi</b>\n\nScegli una delle seguenti opzioni:',
            'fr': '🔐 <b>Connexion</b>\n\nVeuillez choisir l\'une des options suivantes:',
            'pt': '🔐 <b>Entrar</b>\n\nPor favor, escolha uma das seguintes opções:',
            'id': '🔐 <b>Masuk</b>\n\nSilakan pilih salah satu opsi berikut:',
            'kk': '🔐 <b>Кіру</b>\n\nКелесі нұсқалардың бірін таңдаңыз:',
            'de': '🔐 <b>Anmelden</b>\n\nBitte wählen Sie eine der folgenden Optionen:',
            'pl': '🔐 <b>Zaloguj się</b>\n\nProszę wybrać jedną z następujących opcji:'
        }
        
        signup_btn_texts = {
            'fa': '📝 ثبت‌نام',
            'en': '📝 Sign Up',
            'tr': '📝 Kayıt Ol',
            'ar': '📝 التسجيل',
            'ru': '📝 Регистрация',
            'es': '📝 Registrarse',
            'uk': '📝 Реєстрація',
            'hi': '📝 साइन अप करें',
            'it': '📝 Registrati',
            'fr': '📝 S\'inscrire',
            'pt': '📝 Cadastrar',
            'id': '📝 Daftar',
            'kk': '📝 Тіркелу',
            'de': '📝 Registrieren',
            'pl': '📝 Zarejestruj się'
        }
        
        login_btn_texts = {
            'fa': '🔑 ورود',
            'en': '🔑 Login',
            'tr': '🔑 Giriş',
            'ar': '🔑 تسجيل الدخول',
            'ru': '🔑 Войти',
            'es': '🔑 Iniciar sesión',
            'uk': '🔑 Увійти',
            'hi': '🔑 लॉगिन करें',
            'it': '🔑 Accedi',
            'fr': '🔑 Se connecter',
            'pt': '🔑 Entrar',
            'id': '🔑 Masuk',
            'kk': '🔑 Кіру',
            'de': '🔑 Anmelden',
            'pl': '🔑 Zaloguj się'
        }
        
        back_to_menu_texts = {
            'fa': '🔙 بازگشت به منوی اصلی',
            'en': '🔙 Back to Main Menu',
            'tr': '🔙 Ana Menüye Dön',
            'ar': '🔙 العودة إلى القائمة الرئيسية',
            'ru': '🔙 Вернуться в главное меню',
            'es': '🔙 Volver al menú principal',
            'uk': '🔙 Повернутися до головного меню',
            'hi': '🔙 मुख्य मेनू पर वापस जाएं',
            'it': '🔙 Torna al menu principale',
            'fr': '🔙 Retour au menu principal',
            'pt': '🔙 Voltar ao menu principal',
            'id': '🔙 Kembali ke Menu Utama',
            'kk': '🔙 Басты мәзірге оралу',
            'de': '🔙 Zurück zum Hauptmenü',
            'pl': '🔙 Powrót do menu głównego'
        }
        
        signup_login_keyboard = ReplyKeyboardMarkup(
            keyboard=[
                [
                    KeyboardButton(text=signup_btn_texts.get(lang, signup_btn_texts['en'])),
                    KeyboardButton(text=login_btn_texts.get(lang, login_btn_texts['en']))
                ],
                [KeyboardButton(text=back_to_menu_texts.get(lang, back_to_menu_texts['en']))]
            ],
            resize_keyboard=True,
            one_time_keyboard=True
        )
        
        await message.answer(
            login_choice_texts.get(lang, login_choice_texts['en']),
            reply_markup=signup_login_keyboard,
            parse_mode='HTML'
        )
        await state.set_state(RegistrationStates.signup_or_login)
        await state.update_data(is_login_from_guest=True)

@router.message(F.text.contains("خروج از حساب") | F.text.contains("Logout") | 
                F.text.contains("Çıkış") | F.text.contains("تسجيل الخروج") |
                F.text.contains("Выйти") | F.text.contains("Cerrar sesión") |
                F.text.contains("Вийти") | F.text.contains("लॉगआउट") |
                F.text.contains("Esci") | F.text.contains("Se déconnecter") |
                F.text.contains("Sair") | F.text.contains("Keluar") |
                F.text.contains("Шығу") | F.text.contains("Abmelden") |
                F.text.contains("Wyloguj"))
async def logout_handler(message: Message, state: FSMContext):
    """خروج از حساب کاربری و تبدیل به مهمان"""
    user_id = message.from_user.id
    
    async with db.get_session() as session:
        result = await session.execute(select(User).where(User.id == user_id))
        user = result.scalar_one_or_none()
        
        if not user:
            await message.answer("❌ کاربر یافت نشد.")
            return
        
        lang = user.language_code or 'fa'
        
        # بررسی اینکه آیا کاربر مهمان است
        if user.is_guest:
            already_guest_texts = {
                'fa': '⚠️ شما در حال حاضر به عنوان مهمان هستید.',
                'en': '⚠️ You are currently a guest.',
                'tr': '⚠️ Şu anda misafirsiniz.',
                'ar': '⚠️ أنت حاليًا ضيف.',
                'ru': '⚠️ Вы сейчас гость.',
                'es': '⚠️ Actualmente eres un invitado.',
                'uk': '⚠️ Ви зараз гість.',
                'hi': '⚠️ आप वर्तमान में अतिथि हैं।',
                'it': '⚠️ Attualmente sei un ospite.',
                'fr': '⚠️ Vous êtes actuellement un invité.',
                'pt': '⚠️ Você está atualmente como convidado.',
                'id': '⚠️ Anda saat ini adalah tamu.',
                'kk': '⚠️ Сіз қазір қонақсыз.',
                'de': '⚠️ Sie sind derzeit ein Gast.',
                'pl': '⚠️ Jesteś obecnie gościem.'
            }
            await message.answer(already_guest_texts.get(lang, already_guest_texts['en']))
            return
        
        # نمایش پیام تأیید با کیبورد بله/خیر
        confirm_logout_texts = {
            'fa': '⚠️ <b>آیا مطمئن هستید که می‌خواهید از حساب خود خارج شوید؟</b>\n\n'
                  '🔹 با خروج، به حالت مهمان تبدیل می‌شوید\n'
                  '🔹 برای ورود مجدد، باید ایمیل و رمز عبور خود را وارد کنید',
            'en': '⚠️ <b>Are you sure you want to logout?</b>\n\n'
                  '🔹 You will be converted to guest mode\n'
                  '🔹 To login again, you need to enter your email and password',
            'tr': '⚠️ <b>Çıkış yapmak istediğinizden emin misiniz?</b>\n\n'
                  '🔹 Misafir moduna dönüştürüleceksiniz\n'
                  '🔹 Tekrar giriş yapmak için e-posta ve şifrenizi girmeniz gerekir',
            'ar': '⚠️ <b>هل أنت متأكد أنك تريد تسجيل الخروج؟</b>\n\n'
                  '🔹 سيتم تحويلك إلى وضع الضيف\n'
                  '🔹 لتسجيل الدخول مرة أخرى، تحتاج إلى إدخال بريدك الإلكتروني وكلمة المرور',
            'ru': '⚠️ <b>Вы уверены, что хотите выйти?</b>\n\n'
                  '🔹 Вы будете переведены в гостевой режим\n'
                  '🔹 Для повторного входа вам нужно будет ввести email и пароль',
            'es': '⚠️ <b>¿Estás seguro de que quieres cerrar sesión?</b>\n\n'
                  '🔹 Serás convertido a modo invitado\n'
                  '🔹 Para iniciar sesión nuevamente, necesitas ingresar tu correo y contraseña',
            'uk': '⚠️ <b>Ви впевнені, що хочете вийти?</b>\n\n'
                  '🔹 Ви будете переведені в гостьовий режим\n'
                  '🔹 Для повторного входу вам потрібно буде ввести email та пароль',
            'hi': '⚠️ <b>क्या आप सुनिश्चित हैं कि आप लॉगआउट करना चाहते हैं?</b>\n\n'
                  '🔹 आप अतिथि मोड में परिवर्तित हो जाएंगे\n'
                  '🔹 फिर से लॉगिन करने के लिए, आपको अपना ईमेल और पासवर्ड दर्ज करना होगा',
            'it': '⚠️ <b>Sei sicuro di voler uscire?</b>\n\n'
                  '🔹 Sarai convertito in modalità ospite\n'
                  '🔹 Per accedere nuovamente, devi inserire la tua email e password',
            'fr': '⚠️ <b>Êtes-vous sûr de vouloir vous déconnecter?</b>\n\n'
                  '🔹 Vous serez converti en mode invité\n'
                  '🔹 Pour vous reconnecter, vous devrez entrer votre email et mot de passe',
            'pt': '⚠️ <b>Tem certeza de que deseja sair?</b>\n\n'
                  '🔹 Você será convertido para o modo convidado\n'
                  '🔹 Para fazer login novamente, você precisa inserir seu e-mail e senha',
            'id': '⚠️ <b>Apakah Anda yakin ingin keluar?</b>\n\n'
                  '🔹 Anda akan diubah ke mode tamu\n'
                  '🔹 Untuk masuk lagi, Anda perlu memasukkan email dan kata sandi Anda',
            'kk': '⚠️ <b>Шығуды қалайсыз ба?</b>\n\n'
                  '🔹 Сіз қонақ режиміне ауысасыз\n'
                  '🔹 Қайта кіру үшін email және парольді енгізу керек',
            'de': '⚠️ <b>Sind Sie sicher, dass Sie sich abmelden möchten?</b>\n\n'
                  '🔹 Sie werden in den Gastmodus umgewandelt\n'
                  '🔹 Um sich erneut anzumelden, müssen Sie Ihre E-Mail und Ihr Passwort eingeben',
            'pl': '⚠️ <b>Czy na pewno chcesz się wylogować?</b>\n\n'
                  '🔹 Zostaniesz przekonwertowany na tryb gościa\n'
                  '🔹 Aby zalogować się ponownie, musisz wprowadzić swój e-mail i hasło'
        }
        
        await message.answer(
            confirm_logout_texts.get(lang, confirm_logout_texts['en']),
            reply_markup=get_yes_no_keyboard(lang),
            parse_mode='HTML'
        )
        await state.set_state(RegistrationStates.logout_confirm)

@router.message(RegistrationStates.logout_confirm)
async def logout_confirm_handler(message: Message, state: FSMContext):
    """پردازش تأیید یا لغو خروج از حساب"""
    user_id = message.from_user.id
    data = await state.get_data()
    switch_account = data.get('switch_account', False)
    new_email = data.get('new_email')
    
    async with db.get_session() as session:
        result = await session.execute(select(User).where(User.id == user_id))
        user = result.scalar_one_or_none()
        
        if not user:
            await message.answer("❌ کاربر یافت نشد.")
            await state.clear()
            return
        
        lang = user.language_code or 'fa'
        
        # بررسی انتخاب کاربر
        yes_keywords = ['بله', 'Yes', 'Evet', 'نعم', 'Да', 'Sí', 'Так', 'हाँ', 'Sì', 'Oui', 'Sim', 'Ya', 'Иә', 'Ja', 'Tak', '✅']
        no_keywords = ['خیر', 'No', 'Hayır', 'لا', 'Нет', 'Ні', 'नहीं', 'Non', 'Não', 'Tidak', 'Жоқ', 'Nein', 'Nie', '❌']
        
        if any(keyword in message.text for keyword in yes_keywords):
            if switch_account and new_email:
                # حالت تعویض حساب - به‌روزرسانی ایمیل
                user.email = new_email
                user.is_guest = False
                user.email_verified = False
                await session.commit()
                
                # ارسال کد تأیید برای ایمیل جدید
                verification_code = generate_verification_code()
                await send_verification_email(new_email, verification_code, lang)
                await state.update_data(email=new_email, switch_account=False)
                
                switch_success_texts = {
                    'fa': f'✅ حساب شما به ایمیل <b>{new_email}</b> تغییر یافت.\n\nکد تأیید ارسال شد. لطفاً کد 6 رقمی را وارد کنید:',
                    'en': f'✅ Your account has been switched to <b>{new_email}</b>.\n\nVerification code sent. Please enter the 6-digit code:',
                    'tr': f'✅ Hesabınız <b>{new_email}</b> olarak değiştirildi.\n\nDoğrulama kodu gönderildi. Lütfen 6 haneli kodu girin:',
                    'ar': f'✅ تم تبديل حسابك إلى <b>{new_email}</b>.\n\nتم إرسال رمز التحقق. الرجاء إدخال الرمز المكون من 6 أرقام:'
                }
                
                await message.answer(
                    switch_success_texts.get(lang, switch_success_texts['en']),
                    reply_markup=ReplyKeyboardRemove(),
                    parse_mode='HTML'
                )
                await state.set_state(RegistrationStates.verification_code)
            else:
                # حالت خروج عادی - تبدیل به مهمان
                user.is_guest = True
                await session.commit()
                
                logout_success_texts = {
                    'fa': '✅ شما با موفقیت از حساب خود خارج شدید.\n\nاکنون به عنوان مهمان وارد شده‌اید.\nبرای ورود مجدد، از دکمه «ورود به حساب کاربری» استفاده کنید.',
                    'en': '✅ You have successfully logged out.\n\nYou are now logged in as a guest.\nTo login again, use the \"Login to Account\" button.',
                    'tr': '✅ Başarıyla çıkış yaptınız.\n\nŞimdi misafir olarak giriş yaptınız.\nTekrar giriş yapmak için \"Hesaba Giriş\" düğmesini kullanın.',
                    'ar': '✅ لقد قمت بتسجيل الخروج بنجاح.\n\nأنت الآن مسجل كضيف.\nلتسجيل الدخول مرة أخرى، استخدم زر \"تسجيل الدخول\".',
                    'ru': '✅ Вы успешно вышли из аккаунта.\n\nТеперь вы вошли как гость.\nДля повторного входа используйте кнопку \"Войти в аккаунт\".',
                    'es': '✅ Has cerrado sesión exitosamente.\n\nAhora has iniciado sesión como invitado.\nPara iniciar sesión nuevamente, usa el botón \"Iniciar sesión\".',
                    'uk': '✅ Ви успішно вийшли з акаунту.\n\nТепер ви увійшли як гість.\nДля повторного входу використовуйте кнопку \"Увійти в акаунт\".',
                    'hi': '✅ आपने सफलतापूर्वक लॉगआउट कर दिया है।\n\nअब आप अतिथि के रूप में लॉग इन हैं।\nफिर से लॉगिन करने के लिए, \"खाते में लॉगिन करें\" बटन का उपयोग करें।',
                    'it': '✅ Hai effettuato il logout con successo.\n\nOra sei connesso come ospite.\nPer accedere nuovamente, usa il pulsante \"Accedi all\'account\".',
                    'fr': '✅ Vous vous êtes déconnecté avec succès.\n\nVous êtes maintenant connecté en tant qu\'invité.\nPour vous reconnecter, utilisez le bouton \"Se connecter\".',
                    'pt': '✅ Você saiu com sucesso.\n\nAgora você está conectado como convidado.\nPara fazer login novamente, use o botão \"Entrar na conta\".',
                    'id': '✅ Anda telah berhasil keluar.\n\nAnda sekarang masuk sebagai tamu.\nUntuk masuk lagi, gunakan tombol \"Masuk ke Akun\".',
                    'kk': '✅ Сіз сәтті шықтыңыз.\n\nЕнді сіз қонақ ретінде кірдіңіз.\nҚайта кіру үшін \"Аккаунтқа кіру\" батырмасын пайдаланыңыз.',
                    'de': '✅ Sie haben sich erfolgreich abgemeldet.\n\nSie sind jetzt als Gast angemeldet.\nUm sich erneut anzumelden, verwenden Sie die Schaltfläche \"Anmelden\".',
                    'pl': '✅ Pomyślnie się wylogowałeś.\n\nTeraz jesteś zalogowany jako gość.\nAby zalogować się ponownie, użyj przycisku \"Zaloguj się\".'
                }
                
                await message.answer(
                    logout_success_texts.get(lang, logout_success_texts['en']),
                    reply_markup=get_main_menu_keyboard(lang, is_admin=False, is_guest=True)
                )
                await state.clear()
            
        elif any(keyword in message.text for keyword in no_keywords):
            # لغو خروج
            cancel_logout_texts = {
                'fa': '✅ خروج از حساب لغو شد.',
                'en': '✅ Logout cancelled.',
                'tr': '✅ Çıkış iptal edildi.',
                'ar': '✅ تم إلغاء تسجيل الخروج.',
                'ru': '✅ Выход отменен.',
                'es': '✅ Cierre de sesión cancelado.',
                'uk': '✅ Вихід скасовано.',
                'hi': '✅ लॉगआउट रद्द कर दिया गया।',
                'it': '✅ Logout annullato.',
                'fr': '✅ Déconnexion annulée.',
                'pt': '✅ Logout cancelado.',
                'id': '✅ Keluar dibatalkan.',
                'kk': '✅ Шығу болдырылмады.',
                'de': '✅ Abmeldung abgebrochen.',
                'pl': '✅ Wylogowanie anulowane.'
            }
            
            await message.answer(
                cancel_logout_texts.get(lang, cancel_logout_texts['en']),
                reply_markup=get_main_menu_keyboard(lang, is_admin=False, is_guest=False)
            )
            await state.clear()
        else:
            # پاسخ نامعتبر
            invalid_response_texts = {
                'fa': '❌ لطفاً یکی از دکمه‌های «بله» یا «خیر» را انتخاب کنید.',
                'en': '❌ Please select either \"Yes\" or \"No\" button.',
                'tr': '❌ Lütfen \"Evet\" veya \"Hayır\" düğmesini seçin.',
                'ar': '❌ الرجاء تحديد زر \"نعم\" أو \"لا\".',
                'ru': '❌ Пожалуйста, выберите кнопку \"Да\" или \"Нет\".',
                'es': '❌ Por favor selecciona el botón \"Sí\" o \"No\".',
                'uk': '❌ Будь ласка, виберіть кнопку \"Так\" або \"Ні\".',
                'hi': '❌ कृपया \"हाँ\" या \"नहीं\" बटन चुनें।',
                'it': '❌ Seleziona il pulsante \"Sì\" o \"No\".',
                'fr': '❌ Veuillez sélectionner le bouton \"Oui\" ou \"Non\".',
                'pt': '❌ Por favor, selecione o botão \"Sim\" ou \"Não\".',
                'id': '❌ Silakan pilih tombol \"Ya\" atau \"Tidak\".',
                'kk': '❌ \"Иә\" немесе \"Жоқ\" батырмасын таңдаңыз.',
                'de': '❌ Bitte wählen Sie die Schaltfläche \"Ja\" oder \"Nein\".',
                'pl': '❌ Proszę wybrać przycisk \"Tak\" lub \"Nie\".'
            }
            await message.answer(invalid_response_texts.get(lang, invalid_response_texts['en']))
