"""
Settings Handler - هندلر تنظیمات
مدیریت تنظیمات کاربر
"""

from aiogram import Router, F
from aiogram.types import Message, CallbackQuery, InlineKeyboardMarkup, InlineKeyboardButton, ReplyKeyboardMarkup, KeyboardButton
from aiogram.fsm.context import FSMContext
from aiogram.fsm.state import State, StatesGroup
from sqlalchemy import select
from database.database import db
from database.models import User, Preference
from localization.translator import get_text, get_available_languages, get_language_name
import logging

router = Router()
logger = logging.getLogger(__name__)

class SettingsStates(StatesGroup):
    changing_language = State()
    confirming_delete = State()

@router.message((F.text.contains("تنظیمات") | F.text.contains("Settings") | F.text.contains("Ayarlar") | F.text.contains("الإعدادات") | F.text.contains("Настройки") | F.text.contains("Paramètres") | F.text.contains("Configuración") | F.text.contains("Налаштування") | F.text.contains("सेटिंग्स") | F.text.contains("Impostazioni") | F.text.contains("Configurações") | F.text.contains("Pengaturan") | F.text.contains("Баптаулар") | F.text.contains("Einstellungen") | F.text.contains("Ustawienia")))
async def settings_menu(message: Message):
    """نمایش منوی تنظیمات"""
    user_id = message.from_user.id
    
    async with db.get_session() as session:
        result = await session.execute(
            select(User).where(User.id == user_id)
        )
        user = result.scalar_one_or_none()
        
        if not user:
            await message.answer(f"❌ {get_text('fa', 'must_register_first', default='ابتدا باید ثبت‌نام کنید.')}")
            return
        
        user_language = user.language_code or 'fa'
        text = f"⚙️ <b>{get_text(user_language, 'settings')}</b>\n\n"
        text += f"🌍 <b>{get_text(user_language, 'language')}:</b> {get_language_name(user_language)}\n"
        text += f"👤 <b>{get_text(user_language, 'account')}:</b> {get_text(user_language, 'active')}"
        
        # کیبورد ساده با 2 گزینه در یک ردیف
        keyboard = ReplyKeyboardMarkup(
            keyboard=[
                [
                    KeyboardButton(text=f"🌍 {get_text(user_language, 'change_language', default='تغییر زبان')}"),
                    KeyboardButton(text=f"🗑️ {get_text(user_language, 'delete_account', default='حذف حساب')}")
                ],
                [
                    KeyboardButton(text=f"🔙 {get_text(user_language, 'back_to_menu', default='بازگشت به منو')}")
                ]
            ],
            resize_keyboard=True,
            one_time_keyboard=False
        )
        
        await message.answer(text, reply_markup=keyboard, parse_mode='HTML')

# Message handlers برای دکمه‌های کیبورد کاستوم

@router.message(F.text.contains("🌍"))
async def change_language_message(message: Message, state: FSMContext):
    """تغییر زبان از طریق message"""
    await change_language_action(message, state)

@router.message(F.text.contains("🗑️"))
async def delete_account_message(message: Message, state: FSMContext):
    """حذف حساب از طریق message"""
    await delete_account_action(message, state)

@router.message(F.text.contains("🔙"))
async def back_to_main_menu_settings(message: Message):
    """بازگشت به منوی اصلی"""
    from handlers.registration import get_main_menu_keyboard
    import os
    
    user_id = message.from_user.id
    
    async with db.get_session() as session:
        result = await session.execute(select(User).where(User.id == user_id))
        user = result.scalar_one_or_none()
        
        if not user:
            await message.answer(f"❌ {get_text('fa', 'must_register_first', default='ابتدا باید ثبت‌نام کنید.')}")
            return
        
        user_language = user.language_code or 'fa'
    
    # بررسی ادمین بودن
    ADMIN_USER_IDS = [int(uid) for uid in os.getenv('ADMIN_USER_IDS', '').split(',') if uid]
    is_admin = user_id in ADMIN_USER_IDS
    
    await message.answer(
        f"🏠 {get_text(user_language, 'back_to_menu', default='بازگشت به منوی اصلی')}",
        reply_markup=get_main_menu_keyboard(user_language, is_admin=is_admin)
    )

@router.message(F.text.contains("🔄"))
async def start_again_handler(message: Message, state: FSMContext):
    """شروع دوباره - اجرای دستور /start"""
    from handlers.registration import start_command
    await start_command(message, state)

# توابع کمکی

async def change_language_action(message: Message, state: FSMContext):
    """عملکرد تغییر زبان"""
    # دریافت زبان فعلی کاربر
    user_id = message.from_user.id
    async with db.get_session() as session:
        result = await session.execute(select(User).where(User.id == user_id))
        user = result.scalar_one_or_none()
        current_language = user.language_code if user else 'fa'
    
    languages = get_available_languages()
    
    # ایجاد کیبورد زبان‌ها (3 تا در هر سطر)
    keyboard_rows = []
    lang_items = list(languages.items())
    
    for i in range(0, len(lang_items), 3):
        row = []
        for j in range(3):
            if i + j < len(lang_items):
                lang_code, lang_name = lang_items[i + j]
                row.append(KeyboardButton(text=lang_name))
        keyboard_rows.append(row)
    
    # اضافه کردن دکمه بازگشت چندزبانه
    back_text = f"🔙 {get_text(current_language, 'back_to_settings', default='بازگشت به تنظیمات')}"
    keyboard_rows.append([KeyboardButton(text=back_text)])
    
    language_keyboard = ReplyKeyboardMarkup(
        keyboard=keyboard_rows,
        resize_keyboard=True
    )
    
    await message.answer(
        f"🌍 <b>{get_text(current_language, 'select_language', default='انتخاب زبان')}</b>\n\n"
        f"{get_text(current_language, 'select_your_language', default='زبان مورد نظر خود را انتخاب کنید:')}",
        reply_markup=language_keyboard,
        parse_mode='HTML'
    )
    await state.set_state(SettingsStates.changing_language)

async def delete_account_action(message: Message, state: FSMContext):
    """عملکرد حذف حساب با تأییدیه کامل"""
    user_id = message.from_user.id
    
    async with db.get_session() as session:
        result = await session.execute(select(User).where(User.id == user_id))
        user = result.scalar_one_or_none()
        
        if not user:
            await message.answer(f"❌ {get_text('fa', 'must_register_first', default='ابتدا باید ثبت‌نام کنید.')}")
            return
        
        user_language = user.language_code or 'fa'
        
        # متن هشدار کامل
        warning_text = f"🗑️ <b>{get_text(user_language, 'delete_account', default='حذف حساب کاربری')}</b>\n\n"
        warning_text += f"⚠️ <b>{get_text(user_language, 'warning', default='هشدار مهم')}!</b>\n\n"
        warning_text += f"{get_text(user_language, 'delete_warning_message', default='با حذف حساب کاربری، تمام اطلاعات زیر به صورت دائمی حذف خواهند شد')}:\n\n"
        
        warning_text += f"👤 <b>{get_text(user_language, 'profile_info', default='اطلاعات پروفایل')}:</b>\n"
        warning_text += f"   • {get_text(user_language, 'profile_data_items', default='نام، سن، جنسیت')}\n"
        warning_text += f"   • {get_text(user_language, 'bio_interests_items', default='بیو و علایق')}\n"
        warning_text += f"   • {get_text(user_language, 'location_items', default='کشور و شهر')}\n"
        warning_text += f"   • {get_text(user_language, 'all_photos_items', default='تمام عکس‌های آپلود شده')}\n\n"
        
        warning_text += f"💬 <b>{get_text(user_language, 'chats_matches', default='چت‌ها و مچ‌ها')}:</b>\n"
        warning_text += f"   • {get_text(user_language, 'all_matches_items', default='تمام مچ‌های شما')}\n"
        warning_text += f"   • {get_text(user_language, 'all_messages_items', default='تمام پیام‌های ارسالی و دریافتی')}\n"
        warning_text += f"   • {get_text(user_language, 'likes_history_items', default='تاریخچه لایک‌ها')}\n\n"
        
        warning_text += f"💎 <b>{get_text(user_language, 'premium_subscription', default='اشتراک پریمیوم')}:</b>\n"
        if user.is_premium:
            warning_text += f"   • ⚠️ {get_text(user_language, 'premium_active', default='شما اشتراک پریمیوم فعال دارید')}\n"
            warning_text += f"   • {get_text(user_language, 'premium_will_cancel', default='با حذف حساب، اشتراک باطل می‌شود')}\n"
            warning_text += f"   • {get_text(user_language, 'no_refund', default='هزینه پرداختی قابل بازگشت نیست')}\n\n"
        else:
            warning_text += f"   • {get_text(user_language, 'no_premium', default='شما اشتراک پریمیوم ندارید')}\n\n"
        
        warning_text += f"📊 <b>{get_text(user_language, 'statistics', default='آمار و تنظیمات')}:</b>\n"
        warning_text += f"   • {get_text(user_language, 'search_settings_items', default='تمام تنظیمات جستجو')}\n"
        warning_text += f"   • {get_text(user_language, 'views_likes_stats', default='آمار بازدیدها و لایک‌ها')}\n"
        warning_text += f"   • {get_text(user_language, 'activity_history_items', default='تاریخچه فعالیت‌ها')}\n\n"
        
        warning_text += f"🔴 <b>{get_text(user_language, 'irreversible', default='این عمل غیرقابل بازگشت است')}!</b>\n"
        warning_text += f"❌ {get_text(user_language, 'no_data_recovery', default='پس از حذف، امکان بازیابی اطلاعات وجود ندارد.')}\n\n"
        warning_text += f"❓ {get_text(user_language, 'confirm_delete_question', default='آیا مطمئن هستید که می‌خواهید حساب خود را حذف کنید؟')}"
        
        # کیبورد تأیید/لغو
        confirm_keyboard = ReplyKeyboardMarkup(
            keyboard=[
                [
                    KeyboardButton(text=f"✅ {get_text(user_language, 'yes_delete', default='بله، حذف کن')}"),
                    KeyboardButton(text=f"❌ {get_text(user_language, 'no_cancel', default='خیر، انصراف')}")
                ]
            ],
            resize_keyboard=True,
            one_time_keyboard=True
        )
        
        await message.answer(warning_text, reply_markup=confirm_keyboard, parse_mode='HTML')
        await state.set_state(SettingsStates.confirming_delete)

# State handlers
@router.message(SettingsStates.changing_language)
async def save_language(message: Message, state: FSMContext):
    """ذخیره زبان جدید"""
    user_id = message.from_user.id
    
    # دریافت زبان فعلی کاربر
    async with db.get_session() as session:
        result = await session.execute(select(User).where(User.id == user_id))
        user = result.scalar_one_or_none()
        
        if not user:
            await message.answer(f"❌ {get_text('fa', 'language_change_error', default='خطا در تغییر زبان.')}")
            await state.clear()
            return
        
        current_language = user.language_code or 'fa'
    
    # بررسی دکمه بازگشت چندزبانه
    if "🔙" in message.text and ("بازگشت" in message.text or "Back" in message.text or "Dön" in message.text or "العودة" in message.text or "Назад" in message.text or "Volver" in message.text or "Retour" in message.text or "Torna" in message.text or "Voltar" in message.text or "Kembali" in message.text or "оралу" in message.text or "Zurück" in message.text or "Powrót" in message.text or "वापस" in message.text):
        await state.clear()
        await settings_menu(message)
        return
    
    # پیدا کردن کد زبان بر اساس نام
    languages = get_available_languages()
    selected_lang_code = None
    
    for lang_code, lang_name in languages.items():
        if message.text == lang_name:
            selected_lang_code = lang_code
            break
    
    # اگر زبان نامعتبر است
    if not selected_lang_code:
        # بررسی اینکه آیا زبان از قبل انتخاب شده است یا نه
        if current_language:
            # زبان از قبل انتخاب شده - حفظ زبان قبلی و بازگشت به تنظیمات
            await state.clear()
            await settings_menu(message)
            return
        else:
            # زبان هنوز انتخاب نشده - نمایش پیام خطا
            await message.answer(f"❌ {get_text(current_language, 'invalid_language', default='زبان نامعتبر است. لطفاً از دکمه‌های موجود استفاده کنید.')}")
            return
    
    # ذخیره زبان جدید
    async with db.get_session() as session:
        result = await session.execute(select(User).where(User.id == user_id))
        user = result.scalar_one_or_none()
        
        if user:
            user.language_code = selected_lang_code
            await session.commit()
            
            # Update bot menu for this user
            try:
                from handlers.bottom_menu import update_user_menu
                await update_user_menu(message.bot, user_id, selected_lang_code)
            except Exception as e:
                logger.error(f"Error updating user menu: {e}")
            
            # پیام تأیید به زبان جدید و بازگشت به تنظیمات
            await message.answer(
                f"✅ {get_text(selected_lang_code, 'language', default='زبان')} {get_text(selected_lang_code, 'changed', default='تغییر یافت')} "
                f"{get_language_name(selected_lang_code)}",
                parse_mode='HTML'
            )
            
            # بازگشت به منوی تنظیمات با زبان جدید
            await state.clear()
            await settings_menu(message)
            return
        else:
            await message.answer(f"❌ {get_text(current_language, 'language_change_error', default='خطا در تغییر زبان.')}")
    
    await state.clear()

# State handler for delete confirmation
@router.message(SettingsStates.confirming_delete)
async def confirm_delete_account(message: Message, state: FSMContext):
    """تأیید یا لغو حذف حساب"""
    user_id = message.from_user.id
    
    async with db.get_session() as session:
        result = await session.execute(select(User).where(User.id == user_id))
        user = result.scalar_one_or_none()
        
        if not user:
            await message.answer(f"❌ {get_text('fa', 'user_not_found', default='کاربر یافت نشد')}")
            await state.clear()
            return
        
        user_language = user.language_code or 'fa'
        
        # بررسی انتخاب کاربر
        if "✅" in message.text or "بله" in message.text or "Yes" in message.text or "Evet" in message.text or "نعم" in message.text or "Да" in message.text or "Sí" in message.text or "Так" in message.text or "हाँ" in message.text or "Sì" in message.text or "Oui" in message.text or "Sim" in message.text or "Ya" in message.text or "Иә" in message.text or "Ja" in message.text or "Tak" in message.text:
            # کاربر تأیید کرد - حذف حساب
            try:
                # حذف کامل از دیتابیس
                from sqlalchemy import delete
                from database.models import (
                    Swipe, Match, ChatMessage, Preference, Report, 
                    Payment, Event, Notification, ProfileView
                )
                
                # حذف تمام swipe ها
                await session.execute(delete(Swipe).where(
                    (Swipe.from_user == user_id) | (Swipe.to_user == user_id)
                ))
                
                # حذف تمام match ها
                await session.execute(delete(Match).where(
                    (Match.user_a == user_id) | (Match.user_b == user_id)
                ))
                
                # حذف تمام پیام‌ها
                await session.execute(delete(ChatMessage).where(
                    ChatMessage.sender_id == user_id
                ))
                
                # حذف گزارشات
                await session.execute(delete(Report).where(
                    (Report.reporter == user_id) | (Report.target == user_id)
                ))
                
                # حذف پرداخت‌ها
                await session.execute(delete(Payment).where(Payment.user_id == user_id))
                
                # حذف رویدادها
                await session.execute(delete(Event).where(Event.user_id == user_id))
                
                # حذف نوتیفیکیشن‌ها
                await session.execute(delete(Notification).where(
                    (Notification.user_id == user_id) | (Notification.related_user_id == user_id)
                ))
                
                # حذف بازدیدهای پروفایل
                await session.execute(delete(ProfileView).where(
                    (ProfileView.viewer_id == user_id) | (ProfileView.viewed_user_id == user_id)
                ))
                
                # حذف preferences
                await session.execute(delete(Preference).where(Preference.user_id == user_id))
                
                # حذف کاربر
                await session.delete(user)
                await session.commit()
                
                # پیام نهایی با کیبورد استارت
                goodbye_text = f"👋 <b>{get_text(user_language, 'goodbye', default='خداحافظ')}!</b>\n\n"
                goodbye_text += f"✅ {get_text(user_language, 'account_deleted', default='حساب کاربری شما با موفقیت حذف شد')}.\n\n"
                goodbye_text += f"💔 {get_text(user_language, 'sorry_to_see_you_go', default='متأسفیم که شما را از دست دادیم')}.\n"
                goodbye_text += f"🌹 {get_text(user_language, 'hope_to_see_again', default='امیدواریم دوباره به ما بپیوندید')}."
                
                # کیبورد استارت با متن ترجمه شده
                from aiogram.types import ReplyKeyboardMarkup, KeyboardButton
                start_keyboard = ReplyKeyboardMarkup(
                    keyboard=[
                        [KeyboardButton(text=f"🔄 {get_text(user_language, 'start_again', default='شروع دوباره')}")]
                    ],
                    resize_keyboard=True
                )
                
                await message.answer(goodbye_text, reply_markup=start_keyboard, parse_mode='HTML')
                await state.clear()
                
                logger.info(f"User {user_id} deleted their account successfully")
                
            except Exception as e:
                logger.error(f"Error deleting user {user_id}: {e}")
                await message.answer(
                    f"❌ {get_text(user_language, 'delete_account_error', default='خطا در حذف حساب. لطفاً با پشتیبانی تماس بگیرید.')}",
                    parse_mode='HTML'
                )
                await state.clear()
        
        else:
            # کاربر انصراف داد
            cancel_text = f"✅ <b>{get_text(user_language, 'cancelled', default='لغو شد')}</b>\n\n"
            cancel_text += f"{get_text(user_language, 'account_not_deleted', default='حساب کاربری شما حذف نشد')}.\n"
            cancel_text += f"{get_text(user_language, 'everything_normal', default='همه چیز مثل قبل است')}! 🌹"
            
            await state.clear()
            
            # بازگشت به منوی تنظیمات
            await settings_menu(message)

# State handlers



# ==================== پایان فایل ====================
# تمام callback handlerها حذف شدند - فقط کیبورد کاستوم استفاده می‌شود
