#!/usr/bin/env python3
"""سیستم ساده اتاق گفتگو - ایجاد گروه و ارسال لینک دعوت"""

from aiogram import Router, F, Bot
from aiogram.types import Message, InlineKeyboardMarkup, InlineKeyboardButton
from aiogram.fsm.context import FSMContext
from database.database import db
from database.models import User, Match, Swipe
from sqlalchemy import select, and_, or_
from datetime import datetime
import logging
import hashlib
import random

router = Router()
logger = logging.getLogger(__name__)

# ذخیره اطلاعات گروه‌های فعال
active_rooms = {}  # {room_id: {"group_id": id, "user1": id, "user2": id, "created": datetime}}

def generate_room_code() -> str:
    """ایجاد کد 6 رقمی برای اتاق"""
    return str(random.randint(100000, 999999))

async def create_chat_group(bot: Bot, user1_id: int, user2_id: int, user1_name: str, user2_name: str) -> tuple:
    """
    ایجاد گروه چت برای دو کاربر
    Returns: (success, group_link, room_code, error_message)
    """
    try:
        # ایجاد کد اتاق
        room_code = generate_room_code()
        group_title = f"💬 اتاق گفتگو #{room_code}"
        
        # ایجاد گروه با Bot API
        # توجه: بات باید دسترسی ایجاد گروه داشته باشد
        try:
            # ایجاد لینک دعوت عمومی
            # چون بات نمی‌تواند مستقیماً گروه بسازد، باید از روش دیگری استفاده کنیم
            
            # روش جایگزین: ایجاد لینک دعوت موقت
            invite_message = f"""
🎉 <b>اتاق گفتگو آماده است!</b>

📍 <b>کد اتاق:</b> <code>{room_code}</code>

👥 <b>شرکت‌کنندگان:</b>
• {user1_name}
• {user2_name}

📝 <b>دستورالعمل:</b>
1️⃣ یک گروه جدید در تلگرام بسازید
2️⃣ نام گروه را <code>{group_title}</code> بگذارید
3️⃣ بات @ChatMilosBot را به گروه اضافه کنید
4️⃣ کاربر مقابل را به گروه دعوت کنید
5️⃣ شروع به گفتگو کنید!

⚠️ <b>قوانین:</b>
• احترام متقابل را رعایت کنید
• از ارسال محتوای نامناسب خودداری کنید
• در صورت تخلف، گزارش دهید

💡 برای خروج از اتاق، دستور /leave را در گروه ارسال کنید.
"""
            
            # ذخیره اطلاعات اتاق
            active_rooms[room_code] = {
                "user1": user1_id,
                "user2": user2_id,
                "created": datetime.now(),
                "status": "pending"
            }
            
            return True, None, room_code, invite_message
            
        except Exception as e:
            logger.error(f"Error creating group: {e}")
            return False, None, None, f"خطا در ایجاد گروه: {str(e)}"
            
    except Exception as e:
        logger.error(f"Unexpected error: {e}")
        return False, None, None, f"خطای غیرمنتظره: {str(e)}"

async def send_chat_invitation(bot: Bot, user_id: int, target_user_id: int, room_code: str, invite_message: str):
    """ارسال دعوتنامه چت به هر دو کاربر"""
    
    # ارسال به کاربر اول
    keyboard1 = InlineKeyboardMarkup(inline_keyboard=[
        [InlineKeyboardButton(text="✅ متوجه شدم", callback_data=f"accept_chat_{room_code}")],
        [InlineKeyboardButton(text="❌ انصراف", callback_data=f"cancel_chat_{room_code}")]
    ])
    
    try:
        await bot.send_message(
            user_id,
            invite_message,
            parse_mode='HTML',
            reply_markup=keyboard1
        )
    except Exception as e:
        logger.error(f"Error sending invitation to user {user_id}: {e}")
    
    # ارسال به کاربر دوم
    keyboard2 = InlineKeyboardMarkup(inline_keyboard=[
        [InlineKeyboardButton(text="✅ متوجه شدم", callback_data=f"accept_chat_{room_code}")],
        [InlineKeyboardButton(text="❌ انصراف", callback_data=f"cancel_chat_{room_code}")]
    ])
    
    try:
        await bot.send_message(
            target_user_id,
            invite_message,
            parse_mode='HTML',
            reply_markup=keyboard2
        )
    except Exception as e:
        logger.error(f"Error sending invitation to user {target_user_id}: {e}")

async def check_chat_permission(user_id: int, target_user_id: int) -> tuple:
    """
    بررسی دسترسی چت بین دو کاربر
    Returns: (has_permission, reason)
    """
    async with db.get_session() as session:
        # بررسی مچ
        match_result = await session.execute(
            select(Match).where(
                or_(
                    and_(Match.user_a == user_id, Match.user_b == target_user_id),
                    and_(Match.user_a == target_user_id, Match.user_b == user_id)
                )
            )
        )
        match = match_result.scalar_one_or_none()
        
        if match and match.status == 1:
            return True, "match"
        
        # بررسی سوپر لایک از کاربر
        super_like_result = await session.execute(
            select(Swipe).where(
                and_(
                    Swipe.from_user == user_id,
                    Swipe.to_user == target_user_id,
                    Swipe.action == 2  # سوپر لایک
                )
            )
        )
        
        if super_like_result.scalar_one_or_none():
            return True, "superlike"
        
        # بررسی لایک متقابل
        like1_result = await session.execute(
            select(Swipe).where(
                and_(
                    Swipe.from_user == user_id,
                    Swipe.to_user == target_user_id,
                    Swipe.action == 1  # لایک
                )
            )
        )
        
        like2_result = await session.execute(
            select(Swipe).where(
                and_(
                    Swipe.from_user == target_user_id,
                    Swipe.to_user == user_id,
                    Swipe.action == 1  # لایک
                )
            )
        )
        
        if like1_result.scalar_one_or_none() and like2_result.scalar_one_or_none():
            return True, "mutual_like"
        
        return False, "no_permission"

# Callback handlers
@router.callback_query(F.data.startswith("accept_chat_"))
async def handle_accept_chat(callback: CallbackQuery):
    """پذیرش دعوت چت"""
    room_code = callback.data.split("_")[2]
    
    if room_code in active_rooms:
        active_rooms[room_code]["status"] = "active"
        await callback.answer("✅ اتاق گفتگو فعال شد!", show_alert=True)
        await callback.message.edit_text(
            callback.message.text + "\n\n✅ <b>وضعیت:</b> فعال",
            parse_mode='HTML'
        )
    else:
        await callback.answer("❌ اتاق گفتگو یافت نشد!", show_alert=True)

@router.callback_query(F.data.startswith("cancel_chat_"))
async def handle_cancel_chat(callback: CallbackQuery):
    """لغو دعوت چت"""
    room_code = callback.data.split("_")[2]
    
    if room_code in active_rooms:
        del active_rooms[room_code]
        await callback.answer("❌ اتاق گفتگو لغو شد!", show_alert=True)
        await callback.message.edit_text(
            "❌ اتاق گفتگو لغو شد.",
            parse_mode='HTML'
        )
    else:
        await callback.answer("❌ اتاق گفتگو یافت نشد!", show_alert=True)
