#!/usr/bin/env python3
"""
Migration script to add is_read field to chat_messages table
"""

import asyncio
import logging
from sqlalchemy import text
from database.database import db

logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)

async def add_is_read_field():
    """Add is_read field to chat_messages table"""
    try:
        async with db.get_session() as session:
            # Check if column already exists
            check_query = text("""
            SELECT COUNT(*) as count
            FROM INFORMATION_SCHEMA.COLUMNS 
            WHERE TABLE_SCHEMA = DATABASE()
            AND TABLE_NAME = 'chat_messages'
            AND COLUMN_NAME = 'is_read'
            """)
            
            result = await session.execute(check_query)
            count = result.scalar()
            
            if count > 0:
                logger.info("ℹ️  is_read field already exists in chat_messages table")
                return
            
            # Add is_read column
            alter_query = text("""
            ALTER TABLE chat_messages 
            ADD COLUMN is_read BOOLEAN DEFAULT FALSE
            """)
            
            await session.execute(alter_query)
            await session.commit()
            
            logger.info("✅ Successfully added is_read field to chat_messages table")
            
    except Exception as e:
        logger.error(f"❌ Error adding is_read field: {e}")
        raise

async def main():
    """Main function"""
    logger.info("🔄 Starting migration: Adding is_read field to chat_messages")
    
    # Initialize database
    await db.init()
    
    # Add is_read field
    await add_is_read_field()
    
    # Close database connection
    await db.close()
    
    logger.info("✅ Migration completed successfully!")

if __name__ == "__main__":
    asyncio.run(main())
