import os
import urllib.parse
from sqlalchemy.ext.asyncio import create_async_engine, AsyncSession
from sqlalchemy.orm import sessionmaker
from contextlib import asynccontextmanager
import redis.asyncio as redis
from typing import AsyncGenerator
from dotenv import load_dotenv

load_dotenv()

class Database:
    def __init__(self):
        self.engine = None
        self.async_session = None
        self.redis_pool = None

    async def init(self):
        # Try separate DB variables first
        host = os.getenv('DB_HOST')
        port = os.getenv('DB_PORT', '3306')
        user = os.getenv('DB_USER')
        password = os.getenv('DB_PASSWORD')
        database = os.getenv('DB_NAME')
        
        if all([host, user, password, database]):
            # Build MySQL URL with aiomysql and URL encoding
            encoded_password = urllib.parse.quote(password, safe='')
            database_url = f"mysql+aiomysql://{user}:{encoded_password}@{host}:{port}/{database}"
            print(f"Using separate DB variables: {user}@{host}:{port}/{database}")
        else:
            # Fallback to DATABASE_URL
            database_url = os.getenv('DATABASE_URL')
            if not database_url:
                raise ValueError("Database configuration not found!")
            print("Using DATABASE_URL from .env")
        
        print(f"Connecting to database: {database_url.split('@')[0]}@...")
        
        # MySQL connection with aiomysql
        self.engine = create_async_engine(
            database_url,
            echo=False,
            pool_size=20,
            max_overflow=0,
            pool_pre_ping=True
        )

        self.async_session = sessionmaker(
            bind=self.engine,
            class_=AsyncSession,
            expire_on_commit=False
        )

        # Redis connection (optional)
        use_redis = os.getenv('USE_REDIS', 'false').lower() == 'true'
        if use_redis:
            redis_url = os.getenv('REDIS_URL', 'redis://localhost:6379/0')
            try:
                self.redis_pool = redis.from_url(redis_url)
                await self.redis_pool.ping()
                print("✅ Redis connected")
            except Exception as e:
                print(f"⚠️  Redis connection failed: {e}")
                self.redis_pool = None
        else:
            print("ℹ️  Redis disabled")

    @asynccontextmanager
    async def get_session(self) -> AsyncGenerator[AsyncSession, None]:
        async with self.async_session() as session:
            try:
                yield session
                await session.commit()
            except Exception:
                await session.rollback()
                raise
            finally:
                await session.close()

    async def close(self):
        if self.engine:
            await self.engine.dispose()
        if self.redis_pool:
            await self.redis_pool.close()

# Global database instance
db = Database()