#!/usr/bin/env python3
"""
Database initialization script for Sarium bot
Creates all tables and sets up initial data
"""

import asyncio
import os
import sys
from dotenv import load_dotenv
from sqlalchemy.ext.asyncio import create_async_engine
from database.models import Base

# Load environment variables
load_dotenv()

async def init_database():
    """Initialize database with all tables"""
    database_url = os.getenv('DATABASE_URL')
    if not database_url:
        print("ERROR: DATABASE_URL not found in environment variables")
        sys.exit(1)
    
    # Convert to async URL
    async_url = database_url.replace('postgresql://', 'postgresql+asyncpg://')
    
    # Create engine
    engine = create_async_engine(async_url, echo=True)
    
    try:
        # Create all tables
        async with engine.begin() as conn:
            await conn.run_sync(Base.metadata.create_all)
        
        print("✅ Database tables created successfully!")
        
        # Create indexes for better performance
        async with engine.begin() as conn:
            await conn.execute(text("""
                CREATE INDEX IF NOT EXISTS idx_users_tg_id ON users(tg_id);
                CREATE INDEX IF NOT EXISTS idx_users_location ON users(loc_geohash);
                CREATE INDEX IF NOT EXISTS idx_users_active ON users(last_active);
                CREATE INDEX IF NOT EXISTS idx_matches_users ON matches(user_a, user_b);
                CREATE INDEX IF NOT EXISTS idx_matches_status ON matches(status);
                CREATE INDEX IF NOT EXISTS idx_swipes_users ON swipes(from_user, to_user);
                CREATE INDEX IF NOT EXISTS idx_reports_target ON reports(target);
            """))
        
        print("✅ Database indexes created successfully!")
        
    except Exception as e:
        print(f"❌ Error creating database: {e}")
        sys.exit(1)
    finally:
        await engine.dispose()

if __name__ == "__main__":
    from sqlalchemy import text
    asyncio.run(init_database())
