"""
Database migration script for adding premium and admin features
"""

import asyncio
import logging
from sqlalchemy import text
from database.database import db

logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)

async def migrate():
    """Run database migrations"""
    async with db.get_session() as session:
        try:
            # Add new columns to users table
            migrations = [
                # Premium features
                "ALTER TABLE users ADD COLUMN IF NOT EXISTS premium_until TIMESTAMP",
                "ALTER TABLE users ADD COLUMN IF NOT EXISTS super_likes_today SMALLINT DEFAULT 0",
                "ALTER TABLE users ADD COLUMN IF NOT EXISTS last_super_like_reset TIMESTAMP DEFAULT NOW()",
                "ALTER TABLE users ADD COLUMN IF NOT EXISTS boost_until TIMESTAMP",
                "ALTER TABLE users ADD COLUMN IF NOT EXISTS invisible_mode BOOLEAN DEFAULT FALSE",
                "ALTER TABLE users ADD COLUMN IF NOT EXISTS is_active BOOLEAN DEFAULT TRUE",
                "ALTER TABLE users ADD COLUMN IF NOT EXISTS banned_at TIMESTAMP",
                
                # Add indexes for better performance
                "CREATE INDEX IF NOT EXISTS idx_users_premium ON users(is_premium) WHERE is_premium = TRUE",
                "CREATE INDEX IF NOT EXISTS idx_users_active ON users(is_active) WHERE is_active = TRUE",
                "CREATE INDEX IF NOT EXISTS idx_users_boost ON users(boost_until) WHERE boost_until IS NOT NULL",
                "CREATE INDEX IF NOT EXISTS idx_users_banned ON users(is_banned) WHERE is_banned = TRUE",
                
                # Add super_like column to swipes table
                "ALTER TABLE swipes ADD COLUMN IF NOT EXISTS is_super_like BOOLEAN DEFAULT FALSE",
                "CREATE INDEX IF NOT EXISTS idx_swipes_super ON swipes(is_super_like) WHERE is_super_like = TRUE",
            ]
            
            for migration in migrations:
                try:
                    await session.execute(text(migration))
                    logger.info(f"✅ Executed: {migration[:50]}...")
                except Exception as e:
                    logger.warning(f"⚠️ Skipped (may already exist): {migration[:50]}... - {str(e)}")
            
            await session.commit()
            logger.info("✅ All migrations completed successfully!")
            
        except Exception as e:
            logger.error(f"❌ Migration failed: {str(e)}")
            await session.rollback()
            raise

if __name__ == "__main__":
    asyncio.run(migrate())
