#!/usr/bin/env python3
"""
🚨 URGENT FIX: Chat Messages Database Schema
حل فوری مشکل جدول chat_messages
"""

import asyncio
import sys
import os
from dotenv import load_dotenv

# Load environment variables
load_dotenv()

# Add the project root to the Python path
sys.path.insert(0, os.path.dirname(os.path.abspath(__file__)))

from database.database import db
from sqlalchemy import text

async def fix_chat_messages_table():
    """Fix chat_messages table structure"""
    
    print("🔧 Starting urgent fix for chat_messages table...")
    
    try:
        # Initialize database connection first
        await db.init()
        print("✅ Database connection initialized")
        
        async with db.get_session() as session:
            # Check current table structure
            print("📋 Checking current table structure...")
            result = await session.execute(text("DESCRIBE chat_messages"))
            columns = result.fetchall()
            
            print("Current columns:")
            for col in columns:
                print(f"  - {col[0]} ({col[1]})")
            
            # Check if is_read column exists
            column_names = [col[0] for col in columns]
            
            if 'is_read' in column_names:
                print("⚠️  Found 'is_read' column - removing it...")
                await session.execute(text("ALTER TABLE chat_messages DROP COLUMN is_read"))
                print("✅ Removed 'is_read' column")
            
            if 'from_user' in column_names:
                print("⚠️  Found 'from_user' column - renaming to 'sender_id'...")
                await session.execute(text("ALTER TABLE chat_messages CHANGE from_user sender_id BIGINT NOT NULL"))
                print("✅ Renamed 'from_user' to 'sender_id'")
            
            if 'to_user' in column_names:
                print("⚠️  Found 'to_user' column - removing it...")
                await session.execute(text("ALTER TABLE chat_messages DROP COLUMN to_user"))
                print("✅ Removed 'to_user' column")
            
            if 'message_text' in column_names:
                print("⚠️  Found 'message_text' column - renaming to 'content'...")
                await session.execute(text("ALTER TABLE chat_messages CHANGE message_text content TEXT"))
                print("✅ Renamed 'message_text' to 'content'")
            
            if 'file_id' in column_names:
                print("⚠️  Found 'file_id' column - removing it...")
                await session.execute(text("ALTER TABLE chat_messages DROP COLUMN file_id"))
                print("✅ Removed 'file_id' column")
            
            # Ensure correct structure
            print("🔧 Ensuring correct table structure...")
            
            # Check if all required columns exist
            final_result = await session.execute(text("DESCRIBE chat_messages"))
            final_columns = [col[0] for col in final_result.fetchall()]
            
            required_columns = ['id', 'match_id', 'sender_id', 'message_type', 'content', 'created_at']
            
            for req_col in required_columns:
                if req_col not in final_columns:
                    print(f"❌ Missing required column: {req_col}")
                else:
                    print(f"✅ Found required column: {req_col}")
            
            await session.commit()
            print("💾 Changes committed to database")
            
            # Final verification
            print("\n📋 Final table structure:")
            final_check = await session.execute(text("DESCRIBE chat_messages"))
            final_cols = final_check.fetchall()
            
            for col in final_cols:
                print(f"  ✅ {col[0]} ({col[1]})")
            
            print("\n🎉 Chat messages table fixed successfully!")
            print("🔄 Please restart the bot to apply changes.")
            
    except Exception as e:
        print(f"❌ Error fixing table: {e}")
        import traceback
        traceback.print_exc()
    finally:
        # Close database connection
        try:
            await db.close()
            print("🔌 Database connection closed")
        except:
            pass

if __name__ == "__main__":
    print("🚨 URGENT: Fixing chat_messages table...")
    print("⚠️  This will modify the database structure!")
    
    confirm = input("Continue? (yes/no): ")
    if confirm.lower() in ['yes', 'y']:
        asyncio.run(fix_chat_messages_table())
    else:
        print("❌ Operation cancelled")
