#!/usr/bin/env python3
"""Real Chat functionality for ChatMilos Bot"""

from aiogram import Router, F, Bot
from aiogram.types import Message, CallbackQuery, InlineKeyboardMarkup, InlineKeyboardButton
from aiogram.fsm.context import FSMContext
from aiogram.fsm.state import State, StatesGroup
from sqlalchemy import select, and_, or_, desc
from database.database import db
from database.models import User, Match
from database.models import ChatMessage
import logging
import os
from datetime import datetime

router = Router()
logger = logging.getLogger(__name__)

# Bot instance برای ارسال پیام به کاربران دیگر
bot_token = os.getenv('BOT_TOKEN', '8145304128:AAFkpxUUUpI68vYeQWdVo0JdxnmLbWXd_MQ')
bot = Bot(token=bot_token)

class ChatStates(StatesGroup):
    in_chat = State()
    typing_message = State()

@router.callback_query(F.data.startswith("start_chat_"))
async def start_real_chat(callback: CallbackQuery, state: FSMContext):
    """Start real chat with matched user"""
    target_user_id = int(callback.data.split("_")[-1])
    current_user_id = callback.from_user.id
    
    async with db.get_session() as session:
        # Check if users exist
        current_user_result = await session.execute(
            select(User).where(User.user_id == current_user_id)
        )
        current_user = current_user_result.scalar_one_or_none()
        
        target_user_result = await session.execute(
            select(User).where(User.user_id == target_user_id)
        )
        target_user = target_user_result.scalar_one_or_none()
        
        if not current_user or not target_user:
            await callback.message.edit_text("❌ خطا: کاربر پیدا نشد!")
            await callback.answer()
            return
        
        # Create or find existing match
        match_result = await session.execute(
            select(Match).where(
                or_(
                    and_(Match.user_a == current_user_id, Match.user_b == target_user_id),
                    and_(Match.user_a == target_user_id, Match.user_b == current_user_id)
                )
            )
        )
        match = match_result.scalar_one_or_none()
        
        if not match:
            # Create new match
            match = Match(
                user_a=current_user_id,
                user_b=target_user_id,
                status=1  # connected
            )
            session.add(match)
            await session.commit()
        
        # Set chat state
        await state.update_data(
            chat_partner_id=target_user_id,
            chat_partner_name=target_user.username or target_user.first_name or "کاربر",
            match_id=match.id
        )
        await state.set_state(ChatStates.in_chat)
        
        chat_text = f"""
💬 <b>گفتگو با {target_user.username or target_user.first_name or 'کاربر'}</b>

🎉 شما الان می‌توانید با این کاربر گفتگو کنید!

💡 <b>نکات مهم:</b>
• پیام‌های شما مستقیماً ارسال می‌شود
• محترمانه و مؤدبانه باشید
• از اطلاعات شخصی محافظت کنید
• هر گونه رفتار نامناسب را گزارش دهید

✍️ <b>پیام خود را بنویسید:</b>
"""
        
        keyboard = InlineKeyboardMarkup(inline_keyboard=[
            [
                InlineKeyboardButton(text="👤 پروفایل", callback_data=f"view_profile_{target_user_id}"),
                InlineKeyboardButton(text="🚫 مسدود کردن", callback_data=f"block_user_{target_user_id}")
            ],
            [
                InlineKeyboardButton(text="📝 گزارش", callback_data=f"report_user_{target_user_id}"),
                InlineKeyboardButton(text="❌ پایان گفتگو", callback_data="end_chat")
            ]
        ])
        
        await callback.message.edit_text(chat_text, reply_markup=keyboard, parse_mode='HTML')
        await callback.answer("گفتگو شروع شد! 💬")
        
        # Note: Message read status tracking removed for simplicity

@router.message(ChatStates.in_chat)
async def handle_chat_message(message: Message, state: FSMContext):
    """Handle real messages in chat"""
    data = await state.get_data()
    partner_id = data.get('chat_partner_id')
    partner_name = data.get('chat_partner_name')
    match_id = data.get('match_id')
    current_user_id = message.from_user.id
    
    if not partner_id or not match_id:
        await message.answer("❌ خطا در گفتگو. لطفاً دوباره شروع کنید.")
        await state.clear()
        return
    
    async with db.get_session() as session:
        # ذخیره پیام در دیتابیس
        chat_message = ChatMessage(
            match_id=match_id,
            sender_id=current_user_id,
            content=message.text,
            message_type='text'
        )
        session.add(chat_message)
        await session.commit()
        
        # گرفتن اطلاعات فرستنده برای ارسال به طرف مقابل
        sender_result = await session.execute(
            select(User).where(User.user_id == current_user_id)
        )
        sender = sender_result.scalar_one_or_none()
        sender_name = sender.first_name or sender.username or "کاربر" if sender else "کاربر"
    
    # تأیید ارسال به فرستنده
    await message.answer(
        f"📤 <b>پیام ارسال شد</b>\n\n"
        f"👤 به: {partner_name}\n"
        f"💬 متن: <i>{message.text}</i>\n"
        f"⏰ زمان: {datetime.now().strftime('%H:%M')}\n"
        f"✅ تحویل داده شد",
        parse_mode='HTML'
    )
    
    # ارسال واقعی پیام به طرف مقابل
    try:
        notification_text = f"""
🔔 <b>پیام جدید!</b>

👤 <b>از:</b> {sender_name}
💬 <b>پیام:</b> {message.text}
⏰ <b>زمان:</b> {datetime.now().strftime('%H:%M')}

💡 برای پاسخ دادن، روی دکمه زیر کلیک کنید:
"""
        
        keyboard = InlineKeyboardMarkup(inline_keyboard=[
            [InlineKeyboardButton(text="💬 پاسخ دادن", callback_data=f"start_chat_{current_user_id}")],
            [InlineKeyboardButton(text="👁️ مشاهده گفتگو", callback_data="view_my_chats")]
        ])
        
        await bot.send_message(
            chat_id=partner_id,
            text=notification_text,
            reply_markup=keyboard,
            parse_mode='HTML'
        )
        
        logger.info(f"Message sent from {current_user_id} to {partner_id}: {message.text}")
        
    except Exception as e:
        logger.error(f"Failed to send message to {partner_id}: {e}")
        await message.answer("⚠️ پیام ذخیره شد اما ممکن است طرف مقابل آن را فوراً دریافت نکند.")

@router.callback_query(F.data == "end_chat")
async def end_chat(callback: CallbackQuery, state: FSMContext):
    """End current chat"""
    await state.clear()
    await callback.message.edit_text(
        "❌ <b>گفتگو پایان یافت</b>\n\n"
        "🔄 برای شروع گفتگوی جدید، از منوی اصلی استفاده کنید."
    )
    await callback.answer("گفتگو پایان یافت")

@router.callback_query(F.data.startswith("view_profile_"))
async def view_chat_partner_profile(callback: CallbackQuery):
    """View chat partner's profile"""
    user_id = int(callback.data.split("_")[-1])
    
    async with db.get_session() as session:
        result = await session.execute(select(User).where(User.user_id == user_id))
        user = result.scalar_one_or_none()
        
        if not user:
            await callback.answer("کاربر پیدا نشد!")
            return
        
        age = user.age if user.age else "نامشخص"
        gender_display = {
            1: 'مرد',
            2: 'زن',
            3: 'غیره',
            '1': 'مرد',
            '2': 'زن',
            '3': 'غیره'
        }.get(user.gender, 'نامشخص')
        
        profile_text = f"""
👤 <b>پروفایل {user.username or user.first_name or 'کاربر'}</b>

🎂 <b>سن:</b> {age} ساله
⚧️ <b>جنسیت:</b> {gender_display}
📍 <b>شهر:</b> {user.city or 'نامشخص'}

💭 <b>بیو:</b>
{user.bio or 'بیو ندارد'}

🎯 <b>علایق:</b>
{user.interests or 'علایقی ثبت نشده'}
"""
        
        keyboard = InlineKeyboardMarkup(inline_keyboard=[
            [InlineKeyboardButton(text="🔙 بازگشت به گفتگو", callback_data="back_to_chat")]
        ])
        
        await callback.message.edit_text(profile_text, reply_markup=keyboard, parse_mode='HTML')
        await callback.answer()

@router.callback_query(F.data == "back_to_chat")
async def back_to_chat(callback: CallbackQuery):
    """Return to chat"""
    await callback.message.edit_text(
        "💬 <b>بازگشت به گفتگو</b>\n\n"
        "✍️ پیام خود را بنویسید:"
    )
    await callback.answer()

# Message handlers for different media types
@router.message(ChatStates.in_chat, F.photo)
async def handle_photo_message(message: Message, state: FSMContext):
    """Handle real photo messages in chat"""
    data = await state.get_data()
    partner_id = data.get('chat_partner_id')
    partner_name = data.get('chat_partner_name')
    match_id = data.get('match_id')
    current_user_id = message.from_user.id
    
    if not partner_id or not match_id:
        await message.answer("❌ خطا در گفتگو.")
        return
    
    # گرفتن بزرگترین سایز عکس
    photo = message.photo[-1]
    
    async with db.get_session() as session:
        # ذخیره پیام عکس در دیتابیس
        chat_message = ChatMessage(
            match_id=match_id,
            from_user=current_user_id,
            to_user=partner_id,
            message_text="📸 عکس",
            message_type='photo',
            file_id=photo.file_id,
            created_at=datetime.now()
        )
        session.add(chat_message)
        await session.commit()
        
        # گرفتن نام فرستنده
        sender_result = await session.execute(select(User).where(User.user_id == current_user_id))
        sender = sender_result.scalar_one_or_none()
        sender_name = sender.first_name or sender.username or "کاربر" if sender else "کاربر"
    
    # تأیید ارسال
    await message.answer(
        f"📸 <b>عکس ارسال شد</b>\n\n"
        f"👤 به: {partner_name}\n"
        f"⏰ زمان: {datetime.now().strftime('%H:%M')}\n"
        f"✅ تحویل داده شد",
        parse_mode='HTML'
    )
    
    # ارسال عکس به طرف مقابل
    try:
        await bot.send_photo(
            chat_id=partner_id,
            photo=photo.file_id,
            caption=f"📸 <b>عکس جدید از {sender_name}</b>\n⏰ {datetime.now().strftime('%H:%M')}",
            parse_mode='HTML',
            reply_markup=InlineKeyboardMarkup(inline_keyboard=[
                [InlineKeyboardButton(text="💬 پاسخ دادن", callback_data=f"start_chat_{current_user_id}")]
            ])
        )
    except Exception as e:
        logger.error(f"Failed to send photo to {partner_id}: {e}")

@router.message(ChatStates.in_chat, F.voice)
async def handle_voice_message(message: Message, state: FSMContext):
    """Handle real voice messages in chat"""
    data = await state.get_data()
    partner_id = data.get('chat_partner_id')
    partner_name = data.get('chat_partner_name')
    match_id = data.get('match_id')
    current_user_id = message.from_user.id
    
    if not partner_id or not match_id:
        await message.answer("❌ خطا در گفتگو.")
        return
    
    async with db.get_session() as session:
        # ذخیره پیام صوتی در دیتابیس
        chat_message = ChatMessage(
            match_id=match_id,
            from_user=current_user_id,
            to_user=partner_id,
            message_text=f"🎤 پیام صوتی ({message.voice.duration} ثانیه)",
            message_type='voice',
            file_id=message.voice.file_id,
            created_at=datetime.now()
        )
        session.add(chat_message)
        await session.commit()
        
        # گرفتن نام فرستنده
        sender_result = await session.execute(select(User).where(User.user_id == current_user_id))
        sender = sender_result.scalar_one_or_none()
        sender_name = sender.first_name or sender.username or "کاربر" if sender else "کاربر"
    
    # تأیید ارسال
    await message.answer(
        f"🎤 <b>پیام صوتی ارسال شد</b>\n\n"
        f"👤 به: {partner_name}\n"
        f"⏱️ مدت: {message.voice.duration} ثانیه\n"
        f"⏰ زمان: {datetime.now().strftime('%H:%M')}\n"
        f"✅ تحویل داده شد",
        parse_mode='HTML'
    )
    
    # ارسال پیام صوتی به طرف مقابل
    try:
        await bot.send_voice(
            chat_id=partner_id,
            voice=message.voice.file_id,
            caption=f"🎤 <b>پیام صوتی از {sender_name}</b>\n⏰ {datetime.now().strftime('%H:%M')}",
            parse_mode='HTML',
            reply_markup=InlineKeyboardMarkup(inline_keyboard=[
                [InlineKeyboardButton(text="💬 پاسخ دادن", callback_data=f"start_chat_{current_user_id}")]
            ])
        )
    except Exception as e:
        logger.error(f"Failed to send voice to {partner_id}: {e}")

@router.message(ChatStates.in_chat, F.sticker)
async def handle_sticker_message(message: Message, state: FSMContext):
    """Handle real sticker messages in chat"""
    data = await state.get_data()
    partner_id = data.get('chat_partner_id')
    partner_name = data.get('chat_partner_name')
    match_id = data.get('match_id')
    current_user_id = message.from_user.id
    
    if not partner_id or not match_id:
        await message.answer("❌ خطا در گفتگو.")
        return
    
    async with db.get_session() as session:
        # ذخیره استیکر در دیتابیس
        chat_message = ChatMessage(
            match_id=match_id,
            from_user=current_user_id,
            to_user=partner_id,
            message_text="😄 استیکر",
            message_type='sticker',
            file_id=message.sticker.file_id,
            created_at=datetime.now()
        )
        session.add(chat_message)
        await session.commit()
        
        # گرفتن نام فرستنده
        sender_result = await session.execute(select(User).where(User.user_id == current_user_id))
        sender = sender_result.scalar_one_or_none()
        sender_name = sender.first_name or sender.username or "کاربر" if sender else "کاربر"
    
    # تأیید ارسال
    await message.answer(
        f"😄 <b>استیکر ارسال شد</b>\n\n"
        f"👤 به: {partner_name}\n"
        f"⏰ زمان: {datetime.now().strftime('%H:%M')}\n"
        f"✅ تحویل داده شد",
        parse_mode='HTML'
    )
    
    # ارسال استیکر به طرف مقابل
    try:
        await bot.send_sticker(
            chat_id=partner_id,
            sticker=message.sticker.file_id,
            reply_markup=InlineKeyboardMarkup(inline_keyboard=[
                [InlineKeyboardButton(text="💬 پاسخ دادن", callback_data=f"start_chat_{current_user_id}")]
            ])
        )
        # ارسال پیام توضیحی
        await bot.send_message(
            chat_id=partner_id,
            text=f"😄 <b>استیکر از {sender_name}</b>\n⏰ {datetime.now().strftime('%H:%M')}",
            parse_mode='HTML'
        )
    except Exception as e:
        logger.error(f"Failed to send sticker to {partner_id}: {e}")
