#!/usr/bin/env python3
"""Notification system for ChatMilos Bot"""

from aiogram import Router, F, Bot
from aiogram.types import Message, CallbackQuery, InlineKeyboardMarkup, InlineKeyboardButton, ReplyKeyboardMarkup, KeyboardButton
from sqlalchemy import select
from database.database import db
from database.models import User
from database.simple_models import Notification
import logging
import json
import asyncio
import json
from datetime import datetime

router = Router()
logger = logging.getLogger(__name__)

class NotificationManager:
    def __init__(self, bot: Bot):
        self.bot = bot
        
    async def send_match_notification(self, user_id: int, match_user_name: str):
        """Send match notification"""
        try:
            notification_text = f"""
🎉 <b>مچ جدید!</b>

💕 شما با {match_user_name} مچ شدید!

🔥 الان می‌توانید شروع به گفتگو کنید.
"""
            
            keyboard = InlineKeyboardMarkup(inline_keyboard=[
                [InlineKeyboardButton(text="💬 شروع گفتگو", callback_data=f"start_chat_from_notif")]
            ])
            
            await self.bot.send_message(
                chat_id=user_id,
                text=notification_text,
                reply_markup=keyboard,
                parse_mode='HTML'
            )
            logger.info(f"Match notification sent to user {user_id}")
        except Exception as e:
            logger.error(f"Failed to send match notification to {user_id}: {e}")
    
    async def send_like_notification(self, user_id: int, liker_name: str):
        """Send like notification"""
        try:
            notification_text = f"""
💚 <b>لایک جدید!</b>

👤 {liker_name} شما را لایک کرد!

🌟 برای دیدن پروفایل و لایک متقابل، روی دکمه زیر کلیک کنید.
"""
            
            keyboard = InlineKeyboardMarkup(inline_keyboard=[
                [InlineKeyboardButton(text="👁️ مشاهده پروفایل", callback_data="view_liker_profile")]
            ])
            
            await self.bot.send_message(
                chat_id=user_id,
                text=notification_text,
                reply_markup=keyboard,
                parse_mode='HTML'
            )
            logger.info(f"Like notification sent to user {user_id}")
        except Exception as e:
            logger.error(f"Failed to send like notification to {user_id}: {e}")
    
    async def send_message_notification(self, user_id: int, sender_name: str, message_preview: str):
        """Send new message notification"""
        try:
            # Truncate message preview
            preview = message_preview[:50] + "..." if len(message_preview) > 50 else message_preview
            
            notification_text = f"""
💬 <b>پیام جدید!</b>

👤 از: {sender_name}
📝 پیام: <i>{preview}</i>

🔔 برای مشاهده و پاسخ، روی دکمه زیر کلیک کنید.
"""
            
            keyboard = InlineKeyboardMarkup(inline_keyboard=[
                [InlineKeyboardButton(text="💬 مشاهده پیام", callback_data="view_new_message")]
            ])
            
            await self.bot.send_message(
                chat_id=user_id,
                text=notification_text,
                reply_markup=keyboard,
                parse_mode='HTML'
            )
            logger.info(f"Message notification sent to user {user_id}")
        except Exception as e:
            logger.error(f"Failed to send message notification to {user_id}: {e}")
    
    async def send_profile_view_notification(self, user_id: int, viewer_name: str):
        """Send profile view notification"""
        try:
            notification_text = f"""
👁️ <b>بازدید پروفایل!</b>

👤 {viewer_name} پروفایل شما را دید!

🌟 شاید علاقه‌مند باشد. می‌خواهید پروفایل او را ببینید؟
"""
            
            keyboard = InlineKeyboardMarkup(inline_keyboard=[
                [InlineKeyboardButton(text="👁️ مشاهده پروفایل", callback_data="view_visitor_profile")]
            ])
            
            await self.bot.send_message(
                chat_id=user_id,
                text=notification_text,
                reply_markup=keyboard,
                parse_mode='HTML'
            )
            logger.info(f"Profile view notification sent to user {user_id}")
        except Exception as e:
            logger.error(f"Failed to send profile view notification to {user_id}: {e}")
    
    async def send_premium_expiry_notification(self, user_id: int, days_left: int):
        """Send premium expiry notification"""
        try:
            if days_left <= 0:
                notification_text = """
⭐ <b>اشتراک پریمیوم منقضی شد!</b>

😔 اشتراک پریمیوم شما به پایان رسید.

🔄 برای ادامه استفاده از مزایای پریمیوم، اشتراک خود را تمدید کنید.
"""
            else:
                notification_text = f"""
⚠️ <b>اشتراک پریمیوم به زودی منقضی می‌شود!</b>

📅 {days_left} روز تا پایان اشتراک باقی مانده.

🔄 برای تمدید اشتراک، روی دکمه زیر کلیک کنید.
"""
            
            keyboard = InlineKeyboardMarkup(inline_keyboard=[
                [InlineKeyboardButton(text="⭐ تمدید پریمیوم", callback_data="renew_premium")]
            ])
            
            await self.bot.send_message(
                chat_id=user_id,
                text=notification_text,
                reply_markup=keyboard,
                parse_mode='HTML'
            )
            logger.info(f"Premium expiry notification sent to user {user_id}")
        except Exception as e:
            logger.error(f"Failed to send premium expiry notification to {user_id}: {e}")

# Notification settings handlers
@router.callback_query(F.data.startswith("notif_"))
async def handle_notification_settings(callback: CallbackQuery):
    """Handle notification settings"""
    setting_type = callback.data.split("_")[1]
    user_id = callback.from_user.id
    
    async with db.get_session() as session:
        result = await session.execute(select(User).where(User.user_id == user_id))
        user = result.scalar_one_or_none()
        
        if not user:
            await callback.answer("کاربر پیدا نشد!")
            return
        
        # Get current settings
        try:
            settings = json.loads(user.settings) if user.settings else {}
        except:
            settings = {}
        
        notifications = settings.get('notifications', {
            'matches': True,
            'messages': True,
            'likes': True,
            'views': False
        })
        
        # Toggle the setting
        current_status = notifications.get(setting_type, True)
        notifications[setting_type] = not current_status
        settings['notifications'] = notifications
        
        # Update database
        user.settings = json.dumps(settings)
        await session.commit()
        
        # Create response
        setting_names = {
            'matches': 'مچ‌های جدید',
            'messages': 'پیام‌های جدید',
            'likes': 'لایک‌های جدید',
            'views': 'بازدید پروفایل'
        }
        
        setting_name = setting_names.get(setting_type, setting_type)
        new_status = "فعال" if notifications[setting_type] else "غیرفعال"
        
        # دکمه inline برای بازگشت به اعلان
        keyboard = InlineKeyboardMarkup(inline_keyboard=[
            [InlineKeyboardButton(text="🔙 بازگشت به اعلان", callback_data="back_to_notifications")]
        ])
        
        await callback.message.edit_text(
            f"✅ <b>تنظیمات به‌روزرسانی شد!</b>\n\n"
            f"🔔 {setting_name}: <b>{new_status}</b>\n\n"
            f"💡 تنظیمات شما ذخیره شد و از این پس اعمال می‌شود.",
            reply_markup=keyboard,
            parse_mode='HTML'
        )
        await callback.answer(f"{setting_name} {new_status} شد!")

@router.message(F.text == "🔔 نوتیفیکیشن‌ها")
async def show_notification_center(message: Message):
    """Show real notification center"""
    user_id = message.from_user.id
    
    async with db.get_session() as session:
        # بررسی وجود کاربر
        user_result = await session.execute(
            select(User).where(User.user_id == user_id)
        )
        user = user_result.scalar_one_or_none()
        
        if not user:
            await message.answer("❌ ابتدا باید ثبت‌نام کنید.")
            return
        
        # گرفتن اعلان‌های واقعی از دیتابیس
        notifications_result = await session.execute(
            select(Notification).where(
                Notification.user_id == user_id
            ).order_by(Notification.created_at.desc()).limit(10)
        )
        notifications = notifications_result.scalars().all()
        
        notif_text = "🔔 <b>مرکز اعلان‌ها</b>\n\n"
        
        if not notifications:
            notif_text += "📭 <b>هیچ اعلانی ندارید!</b>\n\n"
            notif_text += "💡 اعلان‌های جدید اینجا نمایش داده می‌شوند:\n"
            notif_text += "• 🎉 مچ‌های جدید\n"
            notif_text += "• 💚 لایک‌های دریافتی\n"
            notif_text += "• 💬 پیام‌های جدید\n"
            notif_text += "• 👁️ بازدید پروفایل\n\n"
        else:
            notif_text += f"📱 <b>{len(notifications)} اعلان اخیر:</b>\n\n"
            
            for notif in notifications:
                # محاسبه زمان
                time_diff = datetime.now() - notif.created_at
                if time_diff.days > 0:
                    time_str = f"{time_diff.days} روز پیش"
                elif time_diff.seconds > 3600:
                    time_str = f"{time_diff.seconds // 3600} ساعت پیش"
                else:
                    time_str = f"{time_diff.seconds // 60} دقیقه پیش"
                
                # آیکون بر اساس نوع
                type_icons = {
                    'match': '🎉',
                    'like': '💚',
                    'message': '💬',
                    'profile_view': '👁️',
                    'premium_expiry': '⭐'
                }
                icon = type_icons.get(notif.type, '🔔')
                
                # وضعیت خوانده شده
                read_status = "" if notif.is_read else "🔴 "
                
                notif_text += f"{read_status}{icon} <b>{notif.title}</b>\n"
                notif_text += f"📝 {notif.message}\n"
                notif_text += f"⏰ {time_str}\n\n"
        
        notif_text += "🔧 از دکمه‌های زیر برای مدیریت اعلان‌ها استفاده کنید."
        
        # کیبورد کاستوم برای نوتیفیکیشن‌ها
        keyboard = ReplyKeyboardMarkup(
            keyboard=[
                [
                    KeyboardButton(text="⚙️ تنظیمات اعلان"),
                    KeyboardButton(text="🗑️ پاک کردن همه")
                ],
                [
                    KeyboardButton(text="✅ علامت‌گذاری همه"),
                    KeyboardButton(text="🔄 به‌روزرسانی اعلان")
                ],
                [
                    KeyboardButton(text="🔙 بازگشت به منو")
                ]
            ],
            resize_keyboard=True,
            one_time_keyboard=False
        )
        
        await message.answer(notif_text, reply_markup=keyboard, parse_mode='HTML')

# Message handlers برای دکمه‌های کیبورد کاستوم
@router.message(F.text == "⚙️ تنظیمات اعلان")
async def notification_settings_message(message: Message):
    """Show notification settings via message"""
    await show_notification_settings_text(message)

@router.message(F.text == "🗑️ پاک کردن همه")
async def clear_all_notifications_message(message: Message):
    """Clear all notifications via message"""
    await clear_all_notifications_text(message)

@router.message(F.text == "✅ علامت‌گذاری همه")
async def mark_all_read_message(message: Message):
    """Mark all notifications as read via message"""
    await mark_all_read_text(message)

@router.message(F.text == "🔄 به‌روزرسانی اعلان")
async def refresh_notifications_message(message: Message):
    """Refresh notifications via message"""
    await show_notification_center(message)

# Message handlers برای تنظیمات اعلان
@router.message(F.text.startswith("🎉 مچ‌ها"))
async def toggle_match_notifications(message: Message):
    """تغییر وضعیت اعلان مچ‌ها"""
    await toggle_notification_setting(message, 'matches', 'مچ‌ها')

@router.message(F.text.startswith("💬 پیام‌ها"))
async def toggle_message_notifications(message: Message):
    """تغییر وضعیت اعلان پیام‌ها"""
    await toggle_notification_setting(message, 'messages', 'پیام‌ها')

@router.message(F.text == "💚 لایک‌ها ✅ فعال")
async def toggle_like_notifications_on(message: Message):
    """تغییر وضعیت اعلان لایک‌ها به غیرفعال"""
    await toggle_notification_setting(message, 'likes', 'لایک‌ها')

@router.message(F.text == "💚 لایک‌ها ❌ غیرفعال")
async def toggle_like_notifications_off(message: Message):
    """تغییر وضعیت اعلان لایک‌ها به فعال"""
    await toggle_notification_setting(message, 'likes', 'لایک‌ها')

@router.message(F.text.startswith("👁️ بازدیدها"))
async def toggle_view_notifications(message: Message):
    """تغییر وضعیت اعلان بازدیدها"""
    await toggle_notification_setting(message, 'views', 'بازدیدها')

@router.message(F.text == "🔙 بازگشت به اعلان")
async def back_to_notifications_from_settings(message: Message):
    """بازگشت به مرکز اعلان‌ها"""
    await show_notification_center(message)

@router.message(F.text == "🔙 بازگشت به منو")
async def back_to_main_menu(message: Message):
    """Back to main menu"""
    from handlers.registration import get_main_menu_keyboard
    
    # بررسی ادمین بودن
    admin_ids = [7203796090]  # ادمین ID از حافظه
    is_admin = message.from_user.id in admin_ids
    
    await message.answer(
        "🏠 بازگشت به منوی اصلی",
        reply_markup=get_main_menu_keyboard('fa', is_admin=is_admin)
    )

# توابع کمکی برای message handlers
async def show_notification_settings_text(message: Message):
    """Show notification settings as text message"""
    user_id = message.from_user.id
    
    async with db.get_session() as session:
        result = await session.execute(select(User).where(User.user_id == user_id))
        user = result.scalar_one_or_none()
        
        try:
            settings = json.loads(user.settings) if user.settings else {}
        except:
            settings = {}
        
        notifications = settings.get('notifications', {
            'matches': True,
            'messages': True,
            'likes': True,
            'views': False
        })
        
        settings_text = f"""
🔔 <b>تنظیمات دقیق اعلان‌ها</b>

📱 <b>وضعیت فعلی:</b>

🎉 مچ‌های جدید: {'✅ فعال' if notifications.get('matches', True) else '❌ غیرفعال'}
💬 پیام‌های جدید: {'✅ فعال' if notifications.get('messages', True) else '❌ غیرفعال'}
💚 لایک‌های جدید: {'✅ فعال' if notifications.get('likes', True) else '❌ غیرفعال'}
👁️ بازدید پروفایل: {'✅ فعال' if notifications.get('views', False) else '❌ غیرفعال'}
"""
        
        # کیبورد کاستوم برای تنظیمات اعلان
        settings_keyboard = ReplyKeyboardMarkup(
            keyboard=[
                [
                    KeyboardButton(text=f"🎉 مچ‌ها {'✅' if notifications.get('matches', True) else '❌'}"),
                    KeyboardButton(text=f"💬 پیام‌ها {'✅' if notifications.get('messages', True) else '❌'}")
                ],
                [
                    KeyboardButton(text=f"💚 لایک‌ها {'✅' if notifications.get('likes', True) else '❌'}"),
                    KeyboardButton(text=f"👁️ بازدیدها {'✅' if notifications.get('views', False) else '❌'}")
                ],
                [
                    KeyboardButton(text="🔙 بازگشت به اعلان")
                ]
            ],
            resize_keyboard=True,
            one_time_keyboard=False
        )
        
        await message.answer(settings_text, reply_markup=settings_keyboard, parse_mode='HTML')

async def toggle_notification_setting(message: Message, setting_type: str, setting_name: str):
    """تغییر وضعیت یک تنظیم اعلان"""
    user_id = message.from_user.id
    
    async with db.get_session() as session:
        result = await session.execute(select(User).where(User.user_id == user_id))
        user = result.scalar_one_or_none()
        
        if not user:
            await message.answer("❌ کاربر یافت نشد.")
            return
        
        try:
            settings = json.loads(user.settings) if user.settings else {}
        except:
            settings = {}
        
        notifications = settings.get('notifications', {
            'matches': True,
            'messages': True,
            'likes': True,
            'views': False
        })
        
        # تغییر وضعیت
        notifications[setting_type] = not notifications.get(setting_type, True)
        settings['notifications'] = notifications
        
        # ذخیره در دیتابیس
        user.settings = json.dumps(settings)
        await session.commit()
        
        new_status = "فعال" if notifications[setting_type] else "غیرفعال"
        
        await message.answer(
            f"✅ اعلان {setting_name} {new_status} شد!",
            parse_mode='HTML'
        )
        
        # بازگشت به تنظیمات
        await show_notification_settings_text(message)

async def clear_all_notifications_text(message: Message):
    """Clear all notifications as text message"""
    user_id = message.from_user.id
    
    async with db.get_session() as session:
        # حذف همه اعلان‌های کاربر
        await session.execute(
            Notification.__table__.delete().where(
                Notification.user_id == user_id
            )
        )
        await session.commit()
    
    await message.answer(
        "🗑️ <b>همه اعلان‌ها پاک شدند!</b>\n\n"
        "✨ صندوق اعلان‌های شما خالی است.\n\n"
        "🔔 اعلان‌های جدید اینجا نمایش داده خواهند شد.",
        parse_mode='HTML'
    )

async def mark_all_read_text(message: Message):
    """Mark all notifications as read as text message"""
    user_id = message.from_user.id
    
    async with db.get_session() as session:
        # علامت‌گذاری همه اعلان‌ها به عنوان خوانده شده
        await session.execute(
            Notification.__table__.update().where(
                Notification.user_id == user_id
            ).values(is_read=True)
        )
        await session.commit()
    
    await message.answer(
        "✅ <b>همه اعلان‌ها خوانده شدند!</b>\n\n"
        "📖 تمام اعلان‌های شما به عنوان خوانده شده علامت‌گذاری شدند.\n\n"
        "🔔 اعلان‌های جدید با علامت 🔴 نمایش داده خواهند شد.",
        parse_mode='HTML'
    )

# ==================== پایان فایل ====================
# تمام callback handlerها حذف شدند - فقط کیبورد کاستوم استفاده می‌شود
