#!/usr/bin/env python3
"""Photo upload functionality for ChatMilos Bot"""

from aiogram import Router, F
from aiogram.types import Message, CallbackQuery, InlineKeyboardMarkup, InlineKeyboardButton
from aiogram.fsm.context import FSMContext
from aiogram.fsm.state import State, StatesGroup
from sqlalchemy import select
from database.database import db
from database.models import User
import logging
import json

router = Router()
logger = logging.getLogger(__name__)

class PhotoStates(StatesGroup):
    uploading_profile_photo = State()
    uploading_additional_photo = State()

@router.callback_query(F.data == "upload_profile_photo")
async def start_photo_upload(callback: CallbackQuery, state: FSMContext):
    """Start profile photo upload process"""
    upload_text = """
📸 <b>آپلود عکس پروفایل</b>

📋 <b>راهنمای آپلود:</b>
• عکس باید واضح و با کیفیت باشد
• چهره شما به وضوح دیده شود
• بدون فیلتر یا ادیت زیاد
• حداکثر ۵ عکس می‌توانید آپلود کنید
• عکس اول، عکس اصلی پروفایل شماست

⚠️ <b>نکات مهم:</b>
• عکس‌های نامناسب حذف می‌شوند
• از عکس‌های شخصی دیگران استفاده نکنید
• عکس‌ها باید مطابق قوانین باشند

📷 <b>الان عکس خود را ارسال کنید:</b>
"""
    
    keyboard = InlineKeyboardMarkup(inline_keyboard=[
        [InlineKeyboardButton(text="❌ لغو", callback_data="cancel_photo_upload")]
    ])
    
    await callback.message.edit_text(upload_text, reply_markup=keyboard, parse_mode='HTML')
    await state.set_state(PhotoStates.uploading_profile_photo)
    await callback.answer()

@router.message(PhotoStates.uploading_profile_photo, F.photo)
async def handle_profile_photo_upload(message: Message, state: FSMContext):
    """Handle profile photo upload"""
    user_id = message.from_user.id
    
    # Get the largest photo size
    photo = message.photo[-1]
    file_id = photo.file_id
    
    async with db.get_session() as session:
        # Get user
        result = await session.execute(select(User).where(User.user_id == user_id))
        user = result.scalar_one_or_none()
        
        if not user:
            await message.answer("❌ خطا: کاربر پیدا نشد!")
            await state.clear()
            return
        
        # Update user photos
        try:
            current_photos = json.loads(user.photos) if user.photos else []
        except:
            current_photos = []
        
        # Add new photo to the beginning (main photo)
        current_photos.insert(0, {
            'file_id': file_id,
            'type': 'profile',
            'uploaded_at': str(message.date)
        })
        
        # Keep only 5 photos max
        current_photos = current_photos[:5]
        
        # Update database
        user.photos = json.dumps(current_photos)
        await session.commit()
        
        success_text = f"""
✅ <b>عکس پروفایل آپلود شد!</b>

📊 <b>وضعیت عکس‌ها:</b>
📷 تعداد عکس‌ها: {len(current_photos)}
🌟 عکس اصلی: تنظیم شد
⏰ زمان آپلود: الان

💡 <b>نکته:</b> عکس شما پس از بررسی فعال می‌شود.

🔄 می‌خواهید عکس دیگری اضافه کنید؟
"""
        
        keyboard = InlineKeyboardMarkup(inline_keyboard=[
            [
                InlineKeyboardButton(text="📷 عکس دیگر", callback_data="upload_another_photo"),
                InlineKeyboardButton(text="👁️ مشاهده عکس‌ها", callback_data="view_my_photos")
            ],
            [InlineKeyboardButton(text="✅ تمام", callback_data="finish_photo_upload")]
        ])
        
        await message.answer(success_text, reply_markup=keyboard, parse_mode='HTML')
        await state.clear()

@router.message(PhotoStates.uploading_profile_photo)
async def handle_invalid_photo_upload(message: Message, state: FSMContext):
    """Handle invalid photo upload attempts"""
    await message.answer(
        "❌ <b>فقط عکس ارسال کنید!</b>\n\n"
        "📷 لطفاً یک عکس واضح از خودتان ارسال کنید.\n"
        "یا روی 'لغو' کلیک کنید.",
        parse_mode='HTML'
    )

@router.callback_query(F.data == "upload_another_photo")
async def upload_another_photo(callback: CallbackQuery, state: FSMContext):
    """Upload additional photos"""
    await callback.message.edit_text(
        "📸 <b>آپلود عکس اضافی</b>\n\n"
        "📷 عکس بعدی خود را ارسال کنید:\n\n"
        "💡 این عکس به عنوان عکس اضافی ذخیره می‌شود.",
        parse_mode='HTML'
    )
    await state.set_state(PhotoStates.uploading_additional_photo)
    await callback.answer()

@router.message(PhotoStates.uploading_additional_photo, F.photo)
async def handle_additional_photo_upload(message: Message, state: FSMContext):
    """Handle additional photo upload"""
    user_id = message.from_user.id
    photo = message.photo[-1]
    file_id = photo.file_id
    
    async with db.get_session() as session:
        result = await session.execute(select(User).where(User.user_id == user_id))
        user = result.scalar_one_or_none()
        
        if not user:
            await message.answer("❌ خطا: کاربر پیدا نشد!")
            await state.clear()
            return
        
        try:
            current_photos = json.loads(user.photos) if user.photos else []
        except:
            current_photos = []
        
        if len(current_photos) >= 5:
            await message.answer(
                "⚠️ <b>حداکثر ۵ عکس مجاز است!</b>\n\n"
                "برای اضافه کردن عکس جدید، ابتدا یکی از عکس‌های قبلی را حذف کنید.",
                parse_mode='HTML'
            )
            await state.clear()
            return
        
        # Add new photo
        current_photos.append({
            'file_id': file_id,
            'type': 'additional',
            'uploaded_at': str(message.date)
        })
        
        user.photos = json.dumps(current_photos)
        await session.commit()
        
        await message.answer(
            f"✅ <b>عکس اضافی آپلود شد!</b>\n\n"
            f"📊 تعداد کل عکس‌ها: {len(current_photos)}\n\n"
            f"🔄 می‌خواهید عکس دیگری اضافه کنید؟",
            reply_markup=InlineKeyboardMarkup(inline_keyboard=[
                [
                    InlineKeyboardButton(text="📷 عکس دیگر", callback_data="upload_another_photo"),
                    InlineKeyboardButton(text="✅ تمام", callback_data="finish_photo_upload")
                ]
            ]),
            parse_mode='HTML'
        )
        await state.clear()

@router.callback_query(F.data == "view_my_photos")
async def view_user_photos(callback: CallbackQuery):
    """View user's uploaded photos"""
    user_id = callback.from_user.id
    
    async with db.get_session() as session:
        result = await session.execute(select(User).where(User.user_id == user_id))
        user = result.scalar_one_or_none()
        
        if not user:
            await callback.answer("کاربر پیدا نشد!")
            return
        
        try:
            photos = json.loads(user.photos) if user.photos else []
        except:
            photos = []
        
        if not photos:
            await callback.message.edit_text(
                "📷 <b>هنوز عکسی آپلود نکرده‌اید!</b>\n\n"
                "برای آپلود عکس، روی دکمه زیر کلیک کنید:",
                reply_markup=InlineKeyboardMarkup(inline_keyboard=[
                    [InlineKeyboardButton(text="📸 آپلود عکس", callback_data="upload_profile_photo")]
                ]),
                parse_mode='HTML'
            )
            await callback.answer()
            return
        
        # حذف پیام قبلی
        try:
            await callback.message.delete()
        except:
            pass
        
        # نمایش عکس‌ها یکی یکی
        for i, photo_id in enumerate(photos, 1):
            # استخراج file_id از عکس
            if isinstance(photo_id, dict) and "file_id" in photo_id:
                file_id = photo_id["file_id"]
            elif isinstance(photo_id, str):
                file_id = photo_id
            else:
                file_id = str(photo_id)
            
            caption = f"📸 <b>عکس {i} از {len(photos)}</b>"
            if i == 1:
                caption += "\n🌟 <b>عکس اصلی پروفایل</b>"
            
            try:
                await callback.message.answer_photo(
                    photo=file_id,
                    caption=caption,
                    parse_mode='HTML'
                )
            except Exception as e:
                logger.error(f"Error sending photo {file_id}: {e}")
                await callback.message.answer(f"❌ خطا در نمایش عکس {i}")
        
        # نمایش منوی مدیریت
        photos_text = f"""
📸 <b>مدیریت عکس‌ها</b>

📊 <b>آمار:</b>
📷 تعداد عکس‌ها: {len(photos)}
🌟 عکس اصلی: {'✅ دارد' if photos else '❌ ندارد'}

💡 <b>نکته:</b> عکس اول، عکس اصلی پروفایل شماست.
"""
        
        keyboard = InlineKeyboardMarkup(inline_keyboard=[
            [
                InlineKeyboardButton(text="📷 عکس جدید", callback_data="upload_profile_photo"),
                InlineKeyboardButton(text="🗑️ حذف عکس", callback_data="delete_photos_menu")
            ],
            [InlineKeyboardButton(text="🔙 بازگشت به پروفایل", callback_data="show_profile")]
        ])
        
        await callback.message.answer(photos_text, reply_markup=keyboard, parse_mode='HTML')
        await callback.answer()

@router.callback_query(F.data == "delete_photos_menu")
async def delete_photos_menu(callback: CallbackQuery):
    """Show delete photos menu"""
    user_id = callback.from_user.id
    
    async with db.get_session() as session:
        result = await session.execute(select(User).where(User.user_id == user_id))
        user = result.scalar_one_or_none()
        
        try:
            photos = json.loads(user.photos) if user.photos else []
        except:
            photos = []
        
        if not photos:
            await callback.answer("عکسی برای حذف وجود ندارد!")
            return
        
        delete_text = f"""
🗑️ <b>حذف عکس‌ها</b>

📊 شما {len(photos)} عکس دارید.

⚠️ <b>هشدار:</b> حذف عکس غیرقابل برگشت است!

🔄 <b>گزینه‌های حذف:</b>
"""
        
        keyboard = InlineKeyboardMarkup(inline_keyboard=[
            [InlineKeyboardButton(text="🗑️ حذف عکس اصلی", callback_data="delete_main_photo")],
            [InlineKeyboardButton(text="🗑️ حذف همه عکس‌ها", callback_data="delete_all_photos")],
            [InlineKeyboardButton(text="🔙 بازگشت", callback_data="view_my_photos")]
        ])
        
        await callback.message.edit_text(delete_text, reply_markup=keyboard, parse_mode='HTML')
        await callback.answer()

@router.callback_query(F.data == "delete_main_photo")
async def delete_main_photo(callback: CallbackQuery):
    """Delete main profile photo"""
    user_id = callback.from_user.id
    
    async with db.get_session() as session:
        result = await session.execute(select(User).where(User.user_id == user_id))
        user = result.scalar_one_or_none()
        
        try:
            photos = json.loads(user.photos) if user.photos else []
        except:
            photos = []
        
        if photos:
            photos.pop(0)  # Remove first photo (main photo)
            user.photos = json.dumps(photos)
            await session.commit()
            
            await callback.message.edit_text(
                "✅ <b>عکس اصلی حذف شد!</b>\n\n"
                f"📊 تعداد عکس‌های باقی‌مانده: {len(photos)}",
                reply_markup=InlineKeyboardMarkup(inline_keyboard=[
                    [InlineKeyboardButton(text="👤 بازگشت به پروفایل", callback_data="show_profile")]
                ]),
                parse_mode='HTML'
            )
        else:
            await callback.message.edit_text(
                "❌ عکسی برای حذف وجود ندارد!",
                reply_markup=InlineKeyboardMarkup(inline_keyboard=[
                    [InlineKeyboardButton(text="👤 بازگشت به پروفایل", callback_data="show_profile")]
                ])
            )
        
        await callback.answer()

@router.callback_query(F.data == "delete_all_photos")
async def delete_all_photos(callback: CallbackQuery):
    """Delete all photos with confirmation"""
    confirm_text = """
⚠️ <b>تأیید حذف همه عکس‌ها</b>

🔥 آیا مطمئن هستید که می‌خواهید همه عکس‌هایتان را حذف کنید؟

❌ این عمل غیرقابل برگشت است!
"""
    
    keyboard = InlineKeyboardMarkup(inline_keyboard=[
        [
            InlineKeyboardButton(text="✅ بله، حذف کن", callback_data="confirm_delete_all"),
            InlineKeyboardButton(text="❌ خیر، لغو", callback_data="view_my_photos")
        ]
    ])
    
    await callback.message.edit_text(confirm_text, reply_markup=keyboard, parse_mode='HTML')
    await callback.answer()

@router.callback_query(F.data == "confirm_delete_all")
async def confirm_delete_all_photos(callback: CallbackQuery):
    """Confirm and delete all photos"""
    user_id = callback.from_user.id
    
    async with db.get_session() as session:
        result = await session.execute(select(User).where(User.user_id == user_id))
        user = result.scalar_one_or_none()
        
        user.photos = json.dumps([])
        await session.commit()
        
        await callback.message.edit_text(
            "✅ <b>همه عکس‌ها حذف شدند!</b>\n\n"
            "📷 برای آپلود عکس جدید، روی دکمه زیر کلیک کنید:",
            reply_markup=InlineKeyboardMarkup(inline_keyboard=[
                [InlineKeyboardButton(text="📸 آپلود عکس", callback_data="upload_profile_photo")],
                [InlineKeyboardButton(text="🔙 بازگشت به پروفایل", callback_data="show_profile")]
            ]),
            parse_mode='HTML'
        )
        await callback.answer("همه عکس‌ها حذف شدند!")

@router.callback_query(F.data.in_(["cancel_photo_upload", "finish_photo_upload"]))
async def finish_photo_process(callback: CallbackQuery, state: FSMContext):
    """Finish photo upload process"""
    await state.clear()
    await callback.message.edit_text(
        "📸 <b>فرآیند آپلود عکس تمام شد!</b>\n\n"
        "🔄 برای مشاهده پروفایل، روی دکمه زیر کلیک کنید:",
        reply_markup=InlineKeyboardMarkup(inline_keyboard=[
            [InlineKeyboardButton(text="👤 مشاهده پروفایل", callback_data="show_profile")]
        ])
    )
    await callback.answer()
