from aiogram import Router, F
from aiogram.types import Message, CallbackQuery, ReplyKeyboardMarkup, KeyboardButton, InlineKeyboardMarkup, InlineKeyboardButton, ReplyKeyboardRemove
from aiogram.fsm.context import FSMContext
from aiogram.fsm.state import State, StatesGroup
from sqlalchemy import select
from database.database import db
from database.models import User, Preference
from localization.translations import get_text
import os
from utils.matching import matching_engine
import logging

router = Router()
logger = logging.getLogger(__name__)

class RegistrationStates(StatesGroup):
    language = State()
    gender = State()
    looking_for = State()
    birth_year = State()
    location = State()
    city_country = State()
    bio = State()
    interests = State()

# Language selection keyboard
def get_language_keyboard():
    return InlineKeyboardMarkup(inline_keyboard=[
        [InlineKeyboardButton(text="🇮🇷 فارسی", callback_data="lang_fa")],
        [InlineKeyboardButton(text="🇺🇸 English", callback_data="lang_en")],
        [InlineKeyboardButton(text="🇹🇷 Türkçe", callback_data="lang_tr")],
        [InlineKeyboardButton(text="🇸🇦 العربية", callback_data="lang_ar")]
    ])

def get_gender_keyboard(lang: str):
    return InlineKeyboardMarkup(inline_keyboard=[
        [InlineKeyboardButton(text=get_text(lang, 'gender_male'), callback_data="gender_1")],
        [InlineKeyboardButton(text=get_text(lang, 'gender_female'), callback_data="gender_2")],
        [InlineKeyboardButton(text=get_text(lang, 'gender_other'), callback_data="gender_3")]
    ])

def get_looking_for_keyboard(lang: str):
    return InlineKeyboardMarkup(inline_keyboard=[
        [InlineKeyboardButton(text=get_text(lang, 'looking_for_male'), callback_data="looking_1")],
        [InlineKeyboardButton(text=get_text(lang, 'looking_for_female'), callback_data="looking_2")],
        [InlineKeyboardButton(text=get_text(lang, 'looking_for_any'), callback_data="looking_0")]
    ])

def get_location_keyboard(lang: str):
    keyboard = ReplyKeyboardMarkup(
        keyboard=[
            [KeyboardButton(text=get_text(lang, 'share_location'), request_location=True)],
            [KeyboardButton(text=get_text(lang, 'skip_location'))]
        ],
        resize_keyboard=True,
        one_time_keyboard=True
    )
    return keyboard

@router.message(F.text == "/start")
async def start_command(message: Message, state: FSMContext):
    """Handle /start command"""
    user_id = message.from_user.id
    
    # Check if user is admin
    ADMIN_USER_IDS = [int(uid) for uid in os.getenv('ADMIN_USER_IDS', '').split(',') if uid]
    is_admin = user_id in ADMIN_USER_IDS
    
    # Check if user already exists
    async with db.get_session() as session:
        result = await session.execute(select(User).where(User.user_id == user_id))
        existing_user = result.scalar_one_or_none()
        
        if existing_user:
            # User exists, show main menu
            lang = existing_user.language_code or 'en'
            
            if is_admin:
                await message.answer(
                    "🔐 خوش آمدید ادمین!\nاز منوی ادمین استفاده کنید.",
                    reply_markup=get_main_menu_keyboard(lang, is_admin=True)
                )
            else:
                await message.answer(
                    get_text(lang, 'welcome'),
                    reply_markup=get_main_menu_keyboard(lang, is_admin=False)
                )
            return
    
    # New user, start registration
    await message.answer(
        "Welcome to Sarium! 👋\nمرحباً بك في Sarium!\nSarium'a hoş geldiniz!\nبه Sarium خوش آمدید!\n\nChoose your language / اختر لغتك / Dilinizi seçin / زبان خود را انتخاب کنید:",
        reply_markup=get_language_keyboard()
    )
    await state.set_state(RegistrationStates.language)

@router.callback_query(F.data.startswith("lang_"))
async def language_selected(callback: CallbackQuery, state: FSMContext):
    """Handle language selection"""
    lang = callback.data.split("_")[1]
    await state.update_data(language=lang)
    
    await callback.message.edit_text(
        get_text(lang, 'welcome') + "\n\n" + get_text(lang, 'setup_profile'),
        reply_markup=get_gender_keyboard(lang)
    )
    await state.set_state(RegistrationStates.gender)
    await callback.answer()

@router.callback_query(F.data.startswith("gender_") & ~F.data.startswith("gender_pref_"))
async def gender_selected(callback: CallbackQuery, state: FSMContext):
    """Handle gender selection"""
    gender = int(callback.data.split("_")[1])
    data = await state.get_data()
    lang = data.get('language', 'en')
    
    await state.update_data(gender=gender)
    
    await callback.message.edit_text(
        get_text(lang, 'setup_profile') + "\n\n✅ جنسیت انتخاب شد\n\nحالا دنبال چه کسی هستید؟",
        reply_markup=get_looking_for_keyboard(lang)
    )
    await state.set_state(RegistrationStates.looking_for)
    await callback.answer()

@router.callback_query(F.data.startswith("looking_"))
async def looking_for_selected(callback: CallbackQuery, state: FSMContext):
    """Handle looking for selection"""
    looking_for = int(callback.data.split("_")[1])
    data = await state.get_data()
    lang = data.get('language', 'en')
    
    await state.update_data(looking_for=looking_for)
    
    await callback.message.edit_text(
        get_text(lang, 'setup_profile') + "\n\n✅ جنسیت انتخاب شد\n✅ ترجیحات تنظیم شد\n\nسال تولدتان؟ (مثال: 1995)"
    )
    await state.set_state(RegistrationStates.birth_year)
    await callback.answer()

@router.message(RegistrationStates.birth_year)
async def birth_year_entered(message: Message, state: FSMContext):
    """Handle birth year input"""
    try:
        birth_year = int(message.text)
        current_year = 2024
        age = current_year - birth_year
        
        if age < 18 or age > 80:
            await message.answer("سن باید بین ۱۸ تا ۸۰ سال باشد. لطفاً سال تولد صحیح وارد کنید:")
            return
        
        data = await state.get_data()
        lang = data.get('language', 'en')
        
        await state.update_data(birth_year=birth_year)
        
        await message.answer(
            get_text(lang, 'setup_profile') + f"\n\n✅ جنسیت انتخاب شد\n✅ ترجیحات تنظیم شد\n✅ سن: {age} سال\n\nحالا موقعیت تقریبی خود را ارسال کنید:",
            reply_markup=get_location_keyboard(lang)
        )
        await state.set_state(RegistrationStates.location)
        
    except ValueError:
        await message.answer("لطفاً سال تولد را به صورت عدد وارد کنید (مثال: 1995):")

@router.message(RegistrationStates.location, F.location)
async def location_received(message: Message, state: FSMContext):
    """Handle location sharing"""
    lat = message.location.latitude
    lon = message.location.longitude
    
    # Encode to geohash with reduced precision for privacy
    geohash = matching_engine.encode_geohash(lat, lon, precision=6)
    
    data = await state.get_data()
    lang = data.get('language', 'en')
    
    await state.update_data(
        latitude=lat,
        longitude=lon,
        geohash=geohash
    )
    
    await message.answer(
        "✅ موقعیت دریافت شد!\n\nحالا شهر و کشور خود را بنویسید (مثال: تهران، ایران):",
        reply_markup=ReplyKeyboardRemove()
    )
    await state.set_state(RegistrationStates.city_country)

@router.message(RegistrationStates.location)
async def location_skipped(message: Message, state: FSMContext):
    """Handle location skip"""
    data = await state.get_data()
    lang = data.get('language', 'en')
    
    # Check for skip button or any text containing skip/رد
    skip_text = get_text(lang, 'skip_location')
    if message.text and (message.text == skip_text or 'رد' in message.text or 'Skip' in message.text.lower() or '⏭' in message.text):
        # Get localized text for city prompt
        city_prompt = {
            'fa': "شهر و کشور خود را بنویسید (مثال: تهران، ایران):",
            'en': "Enter your city and country (e.g., London, UK):",
            'tr': "Şehir ve ülkenizi yazın (örn: İstanbul, Türkiye):",
            'ar': "اكتب مدينتك وبلدك (مثال: القاهرة، مصر):"
        }.get(lang, "Enter your city and country:")
        
        await message.answer(
            city_prompt,
            reply_markup=ReplyKeyboardRemove()
        )
        await state.set_state(RegistrationStates.city_country)
    else:
        # Get localized text for location prompt
        location_prompt = {
            'fa': "لطفاً روی دکمه موقعیت کلیک کنید یا رد کنید:",
            'en': "Please share your location or skip:",
            'tr': "Lütfen konumunuzu paylaşın veya atlayın:",
            'ar': "يرجى مشاركة موقعك أو تخطي:"
        }.get(lang, "Please share your location or skip:")
        
        await message.answer(
            location_prompt,
            reply_markup=get_location_keyboard(lang)
        )

@router.message(RegistrationStates.city_country)
async def city_country_entered(message: Message, state: FSMContext):
    """Handle city/country input"""
    city_country = message.text.strip()
    
    # Simple parsing - you might want to use a geocoding service
    parts = city_country.split(',')
    city = parts[0].strip() if parts else city_country
    country = parts[1].strip() if len(parts) > 1 else ""
    
    # Map common country names to codes
    country_codes = {
        'ایران': 'IR', 'iran': 'IR',
        'ترکیه': 'TR', 'turkey': 'TR', 'türkiye': 'TR',
        'عربستان': 'SA', 'saudi arabia': 'SA',
        'امارات': 'AE', 'uae': 'AE',
        'آلمان': 'DE', 'germany': 'DE',
        'آمریکا': 'US', 'usa': 'US', 'united states': 'US'
    }
    
    country_code = country_codes.get(country.lower(), 'XX')
    
    data = await state.get_data()
    lang = data.get('language', 'en')
    
    await state.update_data(
        city=city,
        country=country,
        country_code=country_code
    )
    
    await message.answer(
        f"✅ مکان: {city}, {country}\n\nیک بیو کوتاه برای خودتان بنویسید (اختیاری - /skip برای رد کردن):"
    )
    await state.set_state(RegistrationStates.bio)

@router.message(RegistrationStates.bio)
async def bio_entered(message: Message, state: FSMContext):
    """Handle bio input"""
    bio = None if message.text == "/skip" else message.text.strip()
    
    data = await state.get_data()
    lang = data.get('language', 'en')
    
    await state.update_data(bio=bio)
    
    await message.answer(
        "علایق خود را با کاما جدا کنید (اختیاری - /skip برای رد کردن):\nمثال: موسیقی, سفر, ورزش, فیلم"
    )
    await state.set_state(RegistrationStates.interests)

@router.message(RegistrationStates.interests)
async def interests_entered(message: Message, state: FSMContext):
    """Handle interests input and complete registration"""
    interests = []
    if message.text != "/skip":
        interests = [interest.strip() for interest in message.text.split(',')]
    
    data = await state.get_data()
    lang = data.get('language', 'en')
    
    # Save user to database
    async with db.get_session() as session:
        try:
            # Calculate age from birth year
            from datetime import datetime
            current_year = datetime.now().year
            birth_year = data.get('birth_year')
            age = current_year - birth_year if birth_year else None
            
            # Format interests as string
            interests_str = ','.join(interests) if interests else ''
            
            # Create user
            user = User(
                user_id=message.from_user.id,
                username=message.from_user.username or f"user_{message.from_user.id}",
                first_name=message.from_user.first_name,
                last_name=message.from_user.last_name,
                gender=data.get('gender'),
                age=age,
                language_code=lang,
                country=data.get('country_code', ''),
                city=data.get('city', ''),
                location_lat=str(data.get('latitude', 0)) if data.get('latitude') else None,
                location_lon=str(data.get('longitude', 0)) if data.get('longitude') else None,
                looking_for=str(data.get('looking_for', 0)),
                bio=data.get('bio', ''),
                interests=interests_str
            )
            
            session.add(user)
            await session.flush()
            
            # Create preferences
            preference = Preference(
                user_id=message.from_user.id,  # Use telegram user_id directly
                age_min=18,
                age_max=45,
                gender_pref=data.get('looking_for', 0),
                radius_km=50,
                country_code=data.get('country_code', ''),
                city=data.get('city', '')
            )
            
            session.add(preference)
            await session.commit()
            
            # Check if user is admin
            ADMIN_USER_IDS = [int(uid) for uid in os.getenv('ADMIN_USER_IDS', '').split(',') if uid]
            is_admin = message.from_user.id in ADMIN_USER_IDS
            
            # Registration complete
            if is_admin:
                await message.answer(
                    "✅ ثبت‌نام تکمیل شد!\n\n🔐 شما به عنوان ادمین شناسایی شدید.\nاز منوی ادمین استفاده کنید.",
                    reply_markup=get_main_menu_keyboard(lang, is_admin=True)
                )
            else:
                await message.answer(
                    get_text(lang, 'profile_complete'),
                    reply_markup=get_main_menu_keyboard(lang, is_admin=False)
                )
            
            await state.clear()
            logger.info(f"User {message.from_user.id} registered successfully")
            
        except Exception as e:
            import traceback
            error_details = traceback.format_exc()
            logger.error(f"Registration error for user {message.from_user.id}: {e}\n{error_details}")
            
            # Send more detailed error message for debugging
            error_msg = f"خطا در ثبت نام: {str(e)[:100]}\n\nلطفاً دوباره تلاش کنید: /start"
            await message.answer(error_msg)
            await state.clear()

def get_main_menu_keyboard(lang: str, is_admin: bool = False):
    """Get main menu keyboard"""
    if is_admin:
        # Special admin menu
        return ReplyKeyboardMarkup(
            keyboard=[
                [KeyboardButton(text="🔐 پنل ادمین"), KeyboardButton(text="📊 آمار سریع")],
                [KeyboardButton(text="👥 مدیریت کاربران"), KeyboardButton(text="🚨 گزارشات")],
                [KeyboardButton(text="📢 پیام همگانی"), KeyboardButton(text="🔧 تنظیمات سیستم")],
                [KeyboardButton(text="🧪 حالت تست"), KeyboardButton(text="❓ راهنما")]
            ],
            resize_keyboard=True
        )
    else:
        # Regular user menu
        return ReplyKeyboardMarkup(
            keyboard=[
                [KeyboardButton(text="🔍 یافتن مچ"), KeyboardButton(text="👤 پروفایل")],
                [KeyboardButton(text="💬 گفتگوهای من"), KeyboardButton(text="💚 لایک‌های من")],
                [KeyboardButton(text="🌟 اکتشاف"), KeyboardButton(text="🔔 نوتیفیکیشن‌ها")],
                [KeyboardButton(text="⭐ پریمیوم"), KeyboardButton(text="⚙️ تنظیمات")],
                [KeyboardButton(text="❓ راهنما")]
            ],
            resize_keyboard=True
        )

# My chats handler - added directly
@router.callback_query(F.data == "my_chats")
async def view_my_chats_direct(callback: CallbackQuery, state: FSMContext):
    """Direct my chats handler"""
    try:
        await callback.message.edit_text(
            "💬 گفتگوهای من\n\nدر حال توسعه...",
            reply_markup=InlineKeyboardMarkup(inline_keyboard=[
                [InlineKeyboardButton(text="🏠 منوی اصلی", callback_data="main_menu")]
            ])
        )
        await callback.answer()
    except Exception as e:
        logger.error(f"Error in my_chats: {e}")
        await callback.answer("❌ خطا")
