from aiogram import Router, F
from aiogram.types import Message, CallbackQuery, InlineKeyboardMarkup, InlineKeyboardButton
from aiogram.fsm.context import FSMContext
from aiogram.fsm.state import State, StatesGroup
from sqlalchemy import select, and_, func
from database.database import db
from database.models import User, Match, Report
from localization.translations import get_text, get_nested_text
import logging
import asyncio

router = Router()
logger = logging.getLogger(__name__)

class SafetyStates(StatesGroup):
    reporting = State()
    report_details = State()

def get_report_keyboard(lang: str):
    """Get report reasons keyboard"""
    return InlineKeyboardMarkup(inline_keyboard=[
        [InlineKeyboardButton(text=get_nested_text(lang, 'report_reasons', 'spam'), callback_data="report_1")],
        [InlineKeyboardButton(text=get_nested_text(lang, 'report_reasons', 'abuse'), callback_data="report_2")],
        [InlineKeyboardButton(text=get_nested_text(lang, 'report_reasons', 'nsfw'), callback_data="report_3")],
        [InlineKeyboardButton(text=get_nested_text(lang, 'report_reasons', 'other'), callback_data="report_4")],
        [InlineKeyboardButton(text="❌ لغو", callback_data="report_cancel")]
    ])

@router.callback_query(F.data == "chat_report")
async def start_report(callback: CallbackQuery, state: FSMContext):
    """Start reporting process"""
    data = await state.get_data()
    partner_id = data.get('partner_id')
    
    if not partner_id:
        await callback.answer("خطا: کاربر برای گزارش پیدا نشد")
        return
    
    async with db.get_session() as session:
        user_result = await session.execute(select(User).where(User.user_id == callback.from_user.id))
        user = user_result.scalar_one_or_none()
        
        if not user:
            await callback.answer("خطا: کاربر پیدا نشد")
            return
        
        lang = user.language_code or 'en'
        
        await callback.message.edit_text(
            "🚨 گزارش کاربر\n\nدلیل گزارش خود را انتخاب کنید:",
            reply_markup=get_report_keyboard(lang)
        )
        
        await state.update_data(reporting_user_id=partner_id)
        await state.set_state(SafetyStates.reporting)
    
    await callback.answer()

@router.callback_query(F.data.startswith("report_"))
async def handle_report(callback: CallbackQuery, state: FSMContext):
    """Handle report submission"""
    if callback.data == "report_cancel":
        await callback.message.edit_text("گزارش لغو شد.")
        await state.clear()
        await callback.answer()
        return
    
    reason = int(callback.data.split("_")[1])
    data = await state.get_data()
    reporting_user_id = data.get('reporting_user_id')
    
    async with db.get_session() as session:
        # Get reporter
        reporter_result = await session.execute(select(User).where(User.user_id == callback.from_user.id))
        reporter = reporter_result.scalar_one_or_none()
        
        if not reporter or not reporting_user_id:
            await callback.answer("خطا در ثبت گزارش")
            return
        
        # Create report
        report = Report(
            reporter=reporter.id,
            target=reporting_user_id,
            reason=reason,
            status=0  # pending
        )
        
        session.add(report)
        await session.commit()
        
        # Check if user should be auto-banned (multiple reports)
        report_count = await session.execute(
            select(func.count(Report.id)).where(
                and_(
                    Report.target == reporting_user_id,
                    Report.created_at >= func.now() - func.interval('24 hours')
                )
            )
        )
        report_count = report_count.scalar()
        
        if report_count >= 3:  # Auto-ban after 3 reports in 24h
            target_user = await session.get(User, reporting_user_id)
            if target_user:
                target_user.is_banned = True
                target_user.ban_reason = "Multiple reports"
                await session.commit()
                logger.warning(f"User {reporting_user_id} auto-banned due to multiple reports")
    
    await callback.message.edit_text(
        "✅ گزارش شما ثبت شد.\n\nتیم پشتیبانی بررسی خواهد کرد.\nاز گزارش شما متشکریم."
    )
    await state.clear()
    await callback.answer()

@router.callback_query(F.data == "chat_destruct")
async def self_destruct_menu(callback: CallbackQuery, state: FSMContext):
    """Show self-destruct options"""
    destruct_keyboard = InlineKeyboardMarkup(inline_keyboard=[
        [
            InlineKeyboardButton(text="💣 5 ثانیه", callback_data="destruct_5"),
            InlineKeyboardButton(text="💣 10 ثانیه", callback_data="destruct_10")
        ],
        [
            InlineKeyboardButton(text="💣 30 ثانیه", callback_data="destruct_30"),
            InlineKeyboardButton(text="💣 60 ثانیه", callback_data="destruct_60")
        ],
        [InlineKeyboardButton(text="❌ لغو", callback_data="destruct_cancel")]
    ])
    
    await callback.message.edit_text(
        "💣 خودتخریب پیام بعدی\n\nپیام بعدی شما پس از مدت زمان انتخابی حذف خواهد شد:",
        reply_markup=destruct_keyboard
    )
    await callback.answer()

@router.callback_query(F.data.startswith("destruct_"))
async def set_self_destruct(callback: CallbackQuery, state: FSMContext):
    """Set self-destruct timer"""
    if callback.data == "destruct_cancel":
        await callback.message.edit_text("خودتخریب لغو شد.")
        await callback.answer()
        return
    
    seconds = int(callback.data.split("_")[1])
    await state.update_data(self_destruct_seconds=seconds)
    
    await callback.message.edit_text(
        f"💣 خودتخریب فعال شد!\n\nپیام بعدی شما پس از {seconds} ثانیه حذف خواهد شد."
    )
    await callback.answer()

@router.callback_query(F.data == "chat_icebreaker")
async def send_icebreaker(callback: CallbackQuery, state: FSMContext):
    """Send an icebreaker message"""
    icebreakers = [
        "سلام! چطوری؟ 😊",
        "چه خبر؟ امروز چیکار کردی؟",
        "علاقه‌مندی‌هات چیه؟",
        "از کجایی؟",
        "چه موسیقی‌هایی گوش میدی؟",
        "فیلم مورد علاقت چیه؟",
        "چه غذایی دوست داری؟",
        "تو اوقات فراغت چیکار میکنی؟",
        "سفر دوست داری؟",
        "چه کتابی خوندی اخیراً؟"
    ]
    
    import random
    icebreaker = random.choice(icebreakers)
    
    data = await state.get_data()
    partner_tg_id = data.get('partner_tg_id')
    
    if partner_tg_id:
        try:
            await callback.bot.send_message(partner_tg_id, f"🧊 {icebreaker}")
            await callback.answer("یخ‌شکن ارسال شد! 🧊")
        except Exception as e:
            logger.error(f"Error sending icebreaker: {e}")
            await callback.answer("خطا در ارسال یخ‌شکن")
    else:
        await callback.answer("خطا: گفتگو پیدا نشد")

@router.message(F.text == "/report")
async def report_command(message: Message, state: FSMContext):
    """Handle report command"""
    current_state = await state.get_state()
    
    if current_state != "MatchingStates:in_chat":
        await message.answer("این دستور فقط در حین گفتگو قابل استفاده است.")
        return
    
    # Trigger report process
    fake_callback = type('obj', (object,), {
        'message': message,
        'from_user': message.from_user,
        'answer': lambda text="": asyncio.create_task(message.answer("گزارش در حال پردازش..."))
    })()
    
    await start_report(fake_callback, state)

# Safety word detection
@router.message(F.text.in_(["STOP", "stop", "استوپ", "توقف", "/safety"]))
async def safety_word_triggered(message: Message, state: FSMContext):
    """Handle safety word"""
    current_state = await state.get_state()
    
    if current_state == "MatchingStates:in_chat":
        data = await state.get_data()
        match_id = data.get('match_id')
        partner_tg_id = data.get('partner_tg_id')
        
        if match_id:
            async with db.get_session() as session:
                match_result = await session.execute(select(Match).where(Match.id == match_id))
                match = match_result.scalar_one_or_none()
                
                if match:
                    match.status = 2  # ended
                    match.ended_at = func.now()
                    match.ended_by = message.from_user.id
                    await session.commit()
            
            await message.answer(
                "🛑 کلمه ایمنی تشخیص داده شد!\n\nگفتگو فوراً پایان یافت.\n\nاگر احساس ناامنی می‌کنید، می‌توانید کاربر را گزارش دهید."
            )
            
            if partner_tg_id:
                try:
                    await message.bot.send_message(
                        partner_tg_id,
                        "🛑 گفتگو به دلایل ایمنی پایان یافت.\n\n/find برای مچ جدید"
                    )
                except:
                    pass
            
            await state.clear()
    else:
        await message.answer("کلمه ایمنی فقط در حین گفتگو قابل استفاده است.")

# Block user functionality
@router.callback_query(F.data == "chat_block")
async def block_user(callback: CallbackQuery, state: FSMContext):
    """Block current chat partner"""
    data = await state.get_data()
    partner_id = data.get('partner_id')
    match_id = data.get('match_id')
    
    if not partner_id or not match_id:
        await callback.answer("خطا: کاربر پیدا نشد")
        return
    
    async with db.get_session() as session:
        # Update match status to blocked
        match_result = await session.execute(select(Match).where(Match.id == match_id))
        match = match_result.scalar_one_or_none()
        
        if match:
            match.status = 3  # blocked
            match.ended_at = func.now()
            match.ended_by = callback.from_user.id
            await session.commit()
    
    await callback.message.edit_text(
        "🚫 کاربر مسدود شد.\n\nدیگر با این کاربر مچ نخواهید شد.\n\n/find برای مچ جدید"
    )
    await state.clear()
    await callback.answer()

# Content moderation helpers
def contains_inappropriate_content(text: str) -> bool:
    """Simple content filter"""
    inappropriate_words = [
        # Add inappropriate words in different languages
        'spam', 'اسپم', 'تبلیغ', 'advertisement',
        # Add more as needed
    ]
    
    text_lower = text.lower()
    return any(word in text_lower for word in inappropriate_words)

def contains_contact_info(text: str) -> bool:
    """Check if message contains contact information"""
    import re
    
    # Phone number patterns
    phone_patterns = [
        r'\+?[\d\s\-\(\)]{10,}',  # General phone pattern
        r'@\w+',  # Telegram username
        r'\b\w+@\w+\.\w+\b',  # Email
    ]
    
    for pattern in phone_patterns:
        if re.search(pattern, text):
            return True
    
    return False
