#!/bin/bash

# Health Check Script for ChatDate Bot
# Monitors the status of all services

set -e

# Colors
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m'

# Function to check service health
check_service() {
    service_name=$1
    container_name=$2
    
    if docker ps --format "table {{.Names}}" | grep -q "$container_name"; then
        status=$(docker inspect -f '{{.State.Status}}' "$container_name" 2>/dev/null || echo "unknown")
        
        if [ "$status" == "running" ]; then
            echo -e "${GREEN}✓${NC} $service_name: ${GREEN}Running${NC}"
            
            # Additional health checks
            case "$service_name" in
                "PostgreSQL")
                    if docker exec "$container_name" pg_isready -U chatdate_user -d chatdate >/dev/null 2>&1; then
                        echo -e "  └─ Database: ${GREEN}Ready${NC}"
                    else
                        echo -e "  └─ Database: ${YELLOW}Starting...${NC}"
                    fi
                    ;;
                "Redis")
                    if docker exec "$container_name" redis-cli ping >/dev/null 2>&1; then
                        echo -e "  └─ Cache: ${GREEN}Ready${NC}"
                    else
                        echo -e "  └─ Cache: ${YELLOW}Starting...${NC}"
                    fi
                    ;;
                "Bot")
                    # Check if bot process is running
                    if docker exec "$container_name" pgrep -f "python main.py" >/dev/null 2>&1; then
                        echo -e "  └─ Process: ${GREEN}Active${NC}"
                    else
                        echo -e "  └─ Process: ${RED}Not running${NC}"
                    fi
                    ;;
            esac
        else
            echo -e "${RED}✗${NC} $service_name: ${RED}$status${NC}"
        fi
    else
        echo -e "${RED}✗${NC} $service_name: ${RED}Not found${NC}"
    fi
}

# Function to check disk usage
check_disk_usage() {
    echo -e "\n${BLUE}Disk Usage:${NC}"
    
    # Check Docker volumes
    for volume in postgres_data redis_data; do
        size=$(docker volume inspect "$volume" --format '{{.Mountpoint}}' 2>/dev/null | xargs du -sh 2>/dev/null | cut -f1 || echo "N/A")
        echo -e "  $volume: $size"
    done
    
    # Check logs directory
    if [ -d "./logs" ]; then
        log_size=$(du -sh ./logs | cut -f1)
        echo -e "  Logs: $log_size"
    fi
}

# Function to check memory usage
check_memory_usage() {
    echo -e "\n${BLUE}Memory Usage:${NC}"
    
    for container in chatdate_postgres chatdate_redis chatdate_bot; do
        if docker ps --format "table {{.Names}}" | grep -q "$container"; then
            mem=$(docker stats --no-stream --format "table {{.Container}}\t{{.MemUsage}}" "$container" 2>/dev/null | tail -n 1 | awk '{print $2}' || echo "N/A")
            echo -e "  $container: $mem"
        fi
    done
}

# Function to check recent errors
check_recent_errors() {
    echo -e "\n${BLUE}Recent Errors (last 10):${NC}"
    
    if docker ps --format "table {{.Names}}" | grep -q "chatdate_bot"; then
        errors=$(docker logs chatdate_bot 2>&1 | grep -i "error" | tail -n 10)
        if [ -n "$errors" ]; then
            echo "$errors" | while IFS= read -r line; do
                echo -e "  ${RED}•${NC} $line"
            done
        else
            echo -e "  ${GREEN}No recent errors${NC}"
        fi
    else
        echo -e "  ${YELLOW}Bot container not running${NC}"
    fi
}

# Function to check database statistics
check_database_stats() {
    echo -e "\n${BLUE}Database Statistics:${NC}"
    
    if docker ps --format "table {{.Names}}" | grep -q "chatdate_postgres"; then
        # Count users
        user_count=$(docker exec chatdate_postgres psql -U chatdate_user -d chatdate -t -c "SELECT COUNT(*) FROM users;" 2>/dev/null || echo "N/A")
        echo -e "  Total Users: $user_count"
        
        # Count active matches
        match_count=$(docker exec chatdate_postgres psql -U chatdate_user -d chatdate -t -c "SELECT COUNT(*) FROM matches WHERE status = 'active';" 2>/dev/null || echo "N/A")
        echo -e "  Active Matches: $match_count"
        
        # Count today's registrations
        today_users=$(docker exec chatdate_postgres psql -U chatdate_user -d chatdate -t -c "SELECT COUNT(*) FROM users WHERE DATE(created_at) = CURRENT_DATE;" 2>/dev/null || echo "N/A")
        echo -e "  Today's Registrations: $today_users"
    else
        echo -e "  ${YELLOW}Database not accessible${NC}"
    fi
}

# Main health check
main() {
    echo -e "${BLUE}═══════════════════════════════════════${NC}"
    echo -e "${BLUE}     ChatDate Bot Health Check${NC}"
    echo -e "${BLUE}═══════════════════════════════════════${NC}"
    echo -e "Time: $(date '+%Y-%m-%d %H:%M:%S')\n"
    
    echo -e "${BLUE}Service Status:${NC}"
    check_service "PostgreSQL" "chatdate_postgres"
    check_service "Redis" "chatdate_redis"
    check_service "Bot" "chatdate_bot"
    check_service "Nginx" "chatdate_nginx"
    
    if [ "$1" == "--full" ]; then
        check_disk_usage
        check_memory_usage
        check_database_stats
        check_recent_errors
    fi
    
    echo -e "\n${BLUE}═══════════════════════════════════════${NC}"
    
    # Exit with appropriate code
    if docker ps --format "table {{.Names}}" | grep -q "chatdate_bot"; then
        if docker exec chatdate_bot pgrep -f "python main.py" >/dev/null 2>&1; then
            exit 0
        else
            exit 1
        fi
    else
        exit 1
    fi
}

# Run main function
main "$@"
