#!/usr/bin/env python3
"""
🚨 ULTIMATE FIX: Complete solution for ChatMessage issue
حل نهایی و قطعی مشکل ChatMessage
"""

import mysql.connector
import os
from dotenv import load_dotenv
from urllib.parse import urlparse

load_dotenv()

def ultimate_fix():
    """Ultimate fix for ChatMessage issue"""
    
    print("🚨 ULTIMATE FIX: ChatMessage Database Issue")
    print("="*60)
    
    try:
        # Parse database URL
        db_url = os.getenv('DATABASE_URL')
        if not db_url:
            print("❌ DATABASE_URL not found")
            return False
        
        parsed = urlparse(db_url)
        
        print("🔗 Connecting to database...")
        
        # Connect to database
        conn = mysql.connector.connect(
            host=parsed.hostname or 'localhost',
            port=parsed.port or 3306,
            user=parsed.username,
            password=parsed.password,
            database=parsed.path.lstrip('/')
        )
        
        cursor = conn.cursor()
        print("✅ Connected to database")
        
        # 1. Drop and recreate table with correct structure
        print("\n🔧 Step 1: Recreating chat_messages table...")
        
        cursor.execute("DROP TABLE IF EXISTS chat_messages")
        print("✅ Dropped old table")
        
        cursor.execute("""
            CREATE TABLE chat_messages (
                id BIGINT AUTO_INCREMENT PRIMARY KEY,
                match_id BIGINT NOT NULL,
                sender_id BIGINT NOT NULL,
                message_type VARCHAR(20) DEFAULT 'text',
                content TEXT,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                INDEX idx_match_id (match_id),
                INDEX idx_sender_id (sender_id),
                INDEX idx_created_at (created_at)
            )
        """)
        print("✅ Created new table with correct structure")
        
        # 2. Verify structure
        print("\n📋 Step 2: Verifying table structure...")
        cursor.execute("DESCRIBE chat_messages")
        columns = cursor.fetchall()
        
        expected_columns = ['id', 'match_id', 'sender_id', 'message_type', 'content', 'created_at']
        actual_columns = [col[0] for col in columns]
        
        print("Expected columns:", expected_columns)
        print("Actual columns:  ", actual_columns)
        
        if set(expected_columns) == set(actual_columns):
            print("✅ Table structure is correct!")
        else:
            print("❌ Table structure mismatch!")
            return False
        
        # 3. Test insert
        print("\n🧪 Step 3: Testing insert...")
        try:
            cursor.execute("""
                INSERT INTO chat_messages (match_id, sender_id, message_type, content) 
                VALUES (1, 123456789, 'text', 'Test message')
            """)
            print("✅ Test insert successful")
            
            # Clean up test data
            cursor.execute("DELETE FROM chat_messages WHERE content = 'Test message'")
            print("✅ Test data cleaned up")
        except Exception as e:
            print(f"❌ Test insert failed: {e}")
            return False
        
        # Commit all changes
        conn.commit()
        print("\n💾 All changes committed to database")
        
        cursor.close()
        conn.close()
        
        print("\n🎉 ULTIMATE FIX COMPLETED SUCCESSFULLY!")
        print("🔄 Now restart the bot and test messaging")
        
        return True
        
    except Exception as e:
        print(f"❌ Error: {e}")
        import traceback
        traceback.print_exc()
        return False

def show_instructions():
    """Show manual fix instructions"""
    print("\n📋 MANUAL FIX INSTRUCTIONS:")
    print("="*40)
    print("""
1. Stop the bot:
   pkill -f main.py

2. Run this script:
   python3 ultimate_fix_chat.py

3. Upload updated files:
   - database/simple_models.py (fixed)
   - handlers/chat.py (fixed)
   - handlers/my_chats.py (fixed)

4. Restart bot:
   nohup python3 main.py > bot.log 2>&1 &

5. Test messaging functionality

6. Check logs:
   tail -f bot.log
""")

if __name__ == "__main__":
    print("🚨 ULTIMATE ChatMessage Fix")
    print("This will completely recreate the chat_messages table!")
    print("⚠️  All existing chat messages will be lost!")
    
    confirm = input("\nContinue? (yes/no): ").strip().lower()
    if confirm == 'yes' or confirm == 'y':
        success = ultimate_fix()
        if success:
            print("\n✅ Fix completed successfully!")
            print("🔄 Restart the bot now")
        else:
            print("\n❌ Fix failed!")
            show_instructions()
    else:
        print("❌ Operation cancelled")
        show_instructions()
