"""
Authentication Handler - هندلر احراز هویت
مدیریت ثبت‌نام، ورود و تأیید ایمیل
"""

import re
import random
import string
import hashlib
from datetime import datetime, timedelta
from typing import Optional, Tuple
import logging

logger = logging.getLogger(__name__)

# ذخیره موقت کدهای تأیید (در محیط واقعی از Redis استفاده کنید)
verification_codes = {}

def is_valid_email(email: str) -> bool:
    """بررسی معتبر بودن ایمیل"""
    pattern = r'^[a-zA-Z0-9._%+-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,}$'
    return re.match(pattern, email) is not None

def is_valid_password(password: str) -> Tuple[bool, str]:
    """
    بررسی معتبر بودن رمز عبور
    
    شرایط:
    - حداقل 8 کاراکتر
    - حداقل یک حرف کوچک (a-z)
    - حداقل یک حرف بزرگ (A-Z)
    - حداقل یک عدد (0-9)
    - حداقل یک کاراکتر خاص (#@!$%^&*)
    
    Returns:
        (bool, str): (معتبر است؟, پیام خطا)
    """
    if len(password) < 8:
        return False, "رمز عبور باید حداقل ۸ کاراکتر باشد"
    
    if not re.search(r'[a-z]', password):
        return False, "رمز عبور باید حداقل یک حرف کوچک داشته باشد"
    
    if not re.search(r'[A-Z]', password):
        return False, "رمز عبور باید حداقل یک حرف بزرگ داشته باشد"
    
    if not re.search(r'[0-9]', password):
        return False, "رمز عبور باید حداقل یک عدد داشته باشد"
    
    if not re.search(r'[#@!$%^&*]', password):
        return False, "رمز عبور باید حداقل یک کاراکتر خاص (#@!$%^&*) داشته باشد"
    
    return True, ""

def hash_password(password: str) -> str:
    """هش کردن رمز عبور با SHA-256"""
    return hashlib.sha256(password.encode()).hexdigest()

def verify_password(password: str, password_hash: str) -> bool:
    """بررسی صحت رمز عبور"""
    return hash_password(password) == password_hash

def generate_verification_code() -> str:
    """تولید کد تأیید 6 رقمی"""
    return ''.join(random.choices(string.digits, k=6))

def store_verification_code(email: str, code: str):
    """ذخیره کد تأیید برای ایمیل (معتبر برای 10 دقیقه)"""
    verification_codes[email] = {
        'code': code,
        'expires_at': datetime.now() + timedelta(minutes=10)
    }
    logger.info(f"Verification code for {email}: {code}")

def verify_code(email: str, code: str) -> bool:
    """بررسی صحت کد تأیید"""
    if email not in verification_codes:
        return False
    
    stored = verification_codes[email]
    
    # بررسی انقضا
    if datetime.now() > stored['expires_at']:
        del verification_codes[email]
        return False
    
    # بررسی کد
    if stored['code'] == code:
        del verification_codes[email]
        return True
    
    return False

async def send_verification_email(email: str, code: str, lang: str = 'fa'):
    """
    ارسال ایمیل تأیید با Gmail SMTP
    """
    # ذخیره کد تأیید
    store_verification_code(email, code)
    
    # ارسال ایمیل واقعی
    try:
        from utils.email_service import email_service
        success = await email_service.send_verification_email(email, code, lang)
        
        if success:
            logger.info(f"Verification email sent successfully to {email}")
        else:
            logger.warning(f"Failed to send email to {email}, but code is stored: {code}")
        
        return True  # همیشه True برمی‌گردانیم چون کد ذخیره شده
        
    except Exception as e:
        logger.error(f"Error sending verification email to {email}: {e}")
        logger.info(f"Verification code for {email}: {code}")
        return True  # کد ذخیره شده، پس ادامه می‌دهیم

def get_password_requirements(lang: str = 'fa') -> str:
    """دریافت متن شرایط رمز عبور"""
    requirements = {
        'fa': """رمز عبور باید حداقل ۸ کاراکتر داشته باشد و شامل موارد زیر باشد:
• حداقل یک حرف کوچک (a-z)
• حداقل یک حرف بزرگ (A-Z)
• حداقل یک عدد (0-9)
• حداقل یک کاراکتر خاص (#@!$%^&*)""",
        'en': """Password must be at least 8 characters and include:
• At least one lowercase letter (a-z)
• At least one uppercase letter (A-Z)
• At least one number (0-9)
• At least one special character (#@!$%^&*)""",
        'tr': """Şifre en az 8 karakter olmalı ve şunları içermelidir:
• En az bir küçük harf (a-z)
• En az bir büyük harf (A-Z)
• En az bir rakam (0-9)
• En az bir özel karakter (#@!$%^&*)""",
        'ar': """يجب أن تتكون كلمة المرور من 8 أحرف على الأقل وتتضمن:
• حرف صغير واحد على الأقل (a-z)
• حرف كبير واحد على الأقل (A-Z)
• رقم واحد على الأقل (0-9)
• حرف خاص واحد على الأقل (#@!$%^&*)"""
    }
    
    return requirements.get(lang, requirements['en'])
