"""
Authentication Handler - Multi-Profile Support
هندلر احراز هویت با پشتیبانی از چند پروفایل
"""

from aiogram import Router, F
from aiogram.types import Message, CallbackQuery, ReplyKeyboardMarkup, KeyboardButton, InlineKeyboardMarkup, InlineKeyboardButton
from aiogram.fsm.context import FSMContext
from aiogram.fsm.state import State, StatesGroup
from utils.profile_manager import (
    get_active_profile,
    get_all_profiles,
    switch_profile,
    link_profile_to_telegram,
    create_user,
    get_user_by_email,
    verify_password,
    create_session,
    is_telegram_linked,
    get_profile_count
)
from database.models_new import PlatformType
from localization.translator import get_text
import logging
import re

router = Router()
logger = logging.getLogger(__name__)


class AuthStates(StatesGroup):
    """حالت‌های احراز هویت"""
    choosing_action = State()  # انتخاب ثبت‌نام یا ورود
    entering_email = State()  # وارد کردن ایمیل
    entering_password = State()  # وارد کردن رمز عبور
    confirming_password = State()  # تأیید رمز عبور (برای ثبت‌نام)
    entering_name = State()  # وارد کردن نام (برای ثبت‌نام)
    selecting_profile = State()  # انتخاب پروفایل (اگر چند تا دارد)
    switching_profile = State()  # تعویض پروفایل
    entering_switch_password = State()  # رمز عبور برای تعویض


# ==================== Helper Functions ====================

def is_valid_email(email: str) -> bool:
    """بررسی معتبر بودن ایمیل"""
    pattern = r'^[a-zA-Z0-9._%+-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,}$'
    return re.match(pattern, email) is not None


def is_valid_password(password: str) -> bool:
    """بررسی معتبر بودن رمز عبور (حداقل 6 کاراکتر)"""
    return len(password) >= 6


def get_auth_keyboard(lang: str) -> ReplyKeyboardMarkup:
    """کیبورد انتخاب ثبت‌نام یا ورود"""
    signup_texts = {
        'fa': '📝 ثبت‌نام',
        'en': '📝 Sign Up',
        'tr': '📝 Kayıt Ol',
        'ar': '📝 التسجيل'
    }
    
    login_texts = {
        'fa': '🔑 ورود',
        'en': '🔑 Login',
        'tr': '🔑 Giriş',
        'ar': '🔑 تسجيل الدخول'
    }
    
    return ReplyKeyboardMarkup(
        keyboard=[
            [KeyboardButton(text=signup_texts.get(lang, signup_texts['en']))],
            [KeyboardButton(text=login_texts.get(lang, login_texts['en']))]
        ],
        resize_keyboard=True,
        one_time_keyboard=True
    )


def get_cancel_keyboard(lang: str) -> ReplyKeyboardMarkup:
    """کیبورد انصراف"""
    cancel_texts = {
        'fa': '❌ انصراف',
        'en': '❌ Cancel',
        'tr': '❌ İptal',
        'ar': '❌ إلغاء'
    }
    
    return ReplyKeyboardMarkup(
        keyboard=[[KeyboardButton(text=cancel_texts.get(lang, cancel_texts['en']))]],
        resize_keyboard=True
    )


# ==================== ورود/ثبت‌نام ====================

@router.message(F.text.in_(['🔐 ورود به حساب کاربری', '🔐 Login to Account', '🔐 Hesaba Giriş']))
async def start_auth(message: Message, state: FSMContext):
    """شروع فرآیند ورود/ثبت‌نام"""
    telegram_chat_id = message.chat.id
    lang = 'fa'  # می‌توانید از دیتابیس بگیرید
    
    # بررسی اینکه آیا قبلاً پروفایلی لینک شده
    profile_count = await get_profile_count(telegram_chat_id)
    
    if profile_count > 0:
        # کاربر قبلاً پروفایل دارد، نمایش لیست
        await show_profile_list(message, state)
        return
    
    # کاربر جدید
    auth_messages = {
        'fa': '🔐 <b>ورود به حساب کاربری</b>\n\nلطفاً یکی از گزینه‌های زیر را انتخاب کنید:',
        'en': '🔐 <b>Login to Account</b>\n\nPlease choose one of the following options:',
        'tr': '🔐 <b>Hesaba Giriş</b>\n\nLütfen aşağıdaki seçeneklerden birini seçin:',
        'ar': '🔐 <b>تسجيل الدخول</b>\n\nالرجاء اختيار أحد الخيارات التالية:'
    }
    
    await message.answer(
        auth_messages.get(lang, auth_messages['en']),
        reply_markup=get_auth_keyboard(lang),
        parse_mode='HTML'
    )
    await state.update_data(language=lang)
    await state.set_state(AuthStates.choosing_action)


@router.message(AuthStates.choosing_action)
async def action_chosen(message: Message, state: FSMContext):
    """انتخاب ثبت‌نام یا ورود"""
    data = await state.get_data()
    lang = data.get('language', 'fa')
    
    # تشخیص ثبت‌نام یا ورود
    is_signup = any(keyword in message.text for keyword in ['ثبت‌نام', 'Sign Up', 'Kayıt', 'التسجيل', '📝'])
    
    await state.update_data(is_signup=is_signup)
    
    # درخواست ایمیل
    email_messages = {
        'fa': '📧 لطفاً ایمیل خود را وارد کنید:',
        'en': '📧 Please enter your email:',
        'tr': '📧 Lütfen e-postanızı girin:',
        'ar': '📧 الرجاء إدخال بريدك الإلكتروني:'
    }
    
    await message.answer(
        email_messages.get(lang, email_messages['en']),
        reply_markup=get_cancel_keyboard(lang)
    )
    await state.set_state(AuthStates.entering_email)


@router.message(AuthStates.entering_email)
async def email_entered(message: Message, state: FSMContext):
    """دریافت ایمیل"""
    data = await state.get_data()
    lang = data.get('language', 'fa')
    
    # بررسی انصراف
    if message.text in ['❌ انصراف', '❌ Cancel', '❌ İptal', '❌ إلغاء']:
        await state.clear()
        await message.answer("عملیات لغو شد")
        return
    
    email = message.text.strip().lower()
    
    # اعتبارسنجی ایمیل
    if not is_valid_email(email):
        error_messages = {
            'fa': '❌ فرمت ایمیل نامعتبر است. لطفاً دوباره تلاش کنید:',
            'en': '❌ Invalid email format. Please try again:',
            'tr': '❌ Geçersiz e-posta formatı. Lütfen tekrar deneyin:',
            'ar': '❌ تنسيق البريد الإلكتروني غير صالح. يرجى المحاولة مرة أخرى:'
        }
        await message.answer(error_messages.get(lang, error_messages['en']))
        return
    
    await state.update_data(email=email)
    
    # بررسی وجود کاربر
    user = await get_user_by_email(email)
    is_signup = data.get('is_signup', False)
    
    if is_signup and user:
        # ایمیل قبلاً ثبت شده
        error_messages = {
            'fa': '❌ این ایمیل قبلاً ثبت شده است.\nاگر حساب دارید، از گزینه "ورود" استفاده کنید.',
            'en': '❌ This email is already registered.\nIf you have an account, use the "Login" option.',
            'tr': '❌ Bu e-posta zaten kayıtlı.\nHesabınız varsa, "Giriş" seçeneğini kullanın.',
            'ar': '❌ هذا البريد الإلكتروني مسجل بالفعل.\nإذا كان لديك حساب، استخدم خيار "تسجيل الدخول".'
        }
        await message.answer(error_messages.get(lang, error_messages['en']))
        await state.set_state(AuthStates.choosing_action)
        await message.answer("لطفاً دوباره انتخاب کنید:", reply_markup=get_auth_keyboard(lang))
        return
    
    if not is_signup and not user:
        # ایمیل ثبت نشده
        error_messages = {
            'fa': '❌ این ایمیل ثبت نشده است.\nلطفاً ابتدا ثبت‌نام کنید.',
            'en': '❌ This email is not registered.\nPlease sign up first.',
            'tr': '❌ Bu e-posta kayıtlı değil.\nLütfen önce kayıt olun.',
            'ar': '❌ هذا البريد الإلكتروني غير مسجل.\nيرجى التسجيل أولاً.'
        }
        await message.answer(error_messages.get(lang, error_messages['en']))
        await state.set_state(AuthStates.choosing_action)
        await message.answer("لطفاً دوباره انتخاب کنید:", reply_markup=get_auth_keyboard(lang))
        return
    
    # درخواست رمز عبور
    password_messages = {
        'fa': '🔐 لطفاً رمز عبور خود را وارد کنید:\n\n' + ('(حداقل 6 کاراکتر)' if is_signup else ''),
        'en': '🔐 Please enter your password:\n\n' + ('(minimum 6 characters)' if is_signup else ''),
        'tr': '🔐 Lütfen şifrenizi girin:\n\n' + ('(en az 6 karakter)' if is_signup else ''),
        'ar': '🔐 الرجاء إدخال كلمة المرور:\n\n' + ('(6 أحرف على الأقل)' if is_signup else '')
    }
    
    await message.answer(
        password_messages.get(lang, password_messages['en']),
        reply_markup=get_cancel_keyboard(lang)
    )
    await state.set_state(AuthStates.entering_password)


@router.message(AuthStates.entering_password)
async def password_entered(message: Message, state: FSMContext):
    """دریافت رمز عبور"""
    data = await state.get_data()
    lang = data.get('language', 'fa')
    is_signup = data.get('is_signup', False)
    email = data['email']
    password = message.text.strip()
    
    # حذف پیام رمز عبور برای امنیت
    try:
        await message.delete()
    except:
        pass
    
    # بررسی انصراف
    if password in ['❌ انصراف', '❌ Cancel', '❌ İptal', '❌ إلغاء']:
        await state.clear()
        await message.answer("عملیات لغو شد")
        return
    
    if is_signup:
        # ثبت‌نام: بررسی طول رمز
        if not is_valid_password(password):
            error_messages = {
                'fa': '❌ رمز عبور باید حداقل 6 کاراکتر باشد. دوباره تلاش کنید:',
                'en': '❌ Password must be at least 6 characters. Try again:',
                'tr': '❌ Şifre en az 6 karakter olmalıdır. Tekrar deneyin:',
                'ar': '❌ يجب أن تكون كلمة المرور 6 أحرف على الأقل. حاول مرة أخرى:'
            }
            await message.answer(error_messages.get(lang, error_messages['en']))
            return
        
        await state.update_data(password=password)
        
        # درخواست تأیید رمز
        confirm_messages = {
            'fa': '🔐 لطفاً رمز عبور را دوباره وارد کنید:',
            'en': '🔐 Please enter your password again:',
            'tr': '🔐 Lütfen şifrenizi tekrar girin:',
            'ar': '🔐 الرجاء إدخال كلمة المرور مرة أخرى:'
        }
        await message.answer(confirm_messages.get(lang, confirm_messages['en']))
        await state.set_state(AuthStates.confirming_password)
    else:
        # ورود: تأیید رمز
        success, user_id = await verify_password(email, password)
        
        if not success:
            error_messages = {
                'fa': '❌ رمز عبور اشتباه است. دوباره تلاش کنید:',
                'en': '❌ Incorrect password. Try again:',
                'tr': '❌ Yanlış şifre. Tekrar deneyin:',
                'ar': '❌ كلمة مرور خاطئة. حاول مرة أخرى:'
            }
            await message.answer(error_messages.get(lang, error_messages['en']))
            return
        
        # ورود موفق - لینک کردن به تلگرام
        telegram_chat_id = message.chat.id
        telegram_user_id = message.from_user.id
        
        success, error = await link_profile_to_telegram(
            telegram_chat_id=telegram_chat_id,
            telegram_user_id=telegram_user_id,
            user_id=user_id,
            set_active=True
        )
        
        if not success:
            await message.answer(f"❌ خطا: {error}")
            return
        
        # ایجاد session
        await create_session(
            user_id=user_id,
            platform=PlatformType.telegram,
            telegram_chat_id=telegram_chat_id,
            telegram_user_id=telegram_user_id
        )
        
        # دریافت اطلاعات پروفایل
        profile = await get_active_profile(telegram_chat_id)
        
        success_messages = {
            'fa': f'✅ ورود موفق!\n\nخوش آمدید {profile["first_name"]} 👋',
            'en': f'✅ Login successful!\n\nWelcome {profile["first_name"]} 👋',
            'tr': f'✅ Giriş başarılı!\n\nHoş geldiniz {profile["first_name"]} 👋',
            'ar': f'✅ تسجيل الدخول ناجح!\n\nمرحباً {profile["first_name"]} 👋'
        }
        
        await message.answer(success_messages.get(lang, success_messages['en']))
        await state.clear()


@router.message(AuthStates.confirming_password)
async def password_confirmed(message: Message, state: FSMContext):
    """تأیید رمز عبور (ثبت‌نام)"""
    data = await state.get_data()
    lang = data.get('language', 'fa')
    password = data['password']
    confirm_password = message.text.strip()
    
    # حذف پیام
    try:
        await message.delete()
    except:
        pass
    
    if password != confirm_password:
        error_messages = {
            'fa': '❌ رمزهای عبور مطابقت ندارند. دوباره وارد کنید:',
            'en': '❌ Passwords do not match. Enter again:',
            'tr': '❌ Şifreler eşleşmiyor. Tekrar girin:',
            'ar': '❌ كلمات المرور غير متطابقة. أدخل مرة أخرى:'
        }
        await message.answer(error_messages.get(lang, error_messages['en']))
        await state.set_state(AuthStates.entering_password)
        return
    
    # درخواست نام
    name_messages = {
        'fa': '👤 لطفاً نام خود را وارد کنید:',
        'en': '👤 Please enter your name:',
        'tr': '👤 Lütfen adınızı girin:',
        'ar': '👤 الرجاء إدخال اسمك:'
    }
    
    await message.answer(name_messages.get(lang, name_messages['en']))
    await state.set_state(AuthStates.entering_name)


@router.message(AuthStates.entering_name)
async def name_entered(message: Message, state: FSMContext):
    """دریافت نام و ایجاد حساب"""
    data = await state.get_data()
    lang = data.get('language', 'fa')
    email = data['email']
    password = data['password']
    first_name = message.text.strip()
    
    if len(first_name) < 2:
        await message.answer("❌ نام باید حداقل 2 کاراکتر باشد")
        return
    
    # ایجاد کاربر
    success, user_id, error = await create_user(
        email=email,
        password=password,
        first_name=first_name,
        language_code=lang
    )
    
    if not success:
        await message.answer(f"❌ خطا: {error}")
        return
    
    # لینک کردن به تلگرام
    telegram_chat_id = message.chat.id
    telegram_user_id = message.from_user.id
    
    await link_profile_to_telegram(
        telegram_chat_id=telegram_chat_id,
        telegram_user_id=telegram_user_id,
        user_id=user_id,
        set_active=True
    )
    
    # ایجاد session
    await create_session(
        user_id=user_id,
        platform=PlatformType.telegram,
        telegram_chat_id=telegram_chat_id,
        telegram_user_id=telegram_user_id
    )
    
    success_messages = {
        'fa': f'✅ ثبت‌نام موفق!\n\nخوش آمدید {first_name} 🎉\n\nحالا می‌توانید پروفایل خود را تکمیل کنید.',
        'en': f'✅ Registration successful!\n\nWelcome {first_name} 🎉\n\nYou can now complete your profile.',
        'tr': f'✅ Kayıt başarılı!\n\nHoş geldiniz {first_name} 🎉\n\nArtık profilinizi tamamlayabilirsiniz.',
        'ar': f'✅ التسجيل ناجح!\n\nمرحباً {first_name} 🎉\n\nيمكنك الآن إكمال ملفك الشخصي.'
    }
    
    await message.answer(success_messages.get(lang, success_messages['en']))
    await state.clear()


# ==================== مدیریت پروفایل‌ها ====================

async def show_profile_list(message: Message, state: FSMContext):
    """نمایش لیست پروفایل‌ها"""
    telegram_chat_id = message.chat.id
    profiles = await get_all_profiles(telegram_chat_id)
    lang = 'fa'
    
    if not profiles:
        await message.answer("❌ هیچ پروفایلی یافت نشد")
        return
    
    await message.answer("پروفایل‌های شما:")
    
    for profile in profiles:
        status = "✅ فعال" if profile['is_active'] else ""
        caption = (
            f"👤 {profile['first_name']}\n"
            f"📧 {profile['email']}\n"
            f"{status}"
        )
        
        keyboard = None
        if not profile['is_active']:
            keyboard = InlineKeyboardMarkup(inline_keyboard=[[
                InlineKeyboardButton(
                    text="🔄 تعویض به این پروفایل",
                    callback_data=f"switch_profile:{profile['user_id']}"
                )
            ]])
        
        await message.answer(caption, reply_markup=keyboard)
    
    # دکمه افزودن پروفایل جدید
    add_keyboard = InlineKeyboardMarkup(inline_keyboard=[[
        InlineKeyboardButton(
            text="➕ افزودن پروفایل جدید",
            callback_data="add_new_profile"
        )
    ]])
    await message.answer("می‌توانید پروفایل جدید اضافه کنید:", reply_markup=add_keyboard)


@router.callback_query(F.data.startswith("switch_profile:"))
async def switch_profile_callback(callback: CallbackQuery, state: FSMContext):
    """درخواست تعویض پروفایل"""
    user_id = int(callback.data.split(':')[1])
    
    await state.update_data(target_user_id=user_id)
    await callback.message.answer("🔐 رمز عبور این پروفایل را وارد کنید:")
    await state.set_state(AuthStates.entering_switch_password)
    await callback.answer()


@router.message(AuthStates.entering_switch_password)
async def switch_password_entered(message: Message, state: FSMContext):
    """تأیید رمز و تعویض پروفایل"""
    password = message.text.strip()
    data = await state.get_data()
    target_user_id = data['target_user_id']
    telegram_chat_id = message.chat.id
    
    # حذف پیام رمز
    try:
        await message.delete()
    except:
        pass
    
    # تعویض پروفایل
    success, error = await switch_profile(telegram_chat_id, target_user_id, password)
    
    if success:
        profile = await get_active_profile(telegram_chat_id)
        await message.answer(
            f"✅ تعویض موفق!\n"
            f"حالا با پروفایل {profile['first_name']} وارد شده‌اید 👋"
        )
    else:
        await message.answer(f"❌ خطا: {error}")
    
    await state.clear()


@router.callback_query(F.data == "add_new_profile")
async def add_new_profile_callback(callback: CallbackQuery, state: FSMContext):
    """افزودن پروفایل جدید"""
    await callback.message.answer(
        "برای افزودن پروفایل جدید، لطفاً ثبت‌نام کنید:",
        reply_markup=get_auth_keyboard('fa')
    )
    await state.set_state(AuthStates.choosing_action)
    await callback.answer()
