#!/usr/bin/env python3
"""مدیریت اتاق‌های گفتگو - ایجاد و مدیریت گروه‌های چت"""

from aiogram import Router, F, Bot
from aiogram.types import Message, InlineKeyboardMarkup, InlineKeyboardButton, CallbackQuery
from aiogram.fsm.context import FSMContext
from aiogram.fsm.state import State, StatesGroup
from database.database import db
from database.models import User
from sqlalchemy import select
from datetime import datetime, timedelta
import logging
import hashlib

router = Router()
logger = logging.getLogger(__name__)

# ذخیره اطلاعات گروه‌های فعال
active_chat_rooms = {}  # {room_id: {"user1": id, "user2": id, "created": datetime}}

class ChatStates(StatesGroup):
    """حالت‌های چت"""
    in_chat = State()
    waiting_partner = State()

def generate_room_id(user1_id: int, user2_id: int) -> str:
    """ایجاد شناسه یکتا برای اتاق گفتگو"""
    # مرتب کردن ID ها برای یکتا بودن
    ids = sorted([str(user1_id), str(user2_id)])
    room_string = f"{ids[0]}_{ids[1]}_{datetime.now().strftime('%Y%m%d%H%M%S')}"
    return hashlib.md5(room_string.encode()).hexdigest()[:10]

async def create_chat_room(user1_id: int, user2_id: int, bot: Bot, chat_reason: str = "مچ"):
    """ایجاد گروه تلگرام با Userbot"""
    
    try:
        # استفاده از Userbot برای ایجاد گروه
        from .chat_room_creator import create_private_group
        
        # ایجاد گروه جدید
        bot_username = bot._bot_info.username if hasattr(bot, '_bot_info') else "ChatMilosBot"
        success, message, chat_id = await create_private_group(user1_id, user2_id, bot_username)
        
        if success:
            logger.info(f"Chat room created successfully: {message}")
            
            # ذخیره اطلاعات اتاق
            room_id = generate_room_id(user1_id, user2_id)
            active_chat_rooms[room_id] = {
                "chat_id": chat_id,
                "user1": user1_id,
                "user2": user2_id,
                "created": datetime.now(),
                "reason": chat_reason,
                "status": "active"
            }
            
            # ارسال پیام خوش‌آمدگویی
            await send_welcome_message(bot, chat_id, room_id, chat_reason)
            
            return True, message
        else:
            logger.error(f"Failed to create chat room: {message}")
            # استفاده از روش جایگزین (چت درون بات)
            return await create_inline_chat(user1_id, user2_id, bot, chat_reason)
            
    except ImportError:
        logger.warning("Userbot module not available, using inline chat")
        # اگر ماژول Userbot موجود نیست، از چت درون بات استفاده کن
        return await create_inline_chat(user1_id, user2_id, bot, chat_reason)
    except Exception as e:
        logger.error(f"Error in create_chat_room: {e}")
        return False, str(e)

async def create_inline_chat(user1_id: int, user2_id: int, bot: Bot, chat_reason: str = "مچ"):
    """روش جایگزین: چت درون بات"""
    try:
        from .inline_chat_room import start_inline_chat
        success, result = await start_inline_chat(user1_id, user2_id, bot, chat_reason)
        return success, result
    except Exception as e:
        logger.error(f"Error in inline chat: {e}")
        return False, str(e)

async def send_welcome_message(bot: Bot, chat_id: int, room_id: str, chat_reason: str):
    """ارسال پیام خوش‌آمدگویی در گروه"""
    welcome_message = (
        "🤖 <b>ربات نظارتگر ChatMilos</b>\n\n"
        "به اتاق گفتگو خوش آمدید! 👋\n\n"
        f"🆔 کد اتاق: <code>{room_id}</code>\n"
        f"📌 دلیل ایجاد: {chat_reason}\n\n"
        "⚠️ <b>قوانین مهم:</b>\n"
        "• هویت طرف مقابل محرمانه است\n"
        "• از افشای اطلاعات شخصی خودداری کنید\n"
        "• احترام متقابل را رعایت کنید\n"
        "• محتوای نامناسب = اخراج فوری\n\n"
        "💡 دستورات:\n"
        "/rules - نمایش قوانین\n"
        "/report - گزارش تخلف\n"
        "/leave - خروج از اتاق\n\n"
        "موفق باشید! 💚"
    )
    
    try:
        await bot.send_message(
            chat_id=chat_id,
            text=welcome_message,
            parse_mode='HTML'
        )
    except Exception as e:
        logger.error(f"Error sending welcome message: {e}")

# Handler برای دستورات در گروه
@router.message(F.text == "/rules")
async def show_rules(message: Message):
    """نمایش قوانین گفتگو"""
    rules = (
        "📜 <b>قوانین اتاق گفتگو:</b>\n\n"
        "1️⃣ احترام متقابل را رعایت کنید\n"
        "2️⃣ از ارسال محتوای نامناسب خودداری کنید\n"
        "3️⃣ اطلاعات شخصی خود را فاش نکنید\n"
        "4️⃣ از تهدید و توهین پرهیز کنید\n"
        "5️⃣ اسپم ممنوع است\n\n"
        "⚠️ تخلف = اخراج فوری"
    )
    await message.answer(rules, parse_mode='HTML')

@router.message(F.text == "/report")
async def report_user(message: Message):
    """گزارش تخلف"""
    keyboard = InlineKeyboardMarkup(inline_keyboard=[
        [InlineKeyboardButton(text="🚫 اسپم", callback_data="report_spam")],
        [InlineKeyboardButton(text="😤 توهین", callback_data="report_abuse")],
        [InlineKeyboardButton(text="🔞 محتوای نامناسب", callback_data="report_inappropriate")],
        [InlineKeyboardButton(text="❌ لغو", callback_data="report_cancel")]
    ])
    
    await message.answer(
        "🚨 <b>گزارش تخلف</b>\n\nدلیل گزارش را انتخاب کنید:",
        reply_markup=keyboard,
        parse_mode='HTML'
    )

@router.message(F.text == "/leave")
async def leave_chat(message: Message, bot: Bot):
    """خروج از اتاق گفتگو"""
    user_id = message.from_user.id
    chat_id = message.chat.id
    
    # پیدا کردن اتاق
    room_to_close = None
    for room_id, room_data in active_chat_rooms.items():
        if room_data.get("chat_id") == chat_id:
            room_to_close = room_id
            break
    
    if room_to_close:
        # بستن اتاق
        active_chat_rooms[room_to_close]["status"] = "closed"
        
        await message.answer(
            "👋 <b>اتاق گفتگو بسته شد</b>\n\n"
            "از گفتگوی شما متشکریم!",
            parse_mode='HTML'
        )
        
        # اطلاع به کاربر دیگر
        other_user_id = (
            active_chat_rooms[room_to_close]["user2"] 
            if active_chat_rooms[room_to_close]["user1"] == user_id 
            else active_chat_rooms[room_to_close]["user1"]
        )
        
        try:
            await bot.send_message(
                other_user_id,
                "⚠️ طرف مقابل از گفتگو خارج شد.\nاتاق بسته شد."
            )
        except:
            pass

# Callback handlers برای گزارش
@router.callback_query(F.data.startswith("report_"))
async def handle_report(callback: CallbackQuery):
    """مدیریت گزارش‌ها"""
    report_type = callback.data.replace("report_", "")
    
    if report_type == "cancel":
        await callback.message.delete()
        return
    
    # ثبت گزارش در دیتابیس
    # TODO: implement report logging
    
    await callback.message.edit_text(
        "✅ گزارش شما ثبت شد.\nتیم پشتیبانی بررسی خواهد کرد.",
        parse_mode='HTML'
    )
    await callback.answer()

# تابع کمکی برای پاکسازی اتاق‌های قدیمی
async def cleanup_old_rooms(bot: Bot, max_age_hours: int = 24):
    """پاکسازی اتاق‌های قدیمی"""
    now = datetime.now()
    rooms_to_remove = []
    
    for room_id, room_data in active_chat_rooms.items():
        created = room_data.get("created")
        if created and (now - created).total_seconds() > max_age_hours * 3600:
            rooms_to_remove.append(room_id)
            
            # اطلاع به کاربران
            try:
                chat_id = room_data.get("chat_id")
                if chat_id:
                    await bot.send_message(
                        chat_id,
                        "⏰ مدت زمان این اتاق به پایان رسید.\nاتاق بسته می‌شود."
                    )
            except:
                pass
    
    # حذف اتاق‌های قدیمی
    for room_id in rooms_to_remove:
        del active_chat_rooms[room_id]
    
    if rooms_to_remove:
        logger.info(f"Cleaned up {len(rooms_to_remove)} old chat rooms")
