#!/usr/bin/env python3
"""مدیریت اتاق‌های گفتگو - ایجاد و مدیریت گروه‌های چت"""

from aiogram import Router, F, Bot
from aiogram.types import Message, InlineKeyboardMarkup, InlineKeyboardButton, CallbackQuery, ChatPermissions
from aiogram.fsm.context import FSMContext
from aiogram.fsm.state import State, StatesGroup
from database.database import db
from database.models import User
from sqlalchemy import select
from datetime import datetime, timedelta
import logging
import hashlib

router = Router()
logger = logging.getLogger(__name__)

# ذخیره اطلاعات گروه‌های فعال
active_chat_rooms = {}  # {room_id: {"user1": id, "user2": id, "created": datetime}}

class ChatStates(StatesGroup):
    """حالت‌های چت"""
    in_chat = State()
    waiting_partner = State()

def generate_room_id(user1_id: int, user2_id: int) -> str:
    """ایجاد شناسه یکتا برای اتاق گفتگو"""
    # مرتب کردن ID ها برای یکتا بودن
    ids = sorted([str(user1_id), str(user2_id)])
    room_string = f"{ids[0]}_{ids[1]}_{datetime.now().strftime('%Y%m%d%H%M%S')}"
    return hashlib.md5(room_string.encode()).hexdigest()[:10]

async def create_chat_room(user1_id: int, user2_id: int, bot: Bot, chat_reason: str = "مچ"):
    """ایجاد گروه تلگرام با Userbot"""
    
    # استفاده از Userbot برای ایجاد گروه
    from .chat_room_creator import create_private_group
    
    # ایجاد گروه جدید
    success, message, chat_id = await create_private_group(user1_id, user2_id, bot.username)
    
    if success:
        logger.info(f"Chat room created successfully: {message}")
        return True, message
    else:
        logger.error(f"Failed to create chat room: {message}")
        return False, message

# Handler برای پیام‌های چت
@router.message(F.text & ~F.text.startswith("/"))
async def handle_chat_message(message: Message, state: FSMContext, bot: Bot):
    """مدیریت پیام‌های چت"""
    from .inline_chat_room import handle_chat_message as process_message
    await process_message(message, state, bot)
            
            # اضافه کردن کاربر اول به گروه
            try:
                await bot.add_chat_member(
                    chat_id=chat_id,
                    user_id=user1_id
                )
            except Exception as e:
                logger.warning(f"Could not add user1 directly: {e}")
                # ایجاد لینک دعوت برای کاربر اول
                invite_link_1 = await bot.create_chat_invite_link(
                    chat_id=chat_id,
                    member_limit=1,
                    expire_date=datetime.now() + timedelta(hours=1),
                    creates_join_request=False
                )
                
                # ارسال لینک به کاربر اول
                await bot.send_message(
                    user1_id,
                    f"✅ <b>اتاق گفتگو ایجاد شد!</b>\n\n"
                    f"🆔 کد اتاق: <code>{room_id}</code>\n"
                    f"📌 دلیل دسترسی: {chat_reason}\n\n"
                    f"⚠️ <b>توجه:</b>\n"
                    f"• هویت طرف مقابل محرمانه است\n"
                    f"• ربات نظارتگر پیام‌ها را کنترل می‌کند\n"
                    f"• از ارسال اطلاعات شخصی خودداری کنید\n\n"
                    f"🔗 برای ورود به اتاق روی لینک زیر کلیک کنید:\n"
                    f"{invite_link_1.invite_link}",
                    parse_mode='HTML'
                )
            
            # اضافه کردن کاربر دوم به گروه
            try:
                await bot.add_chat_member(
                    chat_id=chat_id,
                    user_id=user2_id
                )
            except Exception as e:
                logger.warning(f"Could not add user2 directly: {e}")
                # ایجاد لینک دعوت برای کاربر دوم
                invite_link_2 = await bot.create_chat_invite_link(
                    chat_id=chat_id,
                    member_limit=1,
                    expire_date=datetime.now() + timedelta(hours=1),
                    creates_join_request=False
                )
                
                # ارسال لینک به کاربر دوم
                await bot.send_message(
                    user2_id,
                    f"🔔 <b>دعوت به گفتگو!</b>\n\n"
                    f"🆔 کد اتاق: <code>{room_id}</code>\n"
                    f"📌 دلیل دسترسی: {chat_reason}\n\n"
                    f"⚠️ <b>توجه:</b>\n"
                    f"• هویت طرف مقابل محرمانه است\n"
                    f"• ربات نظارتگر پیام‌ها را کنترل می‌کند\n"
                    f"• قوانین گفتگو را رعایت کنید\n\n"
                    f"🔗 برای ورود به اتاق روی لینک زیر کلیک کنید:\n"
                    f"{invite_link_2.invite_link}",
                    parse_mode='HTML'
                )
            
            # ارسال پیام خوش‌آمدگویی در گروه
            welcome_message = (
                "🤖 <b>ربات نظارتگر</b>\n\n"
                "به اتاق گفتگو خوش آمدید! 👋\n\n"
                f"🆔 کد اتاق: <code>{room_id}</code>\n"
                f"📌 دلیل ایجاد: {chat_reason}\n\n"
                "⚠️ <b>قوانین مهم:</b>\n"
                "• هویت طرف مقابل محرمانه است\n"
                "• از افشای اطلاعات شخصی خودداری کنید\n"
                "• احترام متقابل را رعایت کنید\n"
                "• محتوای نامناسب = اخراج فوری\n\n"
                "💡 دستورات:\n"
                "/rules - نمایش قوانین\n"
                "/report - گزارش تخلف\n"
                "/leave - خروج از اتاق\n\n"
                "موفق باشید! 💚"
            )
            
            await bot.send_message(
                chat_id=chat_id,
                text=welcome_message,
                parse_mode='HTML'
            )
            
            # ذخیره اطلاعات اتاق
            active_chat_rooms[room_id] = {
                "chat_id": chat_id,
                "user1": user1_id,
                "user2": user2_id,
                "created": datetime.now(),
                "reason": chat_reason,
                "status": "active"
            }
            
            return True, room_id
            
        except Exception as e:
            logger.error(f"Error creating chat group: {e}")
            
            # روش جایگزین: استفاده از لینک دعوت
            # اگر نتوانستیم گروه بسازیم، از گروه از پیش ساخته استفاده کنیم
            return await create_chat_room_alternative(user1_id, user2_id, bot, chat_reason, room_id)

async def create_chat_room_alternative(user1_id: int, user2_id: int, bot: Bot, chat_reason: str, room_id: str):
    """روش جایگزین: استفاده از گروه از پیش ساخته شده"""
    
    # لیست گروه‌های از پیش ساخته شده
    # این باید از دیتابیس یا کانفیگ خوانده شود
    pre_made_groups = [
        # {"chat_id": -1001234567890, "in_use": False},
        # اضافه کردن گروه‌های واقعی
    ]
    
    # پیدا کردن گروه خالی
    available_group = None
    for group in pre_made_groups:
        if not group.get("in_use", False):
            available_group = group
            group["in_use"] = True
            break
    
    if not available_group:
        # اگر گروه خالی نداریم
        join_keyboard = InlineKeyboardMarkup(
            inline_keyboard=[
                [InlineKeyboardButton(
                    text="🔄 تلاش مجدد",
                    callback_data=f"retry_chat_{user1_id}_{user2_id}"
                )]
            ]
        )
        
        await bot.send_message(
            user1_id,
            "⚠️ <b>همه اتاق‌های گفتگو پر هستند!</b>\n\n"
            "لطفاً چند دقیقه دیگر تلاش کنید.",
            reply_markup=join_keyboard,
            parse_mode='HTML'
        )
        
        return False, "اتاق خالی موجود نیست"
    
    # ایجاد لینک دعوت برای گروه موجود
    chat_id = available_group["chat_id"]
    
    try:
        # پاک کردن پیام‌های قبلی گروه
        # TODO: implement cleanup
        
        # ایجاد لینک‌های دعوت
        invite_link = await bot.create_chat_invite_link(
            chat_id=chat_id,
            member_limit=2,
            expire_date=datetime.now() + timedelta(hours=1),
            creates_join_request=False
        )
        
        # دکمه ورود به گروه
        join_keyboard = InlineKeyboardMarkup(
            inline_keyboard=[
                [InlineKeyboardButton(
                    text="🚪 ورود به اتاق گفتگو",
                    url=invite_link.invite_link
                )]
            ]
        )
        
        # ارسال به هر دو کاربر
        message_text = (
            f"✅ <b>اتاق گفتگو آماده است!</b>\n\n"
            f"🆔 کد اتاق: <code>{room_id}</code>\n"
            f"📌 دلیل دسترسی: {chat_reason}\n\n"
            f"⚠️ <b>توجه:</b>\n"
            f"• هویت طرف مقابل محرمانه است\n"
            f"• ربات نظارتگر پیام‌ها را کنترل می‌کند\n\n"
            f"برای ورود روی دکمه زیر کلیک کنید:"
        )
        
        await bot.send_message(
            user1_id,
            message_text,
            reply_markup=join_keyboard,
            parse_mode='HTML'
        )
        
        await bot.send_message(
            user2_id,
            message_text,
            reply_markup=join_keyboard,
            parse_mode='HTML'
        )
        
        # ذخیره اطلاعات
        active_chat_rooms[room_id] = {
            "chat_id": chat_id,
            "user1": user1_id,
            "user2": user2_id,
            "created": datetime.now(),
            "reason": chat_reason,
            "status": "active"
        }
        
        return True, room_id
        
    except Exception as e:
        logger.error(f"Error in alternative method: {e}")
        return False, "خطا در ایجاد اتاق"

# ==================== Callback Handlers ====================

@router.callback_query(F.data.startswith("chat_rules_"))
async def handle_chat_rules(callback: CallbackQuery):
    """نمایش قوانین گفتگو"""
    room_id = callback.data.split("_")[2]
    
    rules_text = (
        "📋 <b>قوانین اتاق گفتگو:</b>\n\n"
        "1️⃣ احترام متقابل را رعایت کنید\n"
        "2️⃣ از ارسال محتوای نامناسب خودداری کنید\n"
        "3️⃣ اطلاعات شخصی را با احتیاط به اشتراک بگذارید\n"
        "4️⃣ از ارسال لینک‌های مشکوک پرهیز کنید\n"
        "5️⃣ در صورت مشاهده رفتار نامناسب، گزارش دهید\n\n"
        "⚠️ نقض قوانین منجر به محدودیت یا مسدودی می‌شود\n\n"
        f"🆔 کد اتاق: <code>{room_id}</code>"
    )
    
    await callback.answer(
        rules_text,
        show_alert=True
    )

@router.callback_query(F.data.startswith("cancel_chat_"))
async def handle_cancel_chat(callback: CallbackQuery):
    """انصراف از ورود به چت"""
    room_id = callback.data.split("_")[2]
    user_id = callback.from_user.id
    
    if room_id in active_chat_rooms:
        room = active_chat_rooms[room_id]
        
        # بررسی کاربر
        if user_id in [room["user1"], room["user2"]]:
            # تغییر وضعیت اتاق
            room["status"] = "cancelled"
            
            await callback.message.edit_text(
                "❌ شما از ورود به اتاق گفتگو انصراف دادید.\n\n"
                "می‌توانید در هر زمان دوباره درخواست گفتگو دهید.",
                parse_mode='HTML'
            )
        else:
            await callback.answer(
                "⚠️ این دعوت‌نامه برای شما نیست!",
                show_alert=True
            )
    else:
        await callback.answer(
            "⚠️ این اتاق گفتگو منقضی شده است.",
            show_alert=True
        )

# ==================== پیام‌های گروه ====================

@router.message(F.chat.type.in_(["group", "supergroup"]), F.text == "/rules")
async def handle_rules_command(message: Message):
    """نمایش قوانین گروه"""
    rules_text = (
        "📋 <b>قوانین اتاق گفتگو:</b>\n\n"
        "1️⃣ احترام متقابل را رعایت کنید\n"
        "2️⃣ از افشای اطلاعات شخصی خودداری کنید\n"
        "3️⃣ محتوای نامناسب ممنوع است\n"
        "4️⃣ ارسال لینک و تبلیغات ممنوع\n"
        "5️⃣ هویت طرف مقابل محرمانه است\n\n"
        "⚠️ نقض قوانین = اخراج فوری\n\n"
        "💡 دستورات:\n"
        "/report - گزارش تخلف\n"
        "/leave - خروج از اتاق"
    )
    await message.answer(rules_text, parse_mode='HTML')

@router.message(F.chat.type.in_(["group", "supergroup"]), F.text == "/leave")
async def handle_leave_command(message: Message, bot: Bot):
    """خروج از گروه"""
    user_id = message.from_user.id
    chat_id = message.chat.id
    
    # پیدا کردن اتاق
    room_info = None
    for room_id, info in active_chat_rooms.items():
        if info.get("chat_id") == chat_id:
            room_info = info
            break
    
    if room_info and user_id in [room_info["user1"], room_info["user2"]]:
        await message.answer(
            f"👋 {message.from_user.first_name} از گفتگو خارج شد.\n"
            "اتاق بسته می‌شود.",
            parse_mode='HTML'
        )
        
        # اخراج کاربر از گروه
        try:
            await bot.ban_chat_member(chat_id, user_id)
            await bot.unban_chat_member(chat_id, user_id)  # unban فوری برای امکان join مجدد در آینده
        except:
            pass
        
        # تغییر وضعیت اتاق
        room_info["status"] = "ended"

@router.message(F.chat.type.in_(["group", "supergroup"]), F.text == "/report")
async def handle_report_command(message: Message):
    """گزارش تخلف"""
    report_keyboard = InlineKeyboardMarkup(
        inline_keyboard=[
            [InlineKeyboardButton(text="🚫 محتوای نامناسب", callback_data="report_inappropriate")],
            [InlineKeyboardButton(text="📢 اسپم/تبلیغات", callback_data="report_spam")],
            [InlineKeyboardButton(text="🔒 افشای اطلاعات شخصی", callback_data="report_privacy")],
            [InlineKeyboardButton(text="😤 رفتار توهین‌آمیز", callback_data="report_abuse")],
            [InlineKeyboardButton(text="❌ انصراف", callback_data="report_cancel")]
        ]
    )
    
    await message.answer(
        "🚨 <b>گزارش تخلف</b>\n\n"
        "نوع تخلف را انتخاب کنید:",
        reply_markup=report_keyboard,
        parse_mode='HTML'
    )

@router.message(F.chat.type.in_(["group", "supergroup"]))
async def handle_group_message(message: Message, bot: Bot):
    """مدیریت پیام‌های داخل گروه (ربات نظارتگر)"""
    
    # بررسی دستورات
    if message.text and message.text.startswith('/'):
        return  # دستورات در handler های بالا مدیریت می‌شوند
    
    # بررسی محتوای نامناسب
    if message.text and check_inappropriate_content(message.text):
        await message.delete()
        
        # هشدار به کاربر
        user_id = message.from_user.id
        chat_id = message.chat.id
        
        # شمارش تخلفات (در حافظه موقت)
        violations_key = f"{chat_id}_{user_id}"
        if not hasattr(handle_group_message, 'violations'):
            handle_group_message.violations = {}
        
        handle_group_message.violations[violations_key] = handle_group_message.violations.get(violations_key, 0) + 1
        
        if handle_group_message.violations[violations_key] >= 3:
            # اخراج بعد از 3 تخلف
            await message.answer(
                f"⛔ {message.from_user.first_name} به دلیل نقض مکرر قوانین از گروه اخراج شد.",
                parse_mode='HTML'
            )
            await bot.ban_chat_member(chat_id, user_id)
        else:
            await message.answer(
                f"⚠️ <b>هشدار {handle_group_message.violations[violations_key]}/3</b>\n"
                f"{message.from_user.first_name}, پیام شما حذف شد.\n"
                "در صورت تکرار، اخراج خواهید شد.",
                parse_mode='HTML'
            )
        return
    
    # مخفی کردن username ها در پیام‌ها
    if message.text and '@' in message.text:
        # جایگزینی username ها با ***
        import re
        cleaned_text = re.sub(r'@\w+', '@***', message.text)
        if cleaned_text != message.text:
            await message.delete()
            await message.answer(
                f"👤 <b>کاربر:</b>\n{cleaned_text}\n\n"
                f"<i>💡 نکته: username ها برای حفظ حریم خصوصی مخفی شدند</i>",
                parse_mode='HTML'
            )

def check_inappropriate_content(text: str) -> bool:
    """بررسی محتوای نامناسب در پیام"""
    if not text:
        return False
    
    # لیست کلمات ممنوعه (نمونه)
    banned_words = [
        # اضافه کردن کلمات نامناسب
    ]
    
    text_lower = text.lower()
    for word in banned_words:
        if word in text_lower:
            return True
    
    return False

# Export
__all__ = ['router', 'create_chat_room', 'ChatStates']
