#!/usr/bin/env python3
"""Chat Room Pool Manager - مدیریت استخر گروه‌های آماده"""

import logging
from typing import Dict, List, Optional
from datetime import datetime, timedelta
from aiogram import Bot
from aiogram.types import ChatPermissions, InlineKeyboardMarkup, InlineKeyboardButton
from typing import Tuple

logger = logging.getLogger(__name__)

# لیست گروه‌های از پیش ساخته شده
# این ID ها را باید با ID گروه‌های واقعی خودتان جایگزین کنید
AVAILABLE_CHAT_GROUPS = [
    -1001234567890,  # گروه 1
    -1001234567891,  # گروه 2
    -1001234567892,  # گروه 3
    -1001234567893,  # گروه 4
    -1001234567894,  # گروه 5
    # می‌توانید گروه‌های بیشتری اضافه کنید
]

# وضعیت گروه‌ها (آزاد/اشغال)
chat_groups_status: Dict[int, Dict] = {
    chat_id: {"is_busy": False, "users": [], "started_at": None}
    for chat_id in AVAILABLE_CHAT_GROUPS
}

async def get_available_chat_group() -> Optional[int]:
    """دریافت یک گروه آزاد از استخر"""
    for chat_id, status in chat_groups_status.items():
        if not status["is_busy"]:
            return chat_id
    return None

async def assign_chat_group(chat_id: int, user1_id: int, user2_id: int) -> bool:
    """اختصاص گروه به دو کاربر"""
    if chat_id in chat_groups_status:
        chat_groups_status[chat_id] = {
            "is_busy": True,
            "users": [user1_id, user2_id],
            "started_at": datetime.now()
        }
        return True
    return False

async def release_chat_group(chat_id: int, bot: Bot) -> bool:
    """آزاد کردن گروه و حذف کاربران"""
    if chat_id not in chat_groups_status:
        return False
    
    try:
        # حذف همه کاربران از گروه (به جز بات)
        users = chat_groups_status[chat_id].get("users", [])
        for user_id in users:
            try:
                await bot.ban_chat_member(chat_id, user_id)
                await bot.unban_chat_member(chat_id, user_id)  # برای امکان ورود مجدد در آینده
            except Exception as e:
                logger.warning(f"Could not remove user {user_id} from chat {chat_id}: {e}")
        
        # پاک کردن پیام‌های گروه (اختیاری)
        # این قابلیت در Bot API موجود نیست، باید دستی انجام شود
        
        # آزاد کردن گروه
        chat_groups_status[chat_id] = {
            "is_busy": False,
            "users": [],
            "started_at": None
        }
        return True
        
    except Exception as e:
        logger.error(f"Error releasing chat group {chat_id}: {e}")
        return False

async def setup_chat_group(bot: Bot, chat_id: int, user1_id: int, user2_id: int, 
                          user1_name: str, user2_name: str) -> Tuple[str, str]:
    """تنظیم گروه برای دو کاربر"""
    try:
        # تغییر نام گروه
        room_title = f"💬 گفتگو: {user1_name[:10]} و {user2_name[:10]}"
        await bot.set_chat_title(chat_id, room_title)
        
        # تنظیم دسترسی‌ها
        permissions = ChatPermissions(
            can_send_messages=True,
            can_send_media_messages=True,
            can_send_polls=False,
            can_send_other_messages=True,
            can_add_web_page_previews=True,
            can_change_info=False,
            can_invite_users=False,
            can_pin_messages=False
        )
        await bot.set_chat_permissions(chat_id, permissions)
        
        # ایجاد لینک دعوت برای کاربر اول
        invite_link_1 = await bot.create_chat_invite_link(
            chat_id=chat_id,
            member_limit=1,
            expire_date=datetime.now() + timedelta(hours=24),
            creates_join_request=False
        )
        
        # ایجاد لینک دعوت برای کاربر دوم
        invite_link_2 = await bot.create_chat_invite_link(
            chat_id=chat_id,
            member_limit=1,
            expire_date=datetime.now() + timedelta(hours=24),
            creates_join_request=False
        )
        
        return invite_link_1.invite_link, invite_link_2.invite_link
        
    except Exception as e:
        logger.error(f"Error setting up chat group: {e}")
        raise

async def cleanup_expired_chats(bot: Bot, max_duration_hours: int = 24):
    """پاکسازی گروه‌های منقضی شده"""
    now = datetime.now()
    for chat_id, status in chat_groups_status.items():
        if status["is_busy"] and status["started_at"]:
            duration = now - status["started_at"]
            if duration.total_seconds() > max_duration_hours * 3600:
                await release_chat_group(chat_id, bot)
                logger.info(f"Released expired chat group {chat_id}")
