#!/usr/bin/env python3
"""Inline Chat Room - اتاق گفتگو درون بات (بدون نیاز به گروه)"""

import logging
from typing import Dict, Optional
from datetime import datetime
from aiogram import Router, F, Bot
from aiogram.types import Message, ReplyKeyboardMarkup, KeyboardButton
from aiogram.fsm.context import FSMContext
from aiogram.fsm.state import State, StatesGroup
from database.database import db
from database.models import User, Match
from sqlalchemy import select

router = Router()
logger = logging.getLogger(__name__)

# ذخیره جفت‌های در حال چت
active_chats: Dict[int, int] = {}  # {user1_id: user2_id, user2_id: user1_id}
chat_sessions: Dict[str, Dict] = {}  # اطلاعات جلسات چت

class InlineChatStates(StatesGroup):
    """حالت‌های چت درون بات"""
    in_chat = State()
    waiting_partner = State()

async def start_inline_chat(user1_id: int, user2_id: int, bot: Bot, chat_reason: str = "مچ") -> tuple[bool, str]:
    """شروع چت درون بات بین دو کاربر"""
    
    # بررسی اینکه کاربران در چت دیگری نباشند
    if user1_id in active_chats or user2_id in active_chats:
        return False, "یکی از کاربران در حال حاضر در چت دیگری است"
    
    # ثبت جفت چت
    active_chats[user1_id] = user2_id
    active_chats[user2_id] = user1_id
    
    # ایجاد session ID
    session_id = f"{min(user1_id, user2_id)}_{max(user1_id, user2_id)}_{datetime.now().strftime('%Y%m%d%H%M%S')}"
    chat_sessions[session_id] = {
        "user1_id": user1_id,
        "user2_id": user2_id,
        "started_at": datetime.now(),
        "reason": chat_reason,
        "message_count": 0
    }
    
    # کیبورد چت
    chat_keyboard = ReplyKeyboardMarkup(
        keyboard=[
            [KeyboardButton(text="❌ پایان گفتگو")],
            [KeyboardButton(text="👤 نمایش پروفایل")],
            [KeyboardButton(text="🚫 گزارش و خروج")]
        ],
        resize_keyboard=True
    )
    
    try:
        # دریافت نام‌های کاربران
        async with db.get_session() as session:
            user1_result = await session.execute(
                select(User).where(User.id == user1_id)
            )
            user1 = user1_result.scalar_one_or_none()
            
            user2_result = await session.execute(
                select(User).where(User.id == user2_id)
            )
            user2 = user2_result.scalar_one_or_none()
        
        user1_name = user1.first_name if user1 else "ناشناس"
        user2_name = user2.first_name if user2 else "ناشناس"
        
        # ارسال پیام شروع به کاربر اول
        await bot.send_message(
            user1_id,
            f"🎉 <b>گفتگو با {user2_name} شروع شد!</b>\n\n"
            "💬 پیام‌های شما مستقیماً ارسال می‌شود\n"
            "📝 همه پیام‌ها ناشناس هستند\n"
            "❌ برای پایان از دکمه استفاده کنید\n\n"
            f"📌 دلیل: {chat_reason}",
            reply_markup=chat_keyboard,
            parse_mode='HTML'
        )
        
        # ارسال پیام شروع به کاربر دوم
        await bot.send_message(
            user2_id,
            f"🎉 <b>گفتگو با {user1_name} شروع شد!</b>\n\n"
            "💬 پیام‌های شما مستقیماً ارسال می‌شود\n"
            "📝 همه پیام‌ها ناشناس هستند\n"
            "❌ برای پایان از دکمه استفاده کنید\n\n"
            f"📌 دلیل: {chat_reason}",
            reply_markup=chat_keyboard,
            parse_mode='HTML'
        )
        
        logger.info(f"Inline chat started: {session_id}")
        return True, session_id
        
    except Exception as e:
        logger.error(f"Error starting chat: {e}")
        # پاک کردن در صورت خطا
        active_chats.pop(user1_id, None)
        active_chats.pop(user2_id, None)
        chat_sessions.pop(session_id, None)
        return False, str(e)

@router.message(InlineChatStates.in_chat)
async def handle_chat_message(message: Message, state: FSMContext, bot: Bot):
    """مدیریت پیام‌های چت"""
    user_id = message.from_user.id
    partner_id = active_chats.get(user_id)
    
    if not partner_id:
        await message.answer("❌ گفتگو پایان یافته است")
        await state.clear()
        return
    
    # دکمه‌های کنترل
    if message.text == "❌ پایان گفتگو":
        await end_chat(user_id, partner_id, bot, state)
        return
    elif message.text == "🚫 گزارش و خروج":
        await report_and_exit(user_id, partner_id, bot, state, message)
        return
    elif message.text == "👤 نمایش پروفایل":
        await show_partner_profile(user_id, partner_id, bot)
        return
    
    # ارسال پیام به شریک چت
    try:
        # ارسال بر اساس نوع پیام
        if message.text:
            await bot.send_message(
                partner_id,
                f"👤 <b>پیام جدید:</b>\n{message.text}",
                parse_mode='HTML'
            )
        elif message.photo:
            await bot.send_photo(
                partner_id,
                message.photo[-1].file_id,
                caption="👤 عکس دریافتی"
            )
        elif message.voice:
            await bot.send_voice(
                partner_id,
                message.voice.file_id,
                caption="👤 پیام صوتی"
            )
        elif message.sticker:
            await bot.send_sticker(partner_id, message.sticker.file_id)
        
        # تأیید ارسال
        await message.reply("✅ ارسال شد")
        
        # به‌روزرسانی تعداد پیام‌ها
        for session_id, session_data in chat_sessions.items():
            if user_id in [session_data["user1_id"], session_data["user2_id"]]:
                session_data["message_count"] += 1
                break
                
    except Exception as e:
        logger.error(f"Error sending message: {e}")
        await message.answer("❌ خطا در ارسال پیام")

async def end_chat(user1_id: int, user2_id: int, bot: Bot, state: FSMContext):
    """پایان گفتگو"""
    # حذف از لیست فعال
    active_chats.pop(user1_id, None)
    active_chats.pop(user2_id, None)
    
    # کیبورد منو
    menu_keyboard = ReplyKeyboardMarkup(
        keyboard=[
            [KeyboardButton(text="🔍 یافتن مچ")],
            [KeyboardButton(text="👤 پروفایل من")],
            [KeyboardButton(text="💬 گفتگوهای من")]
        ],
        resize_keyboard=True
    )
    
    # پیام پایان
    end_message = (
        "👋 <b>گفتگو پایان یافت</b>\n\n"
        "امیدواریم گفتگوی خوبی داشته باشید!\n"
        "می‌توانید دوباره جستجو کنید."
    )
    
    await bot.send_message(user1_id, end_message, reply_markup=menu_keyboard, parse_mode='HTML')
    await bot.send_message(user2_id, end_message, reply_markup=menu_keyboard, parse_mode='HTML')
    
    # پاک کردن state
    await state.clear()
    
    logger.info(f"Chat ended between {user1_id} and {user2_id}")

async def show_partner_profile(user_id: int, partner_id: int, bot: Bot):
    """نمایش پروفایل شریک چت"""
    async with db.get_session() as session:
        result = await session.execute(
            select(User).where(User.id == partner_id)
        )
        partner = result.scalar_one_or_none()
        
        if partner:
            profile = (
                f"👤 <b>پروفایل:</b>\n\n"
                f"نام: {partner.first_name or 'ناشناس'}\n"
                f"سن: {partner.age or 'نامشخص'}\n"
                f"شهر: {partner.city or 'نامشخص'}\n"
                f"بیو: {partner.bio or 'ندارد'}"
            )
            await bot.send_message(user_id, profile, parse_mode='HTML')
        else:
            await bot.send_message(user_id, "❌ پروفایل یافت نشد")

async def report_and_exit(user_id: int, partner_id: int, bot: Bot, state: FSMContext, message: Message):
    """گزارش و خروج از چت"""
    # ثبت گزارش در دیتابیس
    # ...
    
    await message.answer("🚫 گزارش ثبت شد و گفتگو پایان یافت")
    await end_chat(user_id, partner_id, bot, state)
