#!/usr/bin/env python3
"""Matching display handlers - مرحله 7 نمایش پروفایل‌ها"""

import json
import logging
import os
from typing import Dict, List, Optional
from aiogram import Router, F
from aiogram.types import Message, ReplyKeyboardMarkup, KeyboardButton, FSInputFile
from aiogram.fsm.context import FSMContext
from database.database import db
from database.models import User, Swipe, Match, ProfileView, Notification
from sqlalchemy import select, and_, or_, func
from datetime import datetime, timedelta
from typing import Dict, List, Optional
import json
import random
import logging

router = Router()
logger = logging.getLogger(__name__)

# ذخیره کاندیدها و ایندکس فعلی
user_candidates: Dict[int, List[User]] = {}
current_candidate_index: Dict[int, int] = {}
current_candidate: Dict[int, User] = {}

def get_default_profile_photo(gender):
    """دریافت مسیر عکس پیش‌فرض بر اساس جنسیت"""
    # gender: 1 = مرد، 2 = زن، 3 = غیره
    base_path = os.path.join(os.path.dirname(os.path.dirname(__file__)), 'assets')
    
    if gender == 1 or gender == "1":
        photo_path = os.path.join(base_path, 'default_male.jpg')
    elif gender == 2 or gender == "2":
        photo_path = os.path.join(base_path, 'default_female.jpg')
    else:
        photo_path = os.path.join(base_path, 'default_other.jpg')
    
    # اگر فایل وجود نداشت، از عکس پیش‌فرض عمومی استفاده کن
    if not os.path.exists(photo_path):
        photo_path = os.path.join(base_path, 'default_other.jpg')
    
    return photo_path if os.path.exists(photo_path) else None

class MatchingManager:
    @staticmethod
    async def get_online_status(user: User) -> str:
        """دریافت وضعیت آنلاین کاربر"""
        # بررسی وجود فیلد last_seen
        if not hasattr(user, 'last_seen') or user.last_seen is None:
            return "⚫ آفلاین"
        
        now = datetime.now()
        diff = now - user.last_seen
        
        if diff.total_seconds() < 300:  # 5 دقیقه
            return "🟢 آنلاین"
        elif diff.total_seconds() < 3600:  # 1 ساعت
            minutes = int(diff.total_seconds() / 60)
            return f"⚪ {minutes} دقیقه پیش"
        elif diff.total_seconds() < 86400:  # 24 ساعت
            hours = int(diff.total_seconds() / 3600)
            return f"⚪ {hours} ساعت پیش"
        else:
            days = int(diff.total_seconds() / 86400)
            return f"⚪ {days} روز پیش"

# ==================== جستجو و نمایش کاندیدها ====================
async def find_and_show_candidates(message: Message, state: FSMContext, filters: Dict):
    """جستجو و نمایش کاندیدها بر اساس فیلترها"""
    user_id = message.from_user.id
    
    async with db.get_session() as session:
        # دریافت کاربران رد شده
        rejected_result = await session.execute(
            select(Swipe.to_user).where(
                and_(
                    Swipe.from_user == user_id,
                    Swipe.action.in_([-1, 0])  # رد شده یا پاس داده شده
                )
            )
        )
        rejected_ids = [row[0] for row in rejected_result.fetchall()]
        rejected_ids.append(user_id)  # خود کاربر
        
        # لیست ادمین‌ها که نباید نمایش داده شوند
        admin_ids = [7203796090]  # اضافه کردن ID های ادمین
        
        # ساخت کوئری
        query = select(User).where(
            and_(
                User.id.not_in(rejected_ids),
                User.id.not_in(admin_ids),  # فیلتر کردن ادمین‌ها
                User.is_active == True
            )
        )
        
        # اعمال فیلترها (فقط اگر وجود داشته باشند)
        if filters.get('gender_pref'):
            query = query.where(User.gender == filters['gender_pref'])
        
        if filters.get('age_min'):
            query = query.where(User.age >= filters['age_min'])
        if filters.get('age_max'):
            query = query.where(User.age <= filters['age_max'])
        
        if filters.get('country'):
            query = query.where(User.country == filters['country'])
        
        if filters.get('city'):
            query = query.where(User.city == filters['city'])
        
        if filters.get('photo_required'):
            query = query.where(User.photos.isnot(None))
        
        # فیلتر بر اساس علایق (اگر انتخاب شده)
        if filters.get('interests'):
            # جستجوی کاربرانی که حداقل یک علاقه مشترک دارند
            interest_conditions = []
            for interest in filters['interests']:
                interest_conditions.append(User.interests.contains(interest))
            if interest_conditions:
                query = query.where(or_(*interest_conditions))
        
        # اجرای کوئری و استخراج فقط ID ها
        result = await session.execute(query.limit(30))
        candidate_objects = result.scalars().all()
        
        # استخراج ID ها در همان session
        candidates = []
        for user in candidate_objects:
            candidates.append(user.user_id)
        
        if not candidates:
            # کیبورد برای زمانی که کاربری پیدا نشد
            no_result_keyboard = ReplyKeyboardMarkup(
                keyboard=[
                    [KeyboardButton(text="🔍 جستجوی جدید"), KeyboardButton(text="⚙️ تغییر فیلترها")],
                    [KeyboardButton(text="🔙 بازگشت به منو")]
                ],
                resize_keyboard=True
            )
            
            await message.answer(
                "😔 <b>کاربری پیدا نشد!</b>\n\n"
                "💡 پیشنهادات:\n"
                "• فیلترها را تغییر دهید\n"
                "• بازه سنی را گسترده‌تر کنید\n"
                "• علایق کمتری انتخاب کنید\n"
                "• کمی بعد دوباره تلاش کنید",
                reply_markup=no_result_keyboard,
                parse_mode='HTML'
            )
            # Import states from filters module
            from .matching_filters import MatchingStates
            await state.set_state(MatchingStates.viewing)
            return
        
    # candidates الان از قبل لیست ID است
    if candidates:
        user_candidates[user_id] = candidates  # ذخیره لیست ID ها
        current_candidate_index[user_id] = 0
        
        # نمایش اولین کاندید با ID
        await show_candidate(message, state, candidates[0])
    else:
        await message.answer("❌ هیچ کاندیدی یافت نشد. لطفاً فیلترها را تغییر دهید.")

async def show_candidate(message: Message, state: FSMContext, candidate_id: int):
    """نمایش پروفایل یک کاندید - فقط با ID"""
    user_id = message.from_user.id
    
    # بررسی ورودی
    if not isinstance(candidate_id, int):
        await message.answer("❌ خطا در شناسایی کاربر.")
        return
    
    current_candidate[user_id] = candidate_id  # ذخیره ID
    
    # دریافت اطلاعات کامل کاربر و ساخت پروفایل در یک session
    async with db.get_session() as session:
        # دریافت کاربر از دیتابیس
        result = await session.execute(
            select(User).where(User.id == candidate_id)
        )
        candidate_fresh = result.scalar_one_or_none()
        
        if not candidate_fresh:
            await message.answer("❌ خطا در بارگذاری پروفایل.")
            return
        
        # استخراج تمام داده‌های مورد نیاز قبل از بستن session
        user_data = {
            'username': candidate_fresh.username,
            'first_name': candidate_fresh.first_name,
            'age': candidate_fresh.age,
            'gender': candidate_fresh.gender,
            'country': candidate_fresh.country,
            'city': candidate_fresh.city,
            'bio': candidate_fresh.bio,
            'interests': candidate_fresh.interests,
            'photos': candidate_fresh.photos,
            'last_seen': candidate_fresh.last_seen if hasattr(candidate_fresh, 'last_seen') else None,
            'user_id': candidate_fresh.user_id
        }
        
        # ثبت بازدید
        profile_view = ProfileView(
            viewer_id=user_id,
            viewed_user_id=candidate_id
        )
        session.add(profile_view)
        await session.commit()
    
    # ساخت متن پروفایل با داده‌های استخراج شده (خارج از session)
    profile_text = await build_profile_text_from_dict(user_data)
    
    # ذخیره اطلاعات مورد نیاز برای نمایش
    user_gender = user_data['gender']
    user_photos = user_data['photos']
    
    # کیبورد عملیات
    action_keyboard = ReplyKeyboardMarkup(
        keyboard=[
            [KeyboardButton(text="👎 دیس لایک"), KeyboardButton(text="💬 چت/گفتگو"), KeyboardButton(text="❤️ لایک")],
            # دکمه تغییر فیلترها موقتاً غیرفعال - تا زمان افزایش کاربران
            # [KeyboardButton(text="➡️ بعدی"), KeyboardButton(text="⚙️ تغییر فیلترها"), KeyboardButton(text="💎 سوپر لایک")],
            [KeyboardButton(text="➡️ بعدی"), KeyboardButton(text="💎 سوپر لایک")],
            [KeyboardButton(text="🔙 بازگشت به منو")]
        ],
        resize_keyboard=True,
        one_time_keyboard=False
    )
    
    # بررسی عکس با اطلاعات ذخیره شده
    has_photo = False
    main_photo_id = None
    
    if user_photos:
        try:
            import json
            photos_data = json.loads(user_photos) if isinstance(user_photos, str) else user_photos
            if photos_data and isinstance(photos_data, list) and len(photos_data) > 0:
                has_photo = True
                main_photo_id = photos_data[0]
        except:
            pass
    
    if has_photo and main_photo_id:
        try:
            await message.answer_photo(
                photo=main_photo_id,
                caption=profile_text,
                reply_markup=action_keyboard,
                parse_mode='HTML'
            )
        except Exception as e:
            logger.error(f"Error sending photo: {e}")
            # اگر ارسال عکس موفق نبود، از عکس دیفالت استفاده کن
            default_photo_path = get_default_profile_photo(user_gender)
            if default_photo_path and os.path.exists(default_photo_path):
                try:
                    photo = FSInputFile(default_photo_path)
                    await message.answer_photo(
                        photo=photo,
                        caption=profile_text,
                        reply_markup=action_keyboard,
                        parse_mode='HTML'
                    )
                except:
                    await message.answer(profile_text, reply_markup=action_keyboard, parse_mode='HTML')
            else:
                await message.answer(profile_text, reply_markup=action_keyboard, parse_mode='HTML')
    else:
        # اگر عکس نداشت، از عکس دیفالت استفاده کن
        default_photo_path = get_default_profile_photo(user_gender)
        if default_photo_path and os.path.exists(default_photo_path):
            try:
                photo = FSInputFile(default_photo_path)
                await message.answer_photo(
                    photo=photo,
                    caption=profile_text,
                    reply_markup=action_keyboard,
                    parse_mode='HTML'
                )
            except Exception as e:
                logger.error(f"Error sending default photo: {e}")
                await message.answer(profile_text, reply_markup=action_keyboard, parse_mode='HTML')
        else:
            await message.answer(profile_text, reply_markup=action_keyboard, parse_mode='HTML')
    
    # تنظیم state
    from .matching_filters import MatchingStates
    await state.set_state(MatchingStates.viewing)

async def build_profile_text_from_dict(user_data: dict) -> str:
    """ساخت متن پروفایل از دیکشنری داده‌ها"""
    # اطلاعات پایه
    name = user_data['username'] or user_data['first_name'] or 'کاربر'
    age = user_data['age'] if user_data['age'] else "نامشخص"
    
    # جنسیت
    gender_map = {
        1: '👨 مرد', '1': '👨 مرد',
        2: '👩 زن', '2': '👩 زن',
        3: '⚧️ غیره', '3': '⚧️ غیره'
    }
    gender = gender_map.get(user_data['gender'], '❓ نامشخص')
    
    # وضعیت آنلاین
    online_status = "⚫ آفلاین"  # پیش‌فرض
    if user_data['last_seen']:
        from datetime import datetime
        now = datetime.now()
        diff = now - user_data['last_seen']
        
        if diff.total_seconds() < 300:  # 5 دقیقه
            online_status = "🟢 آنلاین"
        elif diff.total_seconds() < 3600:  # 1 ساعت
            minutes = int(diff.total_seconds() / 60)
            online_status = f"⚪ {minutes} دقیقه پیش"
        elif diff.total_seconds() < 86400:  # 24 ساعت
            hours = int(diff.total_seconds() / 3600)
            online_status = f"⚪ {hours} ساعت پیش"
        else:
            days = int(diff.total_seconds() / 86400)
            online_status = f"⚪ {days} روز پیش"
    
    # ساخت متن
    text = f"🎯 <b>کاربر پیشنهادی</b>\n\n"
    text += f"👤 <b>نام:</b> {name}\n"
    text += f"🎂 <b>سن:</b> {age} ساله\n"
    text += f"⚧️ <b>جنسیت:</b> {gender}\n"
    
    if user_data['country']:
        text += f"🌍 <b>کشور:</b> {user_data['country']}\n"
    
    if user_data['city']:
        text += f"📍 <b>شهر:</b> {user_data['city']}\n"
    
    if user_data['bio']:
        text += f"\n💭 <b>درباره من:</b>\n<i>{user_data['bio']}</i>\n"
    
    if user_data['interests']:
        interests_list = user_data['interests'].split(',') if isinstance(user_data['interests'], str) else []
        if interests_list and any(i.strip() for i in interests_list):
            interests = ", ".join([i.strip() for i in interests_list if i.strip()])
            text += f"\n🎯 <b>علایق:</b> {interests}\n"
    
    text += f"\n📱 <b>وضعیت:</b> {online_status}\n"
    
    # شمارنده کاندیدها
    candidate_user_id = user_data['user_id']
    # توجه: user_candidates با viewer's user_id کلید می‌خورد نه candidate
    # بنابراین نمی‌توانیم اینجا شمارنده نشان دهیم
    
    text += f"\n💫 این کاربر را چطور می‌بینید؟"
    
    return text

async def get_user_photo(user: User) -> tuple[bool, Optional[str]]:
    """دریافت عکس اصلی کاربر"""
    if not user.photos:
        return False, None
    
    try:
        photos_data = json.loads(user.photos) if isinstance(user.photos, str) else user.photos
        if photos_data and isinstance(photos_data, list) and len(photos_data) > 0:
            photo_item = photos_data[0]
            if isinstance(photo_item, dict) and "file_id" in photo_item:
                return True, photo_item["file_id"]
            elif isinstance(photo_item, str):
                return True, photo_item
    except Exception as e:
        logger.error(f"Error parsing photos: {e}")
    
    return False, None

# ==================== هندلرهای دکمه‌های عملیات ====================
@router.message(F.text == "➡️ بعدی")
async def handle_next_candidate(message: Message, state: FSMContext):
    """نمایش کاندید بعدی"""
    user_id = message.from_user.id
    
    if user_id not in user_candidates:
        await message.answer("❌ ابتدا جستجو کنید.")
        return
    
    candidates = user_candidates[user_id]
    current_index = current_candidate_index.get(user_id, 0)
    
    # به کاندید بعدی برو
    next_index = current_index + 1
    
    if next_index >= len(candidates):
        # به اول لیست برگرد (بدون پیام)
        next_index = 0
    
    current_candidate_index[user_id] = next_index
    
    # user_candidates حالا فقط ID ها را دارد
    candidate_id = candidates[next_index]
    await show_candidate(message, state, candidate_id)

# جستجوی جدید غیرفعال شده - طبق درخواست کاربر
# @router.message(F.text == "🔍 جستجوی جدید")
# async def handle_new_search(message: Message, state: FSMContext):
#     """شروع جستجوی جدید"""
#     # پاک کردن داده‌های قبلی
#     user_id = message.from_user.id
#     if user_id in user_candidates:
#         del user_candidates[user_id]
#     if user_id in current_candidate_index:
#         del current_candidate_index[user_id]
    
#     # پاک کردن فیلترهای قبلی
#     from .matching_filters import user_filters, filters_completed
#     if user_id in user_filters:
#         del user_filters[user_id]
#     if user_id in filters_completed:
#         del filters_completed[user_id]
    
#     await message.answer(
#         "🔄 <b>تغییر فیلترها</b>\n\n"
#         "فیلترهای جدید را انتخاب کنید...",
#         parse_mode='HTML'
#     )
    
#     # شروع مجدد فرآیند فیلترها
#     from .matching_filters import start_matching_filters
#     await start_matching_filters(message, state), skip_welcome=True, force_filters=True)

# Export functions for other modules
__all__ = [
    'router',
    'find_and_show_candidates',
    'user_candidates',
    'current_candidate',
    'MatchingManager'
]
