#!/usr/bin/env python3
"""Matching filters handlers - مراحل 1-6 فیلترهای جستجو"""

from aiogram import Router, F
from aiogram.types import Message, ReplyKeyboardMarkup, KeyboardButton, ReplyKeyboardRemove
from aiogram.fsm.context import FSMContext
from aiogram.fsm.state import State, StatesGroup
from typing import Dict, List, Optional
import logging

router = Router()
logger = logging.getLogger(__name__)

# ذخیره فیلترهای کاربران
user_filters: Dict[int, Dict] = {}
# ذخیره وضعیت تکمیل فیلترها
filters_completed: Dict[int, bool] = {}

# FSM States
class MatchingStates(StatesGroup):
    search_type = State()      # مرحله 1: نوع جستجو
    gender_pref = State()       # مرحله 2: جنسیت
    age_range = State()         # مرحله 3: بازه سنی
    location = State()          # مرحله 4: کشور و شهر
    interests = State()         # مرحله 5: علایق
    photo_filter = State()      # مرحله 6: فیلتر عکس
    viewing = State()           # مرحله 7: مشاهده پروفایل‌ها

# ==================== مرحله 0: شروع فرآیند ====================
async def start_matching_flow_internal(message: Message, state: FSMContext, skip_welcome: bool = False, force_filters: bool = False):
    """شروع فرآیند یافتن مچ با فیلترها - تابع داخلی
    
    Args:
        skip_welcome: اگر True باشد، پیام خوش‌آمد نمایش داده نمی‌شود
        force_filters: اگر True باشد، حتماً فیلترها نمایش داده می‌شود (برای تغییر فیلترها)
    """
    user_id = message.from_user.id
    
    # بررسی ثبت‌نام کاربر
    from database.database import db
    from database.models import User
    from sqlalchemy import select
    
    async with db.get_session() as session:
        result = await session.execute(
            select(User).where(User.id == user_id)
        )
        user = result.scalar_one_or_none()
        
        if not user:
            await message.answer("❌ ابتدا باید ثبت‌نام کنید: /start")
            return
    
    # بررسی آیا فیلترها قبلاً تکمیل شده‌اند (فقط اگر force_filters نباشد)
    if not force_filters and user_id in filters_completed and filters_completed[user_id]:
        # اگر فیلترها قبلاً تکمیل شده، مستقیماً به نمایش پروفایل‌ها برو
        filters = user_filters.get(user_id, {})
        from .matching_display import find_and_show_candidates
        await find_and_show_candidates(message, state, filters)
        return

async def start_matching_flow_internal(message: Message, state: FSMContext, skip_welcome: bool = False):
    """فرآیند داخلی شروع matching - فیلترها موقتاً غیرفعال"""
    user_id = message.from_user.id
    
    # ⚠️ فیلترها موقتاً غیرفعال - مستقیماً پروفایل‌ها نمایش داده می‌شوند
    await message.answer(
        "🔍",
        reply_markup=ReplyKeyboardRemove()
    )
    
    # مستقیماً همه کاربران را نمایش بده (بدون فیلتر)
    from .matching_display import find_and_show_candidates
    await find_and_show_candidates(message, state, {})
    return
    
    # ==================== کد قبلی (موقتاً غیرفعال) ====================
    """
    # اگر اولین بار است، فیلترها را مقداردهی کن
    user_filters[user_id] = {
        'search_type': None,
        'gender_pref': None,
        'age_min': 18,
        'age_max': 99,
        'country': None,
        'city': None,
        'interests': [],
        'photo_required': False
    }
    
    # پیام خوش‌آمد فقط برای بار اول و نه برای تغییر فیلترها
    if not skip_welcome:
        welcome_text = (
            "🔍 <b>یافتن مچ</b>\n\n"
            "در این بخش می‌توانید مطابق با علایق و مشخصات خود، مچ مناسب پیدا کنید.\n\n"
            "📝 مراحل ساده‌ای برای تنظیم فیلترها خواهید داشت.\n"
            "⏭️ در هر مرحله می‌توانید گزینه‌ها را رد کنید و بعداً تغییر دهید.\n\n"
            "بیایید شروع کنیم! 🚀"
        )
        
        await message.answer(welcome_text, parse_mode='HTML')
    
    # مرحله 1: نوع جستجو
    search_type_keyboard = ReplyKeyboardMarkup(
        keyboard=[
            [KeyboardButton(text="👫 دوستیابی"), KeyboardButton(text="💑 پارتنر"), KeyboardButton(text="💍 ازدواج")],
            [KeyboardButton(text="⏭️ رد کردن این مرحله")],
            [KeyboardButton(text="🔙 بازگشت به منو")]
        ],
        resize_keyboard=True,
        one_time_keyboard=True
    )
    
    await message.answer(
        "🎯 <b>مرحله 1: نوع جستجو</b>\n\n"
        "به دنبال چه نوع رابطه‌ای هستید؟",
        reply_markup=search_type_keyboard,
        parse_mode='HTML'
    )
    
    await state.set_state(MatchingStates.search_type)
    """  # بستن کامنت کد قبلی

@router.message(F.text.contains("🔍 یافتن مچ"))
async def start_matching_flow(message: Message, state: FSMContext):
    """شروع فرآیند یافتن مچ با فیلترها - handler اصلی"""
    await start_matching_flow_internal(message, state, skip_welcome=False)

# ==================== مرحله 1: نوع جستجو ====================
@router.message(MatchingStates.search_type)
async def handle_search_type(message: Message, state: FSMContext):
    """انتخاب نوع جستجو"""
    user_id = message.from_user.id
    text = message.text
    
    if text == "🔙 بازگشت به منو":
        await back_to_menu(message, state)
        return
    
    # ذخیره نوع جستجو
    if text == "👫 دوستیابی":
        user_filters[user_id]['search_type'] = 'friendship'
    elif text == "💑 پارتنر":
        user_filters[user_id]['search_type'] = 'partner'
    elif text == "💍 ازدواج":
        user_filters[user_id]['search_type'] = 'marriage'
    elif text == "⏭️ رد کردن این مرحله":
        user_filters[user_id]['search_type'] = None
    else:
        await message.answer("❌ لطفاً یکی از گزینه‌های موجود را انتخاب کنید.")
        return
    
    # مرحله 2: جنسیت
    gender_keyboard = ReplyKeyboardMarkup(
        keyboard=[
            [KeyboardButton(text="👨 مرد"), KeyboardButton(text="👩 زن"), KeyboardButton(text="⚧️ غیره")],
            [KeyboardButton(text="⏭️ رد کردن این مرحله")],
            [KeyboardButton(text="⬅️ قبلی"), KeyboardButton(text="🔙 بازگشت به منو")]
        ],
        resize_keyboard=True,
        one_time_keyboard=True
    )
    
    await message.answer(
        "⚧️ <b>مرحله 2: جنسیت مورد نظر</b>\n\n"
        "به دنبال چه جنسیتی هستید؟",
        reply_markup=gender_keyboard,
        parse_mode='HTML'
    )
    
    await state.set_state(MatchingStates.gender_pref)

# ==================== مرحله 2: جنسیت ====================
@router.message(MatchingStates.gender_pref)
async def handle_gender_pref(message: Message, state: FSMContext):
    """انتخاب جنسیت مورد نظر"""
    user_id = message.from_user.id
    text = message.text
    
    if text == "🔙 بازگشت به منو":
        await back_to_menu(message, state)
        return
    
    if text == "⬅️ قبلی":
        # برگشت به مرحله 1: نوع جستجو
        await show_search_type_step(message, state)
        return
    
    # ذخیره جنسیت
    if text == "👨 مرد":
        user_filters[user_id]['gender_pref'] = 1
    elif text == "👩 زن":
        user_filters[user_id]['gender_pref'] = 2
    elif text == "⚧️ غیره":
        user_filters[user_id]['gender_pref'] = 3
    elif text == "⏭️ رد کردن این مرحله":
        user_filters[user_id]['gender_pref'] = None
    else:
        await message.answer("❌ لطفاً یکی از گزینه‌های موجود را انتخاب کنید.")
        return
    
    # مرحله 3: بازه سنی
    age_keyboard = ReplyKeyboardMarkup(
        keyboard=[
            [KeyboardButton(text="⏭️ رد کردن این مرحله")],
            [KeyboardButton(text="⬅️ قبلی"), KeyboardButton(text="🔙 بازگشت به منو")]
        ],
        resize_keyboard=True,
        one_time_keyboard=True
    )
    
    await message.answer(
        "🎂 <b>مرحله 3: بازه سنی</b>\n\n"
        "بازه سنی مورد نظر را به صورت زیر وارد کنید:\n"
        "📝 مثال: 18-30 یا 25-40\n\n"
        "یا روی دکمه '⏭️ رد کردن' کلیک کنید.",
        reply_markup=age_keyboard,
        parse_mode='HTML'
    )
    
    await state.set_state(MatchingStates.age_range)

# ==================== مرحله 3: بازه سنی ====================
@router.message(MatchingStates.age_range)
async def handle_age_range(message: Message, state: FSMContext):
    """انتخاب بازه سنی"""
    user_id = message.from_user.id
    text = message.text
    
    if text == "🔙 بازگشت به منو":
        await back_to_menu(message, state)
        return
    
    if text == "⬅️ قبلی":
        # برگشت به مرحله 2: جنسیت
        await show_gender_step(message, state)
        return
    
    # ذخیره بازه سنی
    if text == "⏭️ رد کردن این مرحله":
        user_filters[user_id]['age_min'] = 18
        user_filters[user_id]['age_max'] = 99
    else:
        # پردازش ورودی متنی بازه سنی
        import re
        age_pattern = re.match(r'^(\d{1,2})-(\d{1,3})$', text)
        
        if age_pattern:
            age_min = int(age_pattern.group(1))
            age_max = int(age_pattern.group(2))
            
            if age_min < 18 or age_max > 99 or age_min >= age_max:
                await message.answer("❌ بازه سنی نامعتبر است. لطفاً بین 18-99 وارد کنید.")
                return
            
            user_filters[user_id]['age_min'] = age_min
            user_filters[user_id]['age_max'] = age_max
        else:
            await message.answer("❌ لطفاً بازه سنی را به فرمت صحیح وارد کنید (18-30)")
            return
    
    # مرحله 4: موقعیت
    location_keyboard = ReplyKeyboardMarkup(
        keyboard=[
            [KeyboardButton(text="📍 ارسال موقعیت فعلی", request_location=True)],
            [KeyboardButton(text="🌍 انتخاب کشور")],
            [KeyboardButton(text="⏭️ رد کردن این مرحله")],
            [KeyboardButton(text="⬅️ قبلی"), KeyboardButton(text="🔙 بازگشت به منو")]
        ],
        resize_keyboard=True,
        one_time_keyboard=True
    )
    
    await message.answer(
        "📍 <b>مرحله 4: موقعیت مکانی</b>\n\n"
        "برای یافتن افراد نزدیک به شما، موقعیت خود را مشخص کنید:",
        reply_markup=location_keyboard,
        parse_mode='HTML'
    )
    
    await state.set_state(MatchingStates.location)

# ==================== مرحله 4: موقعیت ====================
@router.message(MatchingStates.location)
async def handle_location(message: Message, state: FSMContext):
    """انتخاب موقعیت مکانی"""
    user_id = message.from_user.id
    
    if message.text == "🔙 بازگشت به منو":
        await back_to_menu(message, state)
        return
    
    if message.text == "⬅️ قبلی":
        # برگشت به مرحله 3: بازه سنی
        await show_age_step(message, state)
        return
    
    if message.location:
        # کاربر موقعیت ارسال کرده
        user_filters[user_id]['latitude'] = message.location.latitude
        user_filters[user_id]['longitude'] = message.location.longitude
        await message.answer("📍 موقعیت شما ذخیره شد.")
    elif message.text == "🌍 انتخاب کشور":
        # نمایش لیست کشورها
        country_keyboard = ReplyKeyboardMarkup(
            keyboard=[
                [KeyboardButton(text="🇮🇷 ایران"), KeyboardButton(text="🇹🇷 ترکیه")],
                [KeyboardButton(text="🇦🇪 امارات"), KeyboardButton(text="🇩🇪 آلمان")],
                [KeyboardButton(text="🇺🇸 آمریکا"), KeyboardButton(text="🇨🇦 کانادا")],
                [KeyboardButton(text="⏭️ رد کردن این مرحله")],
                [KeyboardButton(text="🔙 بازگشت به منو")]
            ],
            resize_keyboard=True,
            one_time_keyboard=True
        )
        await message.answer(
            "🌍 کشور خود را انتخاب کنید:",
            reply_markup=country_keyboard
        )
        return
    elif message.text == "⏭️ رد کردن این مرحله":
        user_filters[user_id]['country'] = None
        user_filters[user_id]['city'] = None
    else:
        # ذخیره کشور انتخابی
        country_map = {
            "🇮🇷 ایران": "ایران",
            "🇹🇷 ترکیه": "ترکیه",
            "🇦🇪 امارات": "امارات",
            "🇩🇪 آلمان": "آلمان",
            "🇺🇸 آمریکا": "آمریکا",
            "🇨🇦 کانادا": "کانادا"
        }
        if message.text in country_map:
            user_filters[user_id]['country'] = country_map[message.text]
            await message.answer(f"✅ کشور {country_map[message.text]} ذخیره شد.")
        else:
            user_filters[user_id]['country'] = message.text
            await message.answer(f"✅ کشور {message.text} ذخیره شد.")
    
    # مرحله 5: علایق - استفاده از علایق پروفایل کاربر
    from database.database import db
    from database.models import User
    from sqlalchemy import select
    import json
    
    # دریافت علایق کاربر از پروفایل
    async with db.get_session() as session:
        result = await session.execute(
            select(User).where(User.id == user_id)
        )
        current_user = result.scalar_one_or_none()
    
    if current_user and current_user.interests:
        try:
            # Parse علایق کاربر
            if isinstance(current_user.interests, str):
                if current_user.interests.startswith('['):
                    user_interests = json.loads(current_user.interests)
                else:
                    user_interests = current_user.interests.split(',')
            else:
                user_interests = []
            
            if user_interests:
                # اگر کاربر علایق دارد، از آنها برای فیلتر استفاده کن
                user_filters[user_id]['interests'] = user_interests
                
                interests_keyboard = ReplyKeyboardMarkup(
                    keyboard=[
                        [KeyboardButton(text="✅ استفاده از علایق پروفایل")],
                        [KeyboardButton(text="🔄 تغییر علایق")],
                        [KeyboardButton(text="⏭️ رد کردن این مرحله")],
                        [KeyboardButton(text="⬅️ قبلی"), KeyboardButton(text="🔙 بازگشت به منو")]
                    ],
                    resize_keyboard=True,
                    one_time_keyboard=True
                )
                
                interests_text = ', '.join(user_interests[:5]) + ('...' if len(user_interests) > 5 else '')
                await message.answer(
                    f"🎯 <b>مرحله 5: علایق</b>\n\n"
                    f"علایق شما در پروفایل: {interests_text}\n\n"
                    f"می‌خواهید از همین علایق استفاده کنید؟",
                    reply_markup=interests_keyboard,
                    parse_mode='HTML'
                )
            else:
                # اگر علایق ندارد
                await show_no_interests_message(message, user_id)
        except:
            # در صورت خطا
            await show_no_interests_message(message, user_id)
    else:
        # اگر علایق ندارد
        await show_no_interests_message(message, user_id)
    
    await state.set_state(MatchingStates.interests)

async def show_no_interests_message(message: Message, user_id: int):
    """نمایش پیام برای کاربرانی که علایق ندارند"""
    interests_keyboard = ReplyKeyboardMarkup(
        keyboard=[
            [KeyboardButton(text="📝 ویرایش علایق در پروفایل")],
            [KeyboardButton(text="⏭️ رد کردن این مرحله")],
            [KeyboardButton(text="⬅️ قبلی"), KeyboardButton(text="🔙 بازگشت به منو")]
        ],
        resize_keyboard=True,
        one_time_keyboard=True
    )
    
    await message.answer(
        "🎯 <b>مرحله 5: علایق</b>\n\n"
        "⚠️ شما هنوز علایقی در پروفایل خود ثبت نکرده‌اید.\n\n"
        "برای فیلتر بهتر، ابتدا از منوی پروفایل، علایق خود را ویرایش کنید.",
        reply_markup=interests_keyboard,
        parse_mode='HTML'
    )
    
    await state.set_state(MatchingStates.interests)

# ==================== مرحله 5: علایق ====================
@router.message(MatchingStates.interests)
async def handle_interests(message: Message, state: FSMContext):
    """انتخاب علایق"""
    user_id = message.from_user.id
    text = message.text
    
    if text == "🔙 بازگشت به منو":
        await back_to_menu(message, state)
        return
    
    if text == "⬅️ قبلی":
        # برگشت به مرحله 4: موقعیت
        await show_location_step(message, state)
        return
    
    if text == "📝 ویرایش علایق در پروفایل":
        await message.answer(
            "📝 لطفاً از منوی اصلی وارد بخش 'پروفایل من' شوید و علایق خود را ویرایش کنید.\n"
            "سپس دوباره به بخش یافتن مچ برگردید."
        )
        await back_to_menu(message, state)
        return
    
    if text == "🔄 تغییر علایق":
        await message.answer(
            "📝 برای تغییر علایق، از منوی پروفایل اقدام کنید.\n"
            "فعلاً با علایق فعلی ادامه می‌دهیم."
        )
        text = "✅ استفاده از علایق پروفایل"
    
    if text in ["✅ تایید علایق", "✅ استفاده از علایق پروفایل", "⏭️ رد کردن این مرحله"]:
        # مرحله 6: فیلتر عکس
        photo_keyboard = ReplyKeyboardMarkup(
            keyboard=[
                [KeyboardButton(text="📸 فقط با عکس"), KeyboardButton(text="👤 همه کاربران")],
                [KeyboardButton(text="⬅️ قبلی"), KeyboardButton(text="🔙 بازگشت به منو")]
            ],
            resize_keyboard=True,
            one_time_keyboard=True
        )
        
        await message.answer(
            "📸 <b>مرحله 6: فیلتر عکس</b>\n\n"
            "آیا می‌خواهید فقط کاربرانی که عکس دارند را ببینید؟",
            reply_markup=photo_keyboard,
            parse_mode='HTML'
        )
        
        await state.set_state(MatchingStates.photo_filter)
    else:
        await message.answer("❌ لطفاً یکی از گزینه‌های موجود را انتخاب کنید.")
        return

# ==================== مرحله 6: فیلتر عکس ====================
@router.message(MatchingStates.photo_filter)
async def handle_photo_filter(message: Message, state: FSMContext):
    """انتخاب فیلتر عکس"""
    user_id = message.from_user.id
    text = message.text
    
    if text == "🔙 بازگشت به منو":
        await back_to_menu(message, state)
        return
    
    if text == "⬅️ قبلی":
        # برگشت به مرحله 5: علایق
        await show_interests_step(message, state)
        return
    
    if text == "📸 فقط با عکس":
        user_filters[user_id]['photo_required'] = True
    elif text == "👤 همه کاربران":
        user_filters[user_id]['photo_required'] = False
    else:
        await message.answer("❌ لطفاً یکی از گزینه‌های موجود را انتخاب کنید.")
        return
    
    # نمایش خلاصه فیلترها
    filters = user_filters[user_id]
    summary = "📋 <b>خلاصه فیلترهای شما:</b>\n\n"
    
    if filters['search_type']:
        search_types = {'friendship': 'دوستیابی', 'partner': 'پارتنر', 'marriage': 'ازدواج'}
        summary += f"🎯 نوع: {search_types.get(filters['search_type'], 'همه')}\n"
    
    if filters['gender_pref']:
        genders = {1: 'مرد', 2: 'زن', 3: 'غیره'}
        summary += f"⚧️ جنسیت: {genders.get(filters['gender_pref'], 'همه')}\n"
    
    summary += f"🎂 سن: {filters['age_min']} تا {filters['age_max']} سال\n"
    
    if filters['country']:
        summary += f"📍 کشور: {filters['country']}\n"
    
    if filters['interests']:
        summary += f"🎯 علایق: {', '.join(filters['interests'])}\n"
    
    summary += f"📸 عکس: {'الزامی' if filters['photo_required'] else 'اختیاری'}\n"
    
    await message.answer(summary, parse_mode='HTML')
    
    # ذخیره وضعیت تکمیل فیلترها
    filters_completed[user_id] = True
    
    # شروع جستجو
    await message.answer("🔍")
    
    # انتقال به ماژول نمایش
    from .matching_display import find_and_show_candidates
    await find_and_show_candidates(message, state, filters)

# ==================== Helper Functions ====================
async def show_search_type_step(message: Message, state: FSMContext):
    """نمایش مرحله 1: نوع جستجو"""
    user_id = message.from_user.id
    
    search_type_keyboard = ReplyKeyboardMarkup(
        keyboard=[
            [KeyboardButton(text="👫 دوستیابی"), KeyboardButton(text="💑 پارتنر"), KeyboardButton(text="💍 ازدواج")],
            [KeyboardButton(text="⏭️ رد کردن این مرحله")],
            [KeyboardButton(text="🔙 بازگشت به منو")]
        ],
        resize_keyboard=True,
        one_time_keyboard=True
    )
    
    await message.answer(
        "🎯 <b>مرحله 1: نوع جستجو</b>\n\n"
        "به دنبال چه نوع رابطه‌ای هستید؟",
        reply_markup=search_type_keyboard,
        parse_mode='HTML'
    )
    
    await state.set_state(MatchingStates.search_type)

async def show_gender_step(message: Message, state: FSMContext):
    """نمایش مرحله 2: جنسیت"""
    gender_keyboard = ReplyKeyboardMarkup(
        keyboard=[
            [KeyboardButton(text="👨 مرد"), KeyboardButton(text="👩 زن"), KeyboardButton(text="⚧️ غیره")],
            [KeyboardButton(text="⏭️ رد کردن این مرحله")],
            [KeyboardButton(text="⬅️ قبلی"), KeyboardButton(text="🔙 بازگشت به منو")]
        ],
        resize_keyboard=True,
        one_time_keyboard=True
    )
    
    await message.answer(
        "⚧️ <b>مرحله 2: جنسیت مورد نظر</b>\n\n"
        "به دنبال چه جنسیتی هستید؟",
        reply_markup=gender_keyboard,
        parse_mode='HTML'
    )
    
    await state.set_state(MatchingStates.gender_pref)

async def show_age_step(message: Message, state: FSMContext):
    """نمایش مرحله 3: بازه سنی"""
    age_keyboard = ReplyKeyboardMarkup(
        keyboard=[
            [KeyboardButton(text="⏭️ رد کردن این مرحله")],
            [KeyboardButton(text="⬅️ قبلی"), KeyboardButton(text="🔙 بازگشت به منو")]
        ],
        resize_keyboard=True,
        one_time_keyboard=True
    )
    
    await message.answer(
        "🎂 <b>مرحله 3: بازه سنی</b>\n\n"
        "بازه سنی مورد نظر را به صورت زیر وارد کنید:\n"
        "📝 مثال: 18-30 یا 25-40\n\n"
        "یا روی دکمه '⏭️ رد کردن' کلیک کنید.",
        reply_markup=age_keyboard,
        parse_mode='HTML'
    )
    
    await state.set_state(MatchingStates.age_range)

async def show_location_step(message: Message, state: FSMContext):
    """نمایش مرحله 4: موقعیت"""
    location_keyboard = ReplyKeyboardMarkup(
        keyboard=[
            [KeyboardButton(text="📍 ارسال موقعیت فعلی", request_location=True)],
            [KeyboardButton(text="🌍 انتخاب کشور")],
            [KeyboardButton(text="⏭️ رد کردن این مرحله")],
            [KeyboardButton(text="⬅️ قبلی"), KeyboardButton(text="🔙 بازگشت به منو")]
        ],
        resize_keyboard=True,
        one_time_keyboard=True
    )
    
    await message.answer(
        "📍 <b>مرحله 4: موقعیت مکانی</b>\n\n"
        "برای یافتن افراد نزدیک به شما، موقعیت خود را مشخص کنید:",
        reply_markup=location_keyboard,
        parse_mode='HTML'
    )
    
    await state.set_state(MatchingStates.step6_photo)
    
    # فیلترها را ذخیره می‌کنیم
    data = await state.get_data()
    filters = {
        'search_type': data.get('search_type'),
        'gender': data.get('gender'),
        'age_min': data.get('age_min'),
        'age_max': data.get('age_max'),
        'location': data.get('location'),
        'interests': data.get('interests', []),
        'photo_required': data.get('photo_required', False)
    }
    
    user_filters[user_id] = filters
    filters_completed[user_id] = True
    
    await message.answer(
        "🔍",
        reply_markup=ReplyKeyboardRemove()
    )
    
    # جستجوی کاربران
    async with db.get_session() as session:
        # ⚠️ فیلترها موقتاً غیرفعال - برای تست با کاربران کم
        query = select(User).where(User.id != user_id)
        
        # ==================== فیلترها (موقتاً کامنت) ====================
        # TODO: فیلترها را بعد از افزایش تعداد کاربران فعال کنید
        
        # # اعمال فیلتر جنسیت
        # if filters['gender']:
        #     query = query.where(User.gender == filters['gender'])
        
        # # اعمال فیلتر سن
        # if filters['age_min'] and filters['age_max']:
        #     query = query.where(
        #         and_(
        #             User.age >= filters['age_min'],
        #             User.age <= filters['age_max']
        #         )
        #     )
        
        # # اعمال فیلتر موقعیت
        # if filters['location'] and filters['location'] != "📍 همه شهرها":
        #     query = query.where(User.city == filters['location'])
        
        # # اعمال فیلتر عکس
        # if filters['photo_required']:
        #     query = query.where(User.profile_photo.isnot(None))
        
        # ==================== پایان فیلترها ====================
        
        # فیلتر کردن کاربران رد شده (این فعال می‌ماند)
        rejected_query = select(Swipe.to_user).where(
            and_(
                Swipe.from_user == user_id,
                Swipe.action == -1  # رد شده
            )
        )
        rejected_users = await session.execute(rejected_query)
        rejected_ids = [r for r in rejected_users.scalars()]
        
        if rejected_ids:
            query = query.where(User.id.notin_(rejected_ids))
        
        # نمایش همه کاربران (بدون فیلتر)
        result = await session.execute(query.limit(100))  # افزایش limit
        candidates = result.scalars().all()
    
    if candidates:
        # Import from display module
        from .matching_display import start_viewing_profiles
        await start_viewing_profiles(message, state, candidates)
    else:
        await message.answer(
            "😔 <b>کاربری یافت نشد!</b>\n\n"
            "در حال حاضر کاربر فعالی برای نمایش وجود ندارد.\n"
            "لطفاً بعداً دوباره امتحان کنید.",
            parse_mode='HTML'
        )
        await state.clear()

async def show_no_interests_step(message: Message, user_id: int):
    """نمایش پیام برای کاربرانی که علایق ندارند"""
    interests_keyboard = ReplyKeyboardMarkup(
        keyboard=[
            [KeyboardButton(text="📝 ویرایش علایق در پروفایل")],
            [KeyboardButton(text="⏭️ رد کردن این مرحله")],
            [KeyboardButton(text="⬅️ قبلی"), KeyboardButton(text="🔙 بازگشت به منو")]
        ],
        resize_keyboard=True,
        one_time_keyboard=True
    )
    
    await message.answer(
        "🎯 <b>مرحله 5: علایق</b>\n\n"
        "⚠️ شما هنوز علایقی در پروفایل خود ثبت نکرده‌اید.\n\n"
        "برای فیلتر بهتر، ابتدا از منوی پروفایل، علایق خود را ویرایش کنید.",
        reply_markup=interests_keyboard,
        parse_mode='HTML'
    )

async def back_to_menu(message: Message, state: FSMContext):
    """بازگشت به منوی اصلی"""
    from handlers.registration import get_main_menu_keyboard
    from localization.language_helper import get_user_with_language
    from localization.translator import get_text
    import os
    
    user, user_language = await get_user_with_language(message.from_user.id)
    
    if not user:
        await message.answer("❌ ابتدا باید ثبت‌نام کنید.")
        return
    
    # بررسی ادمین
    ADMIN_USER_IDS = [int(uid) for uid in os.getenv('ADMIN_USER_IDS', '').split(',') if uid]
    is_admin = message.from_user.id in ADMIN_USER_IDS
    
    # پاک کردن state
    await state.clear()
    
    # نمایش منوی اصلی
    await message.answer(
        f"🏠 {get_text(user_language, 'back_to_menu', default='بازگشت به منوی اصلی')}",
        reply_markup=get_main_menu_keyboard(user_language, is_admin=is_admin)
    )

# حذف handlers قدیمی - دیگر نیازی نیست چون مستقیماً پروفایل نمایش داده می‌شود

# Export
__all__ = ['router', 'MatchingStates', 'user_filters', 'filters_completed', 'start_matching_flow_internal']
