from aiogram import Router, F
from aiogram.types import Message, ReplyKeyboardMarkup, KeyboardButton
from aiogram.fsm.context import FSMContext
from aiogram.fsm.state import State, StatesGroup
from sqlalchemy import select, and_, desc
from database.database import db
from database.models import User, Match
from database.simple_models import Swipe
import logging
import json
from datetime import datetime

router = Router()
logger = logging.getLogger(__name__)

# States برای مدیریت لایک‌ها
class LikesStates(StatesGroup):
    viewing_profile = State()
    deciding_action = State()

# متغیر global برای ذخیره کاربر فعلی در لایک‌ها
current_like_users = {}

@router.message(F.text.contains("💚 لایک‌های من"))
async def my_likes_main(message: Message):
    """منوی اصلی لایک‌های من - منطقی و واقعی"""
    user_id = message.from_user.id
    
    async with db.get_session() as session:
        # بررسی وجود کاربر
        user_result = await session.execute(
            select(User).where(User.id == user_id)
        )
        current_user = user_result.scalar_one_or_none()
        
        if not current_user:
            await message.answer("❌ ابتدا باید ثبت‌نام کنید.")
            return
        
        # دریافت لایک‌های دریافتی (کسانی که شما را لایک کرده‌اند)
        received_result = await session.execute(
            select(Swipe, User).join(User, Swipe.from_user == User.id).where(
                and_(
                    Swipe.to_user == user_id,
                    Swipe.action.in_([1, 2])  # like=1, super_like=2
                )
            ).order_by(Swipe.created_at.desc())
        )
        received_list = received_result.all()
        
        # دریافت لایک‌های ارسالی (کسانی که شما آن‌ها را لایک کرده‌اید)
        sent_result = await session.execute(
            select(Swipe, User).join(User, Swipe.to_user == User.id).where(
                and_(
                    Swipe.from_user == user_id,
                    Swipe.action.in_([1, 2])  # like=1, super_like=2
                )
            ).order_by(Swipe.created_at.desc())
        )
        sent_list = sent_result.all()
        
        # پیدا کردن مچ‌ها (لایک‌های متقابل)
        matches_result = await session.execute(
            select(User).where(
                and_(
                    User.id.in_(
                        select(Swipe.from_user).where(
                            and_(
                                Swipe.to_user == user_id,
                                Swipe.action.in_([1, 2])
                            )
                        )
                    ),
                    User.id.in_(
                        select(Swipe.to_user).where(
                            and_(
                                Swipe.from_user == user_id,
                                Swipe.action.in_([1, 2])
                            )
                        )
                    )
                )
            )
        )
        matches = matches_result.scalars().all()
        
        # ساخت متن نمایش
        text = "💚 <b>لایک‌های من</b>\n\n"
        text += f"📊 <b>آمار کلی:</b>\n"
        text += f"💕 لایک‌های دریافتی: <b>{len(received_list)}</b>\n"
        text += f"💚 لایک‌های ارسالی: <b>{len(sent_list)}</b>\n"
        text += f"🎉 مچ‌ها: <b>{len(matches)}</b>\n\n"
        
        # نمایش لایک‌های دریافتی اخیر
        if received_list:
            text += "💕 <b>جدیدترین لایک‌های دریافتی:</b>\n"
            for swipe, user in received_list[:3]:
                like_type = "⭐ سوپر لایک" if swipe.action == 2 else "💚 لایک"
                age = user.age if user.age else "نامشخص"
                user_name = user.username or user.first_name or "کاربر"
                
                # محاسبه زمان
                time_diff = datetime.now() - swipe.created_at
                if time_diff.days > 0:
                    time_str = f"{time_diff.days} روز پیش"
                elif time_diff.seconds > 3600:
                    time_str = f"{time_diff.seconds // 3600} ساعت پیش"
                else:
                    time_str = f"{time_diff.seconds // 60} دقیقه پیش"
                
                text += f"• {like_type} از <b>{user_name}</b> ({age} ساله) - {time_str}\n"
            
            if len(received_list) > 3:
                text += f"📝 و {len(received_list) - 3} لایک دیگر...\n"
            text += "\n"
        else:
            text += "💔 <b>هنوز کسی شما را لایک نکرده است.</b>\n\n"
        
        # نمایش مچ‌ها
        if matches:
            text += "🎉 <b>مچ‌های شما:</b>\n"
            for user in matches[:3]:
                age = user.age if user.age else "نامشخص"
                user_name = user.username or user.first_name or "کاربر"
                text += f"💖 <b>{user_name}</b> ({age} ساله)\n"
            
            if len(matches) > 3:
                text += f"📝 و {len(matches) - 3} مچ دیگر...\n"
            text += "\n💬 با مچ‌ها می‌توانید گفتگو کنید!\n\n"
        
        text += "💡 <b>راهنما:</b>\n"
        text += "• 💕 لایک‌های دریافتی: کسانی که شما را لایک کرده‌اند\n"
        text += "• 💚 لایک‌های ارسالی: کسانی که شما آن‌ها را لایک کرده‌اید\n"
        text += "• 🎉 مچ‌ها: لایک‌های متقابل (هر دو طرف لایک کرده‌اند)"
        
        # کیبورد کاستوم برای لایک‌ها
        keyboard_rows = []
        
        # دکمه‌های اصلی
        if received_list:
            keyboard_rows.append([KeyboardButton(text="💕 مشاهده لایک‌های دریافتی")])
        
        if sent_list:
            keyboard_rows.append([KeyboardButton(text="💚 مشاهده لایک‌های ارسالی")])
        
        if matches:
            keyboard_rows.append([KeyboardButton(text="🎉 مشاهده مچ‌ها")])
        
        # اگر لایک دریافتی دارد، دکمه‌های تعامل اضافه کن
        if received_list:
            keyboard_rows.append([
                KeyboardButton(text="👁️ مشاهده پروفایل‌ها"),
                KeyboardButton(text="💖 پاسخ به لایک‌ها")
            ])
        
        # دکمه‌های اضافی
        keyboard_rows.extend([
            [
                KeyboardButton(text="🔍 یافتن مچ جدید"),
                KeyboardButton(text="📊 آمار کامل")
            ],
            [
                KeyboardButton(text="🔙 بازگشت به منو")
            ]
        ])
        
        likes_keyboard = ReplyKeyboardMarkup(
            keyboard=keyboard_rows,
            resize_keyboard=True,
            one_time_keyboard=False
        )
        
        await message.answer(text, reply_markup=likes_keyboard, parse_mode='HTML')

# ==================== هندلرهای عملکرد واقعی ====================

@router.message(F.text.contains("💕 مشاهده لایک‌های دریافتی"))
async def view_received_likes(message: Message):
    """مشاهده کامل لایک‌های دریافتی با قابلیت تعامل"""
    user_id = message.from_user.id
    
    async with db.get_session() as session:
        # دریافت لایک‌های دریافتی
        received_result = await session.execute(
            select(Swipe, User).join(User, Swipe.from_user == User.id).where(
                and_(
                    Swipe.to_user == user_id,
                    Swipe.action.in_([1, 2])  # like=1, super_like=2
                )
            ).order_by(Swipe.created_at.desc())
        )
        received_list = received_result.all()
        
        if not received_list:
            await message.answer(
                "💕 <b>لایک‌های دریافتی</b>\n\n"
                "هنوز کسی شما را لایک نکرده است.\n\n"
                "💡 برای دریافت لایک بیشتر:\n"
                "• پروفایل خود را کامل کنید\n"
                "• عکس‌های جذاب آپلود کنید\n"
                "• فعال باشید و دیگران را لایک کنید",
                parse_mode='HTML'
            )
            return
        
        text = "💕 <b>لایک‌های دریافتی</b>\n\n"
        text += f"📊 تعداد کل: <b>{len(received_list)}</b> لایک\n\n"
        
        # نمایش لایک‌ها با جزئیات
        for i, (swipe, user) in enumerate(received_list[:10], 1):  # نمایش 10 مورد اول
            like_type = "⭐ سوپر لایک" if swipe.action == 2 else "💚 لایک"
            age = user.age if user.age else "نامشخص"
            user_name = user.username or user.first_name or "کاربر"
            
            # محاسبه زمان
            time_diff = datetime.now() - swipe.created_at
            if time_diff.days > 0:
                time_str = f"{time_diff.days} روز پیش"
            elif time_diff.seconds > 3600:
                time_str = f"{time_diff.seconds // 3600} ساعت پیش"
            else:
                time_str = f"{time_diff.seconds // 60} دقیقه پیش"
            
            text += f"{i}. {like_type} از <b>{user_name}</b>\n"
            text += f"   🎂 {age} ساله • ⏰ {time_str}\n"
            
            # اضافه کردن شهر اگر موجود باشد
            if user.city:
                text += f"   📍 {user.city}\n"
            
            text += "\n"
        
        if len(received_list) > 10:
            text += f"... و {len(received_list) - 10} لایک دیگر\n\n"
        
        text += "💡 <b>نکته:</b> برای مشاهده پروفایل و پاسخ دادن، از دکمه‌های زیر استفاده کنید."
        
        await message.answer(text, parse_mode='HTML')

@router.message(F.text.contains("💚 مشاهده لایک‌های ارسالی"))
async def view_sent_likes(message: Message):
    """مشاهده کامل لایک‌های ارسالی"""
    user_id = message.from_user.id
    
    async with db.get_session() as session:
        # دریافت لایک‌های ارسالی
        sent_result = await session.execute(
            select(Swipe, User).join(User, Swipe.to_user == User.id).where(
                and_(
                    Swipe.from_user == user_id,
                    Swipe.action.in_([1, 2])  # like=1, super_like=2
                )
            ).order_by(Swipe.created_at.desc())
        )
        sent_list = sent_result.all()
        
        if not sent_list:
            await message.answer(
                "💚 <b>لایک‌های ارسالی</b>\n\n"
                "هنوز کسی را لایک نکرده‌اید.\n\n"
                "💡 برای شروع:\n"
                "• روی دکمه 'یافتن مچ' کلیک کنید\n"
                "• پروفایل‌های پیشنهادی را ببینید\n"
                "• کسانی که دوست دارید را لایک کنید",
                parse_mode='HTML'
            )
            return
        
        text = "💚 <b>لایک‌های ارسالی</b>\n\n"
        text += f"📊 تعداد کل: <b>{len(sent_list)}</b> لایک\n\n"
        
        # نمایش لایک‌ها با جزئیات
        for i, (swipe, user) in enumerate(sent_list[:10], 1):  # نمایش 10 مورد اول
            like_type = "⭐ سوپر لایک" if swipe.action == 2 else "💚 لایک"
            age = user.age if user.age else "نامشخص"
            user_name = user.username or user.first_name or "کاربر"
            
            # محاسبه زمان
            time_diff = datetime.now() - swipe.created_at
            if time_diff.days > 0:
                time_str = f"{time_diff.days} روز پیش"
            elif time_diff.seconds > 3600:
                time_str = f"{time_diff.seconds // 3600} ساعت پیش"
            else:
                time_str = f"{time_diff.seconds // 60} دقیقه پیش"
            
            text += f"{i}. {like_type} به <b>{user_name}</b>\n"
            text += f"   🎂 {age} ساله • ⏰ {time_str}\n"
            
            # اضافه کردن شهر اگر موجود باشد
            if user.city:
                text += f"   📍 {user.city}\n"
            
            text += "\n"
        
        if len(sent_list) > 10:
            text += f"... و {len(sent_list) - 10} لایک دیگر\n\n"
        
        text += "⏳ <b>منتظر پاسخ:</b> اگر آن‌ها هم شما را لایک کنند، مچ خواهید شد!"
        
        await message.answer(text, parse_mode='HTML')

@router.message(F.text.contains("🎉 مشاهده مچ‌ها"))
async def view_matches(message: Message):
    """مشاهده کامل مچ‌ها"""
    user_id = message.from_user.id
    
    async with db.get_session() as session:
        # پیدا کردن مچ‌ها (لایک‌های متقابل)
        matches_result = await session.execute(
            select(User).where(
                and_(
                    User.id.in_(
                        select(Swipe.from_user).where(
                            and_(
                                Swipe.to_user == user_id,
                                Swipe.action.in_([1, 2])
                            )
                        )
                    ),
                    User.id.in_(
                        select(Swipe.to_user).where(
                            and_(
                                Swipe.from_user == user_id,
                                Swipe.action.in_([1, 2])
                            )
                        )
                    )
                )
            )
        )
        matches = matches_result.scalars().all()
        
        if not matches:
            await message.answer(
                "🎉 <b>مچ‌های شما</b>\n\n"
                "هنوز مچی ندارید!\n\n"
                "💡 برای پیدا کردن مچ:\n"
                "• کسانی را که دوست دارید لایک کنید\n"
                "• اگر آن‌ها هم شما را لایک کنند = مچ! 🎉\n"
                "• سپس می‌توانید با هم گفتگو کنید",
                parse_mode='HTML'
            )
            return
        
        text = "🎉 <b>مچ‌های شما</b>\n\n"
        text += f"💕 تعداد مچ‌ها: <b>{len(matches)}</b>\n\n"
        
        # نمایش مچ‌ها با جزئیات
        for i, user in enumerate(matches[:8], 1):  # نمایش 8 مچ اول
            age = user.age if user.age else "نامشخص"
            user_name = user.username or user.first_name or "کاربر"
            
            text += f"{i}. 💖 <b>{user_name}</b> ({age} ساله)\n"
            
            # اضافه کردن شهر اگر موجود باشد
            if user.city:
                text += f"   📍 {user.city}\n"
            
            # اضافه کردن بیو کوتاه اگر موجود باشد
            if user.bio:
                bio_short = user.bio[:30] + "..." if len(user.bio) > 30 else user.bio
                text += f"   💭 {bio_short}\n"
            
            text += "\n"
        
        if len(matches) > 8:
            text += f"... و {len(matches) - 8} مچ دیگر\n\n"
        
        text += "💬 <b>برای شروع گفتگو:</b>\n"
        text += "از منوی 'گفتگوهای من' استفاده کنید."
        
        await message.answer(text, parse_mode='HTML')

# ==================== هندلرهای اضافی ====================

@router.message(F.text.contains("👁️ مشاهده پروفایل‌ها"))
async def view_profiles_from_likes(message: Message):
    """مشاهده پروفایل کسانی که لایک کرده‌اند"""
    await message.answer(
        "👁️ <b>مشاهده پروفایل‌ها</b>\n\n"
        "این قابلیت در حال توسعه است.\n\n"
        "💡 فعلاً از دکمه‌های بالا استفاده کنید:"
        "• 💕 مشاهده لایک‌های دریافتی\n"
        "• 🎉 مشاهده مچ‌ها",
        parse_mode='HTML'
    )

@router.message(F.text.contains("💖 پاسخ به لایک‌ها"))
async def respond_to_likes(message: Message):
    """پاسخ به لایک‌ها"""
    await message.answer(
        "💖 <b>پاسخ به لایک‌ها</b>\n\n"
        "برای پاسخ به لایک‌ها:\n\n"
        "• از منوی 'یافتن مچ' استفاده کنید\n"
        "• کسانی که شما را لایک کرده‌اند را لایک کنید\n"
        "• به این ترتیب مچ خواهید شد! 🎉",
        parse_mode='HTML'
    )

@router.message(F.text.contains("🔍 یافتن مچ جدید"))
async def find_new_match_from_likes(message: Message):
    """یافتن مچ جدید از لایک‌ها"""
    await message.answer(
        "🔍 <b>یافتن مچ جدید</b>\n\n"
        "برای یافتن مچ جدید، از منوی اصلی روی دکمه 'یافتن مچ' کلیک کنید.\n\n"
        "💡 نکات برای مچ بهتر:\n"
        "• پروفایل خود را کامل کنید\n"
        "• عکس‌های جذاب آپلود کنید\n"
        "• بیو جالب بنویسید\n"
        "• فعال باشید و دیگران را لایک کنید",
        parse_mode='HTML'
    )

@router.message(F.text.contains("📊 آمار کامل"))
async def full_statistics(message: Message):
    """آمار کامل لایک‌ها"""
    user_id = message.from_user.id
    
    async with db.get_session() as session:
        # آمار کامل
        received_count = await session.execute(
            select(Swipe).where(
                and_(
                    Swipe.to_user == user_id,
                    Swipe.action.in_([1, 2])
                )
            )
        )
        received_total = len(received_count.scalars().all())
        
        sent_count = await session.execute(
            select(Swipe).where(
                and_(
                    Swipe.from_user == user_id,
                    Swipe.action.in_([1, 2])
                )
            )
        )
        sent_total = len(sent_count.scalars().all())
        
        # محاسبه نرخ موفقیت
        success_rate = (received_total / sent_total * 100) if sent_total > 0 else 0
        
        text = "📊 <b>آمار کامل لایک‌ها</b>\n\n"
        text += f"💕 کل لایک‌های دریافتی: <b>{received_total}</b>\n"
        text += f"💚 کل لایک‌های ارسالی: <b>{sent_total}</b>\n"
        text += f"📈 نرخ موفقیت: <b>{success_rate:.1f}%</b>\n\n"
        text += f"💡 <b>نکته:</b> نرخ موفقیت = (لایک‌های دریافتی ÷ لایک‌های ارسالی) × 100"
        
        await message.answer(text, parse_mode='HTML')

@router.message(F.text == "💚 لایک")
async def handle_like_in_my_likes(message: Message):
    """هندلر دکمه لایک در بخش لایک‌های من"""
    await message.answer(
        "💚 <b>دکمه لایک</b>\n\n"
        "شما در بخش 'لایک‌های من' هستید.\n\n"
        "💡 برای لایک کردن کاربران جدید:\n"
        "• روی '🔍 یافتن مچ جدید' کلیک کنید\n"
        "• یا از منوی اصلی 'یافتن مچ' را انتخاب کنید\n\n"
        "📋 برای مشاهده لایک‌ها از دکمه‌های بالا استفاده کنید:",
        parse_mode='HTML'
    )

@router.message(F.text.contains("🔙 بازگشت به منو"))
async def back_to_main_menu_likes(message: Message):
    """بازگشت به منوی اصلی"""
    from handlers.registration import get_main_menu_keyboard
    from localization.language_helper import get_user_with_language
    from localization.translator import get_text
    import os
    
    user, user_language = await get_user_with_language(message.from_user.id)
    
    if not user:
        await message.answer(f"❌ {get_text(user_language, 'must_register_first', default='ابتدا باید ثبت‌نام کنید.')}")
        return
    
    # بررسی ادمین بودن
    ADMIN_USER_IDS = [int(uid) for uid in os.getenv('ADMIN_USER_IDS', '').split(',') if uid]
    is_admin = message.from_user.id in ADMIN_USER_IDS
    
    await message.answer(
        f"🏠 {get_text(user_language, 'back_to_menu', default='بازگشت به منوی اصلی')}",
        reply_markup=get_main_menu_keyboard(user_language, is_admin=is_admin)
    )

# ==================== پایان فایل ====================
# سیستم لایک‌های من کاملاً منطقی و واقعی - فقط کیبورد کاستوم
