"""
Profile Features Handler
مدیریت فیچرهای پیشرفته پروفایل:
- آمار پروفایل (بازدید و لایک)
- حالت نامرئی
- وضعیت آنلاین/آفلاین
- پیش‌نمایش پروفایل
- فیلتر علایق مشترک
"""

from aiogram import Router, F
from aiogram.types import Message, CallbackQuery, InlineKeyboardMarkup, InlineKeyboardButton
from aiogram.fsm.context import FSMContext
from sqlalchemy import select, and_, func
from datetime import datetime, timedelta
import json

from database.database import db
from database.models import User, Swipe
from localization.translator import get_text
from localization.language_helper import get_user_with_language

router = Router()

# ==================== آمار پروفایل ====================

async def increment_profile_view(viewer_id: int, profile_id: int):
    """افزایش شمارنده بازدید پروفایل"""
    if viewer_id == profile_id:
        return  # خودش را نشمارد
    
    async with db.get_session() as session:
        result = await session.execute(
            select(User).where(User.id == profile_id)
        )
        user = result.scalar_one_or_none()
        
        if user:
            user.profile_views = (user.profile_views or 0) + 1
            await session.commit()

async def increment_likes_received(user_id: int):
    """افزایش شمارنده لایک دریافتی"""
    async with db.get_session() as session:
        result = await session.execute(
            select(User).where(User.id == user_id)
        )
        user = result.scalar_one_or_none()
        
        if user:
            user.likes_received = (user.likes_received or 0) + 1
            await session.commit()

async def get_profile_stats(user_id: int) -> dict:
    """دریافت آمار پروفایل"""
    async with db.get_session() as session:
        result = await session.execute(
            select(User).where(User.id == user_id)
        )
        user = result.scalar_one_or_none()
        
        if not user:
            return {'views': 0, 'likes': 0}
        
        # شمارش لایک‌های ارسالی
        likes_sent_result = await session.execute(
            select(func.count(Swipe.id)).where(
                and_(Swipe.from_user == user_id, Swipe.action == 1)
            )
        )
        likes_sent = likes_sent_result.scalar() or 0
        
        return {
            'views': user.profile_views or 0,
            'likes_received': user.likes_received or 0,
            'likes_sent': likes_sent
        }

@router.message(F.text.contains("آمار پروفایل") | F.text.contains("Profile Stats"))
async def show_profile_stats(message: Message):
    """نمایش آمار پروفایل"""
    user, user_language = await get_user_with_language(message.from_user.id)
    
    if not user:
        await message.answer(
            get_text(user_language, 'profile_not_found', default='پروفایل یافت نشد')
        )
        return
    
    stats = await get_profile_stats(user.user_id)
    
    stats_text = (
        f"📊 <b>{get_text(user_language, 'profile_stats', default='آمار پروفایل')}</b>\n\n"
        f"👁️ <b>{get_text(user_language, 'profile_views', default='بازدید پروفایل')}:</b> "
        f"{get_text(user_language, 'views_count', default='{count} بازدید').format(count=stats['views'])}\n\n"
        f"❤️ <b>{get_text(user_language, 'likes_received', default='لایک دریافتی')}:</b> "
        f"{get_text(user_language, 'likes_count', default='{count} لایک').format(count=stats['likes_received'])}\n\n"
        f"💕 <b>لایک ارسالی:</b> {stats['likes_sent']}"
    )
    
    await message.answer(stats_text, parse_mode='HTML')

# ==================== حالت نامرئی ====================

async def toggle_invisible_mode(user_id: int) -> bool:
    """تغییر وضعیت حالت نامرئی"""
    async with db.get_session() as session:
        result = await session.execute(
            select(User).where(User.id == user_id)
        )
        user = result.scalar_one_or_none()
        
        if user:
            user.is_invisible = not (user.is_invisible or False)
            await session.commit()
            return user.is_invisible
    
    return False

@router.callback_query(F.data == "toggle_invisible")
async def toggle_invisible_callback(callback: CallbackQuery):
    """تغییر حالت نامرئی"""
    await callback.answer()
    
    user, user_language = await get_user_with_language(callback.from_user.id)
    
    is_invisible = await toggle_invisible_mode(callback.from_user.id)
    
    if is_invisible:
        message = get_text(user_language, 'invisible_mode_on', default='حالت نامرئی فعال شد')
        status = "🔒 فعال"
    else:
        message = get_text(user_language, 'invisible_mode_off', default='حالت نامرئی غیرفعال شد')
        status = "🔓 غیرفعال"
    
    keyboard = InlineKeyboardMarkup(inline_keyboard=[
        [InlineKeyboardButton(
            text=f"حالت نامرئی: {status}",
            callback_data="toggle_invisible"
        )],
        [InlineKeyboardButton(
            text=f"❌ {get_text(user_language, 'close', default='بستن')}",
            callback_data="close_privacy"
        )]
    ])
    
    await callback.message.edit_text(
        f"✅ {message}",
        reply_markup=keyboard
    )

# ==================== وضعیت آنلاین ====================

async def update_online_status(user_id: int, is_online: bool = True):
    """بروزرسانی وضعیت آنلاین"""
    async with db.get_session() as session:
        result = await session.execute(
            select(User).where(User.id == user_id)
        )
        user = result.scalar_one_or_none()
        
        if user:
            user.is_online = is_online
            user.last_active = func.now()
            await session.commit()

async def get_last_seen_text(user: User, language_code: str = 'fa') -> str:
    """دریافت متن آخرین بازدید"""
    if user.hide_last_seen:
        return get_text(language_code, 'last_seen_hidden', default='مخفی')
    
    if user.is_online:
        return get_text(language_code, 'online', default='آنلاین')
    
    if not user.last_active:
        return get_text(language_code, 'offline', default='آفلاین')
    
    # محاسبه زمان گذشته
    now = datetime.now()
    last_active = user.last_active
    
    if isinstance(last_active, str):
        try:
            last_active = datetime.fromisoformat(last_active)
        except:
            return get_text(language_code, 'offline', default='آفلاین')
    
    diff = now - last_active
    
    if diff < timedelta(minutes=5):
        return get_text(language_code, 'last_seen_recently', default='اخیراً')
    elif diff < timedelta(hours=1):
        minutes = int(diff.total_seconds() / 60)
        return get_text(language_code, 'last_seen_minutes', default='{minutes} دقیقه پیش').format(minutes=minutes)
    elif diff < timedelta(days=1):
        hours = int(diff.total_seconds() / 3600)
        return get_text(language_code, 'last_seen_hours', default='{hours} ساعت پیش').format(hours=hours)
    else:
        days = diff.days
        return get_text(language_code, 'last_seen_days', default='{days} روز پیش').format(days=days)

# ==================== فیلتر علایق مشترک ====================

def calculate_interests_compatibility(user_interests: str, target_interests: str) -> dict:
    """محاسبه سازگاری علایق"""
    try:
        # تبدیل به لیست
        if isinstance(user_interests, str):
            if user_interests.startswith('['):
                user_list = json.loads(user_interests)
            else:
                user_list = [i.strip() for i in user_interests.split(',')]
        else:
            user_list = user_interests or []
        
        if isinstance(target_interests, str):
            if target_interests.startswith('['):
                target_list = json.loads(target_interests)
            else:
                target_list = [i.strip() for i in target_interests.split(',')]
        else:
            target_list = target_interests or []
        
        # پیدا کردن علایق مشترک
        common = set(user_list) & set(target_list)
        common_count = len(common)
        
        # محاسبه درصد سازگاری
        if not user_list or not target_list:
            compatibility = 0
        else:
            total = len(set(user_list) | set(target_list))
            compatibility = (common_count / total) * 100 if total > 0 else 0
        
        # تعیین سطح سازگاری
        if compatibility >= 60:
            level = 'high'
        elif compatibility >= 30:
            level = 'medium'
        else:
            level = 'low'
        
        return {
            'common_interests': list(common),
            'common_count': common_count,
            'compatibility_percent': round(compatibility, 1),
            'compatibility_level': level
        }
    
    except Exception as e:
        return {
            'common_interests': [],
            'common_count': 0,
            'compatibility_percent': 0,
            'compatibility_level': 'low'
        }

async def filter_users_by_interests(user_id: int, min_common: int = 1) -> list:
    """فیلتر کاربران بر اساس علایق مشترک"""
    async with db.get_session() as session:
        # دریافت کاربر فعلی
        result = await session.execute(
            select(User).where(User.id == user_id)
        )
        user = result.scalar_one_or_none()
        
        if not user or not user.interests:
            return []
        
        # دریافت همه کاربران فعال
        result = await session.execute(
            select(User).where(
                and_(
                    User.id != user_id,
                    User.is_active == True,
                    User.is_banned == False,
                    User.interests.isnot(None)
                )
            )
        )
        all_users = result.scalars().all()
        
        # محاسبه سازگاری و فیلتر
        compatible_users = []
        for target_user in all_users:
            compatibility = calculate_interests_compatibility(
                user.interests,
                target_user.interests
            )
            
            if compatibility['common_count'] >= min_common:
                compatible_users.append({
                    'user': target_user,
                    'compatibility': compatibility
                })
        
        # مرتب‌سازی بر اساس سازگاری
        compatible_users.sort(
            key=lambda x: x['compatibility']['common_count'],
            reverse=True
        )
        
        return compatible_users

# ==================== پیش‌نمایش پروفایل ====================

@router.callback_query(F.data == "preview_profile")
async def preview_profile_callback(callback: CallbackQuery, state: FSMContext):
    """نمایش پیش‌نمایش پروفایل"""
    await callback.answer()
    
    user, user_language = await get_user_with_language(callback.from_user.id)
    
    # دریافت تغییرات موقت از state
    data = await state.get_data()
    
    preview_text = (
        f"👁️ <b>{get_text(user_language, 'profile_preview', default='پیش‌نمایش پروفایل')}</b>\n\n"
        f"نام: {data.get('name', user.first_name)}\n"
        f"سن: {data.get('age', user.age)}\n"
        f"کشور: {data.get('country', user.country)}\n"
        f"شهر: {data.get('city', user.city)}\n"
        f"بیو: {data.get('bio', user.bio)}\n"
    )
    
    keyboard = InlineKeyboardMarkup(inline_keyboard=[
        [
            InlineKeyboardButton(
                text=f"✅ {get_text(user_language, 'confirm_changes', default='تایید تغییرات')}",
                callback_data="confirm_profile_changes"
            )
        ],
        [
            InlineKeyboardButton(
                text=f"❌ {get_text(user_language, 'discard_changes', default='لغو تغییرات')}",
                callback_data="discard_profile_changes"
            )
        ]
    ])
    
    await callback.message.edit_text(preview_text, reply_markup=keyboard, parse_mode='HTML')
