from aiogram import Router, F
from aiogram.types import Message, CallbackQuery, InlineKeyboardMarkup, InlineKeyboardButton
from aiogram.fsm.context import FSMContext
from aiogram.fsm.state import State, StatesGroup
from sqlalchemy import select, and_, desc
import logging

from database.database import db
from database.models import User, Match, ChatMessage

logger = logging.getLogger(__name__)
router = Router()

# Define states locally
class RealChatStates(StatesGroup):
    in_chat = State()

chat_manager = None

class RealChatManager:
    def __init__(self, bot):
        self.bot = bot
    
    async def send_message_to_user(self, sender_id: int, receiver_id: int, match_id: int, 
                                 message_type: str, content: str = None):
        async with db.get_session() as session:
            try:
                chat_message = ChatMessage(
                    match_id=match_id,
                    sender_id=sender_id,
                    message_type=message_type,
                    content=content
                )
                session.add(chat_message)
                await session.commit()
                logger.info(f"Message saved: {sender_id} -> {receiver_id}")
            except Exception as e:
                logger.error(f"Error saving message: {e}")
                await session.rollback()

def init_chat_manager(bot):
    global chat_manager
    chat_manager = RealChatManager(bot)

def get_main_menu_keyboard():
    """Simple main menu keyboard"""
    return InlineKeyboardMarkup(inline_keyboard=[
        [InlineKeyboardButton(text="🏠 منوی اصلی", callback_data="main_menu")]
    ])

@router.message(RealChatStates.in_chat)
async def handle_real_text_message(message: Message, state: FSMContext):
    """Handle text messages in real chat"""
    try:
        current_user_id = message.from_user.id
        data = await state.get_data()
        
        match_id = data.get('active_match_id', 1)
        partner_id = data.get('chat_partner_id')
        
        if not partner_id:
            await message.answer("❌ خطا در گفتگو")
            return
            
        # Save message
        if chat_manager:
            await chat_manager.send_message_to_user(
                sender_id=current_user_id,
                receiver_id=partner_id,
                match_id=match_id,
                message_type='text',
                content=message.text
            )
            
        await message.answer("✅ پیام ارسال شد")
        
    except Exception as e:
        logger.error(f"Error in real chat: {e}")
        await message.answer("❌ خطا در ارسال پیام")

@router.callback_query(F.data.startswith("open_chat_"))
async def open_chat(callback: CallbackQuery, state: FSMContext):
    """Open chat with specific user"""
    try:
        target_user_id = int(callback.data.split("_")[2])
        
        # Set chat state
        await state.update_data(
            active_match_id=1,  # Default match ID
            chat_partner_id=target_user_id
        )
        await state.set_state(RealChatStates.in_chat)
        
        # Get user info
        async with db.get_session() as session:
            user_result = await session.execute(
                select(User).where(User.id == target_user_id)
            )
            user = user_result.scalar_one_or_none()
            user_name = user.username or user.first_name or "کاربر" if user else "کاربر"
        
        chat_text = f"""💬 گفتگو با {user_name}

✍️ پیام خود را بنویسید:"""
        
        keyboard = InlineKeyboardMarkup(inline_keyboard=[
            [InlineKeyboardButton(text="🔚 پایان گفتگو", callback_data="end_chat")],
            [InlineKeyboardButton(text="🏠 منوی اصلی", callback_data="main_menu")]
        ])
        
        await callback.message.edit_text(chat_text, reply_markup=keyboard)
        await callback.answer("💬 گفتگو شروع شد")
        
    except Exception as e:
        logger.error(f"Error opening chat: {e}")
        await callback.answer("❌ خطا در باز کردن گفتگو")

@router.callback_query(F.data == "end_chat")
async def end_chat(callback: CallbackQuery, state: FSMContext):
    """End current chat"""
    await state.clear()
    await callback.message.edit_text(
        "🔚 گفتگو پایان یافت",
        reply_markup=get_main_menu_keyboard()
    )
    await callback.answer()

# Compatibility functions
async def start_real_chat(callback: CallbackQuery, state: FSMContext):
    """Compatibility function"""
    await open_chat(callback, state)
