#!/usr/bin/env python3
"""Advanced matching system with filters for ChatMilos Bot"""

from aiogram import Router, F, Bot
from aiogram.types import Message, CallbackQuery, InlineKeyboardMarkup, InlineKeyboardButton, ReplyKeyboardMarkup, KeyboardButton
from aiogram.fsm.context import FSMContext
from aiogram.fsm.state import State, StatesGroup
from sqlalchemy import select, and_, or_, func, not_
from database.database import db
from database.models import User, Match, Preference
from database.simple_models import Swipe, Notification, ProfileView
from handlers.profile_features import increment_profile_view, increment_likes_received
from localization.translator import get_text
from localization.language_helper import get_user_with_language
import logging
import json
from datetime import datetime, timedelta
from typing import Optional, Dict, List

router = Router()
logger = logging.getLogger(__name__)

# ذخیره فیلترها و کاندیدهای کاربران
user_filters: Dict[int, Dict] = {}
user_candidates: Dict[int, List[User]] = {}
current_candidate_index: Dict[int, int] = {}
current_candidate: Dict[int, Optional[User]] = {}

# FSM States for matching flow
class MatchingStates(StatesGroup):
    search_type = State()      # مرحله 1: نوع جستجو
    gender_pref = State()       # مرحله 2: جنسیت
    age_range = State()         # مرحله 3: بازه سنی
    location = State()          # مرحله 4: کشور و شهر
    interests = State()         # مرحله 5: علایق
    photo_filter = State()      # مرحله 6: فیلتر عکس
    viewing = State()           # مرحله 7: مشاهده پروفایل‌ها

class RealMatchingManager:
    def __init__(self, bot: Bot):
        self.bot = bot
    
    async def create_match_notification(self, user_id: int, match_user_id: int):
        """Create match notification in database"""
        async with db.get_session() as session:
            # Get match user info
            match_user_result = await session.execute(
                select(User).where(User.id == match_user_id)
            )
            match_user = match_user_result.scalar_one_or_none()
            
            if match_user:
                notification = Notification(
                    user_id=user_id,
                    type='match',
                    title='🎉 مچ جدید!',
                    message=f'شما با {match_user.username or match_user.first_name or "کاربری"} مچ شدید!',
                    related_user_id=match_user_id
                )
                session.add(notification)
                await session.commit()
    
    async def get_online_status(self, user: User) -> str:
        """Get user online status"""
        if not user.last_seen:
            return "آفلاین"
        
        now = datetime.now()
        diff = now - user.last_seen
        
        if diff.total_seconds() < 300:  # 5 minutes
            return "🟢 آنلاین"
        elif diff.total_seconds() < 3600:  # 1 hour
            minutes = int(diff.total_seconds() / 60)
            return f"⚪ {minutes} دقیقه پیش"
        elif diff.total_seconds() < 86400:  # 24 hours
            hours = int(diff.total_seconds() / 3600)
            return f"⚪ {hours} ساعت پیش"
        else:
            days = int(diff.total_seconds() / 86400)
            return f"⚪ {days} روز پیش"

# Initialize matching manager
matching_manager = None

def set_matching_manager(bot: Bot):
    global matching_manager
    matching_manager = RealMatchingManager(bot)

# Helper functions
async def get_skip_button(text: str = "رد کردن این مرحله ⏭️") -> ReplyKeyboardMarkup:
    """Create skip button keyboard"""
    return ReplyKeyboardMarkup(
        keyboard=[
            [KeyboardButton(text=text)],
            [KeyboardButton(text="🔙 بازگشت به منو")]
        ],
        resize_keyboard=True,
        one_time_keyboard=True
    )

@router.message(F.text.contains("🔍 یافتن مچ"))
async def start_matching_flow(message: Message, state: FSMContext):
    """Start the matching flow with filters"""
    user_id = message.from_user.id
    
    async with db.get_session() as session:
        # Get current user
        current_user_result = await session.execute(
            select(User).where(User.id == user_id)
        )
        current_user = current_user_result.scalar_one_or_none()
        
        if not current_user:
            await message.answer("❌ ابتدا ثبت‌نام کنید.")
            return
    
    # Initialize user filters
    user_filters[user_id] = {
        'search_type': None,
        'gender_pref': None,
        'age_min': 18,
        'age_max': 99,
        'country': None,
        'city': None,
        'interests': [],
        'photo_required': False
    }
    
    # Show welcome message
    welcome_text = (
        "🔍 <b>یافتن مچ</b>\n\n"
        "در این بخش می‌توانید مطابق با علایق و مشخصات خود، مچ مناسب پیدا کنید.\n\n"
        "📝 مراحل ساده‌ای برای تنظیم فیلترها خواهید داشت.\n"
        "⏭️ در هر مرحله می‌توانید گزینه‌ها را رد کنید و بعداً تغییر دهید.\n\n"
        "بیایید شروع کنیم! 🚀"
    )
    
    await message.answer(welcome_text, parse_mode='HTML')
    
    # Start with search type selection
    await show_search_type_selection(message, state)
        
        # Get user preferences
        from database.models import Preference
        pref_result = await session.execute(
            select(Preference).where(Preference.user_id == user_id)
        )
        preferences = pref_result.scalar_one_or_none()
        
        # Set default preferences if not exists
        if not preferences:
            preferences = Preference(
                user_id=user_id,
                age_min=18,
                age_max=50,
                gender_pref=0,  # Any gender
                max_distance=50
            )
            session.add(preferences)
            await session.commit()
        
        # Get only REJECTED users (not liked users)
        rejected_users_result = await session.execute(
            select(Swipe.to_user).where(
                and_(
                    Swipe.from_user == user_id,
                    Swipe.action == -1  # فقط رد شده‌ها
                )
            )
        )
        rejected_user_ids = [row[0] for row in rejected_users_result.fetchall()]
        rejected_user_ids.append(user_id)  # Exclude self
        
        # Build query for potential matches
        query = select(User).where(
            and_(
                User.id.not_in(rejected_user_ids),
                User.is_active == True,
                User.age >= preferences.age_min,
                User.age <= preferences.age_max
            )
        )
        
        # Filter by gender preference
        if preferences.gender_pref != 0:
            query = query.where(User.gender == preferences.gender_pref)
        
        # Execute query
        candidates_result = await session.execute(query.limit(10))
        candidates = candidates_result.scalars().all()
        
        if not candidates:
            await message.answer(
                "😔 <b>کاربری پیدا نشد!</b>\n\n"
                "💡 پیشنهادات:\n"
                "• بازه سنی را گسترده‌تر کنید\n"
                "• ترجیحات جنسیت را تغییر دهید\n"
                "• کمی بعد دوباره تلاش کنید\n\n"
                "⚙️ برای تغییر تنظیمات روی دکمه 'تنظیمات' کلیک کنید.",
                parse_mode='HTML'
            )
            return
        
        # Select random candidate
        import random
        candidate = random.choice(candidates)
        
        # Record profile view
        profile_view = ProfileView(
            viewer_id=user_id,
            viewed_user_id=candidate.user_id
        )
        session.add(profile_view)
        await session.commit()
        
        # Show candidate profile
        age = candidate.age if candidate.age else "نامشخص"
        gender_display = {
            1: 'مرد',
            2: 'زن',
            3: 'غیره',
            '1': 'مرد',
            '2': 'زن',
            '3': 'غیره'
        }.get(candidate.gender, 'نامشخص')
        
        profile_text = f"🎯 <b>کاربر پیشنهادی</b>\n\n"
        profile_text += f"👤 <b>نام:</b> {candidate.username or candidate.first_name or 'کاربر'}\n"
        profile_text += f"🎂 <b>سن:</b> {age} ساله\n"
        profile_text += f"⚧️ <b>جنسیت:</b> {gender_display}\n"
        
        if candidate.city:
            profile_text += f"📍 <b>شهر:</b> {candidate.city}\n"
        
        if candidate.bio:
            profile_text += f"\n💭 <b>بیو:</b>\n<i>{candidate.bio}</i>\n"
        
        if candidate.interests:
            interests_list = candidate.interests.split(',') if isinstance(candidate.interests, str) else []
            if interests_list and any(i.strip() for i in interests_list):
                interests = ", ".join([i.strip() for i in interests_list if i.strip()])
                profile_text += f"\n🎯 <b>علایق:</b> {interests}\n"
        
        profile_text += f"\n💫 این کاربر را چطور می‌بینید?"
        
        # کیبورد کاستوم برای یافتن مچ
        find_match_keyboard = ReplyKeyboardMarkup(
            keyboard=[
                [
                    KeyboardButton(text="💚 لایک"),
                    KeyboardButton(text="⭐ سوپر لایک")
                ],
                [
                    KeyboardButton(text="👎 رد کردن"),
                    KeyboardButton(text="🚫 گزارش")
                ],
                [
                    KeyboardButton(text="🔍 کاربر بعدی"),
                    KeyboardButton(text="👁️ مشاهده دوباره")
                ],
                [
                    KeyboardButton(text="🔙 بازگشت به منو")
                ]
            ],
            resize_keyboard=True,
            one_time_keyboard=False
        )
        
        # بررسی عکس پروفایل
        has_photo = False
        main_photo_id = None
        
        if candidate.photos:
            try:
                photos_data = json.loads(candidate.photos) if isinstance(candidate.photos, str) else candidate.photos
                if photos_data and isinstance(photos_data, list) and len(photos_data) > 0:
                    photo_item = photos_data[0]
                    main_photo_id = photo_item["file_id"] if isinstance(photo_item, dict) and "file_id" in photo_item else (photo_item if isinstance(photo_item, str) else str(photo_item))
                    has_photo = True
            except Exception as e:
                logger.error(f"Error parsing photos: {e}")
        
        # ذخیره candidate_id برای استفاده در دکمه‌های کیبورد سفارشی
        current_user_candidates[user_id] = candidate.user_id
        
        # افزایش شمارنده بازدید پروفایل
        await increment_profile_view(user_id, candidate.user_id)
        
        # ارسال پروفایل با یا بدون عکس + کیبورد سفارشی
        if has_photo and main_photo_id:
            try:
                await message.answer_photo(
                    photo=main_photo_id,
                    caption=profile_text,
                    reply_markup=find_match_keyboard,
                    parse_mode='HTML'
                )
            except Exception as e:
                logger.error(f"Error sending photo {main_photo_id}: {e}")
                await message.answer(profile_text, reply_markup=find_match_keyboard, parse_mode='HTML')
        else:
            await message.answer(profile_text, reply_markup=find_match_keyboard, parse_mode='HTML')

# ==================== هندلرهای کیبورد کاستوم یافتن مچ ====================

@router.message(F.text == "💚 لایک")
async def handle_custom_like(message: Message):
    """هندلر دکمه لایک در کیبورد کاستوم"""
    user_id = message.from_user.id
    
    # بررسی وجود candidate فعلی
    if user_id not in current_user_candidates or not current_user_candidates[user_id]:
        await message.answer("❌ کاربری برای لایک کردن وجود ندارد. ابتدا یک کاربر پیدا کنید.")
        return
    
    target_user_id = current_user_candidates[user_id]
    
    async with db.get_session() as session:
        # بررسی لایک قبلی
        existing_swipe = await session.execute(
            select(Swipe).where(
                and_(
                    Swipe.from_user == user_id,
                    Swipe.to_user == target_user_id
                )
            )
        )
        
        if existing_swipe.scalar_one_or_none():
            await message.answer("❌ شما قبلاً این کاربر را لایک کرده‌اید!")
            return
        
        # ثبت لایک (1=like, 0=pass, 2=super_like)
        swipe = Swipe(
            from_user=user_id,
            to_user=target_user_id,
            action=1,  # like
            created_at=datetime.now()
        )
        session.add(swipe)
        
        # افزایش شمارنده لایک دریافتی برای کاربر هدف
        await increment_likes_received(target_user_id)
        
        # بررسی مچ (آیا طرف مقابل هم لایک کرده؟)
        mutual_swipe = await session.execute(
            select(Swipe).where(
                and_(
                    Swipe.from_user == target_user_id,
                    Swipe.to_user == user_id,
                    Swipe.action.in_([1, 2])  # like or super_like
                )
            )
        )
        
        is_match = mutual_swipe.scalar_one_or_none() is not None
        
        await session.commit()
        
        if is_match:
            await message.answer(
                "🎉 <b>مچ جدید!</b>\n\n"
                "شما و این کاربر همدیگر را لایک کردید!\n"
                "حالا می‌توانید با هم گفتگو کنید.\n\n"
                "💬 برای شروع گفتگو از منوی 'گفتگوهای من' استفاده کنید.",
                parse_mode='HTML'
            )
        else:
            await message.answer(
                "💚 <b>لایک ارسال شد!</b>\n\n"
                "اگر این کاربر هم شما را لایک کند، مچ خواهید شد! 🎉",
                parse_mode='HTML'
            )
        
        # پاک کردن candidate فعلی
        current_user_candidates[user_id] = None

@router.message(F.text.contains("⭐ سوپر لایک"))
async def handle_custom_super_like(message: Message):
    """هندلر دکمه سوپر لایک در کیبورد کاستوم"""
    user_id = message.from_user.id
    
    # بررسی وجود candidate فعلی
    if user_id not in current_user_candidates or not current_user_candidates[user_id]:
        await message.answer("❌ کاربری برای سوپر لایک کردن وجود ندارد. ابتدا یک کاربر پیدا کنید.")
        return
    
    # بررسی پریمیوم بودن کاربر
    async with db.get_session() as session:
        user_result = await session.execute(select(User).where(User.id == user_id))
        user = user_result.scalar_one_or_none()
        
        if not user or not user.is_premium:
            await message.answer(
                "⭐ <b>سوپر لایک</b>\n\n"
                "این قابلیت ویژه کاربران پریمیوم است!\n\n"
                "💎 با خرید پریمیوم:\n"
                "• سوپر لایک نامحدود\n"
                "• اولویت در نمایش پروفایل\n"
                "• مشاهده همه لایک‌ها\n\n"
                "⭐ برای خرید پریمیوم روی دکمه 'پریمیوم' کلیک کنید.",
                parse_mode='HTML'
            )
            return
        
        target_user_id = current_user_candidates[user_id]
        
        # بررسی سوپر لایک قبلی
        existing_swipe = await session.execute(
            select(Swipe).where(
                and_(
                    Swipe.from_user == user_id,
                    Swipe.to_user == target_user_id
                )
            )
        )
        
        if existing_swipe.scalar_one_or_none():
            await message.answer("❌ شما قبلاً این کاربر را لایک کرده‌اید!")
            return
        
        # ثبت سوپر لایک (2=super_like)
        swipe = Swipe(
            from_user=user_id,
            to_user=target_user_id,
            action=2,  # super_like
            created_at=datetime.now()
        )
        session.add(swipe)
        
        # افزایش شمارنده لایک دریافتی برای کاربر هدف
        await increment_likes_received(target_user_id)
        
        # بررسی مچ
        mutual_swipe = await session.execute(
            select(Swipe).where(
                and_(
                    Swipe.from_user == target_user_id,
                    Swipe.to_user == user_id,
                    Swipe.action.in_([1, 2])
                )
            )
        )
        
        is_match = mutual_swipe.scalar_one_or_none() is not None
        
        await session.commit()
        
        if is_match:
            await message.answer(
                "🎉⭐ <b>سوپر مچ!</b>\n\n"
                "شما این کاربر را سوپر لایک کردید و او هم شما را لایک کرده!\n"
                "این یک مچ ویژه است! 🌟\n\n"
                "💬 برای شروع گفتگو از منوی 'گفتگوهای من' استفاده کنید.",
                parse_mode='HTML'
            )
        else:
            await message.answer(
                "⭐ <b>سوپر لایک ارسال شد!</b>\n\n"
                "این کاربر اطلاع خواهد یافت که شما او را سوپر لایک کرده‌اید!\n"
                "احتمال مچ شدن بیشتر است! 🌟",
                parse_mode='HTML'
            )
        
        # پاک کردن candidate فعلی
        current_user_candidates[user_id] = None

@router.message(F.text.contains("👎 رد کردن"))
async def handle_custom_pass(message: Message):
    """هندلر دکمه رد کردن در کیبورد کاستوم"""
    user_id = message.from_user.id
    
    # بررسی وجود candidate فعلی
    if user_id not in current_user_candidates or not current_user_candidates[user_id]:
        await message.answer("❌ کاربری برای رد کردن وجود ندارد. ابتدا یک کاربر پیدا کنید.")
        return
    
    target_user_id = current_user_candidates[user_id]
    
    async with db.get_session() as session:
        # ثبت رد کردن (0=pass)
        swipe = Swipe(
            from_user=user_id,
            to_user=target_user_id,
            action=0,  # pass
            created_at=datetime.now()
        )
        session.add(swipe)
        await session.commit()
        
        await message.answer(
            "👎 <b>رد شد</b>\n\n"
            "این کاربر رد شد و دیگر به شما نشان داده نخواهد شد.\n\n"
            "🔍 برای یافتن کاربر بعدی روی دکمه 'کاربر بعدی' کلیک کنید.",
            parse_mode='HTML'
        )
        
        # پاک کردن candidate فعلی
        current_user_candidates[user_id] = None

@router.message(F.text.contains("🚫 گزارش"))
async def handle_custom_report(message: Message):
    """هندلر دکمه گزارش در کیبورد کاستوم"""
    user_id = message.from_user.id
    
    # بررسی وجود candidate فعلی
    if user_id not in current_user_candidates or not current_user_candidates[user_id]:
        await message.answer("❌ کاربری برای گزارش وجود ندارد. ابتدا یک کاربر پیدا کنید.")
        return
    
    await message.answer(
        "🚫 <b>گزارش کاربر</b>\n\n"
        "گزارش شما ثبت شد و توسط تیم پشتیبانی بررسی خواهد شد.\n\n"
        "⚠️ گزارش‌های کاذب منجر به مسدود شدن حساب شما خواهد شد.\n\n"
        "🔍 برای یافتن کاربر بعدی روی دکمه 'کاربر بعدی' کلیک کنید.",
        parse_mode='HTML'
    )
    
    # پاک کردن candidate فعلی
    current_user_candidates[user_id] = None

@router.message(F.text.contains("🔍 کاربر بعدی"))
async def handle_next_user(message: Message):
    """هندلر دکمه کاربر بعدی"""
    # فراخوانی مجدد تابع یافتن مچ
    await real_find_match(message)

@router.message(F.text.contains("👁️ مشاهده دوباره"))
async def handle_view_again(message: Message):
    """هندلر دکمه مشاهده دوباره"""
    user_id = message.from_user.id
    
    # بررسی وجود candidate فعلی
    if user_id not in current_user_candidates or not current_user_candidates[user_id]:
        await message.answer("❌ کاربری برای مشاهده وجود ندارد. ابتدا یک کاربر پیدا کنید.")
        return
    
    await message.answer(
        "👁️ <b>مشاهده دوباره</b>\n\n"
        "پروفایل کاربر بالا نمایش داده شده است.\n\n"
        "💡 از دکمه‌های زیر برای انتخاب استفاده کنید:\n"
        "• 💚 لایک - اگر کاربر را دوست دارید\n"
        "• ⭐ سوپر لایک - برای علاقه ویژه (پریمیوم)\n"
        "• 👎 رد کردن - اگر علاقه‌ای ندارید\n"
        "• 🔍 کاربر بعدی - برای دیدن کاربر جدید",
        parse_mode='HTML'
    )

@router.message(F.text.contains("🔙 بازگشت به منو"))
async def back_to_main_menu_matching(message: Message):
    """بازگشت به منوی اصلی"""
    from handlers.registration import get_main_menu_keyboard
    from localization.language_helper import get_user_with_language
    from localization.translator import get_text
    import os
    
    user, user_language = await get_user_with_language(message.from_user.id)
    
    if not user:
        await message.answer(f"❌ {get_text(user_language, 'must_register_first', default='ابتدا باید ثبت‌نام کنید.')}")
        return
    
    # بررسی ادمین بودن
    ADMIN_USER_IDS = [int(uid) for uid in os.getenv('ADMIN_USER_IDS', '').split(',') if uid]
    is_admin = message.from_user.id in ADMIN_USER_IDS
    
    # پاک کردن candidate فعلی
    user_id = message.from_user.id
    if user_id in current_user_candidates:
        current_user_candidates[user_id] = None
    
    await message.answer(
        f"🏠 {get_text(user_language, 'back_to_menu', default='بازگشت به منوی اصلی')}",
        reply_markup=get_main_menu_keyboard(user_language, is_admin=is_admin)
    )

# ==================== پایان فایل ====================
# تمام inline keyboardها حذف شدند - فقط کیبورد کاستوم استفاده می‌شود
