"""
Registration Handler - هندلر ثبت‌نام
سناریوی کامل ثبت‌نام با کیبورد کاستوم (بدون inline button)

ترتیب مراحل:
1. انتخاب زبان (متن پیش‌فرض انگلیسی)
2. پیام خوش‌آمدگویی + دکمه ادامه
3. قوانین و حریم خصوصی
4. انتخاب جنسیت
5. دنبال چه کسی هستید
6. نام
7. سن (18-80)
8. روش انتخاب کشور (دستی یا لوکیشن)
9. بیو (اختیاری)
10. علایق (کیبورد کاستوم مثل پروفایل)
11. پایان ثبت‌نام
"""

from aiogram import Router, F, Bot
from aiogram.types import Message, ReplyKeyboardMarkup, KeyboardButton, ReplyKeyboardRemove, InputFile, FSInputFile
from aiogram.fsm.context import FSMContext
from aiogram.fsm.state import State, StatesGroup
from sqlalchemy import select
from database.database import db
from database.models import User, Preference
from localization.translator import get_text
from localization.translations import LANGUAGE_CODES
import os
import json
from utils.matching import matching_engine
import logging
from datetime import datetime

router = Router()
logger = logging.getLogger(__name__)

class RegistrationStates(StatesGroup):
    language = State()
    welcome = State()
    terms = State()
    gender = State()
    looking_for = State()
    name = State()
    age = State()
    country_method = State()
    country = State()
    city = State()
    bio = State()
    interests = State()
    photo_method = State()
    photo_upload = State()
    photo_confirm = State()

# ==================== Helper Functions ====================

def get_language_keyboard():
    """کیبورد انتخاب زبان - 3 زبان در هر سطر"""
    lang_items = list(LANGUAGE_CODES.items())
    keyboard_rows = []
    
    for i in range(0, len(lang_items), 3):
        row = []
        for j in range(3):
            if i + j < len(lang_items):
                lang_code, lang_name = lang_items[i + j]
                row.append(KeyboardButton(text=lang_name))
        keyboard_rows.append(row)
    
    return ReplyKeyboardMarkup(
        keyboard=keyboard_rows,
        resize_keyboard=True,
        one_time_keyboard=True
    )

def get_continue_keyboard(lang: str):
    """کیبورد ادامه"""
    continue_texts = {
        'fa': '▶️ ادامه',
        'en': '▶️ Continue',
        'tr': '▶️ Devam',
        'ar': '▶️ متابعة',
        'ru': '▶️ Продолжить',
        'es': '▶️ Continuar'
    }
    return ReplyKeyboardMarkup(
        keyboard=[
            [KeyboardButton(text=continue_texts.get(lang, '▶️ Continue'))]
        ],
        resize_keyboard=True,
        one_time_keyboard=True
    )

def get_gender_keyboard(lang: str):
    """کیبورد انتخاب جنسیت"""
    return ReplyKeyboardMarkup(
        keyboard=[
            [KeyboardButton(text=get_text(lang, 'gender_male'))],
            [KeyboardButton(text=get_text(lang, 'gender_female'))],
            [KeyboardButton(text=get_text(lang, 'gender_other'))]
        ],
        resize_keyboard=True,
        one_time_keyboard=True
    )

def get_looking_for_keyboard(lang: str):
    """کیبورد انتخاب دنبال چه کسی"""
    return ReplyKeyboardMarkup(
        keyboard=[
            [KeyboardButton(text=get_text(lang, 'looking_for_male'))],
            [KeyboardButton(text=get_text(lang, 'looking_for_female'))],
            [KeyboardButton(text=get_text(lang, 'looking_for_any'))]
        ],
        resize_keyboard=True,
        one_time_keyboard=True
    )

def get_country_method_keyboard(lang: str):
    """کیبورد انتخاب روش وارد کردن کشور"""
    manual_texts = {
        'fa': '✍️ دستی',
        'en': '✍️ Manual',
        'tr': '✍️ Manuel',
        'ar': '✍️ يدوي',
        'ru': '✍️ Вручную',
        'es': '✍️ Manual'
    }
    location_texts = {
        'fa': '📍 لوکیشن',
        'en': '📍 Location',
        'tr': '📍 Konum',
        'ar': '📍 الموقع',
        'ru': '📍 Местоположение',
        'es': '📍 Ubicación'
    }
    return ReplyKeyboardMarkup(
        keyboard=[
            [KeyboardButton(text=manual_texts.get(lang, '✍️ Manual'))],
            [KeyboardButton(text=location_texts.get(lang, '📍 Location'), request_location=True)]
        ],
        resize_keyboard=True,
        one_time_keyboard=True
    )

def get_skip_keyboard(lang: str):
    """کیبورد رد کردن"""
    return ReplyKeyboardMarkup(
        keyboard=[
            [KeyboardButton(text=get_text(lang, 'skip'))]
        ],
        resize_keyboard=True,
        one_time_keyboard=True
    )

def get_photo_method_keyboard(lang: str):
    """کیبورد انتخاب روش آپلود عکس"""
    photo_methods = {
        'fa': ['📱 استفاده از عکس پروفایل تلگرام', '📤 آپلود عکس جدید', '⏭️ رد کردن'],
        'en': ['📱 Use Telegram Profile Photo', '📤 Upload New Photo', '⏭️ Skip'],
        'tr': ['📱 Telegram Profil Fotoğrafını Kullan', '📤 Yeni Fotoğraf Yükle', '⏭️ Atla'],
        'ar': ['📱 استخدام صورة الملف الشخصي', '📤 تحميل صورة جديدة', '⏭️ تخطي'],
        'ru': ['📱 Использовать фото профиля', '📤 Загрузить новое фото', '⏭️ Пропустить'],
        'es': ['📱 Usar foto de perfil', '📤 Subir nueva foto', '⏭️ Omitir']
    }
    
    buttons = photo_methods.get(lang, photo_methods['en'])
    
    return ReplyKeyboardMarkup(
        keyboard=[
            [KeyboardButton(text=buttons[0])],
            [KeyboardButton(text=buttons[1])],
            [KeyboardButton(text=buttons[2])]
        ],
        resize_keyboard=True,
        one_time_keyboard=True
    )

def get_photo_confirm_keyboard(lang: str):
    """کیبورد تایید/حذف عکس"""
    confirm_texts = {
        'fa': ['✅ تایید', '❌ حذف'],
        'en': ['✅ Confirm', '❌ Delete'],
        'tr': ['✅ Onayla', '❌ Sil'],
        'ar': ['✅ تأكيد', '❌ حذف'],
        'ru': ['✅ Подтвердить', '❌ Удалить'],
        'es': ['✅ Confirmar', '❌ Eliminar']
    }
    
    buttons = confirm_texts.get(lang, confirm_texts['en'])
    
    return ReplyKeyboardMarkup(
        keyboard=[
            [KeyboardButton(text=buttons[0])],
            [KeyboardButton(text=buttons[1])]
        ],
        resize_keyboard=True,
        one_time_keyboard=True
    )

def get_default_photo_path(gender: int):
    """دریافت مسیر عکس پیش‌فرض بر اساس جنسیت"""
    # 1 = مرد، 2 = زن، 3 = غیره
    if gender == 1:
        return "assets/default_male.jpg"
    elif gender == 2:
        return "assets/default_female.jpg"
    else:
        return "assets/default_other.jpg"

def get_interests_keyboard(lang: str, selected_interests: list = None):
    """کیبورد انتخاب علایق - مشابه پروفایل"""
    if selected_interests is None:
        selected_interests = []
    
    interests_dict = get_text(lang, 'interests_list', default={})
    
    if not interests_dict:
        interests_dict = {
            'sports': 'ورزش', 'music': 'موسیقی', 'movies': 'فیلم', 'books': 'کتاب',
            'travel': 'سفر', 'cooking': 'آشپزی', 'art': 'هنر', 'photography': 'عکاسی',
            'gaming': 'بازی', 'fitness': 'تناسب اندام', 'nature': 'طبیعت', 'technology': 'فناوری',
            'fashion': 'مد', 'dancing': 'رقص', 'reading': 'مطالعه', 'writing': 'نویسندگی'
        }
    
    keyboard = []
    row = []
    
    for key, value in list(interests_dict.items())[:30]:  # حداکثر 30 علاقه
        if key in selected_interests:
            button_text = f"✅ {value}"
        else:
            button_text = value
        
        row.append(KeyboardButton(text=button_text))
        
        if len(row) == 3:
            keyboard.append(row)
            row = []
    
    if row:
        keyboard.append(row)
    
    # دکمه تایید و رد کردن
    if selected_interests:
        confirm_text = get_text(lang, 'confirm_interests', default='تایید علایق')
        keyboard.append([
            KeyboardButton(text=f"✅ {confirm_text} ({len(selected_interests)})")
        ])
    
    keyboard.append([
        KeyboardButton(text=get_text(lang, 'skip'))
    ])
    
    return ReplyKeyboardMarkup(keyboard=keyboard, resize_keyboard=True)

def get_main_menu_keyboard(lang: str, is_admin: bool = False):
    """منوی اصلی"""
    if is_admin:
        return ReplyKeyboardMarkup(
            keyboard=[
                [KeyboardButton(text="🔐 پنل ادمین"), KeyboardButton(text="📊 آمار سریع")],
                [KeyboardButton(text="👥 مدیریت کاربران"), KeyboardButton(text="🚨 گزارشات")],
                [KeyboardButton(text="📢 پیام همگانی"), KeyboardButton(text="🔧 تنظیمات سیستم")],
                [KeyboardButton(text="🧪 حالت تست"), KeyboardButton(text="❓ راهنما")]
            ],
            resize_keyboard=True
        )
    else:
        return ReplyKeyboardMarkup(
            keyboard=[
                [KeyboardButton(text=f"🔍 {get_text(lang, 'find_match', default='یافتن مچ')}"), KeyboardButton(text=get_text(lang, 'profile'))],
                [KeyboardButton(text=f"💬 {get_text(lang, 'my_chats', default='گفتگوهای من')}"), KeyboardButton(text=f"💚 {get_text(lang, 'my_likes', default='لایک‌های من')}")],
                [KeyboardButton(text=f"🌟 {get_text(lang, 'discovery', default='اکتشاف')}"), KeyboardButton(text=f"🔔 {get_text(lang, 'notifications', default='نوتیفیکیشن‌ها')}")],
                [KeyboardButton(text=f"⭐ {get_text(lang, 'premium', default='پریمیوم')}"), KeyboardButton(text=get_text(lang, 'settings'))],
                [KeyboardButton(text=f"❓ {get_text(lang, 'help', default='راهنما')}")]
            ],
            resize_keyboard=True
        )

# ==================== Registration Flow ====================

@router.message(F.text == "/start")
async def start_command(message: Message, state: FSMContext):
    """مرحله 1: شروع ثبت‌نام یا ورود به منوی اصلی"""
    user_id = message.from_user.id
    
    # بررسی ادمین بودن
    ADMIN_USER_IDS = [int(uid) for uid in os.getenv('ADMIN_USER_IDS', '').split(',') if uid]
    is_admin = user_id in ADMIN_USER_IDS
    
    # بررسی وجود کاربر
    async with db.get_session() as session:
        result = await session.execute(select(User).where(User.id == user_id))
        existing_user = result.scalar_one_or_none()
        
        if existing_user:
            # کاربر وجود دارد، نمایش منوی اصلی
            lang = existing_user.language_code or 'fa'
            
            if is_admin:
                await message.answer(
                    "🔐 خوش آمدید ادمین!\nاز منوی ادمین استفاده کنید.",
                    reply_markup=get_main_menu_keyboard(lang, is_admin=True)
                )
            else:
                await message.answer(
                    get_text(lang, 'welcome'),
                    reply_markup=get_main_menu_keyboard(lang, is_admin=False)
                )
            return
    
    # کاربر جدید - شروع ثبت‌نام
    # مرحله 1: انتخاب زبان (متن پیش‌فرض انگلیسی)
    await message.answer(
        "🌍 <b>Choose Language</b>\n\n"
        "Please select your preferred language:",
        reply_markup=get_language_keyboard(),
        parse_mode='HTML'
    )
    await state.set_state(RegistrationStates.language)

@router.message(RegistrationStates.language)
async def language_selected(message: Message, state: FSMContext):
    """مرحله 2: انتخاب زبان"""
    # پیدا کردن کد زبان از نام
    selected_lang = None
    for code, name in LANGUAGE_CODES.items():
        if name == message.text:
            selected_lang = code
            break
    
    if not selected_lang:
        await message.answer("Please select one of the available languages.")
        return
    
    await state.update_data(language=selected_lang)
    
    # مرحله 3: پیام خوش‌آمدگویی با دکمه ادامه
    welcome_messages = {
        'fa': '👋 به ChatMilos خوش آمدید!\n\n💫 جایی که افراد واقعی با هم آشنا می‌شوند\n\n🌟 آماده‌اید تا دنیای جدیدی از آشنایی‌ها را تجربه کنید؟',
        'en': '👋 Welcome to ChatMilos!\n\n💫 Where real people meet\n\n🌟 Ready to experience a new world of connections?',
        'tr': '👋 ChatMilos\'a hoş geldiniz!\n\n💫 Gerçek insanların buluştuğu yer\n\n🌟 Yeni bir tanışma dünyasını deneyimlemeye hazır mısınız?',
        'ar': '👋 مرحباً بك في ChatMilos!\n\n💫 حيث يلتقي الأشخاص الحقيقيون\n\n🌟 هل أنت مستعد لتجربة عالم جديد من الاتصالات؟',
        'ru': '👋 Добро пожаловать в ChatMilos!\n\n💫 Где встречаются настоящие люди\n\n🌟 Готовы испытать новый мир знакомств?',
        'es': '👋 ¡Bienvenido a ChatMilos!\n\n💫 Donde las personas reales se encuentran\n\n🌟 ¿Listo para experimentar un nuevo mundo de conexiones?',
    }
    
    welcome_text = welcome_messages.get(selected_lang, welcome_messages['en'])
    
    await message.answer(
        welcome_text,
        reply_markup=get_continue_keyboard(selected_lang)
    )
    await state.set_state(RegistrationStates.welcome)

@router.message(RegistrationStates.welcome)
async def welcome_continue(message: Message, state: FSMContext):
    """مرحله 4: دکمه ادامه و نمایش قوانین"""
    data = await state.get_data()
    lang = data.get('language', 'fa')
    
    # نمایش قوانین و حریم خصوصی
    terms_keyboard = ReplyKeyboardMarkup(
        keyboard=[[KeyboardButton(text=get_text(lang, 'accept_terms'))]],
        resize_keyboard=True,
        one_time_keyboard=True
    )
    
    await message.answer(
        get_text(lang, 'terms_text'),
        reply_markup=terms_keyboard,
        parse_mode='HTML'
    )
    await state.set_state(RegistrationStates.terms)

@router.message(RegistrationStates.terms)
async def terms_accepted(message: Message, state: FSMContext):
    """مرحله 5: قبول قوانین"""
    data = await state.get_data()
    lang = data.get('language', 'fa')
    
    accept_term = get_text(lang, 'accept_terms')
    if "✅" not in message.text and accept_term not in message.text:
        await message.answer(f"لطفاً روی دکمه '{accept_term}' کلیک کنید.")
        return
    
    # نمایش انتخاب جنسیت
    await message.answer(
        get_text(lang, 'select_gender'),
        reply_markup=get_gender_keyboard(lang)
    )
    await state.set_state(RegistrationStates.gender)

@router.message(RegistrationStates.gender)
async def gender_selected(message: Message, state: FSMContext):
    """مرحله 6: انتخاب جنسیت"""
    data = await state.get_data()
    lang = data.get('language', 'fa')
    
    # تشخیص جنسیت از متن
    gender_male = get_text(lang, 'gender_male')
    gender_female = get_text(lang, 'gender_female')
    gender_other = get_text(lang, 'gender_other')
    
    if gender_male in message.text or "مرد" in message.text or "Male" in message.text or "Erkek" in message.text:
        gender = 1
    elif gender_female in message.text or "زن" in message.text or "Female" in message.text or "Kadın" in message.text:
        gender = 2
    elif gender_other in message.text or "غیره" in message.text or "Other" in message.text or "Diğer" in message.text:
        gender = 3
    else:
        await message.answer(get_text(lang, 'select_gender'), reply_markup=get_gender_keyboard(lang))
        return
    
    await state.update_data(gender=gender)
    
    # نمایش انتخاب دنبال چه کسی
    await message.answer(
        get_text(lang, 'select_looking_for'),
        reply_markup=get_looking_for_keyboard(lang)
    )
    await state.set_state(RegistrationStates.looking_for)

@router.message(RegistrationStates.looking_for)
async def looking_for_selected(message: Message, state: FSMContext):
    """مرحله 7: انتخاب دنبال چه کسی"""
    data = await state.get_data()
    lang = data.get('language', 'fa')
    
    # تشخیص ترجیح از متن
    looking_male = get_text(lang, 'looking_for_male')
    looking_female = get_text(lang, 'looking_for_female')
    looking_any = get_text(lang, 'looking_for_any')
    
    if looking_male in message.text or "مرد" in message.text or "Male" in message.text:
        looking_for = 1
    elif looking_female in message.text or "زن" in message.text or "Female" in message.text:
        looking_for = 2
    elif looking_any in message.text or "فرقی" in message.text or "Anyone" in message.text or "Any" in message.text:
        looking_for = 0
    else:
        await message.answer(get_text(lang, 'select_looking_for'), reply_markup=get_looking_for_keyboard(lang))
        return
    
    await state.update_data(looking_for=looking_for)
    
    # درخواست نام
    await message.answer(
        get_text(lang, 'enter_name'),
        reply_markup=ReplyKeyboardRemove()
    )
    await state.set_state(RegistrationStates.name)

@router.message(RegistrationStates.name)
async def name_entered(message: Message, state: FSMContext):
    """مرحله 8: وارد کردن نام"""
    data = await state.get_data()
    lang = data.get('language', 'fa')
    
    name = message.text.strip()
    if len(name) < 2:
        await message.answer("نام باید حداقل 2 کاراکتر باشد.")
        return
    
    await state.update_data(name=name)
    
    # درخواست سن
    await message.answer(get_text(lang, 'enter_age'))
    await state.set_state(RegistrationStates.age)

@router.message(RegistrationStates.age)
async def age_entered(message: Message, state: FSMContext):
    """مرحله 9: وارد کردن سن"""
    data = await state.get_data()
    lang = data.get('language', 'fa')
    
    try:
        age = int(message.text)
        if age < 18 or age > 80:
            await message.answer(get_text(lang, 'age_invalid'))
            return
        
        await state.update_data(age=age)
        
        # روش انتخاب کشور
        country_method_texts = {
            'fa': 'روش انتخاب کشور خود را مشخص کنید:',
            'en': 'Choose how to select your country:',
            'tr': 'Ülkenizi seçme yöntemini belirleyin:',
            'ar': 'حدد طريقة اختيار بلدك:',
            'ru': 'Выберите способ выбора вашей страны:',
            'es': 'Elige cómo seleccionar tu país:'
        }
        
        await message.answer(
            country_method_texts.get(lang, 'Choose how to select your country:'),
            reply_markup=get_country_method_keyboard(lang)
        )
        await state.set_state(RegistrationStates.country_method)
        
    except ValueError:
        await message.answer("لطفاً سن را به صورت عدد وارد کنید (مثال: 25)")

@router.message(RegistrationStates.country_method, F.location)
async def location_received_for_country(message: Message, state: FSMContext):
    """مرحله 10: دریافت لوکیشن برای کشور و شهر"""
    data = await state.get_data()
    lang = data.get('language', 'fa')
    
    lat = message.location.latitude
    lon = message.location.longitude
    
    # Encode to geohash
    geohash = matching_engine.encode_geohash(lat, lon, precision=6)
    
    await state.update_data(
        latitude=lat,
        longitude=lon,
        geohash=geohash,
        country="Unknown",  # Will be geocoded later
        city="Unknown"
    )
    
    # درخواست بیو
    await message.answer(
        get_text(lang, 'enter_bio'),
        reply_markup=get_skip_keyboard(lang)
    )
    await state.set_state(RegistrationStates.bio)

@router.message(RegistrationStates.country_method)
async def country_method_selected(message: Message, state: FSMContext):
    """مرحله 10: انتخاب روش دستی"""
    data = await state.get_data()
    lang = data.get('language', 'fa')
    
    if "دستی" in message.text or "Manual" in message.text or "Manuel" in message.text or "✍️" in message.text:
        # وارد کردن دستی کشور
        country_prompts = {
            'fa': 'نام کشور خود را وارد کنید:',
            'en': 'Enter your country name:',
            'tr': 'Ülke adınızı girin:',
            'ar': 'أدخل اسم بلدك:',
            'ru': 'Введите название вашей страны:',
            'es': 'Ingresa el nombre de tu país:'
        }
        
        await message.answer(
            country_prompts.get(lang, 'Enter your country name:'),
            reply_markup=ReplyKeyboardRemove()
        )
        await state.set_state(RegistrationStates.country)
    else:
        await message.answer(
            "لطفاً یکی از گزینه‌ها را انتخاب کنید.",
            reply_markup=get_country_method_keyboard(lang)
        )

@router.message(RegistrationStates.country)
async def country_entered(message: Message, state: FSMContext):
    """مرحله 11: وارد کردن کشور (دستی)"""
    data = await state.get_data()
    lang = data.get('language', 'fa')
    
    country = message.text.strip()
    
    # Map common country names to codes
    country_codes = {
        'ایران': 'IR', 'iran': 'IR',
        'ترکیه': 'TR', 'turkey': 'TR', 'türkiye': 'TR',
        'عربستان': 'SA', 'saudi arabia': 'SA',
        'امارات': 'AE', 'uae': 'AE',
        'آلمان': 'DE', 'germany': 'DE',
        'آمریکا': 'US', 'usa': 'US', 'united states': 'US'
    }
    
    country_code = country_codes.get(country.lower(), 'XX')
    
    await state.update_data(country=country, country_code=country_code)
    
    # درخواست شهر
    city_prompts = {
        'fa': 'نام شهر خود را وارد کنید:',
        'en': 'Enter your city name:',
        'tr': 'Şehir adınızı girin:',
        'ar': 'أدخل اسم مدينتك:',
        'ru': 'Введите название вашего города:',
        'es': 'Ingresa el nombre de tu ciudad:'
    }
    
    await message.answer(
        city_prompts.get(lang, 'Enter your city name:'),
        reply_markup=ReplyKeyboardRemove()
    )
    await state.set_state(RegistrationStates.city)

@router.message(RegistrationStates.city)
async def city_entered(message: Message, state: FSMContext):
    """مرحله 12: وارد کردن شهر"""
    data = await state.get_data()
    lang = data.get('language', 'fa')
    
    city = message.text.strip()
    await state.update_data(city=city)
    
    # درخواست بیو
    await message.answer(
        get_text(lang, 'enter_bio'),
        reply_markup=get_skip_keyboard(lang)
    )
    await state.set_state(RegistrationStates.bio)

@router.message(RegistrationStates.bio)
async def bio_entered(message: Message, state: FSMContext):
    """مرحله 13: وارد کردن بیو"""
    data = await state.get_data()
    lang = data.get('language', 'fa')
    
    skip_text = get_text(lang, 'skip')
    
    if skip_text in message.text or "رد" in message.text or "Skip" in message.text:
        bio = None
    else:
        bio = message.text.strip()
        if len(bio) > 200:
            await message.answer(get_text(lang, 'bio_too_long'))
            return
    
    await state.update_data(bio=bio)
    
    # درخواست علایق با کیبورد کاستوم
    await message.answer(
        get_text(lang, 'select_interests'),
        reply_markup=get_interests_keyboard(lang, [])
    )
    await state.set_state(RegistrationStates.interests)

@router.message(RegistrationStates.interests)
async def interests_selected(message: Message, state: FSMContext):
    """مرحله 14: انتخاب علایق"""
    data = await state.get_data()
    lang = data.get('language', 'fa')
    selected_interests = data.get('selected_interests', [])
    
    skip_text = get_text(lang, 'skip')
    confirm_text = get_text(lang, 'confirm_interests', default='تایید علایق')
    
    # بررسی رد کردن یا تایید
    if skip_text in message.text or "رد" in message.text or "Skip" in message.text:
        # رد کردن علایق
        await complete_registration(message, state, [])
        return
    elif "✅" in message.text and (confirm_text in message.text or "تایید" in message.text):
        # تایید علایق
        await complete_registration(message, state, selected_interests)
        return
    
    # انتخاب/لغو انتخاب علاقه
    interests_dict = get_text(lang, 'interests_list', default={})
    
    if not interests_dict:
        interests_dict = {
            'sports': 'ورزش', 'music': 'موسیقی', 'movies': 'فیلم', 'books': 'کتاب',
            'travel': 'سفر', 'cooking': 'آشپزی', 'art': 'هنر', 'photography': 'عکاسی',
            'gaming': 'بازی', 'fitness': 'تناسب اندام', 'nature': 'طبیعت', 'technology': 'فناوری',
            'fashion': 'مد', 'dancing': 'رقص', 'reading': 'مطالعه', 'writing': 'نویسندگی'
        }
    
    # پیدا کردن علاقه انتخاب شده
    selected_key = None
    for key, value in interests_dict.items():
        if value in message.text.replace('✅ ', ''):
            selected_key = key
            break
    
    if selected_key:
        if selected_key in selected_interests:
            selected_interests.remove(selected_key)
        else:
            if len(selected_interests) < 10:
                selected_interests.append(selected_key)
            else:
                await message.answer("حداکثر 10 علاقه می‌توانید انتخاب کنید.")
                return
        
        await state.update_data(selected_interests=selected_interests)
        
        # به‌روزرسانی کیبورد
        await message.answer(
            get_text(lang, 'select_interests'),
            reply_markup=get_interests_keyboard(lang, selected_interests)
        )

async def complete_registration(message: Message, state: FSMContext, interests: list):
    """تکمیل ثبت‌نام و ذخیره در دیتابیس"""
    data = await state.get_data()
    lang = data.get('language', 'fa')
    
    # ذخیره کاربر در دیتابیس
    async with db.get_session() as session:
        try:
            interests_str = ','.join(interests) if interests else ''
            
            # ایجاد کاربر
            user = User(
                user_id=message.from_user.id,
                username=message.from_user.username or f"user_{message.from_user.id}",
                first_name=data.get('name'),
                last_name=message.from_user.last_name,
                gender=str(data.get('gender')),
                age=data.get('age'),
                language_code=lang,
                country=data.get('country_code', data.get('country', '')),
                city=data.get('city', ''),
                location_lat=str(data.get('latitude', 0)) if data.get('latitude') else None,
                location_lon=str(data.get('longitude', 0)) if data.get('longitude') else None,
                looking_for=str(data.get('looking_for', 0)),
                bio=data.get('bio', ''),
                interests=interests_str
            )
            
            session.add(user)
            await session.flush()
            
            # ایجاد تنظیمات جستجو
            preference = Preference(
                user_id=message.from_user.id,
                age_min=18,
                age_max=45,
                gender_pref=data.get('looking_for', 0),
                radius_km=50,
                country_code=data.get('country_code', ''),
                city=data.get('city', '')
            )
            
            session.add(preference)
            await session.commit()
            
            # بررسی ادمین بودن
            ADMIN_USER_IDS = [int(uid) for uid in os.getenv('ADMIN_USER_IDS', '').split(',') if uid]
            is_admin = message.from_user.id in ADMIN_USER_IDS
            
            # پایان ثبت‌نام
            completion_messages = {
                'fa': '🎉 <b>پروفایل شما تکمیل شد!</b>',
                'en': '🎉 <b>Your profile is complete!</b>',
                'tr': '🎉 <b>Profiliniz tamamlandı!</b>',
                'ar': '🎉 <b>اكتمل ملفك الشخصي!</b>',
                'ru': '🎉 <b>Ваш профиль завершен!</b>',
                'es': '🎉 <b>¡Tu perfil está completo!</b>'
            }
            
            await message.answer(
                completion_messages.get(lang, completion_messages['en']),
                parse_mode='HTML'
            )
            
            if is_admin:
                await message.answer(
                    "🔐 شما به عنوان ادمین شناسایی شدید.",
                    reply_markup=get_main_menu_keyboard(lang, is_admin=True)
                )
            else:
                await message.answer(
                    "✨",
                    reply_markup=get_main_menu_keyboard(lang, is_admin=False)
                )
            
            await state.clear()
            logger.info(f"User {message.from_user.id} registered successfully")
            
        except Exception as e:
            import traceback
            error_details = traceback.format_exc()
            logger.error(f"Registration error for user {message.from_user.id}: {e}\n{error_details}")
            
            await message.answer(
                f"❌ خطا در ثبت‌نام: {str(e)[:100]}\n\nلطفاً دوباره تلاش کنید: /start"
            )
            await state.clear()
